# Copyright (c) 2010 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

#-------------------------------------------------------------------------------
# This module implements ipv4 and ipv6 routing commands
#-------------------------------------------------------------------------------
'''Routing configuration commands support for IPv4 and IPv6'''

import Tac
import ConfigMount
import Tracing
import struct
from socket import socket, AF_UNIX, SOCK_DGRAM
import QuickTrace

qv = QuickTrace.Var
qt0 = QuickTrace.trace0

t0 = Tracing.trace0

# wrapper around quicktracing an exception
def qtraceException( e ):
   # split the ouput into multiple messages
   # if it overflows the quicktrace string size
   def _qtfull( output ):
      qtrace_string_size = QuickTrace.qtrace_string_size
      n = len( output )
      start = 0
      end = start + qtrace_string_size
      while start < n:
         qt0( qv( output[ start : end ] ) )
         start = end
         end += qtrace_string_size
         end = min( end, n )

   qt0( type( e ).__name__ )        # exception class name
   qt0( "error:", qv( e.error ) )   # exception error
   _qtfull( str( e ) )              # exception message

   # The error message could span multiple lines
   # QuickTrace does not handle it, so break at newlines
   #
   # For example, this is printed when the command fails ...
   #
   # sudo -E ip -4 neigh flush to 0.0.0.0/0
   # Failed to send flush request: No such device
   # Flush terminated
   for s in e.output.split( '\n' ): # exception output
      _qtfull( s )

def ipDottedToLong( ipAddress ):
   ipaddr = ipAddress.split( '.' )
   binaryipOuter = int( ipaddr[ 0 ] )
   for i in range( 1, 4 ):
      binaryipOuter = binaryipOuter << 8 | int( ipaddr[ i ] )
   return binaryipOuter

class Arp:
   def __init__ ( self ):
      self.arpInputConfig = None

   def plugin( self, entityManager ):
      self.arpInputConfig = ConfigMount.mount( entityManager,
                                               "arp/input/config/cli",
                                               "Arp::InputConfig",
                                               "wS" )

   def arpInputConfigVrf( self, vrfName ):
      assert vrfName and vrfName != ''
      return self.arpInputConfig.vrf[ vrfName ]

   def cliSource( self ):
      return self.arpInputConfig.source

def sendArpRefreshReqMsg( mode, af, vrfName, intfName, ipAddrStr ):

   # Build the message using host byte order, standard size, no alignment.
   # This is indicated by the '=' character in struct.pack's format argument.

   # Address family field is 1 byte
   if af == Tac.Type( 'Arnet::AddressFamily' ).ipv4:
      addressFamily = 1 # Value of Arnet::AddressFamily::ipv4_
   else:
      addressFamily = 2 # Value of Arnet::AddressFamily::ipv6_
   addressFamilyField = struct.pack( '=B', addressFamily )

   # Interface ID field is 8 bytes
   if not intfName:
      intfId = Tac.Type( 'Arnet::IntfId' ).emptyIntfId
   else:
      intfId = Tac.Value( 'Arnet::IntfId', intfName ).intfId
   intfIdField = struct.pack( '=Q', intfId )

   # IP address field is 16 bytes (array of 4 words)
   if not ipAddrStr:
      ipAddrField = struct.pack( '=IIII', 0, 0, 0, 0 )
   else:
      if af == Tac.Type( 'Arnet::AddressFamily' ).ipv4:
         ipv4Addr = Tac.Value( 'Arnet::IpAddr', stringValue=ipAddrStr )
         ipAddrField = struct.pack( '=IIII', ipv4Addr.value, 0, 0, 0 )
      else:
         ipv6Addr = Tac.Value( 'Arnet::Ip6Addr', stringValue=ipAddrStr )
         ipAddrField = struct.pack( '=IIII', ipv6Addr.word0, ipv6Addr.word1,
                                             ipv6Addr.word2, ipv6Addr.word3 )

   # VRF name field is a null-terminated string
   vrfNameField = ( vrfName + '\0' ).encode()

   msg = addressFamilyField + intfIdField + ipAddrField + vrfNameField

   clientSocket = socket( AF_UNIX, SOCK_DGRAM )

   # Prepend a null byte to socket name since it's an abstract socket
   serverSocket = b'\0AristaArpRefreshRequestSocket'
   errorCode = clientSocket.connect_ex( serverSocket )
   if errorCode:
      mode.addError( 'ARP refresh service is not available. '
                     'Make sure the Arp Agent is running.' )
      clientSocket.close()
      return

   clientSocket.send( msg )

   clientSocket.close()

minArpTimeout = Tac.enumValue( 'Arp::ArpTimeoutConsts', 'minArpTimeout' )
maxArpTimeout = Tac.enumValue( 'Arp::ArpTimeoutConsts', 'maxArpTimeout' )
defaultEthernetTimeout = Tac.Type( 'Arp::ArpIntfConfig' ).defaultEthernetTimeout
arpCacheCapacity = Tac.Value( 'Arp::ArpCacheCapacity', )
