#!/usr/bin/env python3
# Copyright (c) 2023 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import BasicCli
import CliCommand
import CliParser
import ShowCommand
from CliPlugin import EthIntfCli
from CliPlugin import ArpModel
from CliPlugin.VirtualIntfRule import IntfMatcher
from CliPlugin.SubIntfCli import subMatcher
import LazyMount
import Tac
import SmashLazyMount

from collections import defaultdict

arpCounterDir = None
arpHwStatus = None

intfMatcher = IntfMatcher()
intfMatcher |= subMatcher
intfMatcher |= EthIntfCli.EthPhyIntf.ethMatcher

def arpCountSupport( mode, token ):
   if not arpHwStatus.arpCountersSupported:
      return CliParser.guardNotThisPlatform
   return None

class ShowArpCounters( ShowCommand.ShowCliCommandClass ):
   syntax = "show arp counters hardware ingress [ interface INTF ]"
   data = {
      'arp': 'ARP',
      'counters': CliCommand.guardedKeyword( 'counters',
                                              helpdesc='counters',
                                              guard=arpCountSupport ),
      'hardware': 'Show hardware counters',
      'ingress': 'Show ingress packet counters',
      'interface': 'Interface selector',
      'INTF': intfMatcher,
      }
   cliModel = ArpModel.ArpCounters

   @staticmethod
   def handler( mode, args ):
      intf = args.get( 'INTF' )
      intfCounters = defaultdict( ArpModel.ArpCounter )

      if intf:
         intfId = Tac.Value( 'Arnet::IntfId', str( intf ) )
         reqEntry = Tac.Value( 'Arp::ArpCounterKey', intf=intfId, opcode=1 )
         if reqEntry in arpCounterDir.arpCounter:
            counter = arpCounterDir.arpCounter[ reqEntry ]
            intfCounters[ str( intf ) ].requests = counter.pkts
         repEntry = Tac.Value( 'Arp::ArpCounterKey', intf=intfId, opcode=0 )
         if repEntry in arpCounterDir.arpCounter:
            counter = arpCounterDir.arpCounter[ repEntry ]
            intfCounters[ str( intf ) ].replies = counter.pkts
         return ArpModel.ArpCounters( interfaces=dict( intfCounters ) )

      for key in ( arpCounterDir.arpCounter ):
         intfName = str( key.intf )
         counter = arpCounterDir.arpCounter[ key ]
         if key.opcode == 1:
            intfCounters[ intfName ].requests = counter.pkts
         else:
            intfCounters[ intfName ].replies = counter.pkts
      return ArpModel.ArpCounters( interfaces=dict( intfCounters ) )

BasicCli.addShowCommandClass( ShowArpCounters )

# -------------------------------------------------------------------------------
# Have the Cli Agent mount all needed state from sysdb
# -------------------------------------------------------------------------------
def Plugin( entityManager ):
   global arpCounterDir
   global arpHwStatus

   arpCounterDir = SmashLazyMount.mount( entityManager, "arp/rxcounter",
                                       "Arp::ArpCounterDir",
                                       SmashLazyMount.mountInfo( 'reader' ) )
   arpHwStatus = LazyMount.mount( entityManager, "arp/hardware/status",
                                  "Arp::Hardware::Status", "r" )
