# Copyright (c) 2005-2010, 2011 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

#-------------------------------------------------------------------------------
# This module implements non-interface-specific IP configuration.
#-------------------------------------------------------------------------------
"""Configuration commands supported for IP"""

import Arnet
import ArpLogMsgs
import CliParser
from CliPlugin import ArpCommon
from CliPlugin.ArpCommon import defaultEthernetTimeout
from CliPlugin.ArpCommon import qtraceException
from CliPlugin.ArpCommon import sendArpRefreshReqMsg
from CliPlugin import VrfCli
from CliPlugin.VrfCli import getAllVrfNames, vrfExists, ALL_VRF_NAME, DEFAULT_VRF
import LazyMount
import Logging
import Tac
import UtmpDump
import Cell
from ArpLibTypes import (
   tacInputConfigKey,
   )

arpConfig = None
ipStatus = None
allVrfStatusLocal = None
arpHwStatus = None
noArpTableForVrfMsg = "ARP table for VRF %s does not exist."
ipv4Af = Tac.Type( 'Arnet::AddressFamily' ).ipv4
arp = ArpCommon.Arp()

def addStaticArp( mode, args ):
   ipAddr = args[ 'IPADDR' ]
   macAddr = args[ 'MACADDR' ]
   vrfName = args.get( 'VRF', DEFAULT_VRF )

   if not vrfExists( vrfName ):
      mode.addError( noArpTableForVrfMsg % vrfName )
      return
   configKey = tacInputConfigKey( ipAddr, '' )
   arp.arpInputConfigVrf( vrfName ).afEntryPermanentIs( configKey,
                                                        macAddr,
                                                        arp.cliSource() )

def delStaticArp( mode, args ):
   ipAddr = args[ 'IPADDR' ]
   vrfName = args.get( 'VRF', DEFAULT_VRF )
   # XXX The 'no arp <ip-address>' command deletes all ARP entries for the given IP
   # address.  It's not possible to remove an ARP entry just for one interface.  This
   # is pretty lame, but it's the industry-standard.
   configKey = tacInputConfigKey( ipAddr, '' )
   try:
      del arp.arpInputConfigVrf( vrfName ).ipv4[ configKey ]
   except KeyError: # VRF DNE.
      pass

def enableMonitorMac( mode, args ):
   arpConfig.globalMonitorMac = True

def disableMonitorMac( mode, args ):
   arpConfig.globalMonitorMac = False

def enableArpPersistent( mode, args ):
   arpConfig.restoreOnReboot = True

def disableArpPersistent( mode, args ):
   arpConfig.restoreOnReboot = False

def enableIpv4Persistent( mode, args ):
   refreshDelay = args.get( 'SECONDS', arpConfig.refreshDelayDefault )
   arpConfig.ip4PersistanceEnabled = True
   arpConfig.refreshDelay = refreshDelay

def disableIpv4Persistent( mode, args ):
   arpConfig.ip4PersistanceEnabled = False
   arpConfig.refreshDelay = arpConfig.refreshDelayDefault

def setProxyMaxDelay( mode, args ):
   arpConfig.proxyMaxDelay = args[ 'MILLISECONDS' ]

def unsetProxyMaxDelay( mode, args ):
   arpConfig.proxyMaxDelay = arpConfig.proxyMaxDelayDefault

def runClearArpCmd( cmd, vrfName=DEFAULT_VRF ):
   if not vrfName:
      vrfName = DEFAULT_VRF
   try:
      if vrfName != DEFAULT_VRF:
         vrf = allVrfStatusLocal.vrf.get( vrfName )
         if vrf and vrf.state == 'active':
            Arnet.NsLib.runMaybeInNetNs( vrf.networkNamespace,
                                         cmd,
                                         stdout=Tac.CAPTURE,
                                         stderr=Tac.CAPTURE,
                                         asRoot=True )
            return True
         else:
            return False
      else:
         Tac.run( cmd, asRoot=True, stdout=Tac.CAPTURE, stderr=Tac.CAPTURE )
         return True
   except Tac.SystemCommandError as e:
      qtraceException( e )
   return True

def clearArpCache( mode, args ):
   vrfName = args.get( 'VRF' )
   vrfName = VrfCli.vrfMap.lookupCliModeVrf( mode, vrfName )
   uInfo = UtmpDump.getUserInfo()
   cmd = [ "ip", "-4", "neigh", "flush", "to", "0.0.0.0/0" ]
   # No state needs to be saved, and we never run Cli remotely, so
   # we'll just do this:
   if vrfName == ALL_VRF_NAME:
      vrfList = getAllVrfNames( mode )
      for vrf in vrfList:
         runClearArpCmd( cmd, vrf )
      Logging.log( ArpLogMsgs.ETH_ARP_CACHE_CLEAR,
            vrfName, uInfo[ 'user' ], uInfo[ 'tty' ], uInfo[ 'ipAddr' ] )
   else:
      if runClearArpCmd( cmd, vrfName ):
         Logging.log( ArpLogMsgs.ETH_ARP_CACHE_CLEAR,
               vrfName, uInfo[ 'user' ], uInfo[ 'tty' ], uInfo[ 'ipAddr' ] )


def clearArpInterface( mode, args ):
   intf = args[ 'INTF' ]
   vrfName = args.get( 'VRF' )
   if not intf.lookup():
      mode.addError( "Interface does not exist" )
      return
   ipIntfStatus = ipStatus.ipIntfStatus.get( intf.name_ )
   if not ipIntfStatus:
      mode.addError( f"{intf.name_} is not a layer 3 interface" )
      return
   vrfName = VrfCli.vrfMap.lookupCliModeVrf( mode, vrfName )
   if vrfName != ipIntfStatus.vrf:
      mode.addError( "Interface vrf does not match" )
      return
   cmd = [ "ip", "-4", "neigh", "flush", "dev", intf.status().deviceName ]
   if runClearArpCmd( cmd, vrfName ):
      uInfo = UtmpDump.getUserInfo()
      Logging.log( ArpLogMsgs.ETH_ARP_CACHE_INTERFACE_CLEAR, vrfName, intf.name_,
                   uInfo[ 'user' ], uInfo[ 'tty' ], uInfo[ 'ipAddr' ] )

def clearIpArp( mode, args ):
   arpIpAddr = args[ 'IPADDR' ]
   vrfName = args.get( 'VRF' )
   vrfName = VrfCli.vrfMap.lookupCliModeVrf( mode, vrfName )
   uInfo = UtmpDump.getUserInfo()
   cmd = [ "ip", "-4", "neigh", "flush", arpIpAddr ]
   if runClearArpCmd( cmd, vrfName ):
      Logging.log( ArpLogMsgs.ETH_ARP_CACHE_IP_CLEAR,
            arpIpAddr, vrfName, uInfo[ 'user' ], uInfo[ 'tty' ], uInfo[ 'ipAddr' ] )

def refreshArpByVrf( mode, args ):
   vrfName = args.get( 'VRF' )
   vrfName = VrfCli.vrfMap.lookupCliModeVrf( mode, vrfName )
   sendArpRefreshReqMsg( mode, ipv4Af, vrfName, None, None )

def refreshArpByIntf( mode, args ):
   intf = args[ 'INTF' ]
   vrfName = args.get( 'VRF' )
   vrfName = VrfCli.vrfMap.lookupCliModeVrf( mode, vrfName )
   ipIntfStatus = ipStatus.ipIntfStatus.get( intf.name )
   if not ipIntfStatus:
      return
   if vrfName != ipIntfStatus.vrf:
      mode.addError( "Interface vrf does not match" )
      return
   sendArpRefreshReqMsg( mode, ipv4Af, vrfName, intf.name, None )

def refreshIpArp( mode, args ):
   arpIpAddr = args[ 'IPADDR' ]
   vrfName = args.get( 'VRF' )
   vrfName = VrfCli.vrfMap.lookupCliModeVrf( mode, vrfName )
   sendArpRefreshReqMsg( mode, ipv4Af, vrfName, None, arpIpAddr )

def setGlobalArpTimeOut( mode, args ):
   arpConfig.globalArpTimeout = args[ 'SECONDS' ]

def noSetGlobalArpTimeOut( mode, args ):
   arpConfig.globalArpTimeout = defaultEthernetTimeout

clearArpHwCountersHooks = []

def registerClearArpCountersHook( hook ):
   # Hooks should have two arguments, one for mode and another for the command issued
   # The hook should use AgentCommandRequest to handle counter clear on PD agent
   clearArpHwCountersHooks.append( hook )

def arpCountersSupported( mode, token ):
   if not arpHwStatus.arpCountersSupported:
      return CliParser.guardNotThisPlatform
   return None

def clearArpCountersHardware( mode, args ):
   cmd = args[ 'cmd' ]
   for clearArpHwCounters in clearArpHwCountersHooks:
      clearArpHwCounters( mode, cmd )

#-------------------------------------------------------------------------------
# Have the Cli Agent mount all needed state from sysdb
#-------------------------------------------------------------------------------
myEntManager = None
def Plugin( entityManager ):
   arp.plugin( entityManager )
   global allVrfStatusLocal
   global ipStatus
   global arpConfig
   global myEntManager
   global arpHwStatus

   myEntManager = entityManager

   ipStatus = LazyMount.mount( entityManager, "ip/status", "Ip::Status", "r" )
   allVrfStatusLocal = LazyMount.mount( entityManager,
                                        Cell.path( "ip/vrf/status/local" ),
                                        "Ip::AllVrfStatusLocal", "r" )
   arpConfig = LazyMount.mount( entityManager, "arp/config", "Arp::Config", "w" )
   arpHwStatus = LazyMount.mount( entityManager, 'arp/hardware/status',
                                  'Arp::Hardware::Status', 'r' )
