#!/usr/bin/env python3
# Copyright (c) 2017 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import sys
import BasicCli
import CliCommand
import CliMatcher
from CliPlugin import TechSupportCli
from CliPlugin.ArpSuppressionModel import ArpBindingModel
from CliPlugin.Ip6AddrMatcher import Ip6AddrMatcher
from CliPlugin.IpAddrMatcher import IpAddrMatcher
from CliPlugin.IpEth import ShowArpExpr, arpAfterShowKw, neighborAfterShowKw
from CliPlugin.VlanCli import vlanSetMatcher
import CliPrint
from CliToken.Clear import clearKwNode
import CliToken.Ip
import CliToken.Ipv6
import LazyMount
from MultiRangeRule import multiRangeToCanonicalString
import SharedMem
import SharkLazyMount
import ShowCommand
import Smash
import Tac

cprinter = CliPrint.CliPrint().lib
bridgingHwCapabilities = None
remoteBindingConfig = None
localMacIpAddrTableConfig = None
localMacIpAddrTable = None
vcsRemoteBindingConfig = None

IpGenAddr = Tac.Type( 'Arnet::IpGenAddr' )

matcherArpClear = CliMatcher.KeywordMatcher(
   'arp', helpdesc='ARP entries to clear' )
matcherNeighborsClear = CliMatcher.KeywordMatcher(
   'neighbors', helpdesc='Neighbor entries to clear' )
matcherBridgedClear = CliMatcher.KeywordMatcher(
   "bridged", helpdesc="Clear entries learned from bridged traffic" )
matcherBridgedShow = CliMatcher.KeywordMatcher(
   "bridged", helpdesc="Show entries learned from bridged traffic" )
matcherRemoteShow = CliMatcher.KeywordMatcher(
   "remote", helpdesc="Show remote host bindings" )

def _showTechGuard():
   return bridgingHwCapabilities.vxlanBridgedArpNdLearningSupported

def showArp( mode, args ):
   filterAf = 'ipv6' if 'ipv6' in args else 'ipv4'
   arpBridged = 'bridged' in args
   filterVlanId = args.get( 'VLAN', 0 )

   fmt = CliPrint.TEXT
   if mode:
      outputFormat = cprinter.stringToOutputFormat( mode.session_.outputFormat_ )
      fmt = outputFormat.value
   fd = sys.stdout.fileno()

   # We're hopping into c++, make sure the LazyMount is completed and
   # get the actual underlying entity to pass as a constructor param.
   localMacIpAddrTableEntity = SharkLazyMount.force( localMacIpAddrTable )

   helper = Tac.newInstance( "ArpSuppression::ArpSuppressionShowHelper",
                             arpBridged, localMacIpAddrTableEntity,
                             remoteBindingConfig, vcsRemoteBindingConfig )
   helper.showArp( fd, fmt, filterAf, filterVlanId )

   return ArpBindingModel

def clearArp( mode, args ):
   vlanSet = args.get( 'VLANS' )
   ipv4Addr = args.get( 'IPADDR', '' )
   ipv6Addr = args.get( 'IPV6ADDR', '' )
   if ( ipv4Addr or ipv6Addr ) and ( not vlanSet or len( vlanSet ) != 1 ):
      mode.addError( "Clearing an IP address requires one VLAN to be specified" )
      return

   vlanStr = multiRangeToCanonicalString( vlanSet ) if vlanSet else '1-4094'
   ipAddr = IpGenAddr( ipv6Addr.stringValue if ipv6Addr else ipv4Addr )
   af = 'ipv6' if 'ipv6' in args else 'ipv4'
   clearRequest = Tac.Value( "ArpSuppression::ClearRequest", vlanStr, ipAddr, af )
   clearRequest.requestId = localMacIpAddrTableConfig.clearRequest.requestId + 1
   localMacIpAddrTableConfig.clearRequest = clearRequest

#--------------------------------------------------------------------------------
# show arp bridged [ vlan VLAN ]
# show ipv6 neighbors bridged [ vlan VLAN ]
#--------------------------------------------------------------------------------
class ShowArpBridged( ShowCommand.ShowCliCommandClass ):
   syntax = "show arp | ( ipv6 neighbors ) bridged [ vlan VLAN ]"
   data = {
      'arp' : arpAfterShowKw,
      'ipv6' : CliToken.Ipv6.ipv6MatcherForShow,
      'neighbors' : neighborAfterShowKw,
      'bridged' : matcherBridgedShow,
      'vlan' : 'Show information for specific VLAN',
      'VLAN' : CliMatcher.IntegerMatcher( 1, 4094,
                                          helpdesc='Identifier for a Virtual LAN' )
      }
   handler = showArp
   cliModel = ArpBindingModel

BasicCli.addShowCommandClass( ShowArpBridged )

#--------------------------------------------------------------------------------
# clear arp bridged [ vlan VLANS [ IPADDR ] ]
# clear ipv6 neighbors bridged [ vlan VLANS [ IPV6ADDR ] ]
#--------------------------------------------------------------------------------
class ClearArpBridged( CliCommand.CliCommandClass ):
   syntax = "clear arp bridged [ vlan VLANS [ IPADDR ] ]"
   data = {
      'clear' : clearKwNode,
      'arp' : matcherArpClear,
      'bridged' : matcherBridgedClear,
      'vlan' : 'Clear entries associated with single or range(s) of VLANs',
      'VLANS' : vlanSetMatcher,
      'IPADDR' : IpAddrMatcher( helpdesc='IP address to be cleared' ),
      }
   handler = clearArp

class ClearIpv6NeighborsBridged( CliCommand.CliCommandClass ):
   syntax = "clear ipv6 neighbors bridged [ vlan VLANS [ IPV6ADDR ] ]"
   data = {
      'clear' : clearKwNode,
      'ipv6' : CliToken.Ipv6.ipv6MatcherForClear,
      'neighbors' : matcherNeighborsClear,
      'bridged' : matcherBridgedClear,
      'vlan' : 'Clear entries associated with single or range(s) of VLANs',
      'VLANS' : vlanSetMatcher,
      'IPV6ADDR' : Ip6AddrMatcher( helpdesc='IPv6 address to be cleared' ),
      }
   handler = clearArp

BasicCli.EnableMode.addCommandClass( ClearArpBridged )
BasicCli.EnableMode.addCommandClass( ClearIpv6NeighborsBridged )

#--------------------------------------------------------------------------------
# show arp remote [ vlan VLAN ]
# show ip arp remote [ vlan VLAN ]
# show ipv6 neighbors remote [ vlan VLAN ]
#--------------------------------------------------------------------------------
class ShowArpRemote( ShowCommand.ShowCliCommandClass ):
   syntax = "show arp | ( ipv6 neighbors ) remote [ vlan VLAN ]"
   data = {
      'arp' : ShowArpExpr,
      'ipv6' : CliToken.Ipv6.ipv6MatcherForShow,
      'neighbors' : neighborAfterShowKw,
      'remote' : matcherRemoteShow,
      'vlan' : 'Show information for specific VLAN',
      'VLAN' : CliMatcher.IntegerMatcher( 1, 4094,
                                          helpdesc='Identifier for a Virtual LAN' )
      }
   handler = showArp
   cliModel = ArpBindingModel

BasicCli.addShowCommandClass( ShowArpRemote )

def Plugin( entityManager ):
   smashEm = SharedMem.entityManager( sysdbEm=entityManager )
   readerInfo = Smash.mountInfo( 'reader' )

   global bridgingHwCapabilities
   bridgingHwCapabilities = LazyMount.mount( entityManager,
                                             "bridging/hwcapabilities",
                                             "Bridging::HwCapabilities", "r" )

   global remoteBindingConfig
   remoteBindingConfig = smashEm.doMount( 'arpSuppression/remoteBindingConfig/evpn',
                                          'ArpSuppression::RemoteBindingConfig',
                                          readerInfo )

   global localMacIpAddrTableConfig
   localMacIpAddrTableConfig = \
      LazyMount.mount( entityManager,
                       "arpSuppression/localMacIpAddrTableConfig/vxlan",
                       "ArpSuppression::LocalMacIpAddrTableConfig", "w" )

   global vcsRemoteBindingConfig
   vcsRemoteBindingConfig = \
         smashEm.doMount( 'arpSuppression/remoteBindingConfig/vcs',
                          'ArpSuppression::RemoteBindingConfig',
                          readerInfo )

   # After a period of not being used, the underlying entity in the
   # SharkLazyMount will get unmounted to release resources. It will
   # seamlessly get re-mounted under-the-hood on subsequent access.
   global localMacIpAddrTable
   autoUnmount = True
   localMacIpAddrTable = SharkLazyMount.mount(
      entityManager,
      "arpSuppression/vxlan/localMacIpAddrTable",
      "ArpSuppression::Status::LocalMacIpAddrTable",
      SharkLazyMount.mountInfo( 'shadow' ),
      autoUnmount )

   # Register commands in show tech-support
   # Timestamps are made up to maintain historical order within show tech-support
   TechSupportCli.registerShowTechSupportCmd(
      '2021-05-18 12:35:46',
      cmds=[ 'show arp bridged',
             'show ipv6 neighbors bridged'
           ],
      cmdsGuard=_showTechGuard )

   TechSupportCli.registerShowTechSupportCmd(
      '2022-06-23 15:16:30',
      cmds=[ 'show arp remote',
             'show ipv6 neighbors remote'
           ],
      )

   # Register commands in show tech-support extended evpn
   TechSupportCli.registerShowTechSupportCmd(
      '2021-05-18 12:35:57',
      cmds=[ 'show arp bridged',
             'show ipv6 neighbors bridged'
           ],
      cmdsGuard=_showTechGuard,
      extended='evpn' )

   TechSupportCli.registerShowTechSupportCmd(
      '2021-05-18 12:36:57',
      cmds=[ 'show arp remote',
             'show ipv6 neighbors remote'
           ],
      extended='evpn' )
