# Copyright (c) 2018 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import Tac
import BasicCli
import CliParser
from CliMode.RouterL2Vpn import RoutingL2VpnMode
import ConfigMount
import CliCommand
import CliMatcher
from CliToken.Router import routerMatcherForConfig as tokenRouter
import LazyMount

l2VpnConfig = None
bridgingHwCapabilities = None

def selectiveFloodingSupportedGuard( mode, token ):
   if bridgingHwCapabilities.vxlanSelectiveFloodingSupported:
      return None
   return CliParser.guardNotThisPlatform

def bridgedArpNdLearningSupportedGuard( mode, token ):
   if bridgingHwCapabilities.vxlanBridgedArpNdLearningSupported:
      return None
   return CliParser.guardNotThisPlatform

prefixListNameMatcher = CliMatcher.PatternMatcher(
      pattern=r'([A-Za-z0-9_:{}\[\]=\+-])([A-Za-z0-9_:{}\.\[\]=\+-]*)',
      helpdesc='Prefix-list name', helpname='WORD' )

class RouterL2VpnConfigMode( RoutingL2VpnMode, BasicCli.ConfigModeBase ):
   name = 'l2-vpn configuration'

   def __init__( self, parent, session ):
      RoutingL2VpnMode.__init__( self, None )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

# --------------------------------------------------------------------------
# 'router l2-vpn' to enter config-rtr-l2-vpn mode
# --------------------------------------------------------------------------
class RouterL2VpnModeCmd( CliCommand.CliCommandClass ):
   syntax = 'router l2-vpn'
   noOrDefaultSyntax = syntax
   data = {
         'router' : tokenRouter,
         'l2-vpn' : 'l2-vpn configuration',
   }

   @staticmethod
   def handler( mode, args ):
      childMode = mode.childMode( RouterL2VpnConfigMode )
      mode.session_.gotoChildMode( childMode )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      defaultFloodTrafficConfig = Tac.Value( 'Routing::L2Vpn::FloodTraffic',
                                             False, False, False )
      l2VpnConfig.floodTrafficConfig = defaultFloodTrafficConfig
      l2VpnConfig.arpL2LearningEnabled = False
      l2VpnConfig.arpProxyPrefixList = ''
      l2VpnConfig.arpSelectiveInstall = False
      l2VpnConfig.ndL2LearningEnabled = False
      l2VpnConfig.ndProxyPrefixList = ''
      l2VpnConfig.rsFloodingDisabled = False
      l2VpnConfig.dadFloodingDisabled = False
      l2VpnConfig.virtualRouterNaFloodingDisabled = False
      l2VpnConfig.virtualRouterGarpFloodingDisabled = False

BasicCli.GlobalConfigMode.addCommandClass( RouterL2VpnModeCmd )

# --------------------------------------------------------------------------
# [ no } default ] arp flooding
# [ no | default ] arp learning bridged
# [ no | default ] arp proxy prefix-list WORD
# [ no | default ] arp selective-install
# --------------------------------------------------------------------------
class RouterL2VpnArpConfigCmd( CliCommand.CliCommandClass ):
   _toggleCmdsStr = ''
   _toggleCmdsStr += '( FLOODING_KW ) | '
   _toggleCmdsStr += '( learning bridged ) | '

   # pylint: disable-next=consider-using-f-string
   syntax = '''arp ( %s( proxy prefix-list WORD ) |
                       ( selective-install ) )''' % _toggleCmdsStr
   # pylint: disable-next=consider-using-f-string
   noOrDefaultSyntax = '''arp ( %s( proxy prefix-list ... ) |
                                  ( selective-install ) )''' % _toggleCmdsStr

   data = {
      'arp' : 'ARP configuration',
      'FLOODING_KW' : CliCommand.guardedKeyword(
         'flooding',
         helpdesc='Enable ARP flooding into tunnel',
         guard=selectiveFloodingSupportedGuard ),
      'learning' : 'Learn ARP entries',
      'bridged' : CliCommand.guardedKeyword(
         'bridged',
         helpdesc='Learn ARP entries from bridged traffic',
         guard=bridgedArpNdLearningSupportedGuard ),
      'proxy' : 'Proxy ARP',
      'prefix-list' : 'Prefix list',
      'WORD' : prefixListNameMatcher,
      'selective-install' : 'Install ARP entries for remote hosts on demand'
   }

   @staticmethod
   def handler( mode, args ):
      if 'FLOODING_KW' in args:
         enabled = l2VpnConfig.floodTrafficConfig.enabled
         nd = l2VpnConfig.floodTrafficConfig.nd
         value = Tac.Value( 'Routing::L2Vpn::FloodTraffic', enabled, True, nd )
         l2VpnConfig.floodTrafficConfig = value
      elif 'learning' in args:
         l2VpnConfig.arpL2LearningEnabled = True
      elif 'proxy' in args:
         l2VpnConfig.arpProxyPrefixList = args[ 'WORD' ] # prefix-list name
      elif 'selective-install' in args:
         l2VpnConfig.arpSelectiveInstall = True

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      if 'FLOODING_KW' in args:
         enabled = l2VpnConfig.floodTrafficConfig.enabled
         nd = l2VpnConfig.floodTrafficConfig.nd
         value = Tac.Value( 'Routing::L2Vpn::FloodTraffic', enabled, False, nd )
         l2VpnConfig.floodTrafficConfig = value
      elif 'learning' in args:
         l2VpnConfig.arpL2LearningEnabled = False
      elif 'proxy' in args:
         l2VpnConfig.arpProxyPrefixList = ''
      elif 'selective-install' in args:
         l2VpnConfig.arpSelectiveInstall = False

RouterL2VpnConfigMode.addCommandClass( RouterL2VpnArpConfigCmd )

# --------------------------------------------------------------------------
# [ no | default ] nd flooding
# [ no | default ] nd learning bridged
# [ no | default ] nd proxy prefix-list WORD
# [ no | default ] nd ( rs flooding disabled | dad flooding disabled )
# --------------------------------------------------------------------------
class RouterL2VpnNdConfigCmd( CliCommand.CliCommandClass ):
   _toggleCmdsStr = ''
   _toggleCmdsStr += '( FLOODING_KW ) | '
   _toggleCmdsStr += '( learning bridged ) | '

   # pylint: disable-next=consider-using-f-string
   syntax = '''nd ( %s( proxy prefix-list WORD ) |
                      ( rs flooding disabled ) |
                      ( dad flooding disabled ) )''' % _toggleCmdsStr
   # pylint: disable-next=consider-using-f-string
   noOrDefaultSyntax = '''nd ( %s( proxy prefix-list ... ) |
                                 ( rs flooding disabled ) |
                                 ( dad flooding disabled ) )''' % _toggleCmdsStr

   data = {
      'nd' : 'Neighbor Discovery configuration',
      'FLOODING_KW' : CliCommand.guardedKeyword(
         'flooding',
         helpdesc='Enable Neighbor Discovery flooding into tunnel',
         guard=selectiveFloodingSupportedGuard ),
      'learning' : 'Learn neighbor entries',
      'bridged' : CliCommand.guardedKeyword(
         'bridged',
         helpdesc='Learn neighbor entries from bridged traffic',
         guard=bridgedArpNdLearningSupportedGuard ),
      'proxy' : 'Proxy ND',
      'prefix-list' : 'Prefix list',
      'rs' : 'Router solicitation',
      'flooding' : 'Vtep flooding',
      'disabled' : 'Disabled',
      'dad' : 'Duplicate address detection',
      'WORD' : prefixListNameMatcher,
   }

   @staticmethod
   def handler( mode, args ):
      if 'FLOODING_KW' in args:
         enabled = l2VpnConfig.floodTrafficConfig.enabled
         arp = l2VpnConfig.floodTrafficConfig.arp
         value = Tac.Value( 'Routing::L2Vpn::FloodTraffic', enabled, arp, True )
         l2VpnConfig.floodTrafficConfig = value
      elif 'learning' in args:
         l2VpnConfig.ndL2LearningEnabled = True
      elif 'proxy' in args:
         l2VpnConfig.ndProxyPrefixList = args[ 'WORD' ] # prefix-list name
      elif 'rs' in args:
         l2VpnConfig.rsFloodingDisabled = True
      elif 'dad' in args:
         l2VpnConfig.dadFloodingDisabled = True

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      if 'FLOODING_KW' in args:
         enabled = l2VpnConfig.floodTrafficConfig.enabled
         arp = l2VpnConfig.floodTrafficConfig.arp
         value = Tac.Value( 'Routing::L2Vpn::FloodTraffic', enabled, arp, False )
         l2VpnConfig.floodTrafficConfig = value
      elif 'learning' in args:
         l2VpnConfig.ndL2LearningEnabled = False
      elif 'proxy' in args:
         l2VpnConfig.ndProxyPrefixList = ''
      elif 'rs' in args:
         l2VpnConfig.rsFloodingDisabled = False
      elif 'dad' in args:
         l2VpnConfig.dadFloodingDisabled = False

RouterL2VpnConfigMode.addCommandClass( RouterL2VpnNdConfigCmd )

# --------------------------------------------------------------------------
# [ no | default ] virtual-router ( neighbor advertisement flooding disabled |
#                                   arp advertisement flooding disabled )
# --------------------------------------------------------------------------
class RouterL2VpnVirtualRouterConfigCmd( CliCommand.CliCommandClass ):
   syntax = 'virtual-router ( ( neighbor advertisement flooding disabled ) |' \
                             '( arp advertisement flooding disabled ) )'
   noOrDefaultSyntax = syntax

   data = {
      'virtual-router' : 'Virtual router configuration',
      'neighbor' : 'Neighbor',
      'advertisement' : 'Advertisement',
      'flooding' : 'Vtep flooding',
      'disabled' : 'Disabled',
      'arp' : 'Arp',
   }

   @staticmethod
   def handler( mode, args ):
      neighbor = args.get( 'neighbor' )
      if neighbor:
         l2VpnConfig.virtualRouterNaFloodingDisabled = True
      arp = args.get( 'arp' )
      if arp:
         l2VpnConfig.virtualRouterGarpFloodingDisabled = True

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      neighbor = args.get( 'neighbor' )
      if neighbor:
         l2VpnConfig.virtualRouterNaFloodingDisabled = False
      arp = args.get( 'arp' )
      if arp:
         l2VpnConfig.virtualRouterGarpFloodingDisabled = False

RouterL2VpnConfigMode.addCommandClass( RouterL2VpnVirtualRouterConfigCmd )

# CLI: tunnel broadcast-traffic arp
# CLI to enable only Arp broadcast traffic into the Vxlan or MPLS tunnel
class FloodTrafficCmd( CliCommand.CliCommandClass ):
   syntax = 'flooding default disabled'
   noOrDefaultSyntax = syntax
   data = {
         'flooding' : CliCommand.guardedKeyword(
            'flooding',
            helpdesc='Modify flooding behavior of traffic into tunnel',
            guard=selectiveFloodingSupportedGuard ),
         'default' : 'Default flooding behavior',
         'disabled' : 'Disable default flooding behavior',
   }

   @staticmethod
   def handler( mode, args ):
      arp = l2VpnConfig.floodTrafficConfig.arp
      nd = l2VpnConfig.floodTrafficConfig.nd
      value = Tac.Value( 'Routing::L2Vpn::FloodTraffic', True, arp, nd )
      l2VpnConfig.floodTrafficConfig = value

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      arp = l2VpnConfig.floodTrafficConfig.arp
      nd = l2VpnConfig.floodTrafficConfig.nd
      value = Tac.Value( 'Routing::L2Vpn::FloodTraffic', False, arp, nd )
      l2VpnConfig.floodTrafficConfig = value

RouterL2VpnConfigMode.addCommandClass( FloodTrafficCmd )


def Plugin( entityManager ):
   global l2VpnConfig
   global bridgingHwCapabilities

   l2VpnConfig = ConfigMount.mount( entityManager, "routing/l2vpn/config",
                                    "Routing::L2Vpn::Config", "w" )
   bridgingHwCapabilities = LazyMount.mount( entityManager,
                                             "bridging/hwcapabilities",
                                             "Bridging::HwCapabilities", "r" )
