#!/usr/bin/env python3
# Copyright (c) 2017 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from __future__ import absolute_import, division, print_function

import json
import threading
import os

import Ark
from CliPlugin import AssetTagsModel
import LazyMount
from Tracing import Handle, t0
import six

__defaultTraceHandle__ = Handle( 'AssetTagsCliPlugin' )

TAGS_FILE = '/mnt/flash/.assetTags'
TAG_NOT_SET = '<not set>'
NOT_INSERTED = '<not inserted>'
NO_SERIAL = ( 'N/A', '' )
TAG_KW_MAP = {
   'FixedSystem' : 'switch',
   'Chassis' : 'chassis',
}

fileLock = threading.RLock()
entmib = None

def normalizeTag( tag ):
   if tag == 'SwitchcardCes':
      return 'Switchcard'
   return tag

@Ark.synchronized( fileLock )
def _readAllAssetTags( mode ):
   ''' Try reading all of the asset tags that are currently stored in the system.
   If we get an error in the process reading the file we create a new file and
   save the previous version. Basically to help deal with corruption.'''
   t0( '_readAllAssetTags' )
   try:
      with open( TAGS_FILE ) as f:
         result = f.read()
         return json.loads( result ) if result else {}
   except EnvironmentError:
      return {}
   except ValueError:
      mode.addError( 'The tags file is corrupted. Data may have been lost.' )
      return {}

@Ark.synchronized( fileLock )
def _updateAssetTag( mode, serial, module, tag=None ):
   ''' Update a single module's asset tag. Update in file.
   If no tag is provided then the serial number is removed from the file'''
   t0( '_updateAssetTag' )
   tags = _readAllAssetTags( mode ) or {}
   if tag is None: # remove tag
      tags.pop( serial, None )
   else:
      tags[ serial ] = { 'tag' : tag, 'module' : module }

   try:
      with open( TAGS_FILE, 'w' ) as f:
         json.dump( tags, f )
   except IOError:
      # pylint: disable-next=consider-using-f-string
      mode.addError( 'Could not save tags in %s' % TAGS_FILE )

def _getFruFromModule( module ):
   t0( '_getFruFromModule' )
   # Here `module` is either `cardSlot` or `powerSupplySlot`
   # Assuming cards are more likely to be tagged, we try to get the `card` attribute.
   # If not, we try `powerSupply`, if that also fails, there's no FRU.
   card = getattr( module, 'card', None )
   if card is not None:
      return card

   powersup = getattr( module, 'powerSupply', None )
   if powersup is not None:
      return powersup

   return None

def _generateModel( tags, keySerialTagList, showMissing ):
   t0( '_generateModel' )
   result = AssetTagsModel.Assets()
   for module, serial, tag in keySerialTagList:
      result.assets[ module ] = AssetTagsModel.Asset( serial=serial, tag=tag )
      tags.pop( serial, None ) # remove present frus to reveal the missing ones

   if showMissing:
      for sn, val in six.iteritems( tags ):
         loc = val[ 'module' ].split( '/', 1 )[ 0 ]
         tag = val[ 'tag' ]
         result.missing[ sn ] = AssetTagsModel.Missing( fruType=loc, tag=tag )
   return result

def _getSystemAssetTag( tags ):
   t0( '_getSystemAssetTag' )
   serial = entmib.root.serialNum
   systemKey = TAG_KW_MAP[ entmib.root.tag ]
   tag = tags.get( serial, {} ).get( 'tag', TAG_NOT_SET )
   return [ ( systemKey, serial, tag ) ]

def _getFruAssetTag( tags, module ):
   t0( '_getFruAssetTag' )
   # pylint: disable-next=consider-using-f-string
   location = '%s/%s' % ( normalizeTag( module.tag ), module.label )
   serial = NOT_INSERTED
   tag = TAG_NOT_SET
   fru = _getFruFromModule( module )
   if fru:
      serial = fru.serialNum
      tag = tags.get( serial, {} ).get( 'tag', TAG_NOT_SET )
      # pylint: disable-next=consider-using-f-string
      location = '%s/%s' % ( normalizeTag( fru.tag ), fru.label )
   return [ ( location, serial, tag ) ]

def _getAllAssetTags( tags ):
   t0( '_getAllAssetTags' )
   result = _getSystemAssetTag( tags )

   system = entmib.root

   for slot in system.powerSupplySlot.values():
      result += _getFruAssetTag( tags, slot )
   if entmib.chassis:
      for slot in system.cardSlot.values():
         result += _getFruAssetTag( tags, slot )
   return result

# show hardware asset-tag
def doShowAssetTag( mode, args ):
   if entmib.root is None:
      return AssetTagsModel.Assets()

   module = args.get( 'MODULE' )
   t0( 'doShowAssetTag' )
   tags = _readAllAssetTags( mode ) or {}
   showAll = module is None
   if showAll:
      result = _getAllAssetTags( tags )
   elif module.tag in TAG_KW_MAP:
      result = _getSystemAssetTag( tags )
   else:
      # FruCli.slotRule gives us a slot, but powerSupplyRule gives us a powerSupply,
      # so we have to look for 'slot' if the command was invoked for a card.
      result = _getFruAssetTag( tags, getattr( module, 'slot', module ) )
   return _generateModel( tags, result, showAll )

# hardware asset-tag MODULE TAG
def doSetAssetTag( mode, args ):
   t0( 'doSetAssetTag' )
   module = args[ 'MODULE' ]
   tag = args.get( 'TAG' )
   system = TAG_KW_MAP.get( module.tag )
   if system:
      serial = entmib.root.serialNum
      _updateAssetTag( mode, serial, system, tag )
   else:
      # FruCli.slotRule gives us a slot, but powerSupplyRule gives us a powerSupply,
      # so we have to look for 'slot' if the command was invoked for a card.
      fru = _getFruFromModule( getattr( module, 'slot', module ) )
      if fru:
         if fru.serialNum in NO_SERIAL:
            mode.addError( 'Only hardware with a serial number can be tagged.' )
         else:
            # module.tag is the location. e.g., 'linecard'
            _updateAssetTag( mode, fru.serialNum, module.tag, tag )
      else:
         mode.addError( 'No hardware in that slot.' )

def Plugin( em ):
   global entmib
   entmib = LazyMount.mount( em, 'hardware/entmib', 'EntityMib::Status', 'r' )
   if os.path.exists( os.path.dirname( TAGS_FILE ) ):
      try:
         open( TAGS_FILE, 'a' ).close() # pylint: disable=consider-using-with
         os.chmod( TAGS_FILE, 0o666 )
      except ( IOError, OSError ):
         pass
