# Copyright (c) 2019 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import sys
import BasicCli
import CliCommand
import CliMatcher
import CliParser
import BasicCliUtil
import DateTimeRule
import subprocess
import AsuUtil

from CliPlugin.ReloadCli import reloadKwNode
from CliPlugin.AsuReloadCli import ( _asuData, _asuReloadCliHelper,
                                     generateProcOutput, rebootHitlessGuard,
                                     isReloadScheduled, scheduleReload,
                                     getFastbootFeatureDisruption )
from CliPlugin.AsuReloadShowCommands import showFastbootFeatureDisruption
from CliPlugin.ReloadConfigSaveCli import doConfigSaveBeforeReload
from Toggles import AsuCliToggleLib

def isReloadNow( mode, args ):
   return "now" in args

# Silly pylint, don't complain about space after = in decorators:
# pylint: disable-msg=C0322 # pylint: disable=bad-option-value
@BasicCliUtil.EapiIncompatible( overrideFn=isReloadNow,
                            additionalMsg="To reload the machine over the API, "
                            "please use 'reload fast-boot now' instead." )
def doFastReboot( mode, args ):
   now = "now" in args
   cmdInvoked = _asuData.AsuMode.fastboot
   doRebootHitlessCommon( mode, now, cmdInvoked )

def fastRebootGuard( mode, token ):
   if _asuData.asuHwStatus.asuModeSupported == _asuData.AsuMode.fastboot:
      return None
   # If Asu fastboot is not supported then check for Asu hitless support
   return rebootHitlessGuard( mode, token )

def fastRebootInGuard( mode, args ):
   return None if AsuCliToggleLib.toggleDelayedFastBootReloadEnabled() else \
      CliParser.guardNotThisEosVersion

def doRebootHitlessSim( mode ):
   '''
   Simulate a minimal reboot hitless, by running the AsuPStore save,
   and creating a commandline sim file to simulate hitless boot stages.
   A ptest can then issue a "service ProcMgr stop" "service ProcMgr start"
   to simulate the hitless startup of the agents.

   The "relaod hitless sim cleanup" can be used by the ptest to remove the sim
   files.
   '''
   AsuUtil.doSaveAsuState( mode.entityManager )

   fs = lambda: None
   setattr( fs, 'realFilename_', "simFile" )
   procOutput = generateProcOutput( fs )
   with open( "/var/tmp/asuCmdlineSim", "w" ) as f:
      f.write( " ".join( procOutput ) )

def doRebootHitlessSimCleanup( mode ):
   subprocess.call( [ "rm", "-f", "/var/tmp/asuCmdlineSim" ] )
   subprocess.call( [ "rm", "-f", "/mnt/flash/asu-persistent-store" ] )
   subprocess.call( [ "rm", "-f", "/mnt/flash/asu-persistent-store.json" ] )

# Silly pylint, don't complain about space after = in decorators:
# pylint: disable-msg=C0322 # pylint: disable=bad-option-value
@BasicCliUtil.EapiIncompatible( overrideFn=isReloadNow,
                            additionalMsg="To reload the machine over the API, "
                            "please use 'reload fast-boot now' instead." )
def doRebootHitless( mode, args ):
   now = "now" in args
   cmdInvoked = _asuData.AsuMode.hitless
   simOpt = 'sim' in args
   cleanOpt = 'cleanup' in args
   doRebootHitlessCommon( mode, now, cmdInvoked, simOpt, cleanOpt )

def doRebootHitlessCommon( mode, now, cmdInvoked, simOpt=None, cleanOpt=None ):
   # Set reboot mode to hitless, regardless of cmd invoked
   # hitless is treated as ASU2 in product code, see go/asuprimer for more info
   _asuData.asuCliConfig.asuRebootMode = _asuData.AsuMode.hitless

   if simOpt:
      if cleanOpt:
         doRebootHitlessSimCleanup( mode )
      else:
         doRebootHitlessSim( mode )
   else:
      _asuReloadCliHelper.doFastBootCommon( mode, now, cmdInvoked )

def doFastRebootLater( mode, args ):
   if isReloadScheduled():
      print( 'Reload is already scheduled, please cancel it' )
      return
   # Get time/reason/force
   reloadReason = args.get( 'REASON' )
   force = 'force' in args
   schedule = args.get( 'MMM' ) or args[ 'HHHMM' ]
   hhh, mm = 0, schedule[ -1 ]
   if len( schedule ) > 1:
      if mm >= 60:
         print( 'Invalid time/date' )
         return
      hhh = schedule[ 0 ]
   seconds = hhh * 60 * 60 + mm * 60
   # 1. check if fast-boot reload can be proceeded
   fastbootFeatureDisruption = getFastbootFeatureDisruption( mode, args )
   if fastbootFeatureDisruption.blockingList:
      fastbootFeatureDisruption.render()
      return
   # 2. ask for saving changed configuration. "no" will stop scheduling.
   if not doConfigSaveBeforeReload( mode, power=False, now=False,
                                    noMeansFailure=True ):
      return
   # 3. confirm to procceed
   if not force:
      answer = BasicCliUtil.getSingleChar( "Proceed with reload? [confirm]" )
      print( answer )
      sys.stdout.flush()
      if answer.lower() not in 'y\r\n':
         return
   if scheduleReload( seconds, reloadReason, mode ):
      showFastbootFeatureDisruption( mode, args ).render()

fastbootGuardedKwNode = CliCommand.guardedKeyword( "fast-boot",
                                                   helpdesc="Perform hitless "
                                                            "reload with image",
                                                   guard=fastRebootGuard )

#--------------------------------------------------------------------------------
# reload fast-boot [ now ]
#--------------------------------------------------------------------------------
class DoFastRebootCmd( CliCommand.CliCommandClass ):
   syntax = 'reload fast-boot [ now ]'
   data = {
      'reload' : reloadKwNode,
      'fast-boot' : fastbootGuardedKwNode,
      'now' : "Perform action immediately without prompting"
   }
   syncAcct = True
   handler = doFastReboot

BasicCli.EnableMode.addCommandClass( DoFastRebootCmd )

#--------------------------------------------------------------------------------
# reload hitless [ sim ] [ cleanup ] [ now ]
#--------------------------------------------------------------------------------
class DoRebootHitlessCmd( CliCommand.CliCommandClass ):
   syntax = 'reload hitless [ sim ] [ cleanup ] [ now ]'
   data = {
      'reload' : reloadKwNode,
      'hitless' : CliCommand.Node( CliMatcher.KeywordMatcher( "hitless",
                                                 helpdesc="Perform hitless "
                                                          "reload with image" ),
                                   guard=rebootHitlessGuard,
                                   hidden=True ),
      'sim' : CliCommand.Node( CliMatcher.KeywordMatcher( "sim",
                                             helpdesc="Perform hitless "
                                                      "reload simulation" ),
                               hidden=True ),
      'cleanup' : CliCommand.Node( CliMatcher.KeywordMatcher( "cleanup",
                                             helpdesc="Remove hitless "
                                                      "reload simulation files" ),
                                   hidden=True ),
      'now' : "Perform action immediately without prompting"
   }
   syncAcct = True
   handler = doRebootHitless

BasicCli.EnableMode.addCommandClass( DoRebootHitlessCmd )

#--------------------------------------------------------------------------------
# reload fast-boot in ( MM | HHHMM ) [ force ] [ reason REASON ]
#--------------------------------------------------------------------------------
class DoFastRebootLaterCmd( CliCommand.CliCommandClass ):
   syntax = 'reload fast-boot in ( MMM | HHHMM ) [ force ] [ reason REASON ]'
   data = {
      'reload' : reloadKwNode,
      'fast-boot' : CliCommand.guardedKeyword( "fast-boot",
                                               helpdesc="Perform hitless "
                                                        "reload with image",
                                               guard=fastRebootGuard ),
      'in' : CliCommand.guardedKeyword( "in",
                                        helpdesc="Perform the reload after "
                                                 "specified delay",
                                        guard=fastRebootInGuard ),
      'MMM' : DateTimeRule.TimeMatcher( name="mmm", numPattern="mmm",
                                        helpdesc="Delay in minutes" ),
      'HHHMM' : DateTimeRule.TimeMatcher( name="hhh:mm", numPattern="hhh:mm",
                  helpdesc="Delay in hours and minutes mm<0-59>" ),
      'force' : 'Perform action immediately without prompting',
      'reason' : 'String to display explaining the purpose of the reload',
      'REASON' : CliMatcher.StringMatcher( helpname="TEXT",
                  helpdesc='String to display explaining the purpose of the reload' )
   }
   syncAcct = True
   handler = doFastRebootLater

BasicCli.EnableMode.addCommandClass( DoFastRebootLaterCmd )
