# Copyright (c) 2021 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# pylint: disable=ungrouped-imports

import BasicCli
import CliCommand
import CliMatcher
import CliParser
import CliGlobal
import ConfigMount
import LazyMount
from CliToken.Router import routerMatcherForConfig
from CliMode.Avt import ( RouterAdaptiveVirtualTopologyConfigMode,
                          RouterAvtPolicyAppProfileConfigMode,
                          RouterAvtPolicyConfigMode,
                          RouterAvtProfileConfigMode,
                          RouterAvtVrfConfigMode,
                        )
from CliPlugin.DpsCli import loadBalanceProfileNameMatcher
from CliPlugin.DpsCli import matcherTrafficClass
from CliPlugin.DpsCli import matcherTrafficClassNum
from CliPlugin.DpsCli import matcherDscp
from CliPlugin.DpsCli import matcherDscpValue
import Tac
import Toggles.DpsToggleLib
import Toggles.BgpCommonToggleLib
from Toggles.WanTECommonToggleLib import ( toggleAvtPathOutlierEliminationEnabled,
                                           toggleAvtLowestLoadMetricEnabled )
from CliPlugin.WanTEShowCli import dpsSupported
from CliPlugin.ShowConfigConsistencyCliModels import (
   ConfigConsistencyMultiTableModelBuilder,
)
from ConfigConsistencyChecker import (
   configConsistencyCategory,
   configConsistencyCheck,
   configConsistencyCluster,
)

gv = CliGlobal.CliGlobal( dict( avtCliConfig=None,
                                ipConfig=None,
                                ipStatus=None,
                                dpsIntfConfig=None,
                                dpsIntfStatus=None ) )

tokenPercentage = CliMatcher.FloatMatcher( 0, 100,
                                  helpdesc='percentage',
                                  precisionString='%.2f' )

tokenMilliSecond = CliMatcher.IntegerMatcher( 0, 10000, helpdesc='millisecond' )

avtMatcher = CliMatcher.KeywordMatcher( 'adaptive-virtual-topology',
      helpdesc='Configure adaptive virtual topology' )
roleConfigNameMatcher = CliMatcher.EnumMatcher({
         'transit': 'Transit router profile',
         'edge': 'Edge router profile',
         'pathfinder': 'Pathfinder router profile',
      } )
regionConfigNameMatcher = CliMatcher.DynamicNameMatcher(
      lambda mode: gv.avtCliConfig.region,
      helpdesc='Name of the region',
      helpname='WORD',
      priority=CliParser.PRIO_LOW )
regionConfigIdMatcher = CliMatcher.IntegerMatcher( 1, 255, helpdesc='Region ID' )
zoneConfigNameMatcher = CliMatcher.DynamicNameMatcher(
      lambda mode: gv.avtCliConfig.zone,
      helpdesc='Name of the zone',
      helpname='WORD',
      priority=CliParser.PRIO_LOW )
zoneConfigIdMatcher = CliMatcher.IntegerMatcher( 1, 10000, helpdesc='Zone ID' )
siteConfigNameMatcher = CliMatcher.DynamicNameMatcher(
      lambda mode: gv.avtCliConfig.site,
      helpdesc='Name of the site',
      helpname='WORD',
      priority=CliParser.PRIO_LOW )
siteConfigIdMatcher = CliMatcher.IntegerMatcher( 1, 10000, helpdesc='Site ID' )
profileConfigNameMatcher = CliMatcher.DynamicNameMatcher(
      lambda mode: gv.avtCliConfig.avtProfile,
      helpdesc='Name of the profile',
      helpname='WORD',
      priority=CliParser.PRIO_LOW )
profileIePolicyNameMatcher = CliMatcher.DynamicNameMatcher(
      lambda mode: [],
      helpdesc='Name of the internet exit policy',
      helpname='WORD',
      priority=CliParser.PRIO_LOW )
vrfConfigNameMatcher = CliMatcher.DynamicNameMatcher(
      lambda mode: gv.avtCliConfig.vrfConfig,
      helpdesc='Name of the VRF Configuration',
      helpname='WORD',
      priority=CliParser.PRIO_LOW )
vrfPolicyNameMatcher = CliMatcher.DynamicNameMatcher(
      lambda mode: gv.avtCliConfig.vrfConfig,
      helpdesc='Name of the AVT VRF policy',
      helpname='WORD',
      priority=CliParser.PRIO_LOW )
vrfProfileNameMatcher = CliMatcher.DynamicNameMatcher(
      lambda mode: gv.avtCliConfig.vrfConfig,
      helpdesc='Name of the AVT VRF profile',
      helpname='WORD',
      priority=CliParser.PRIO_LOW )
vrfProfileIdMatcher = CliMatcher.IntegerMatcher(
   1, Tac.Type( "Avt::AvtId" ).maxConfiguredAvtId, helpdesc='AVT ID' )
policyConfigNameMatcher = CliMatcher.DynamicNameMatcher(
      lambda mode: gv.avtCliConfig.policyConfig,
      helpdesc='Name of the policy',
      helpname='WORD',
      priority=CliParser.PRIO_LOW )
policyConfigProfileNamematcher = CliMatcher.DynamicNameMatcher(
      lambda mode: gv.avtCliConfig.policyConfig,
      helpdesc='Name of the profile',
      helpname='WORD',
      priority=CliParser.PRIO_LOW )
policyConfigProfileAvtNameMatcher = CliMatcher.DynamicNameMatcher(
      lambda mode: gv.avtCliConfig.policyConfig,
      helpdesc='Name of the AVT profile',
      helpname='WORD',
      priority=CliParser.PRIO_LOW )
policyConfigTrafficClassMatcher = CliMatcher.IntegerMatcher( 0, 2**16 - 1,
                                          helpdesc='App profile traffic class' )
policyConfigProfileOrderMatcher = CliMatcher.EnumMatcher( {
         'after': 'Insert the previous application profile '
                  'AFTER the following application profile',
         'before': 'Insert the previous application profile '
                  'BEFORE the following application profile',
      } )
policyConfigProfileName2Matcher = CliMatcher.DynamicNameMatcher(
      lambda mode: gv.avtCliConfig.policyConfig,
      helpdesc='Name of the policy',
      helpname='WORD',
      priority=CliParser.PRIO_LOW )

class RouterAdaptiveVirtualTopologyConfigCmd( CliCommand.CliCommandClass ):
   syntax = '''router adaptive-virtual-topology'''
   noOrDefaultSyntax = syntax
   data = {
         'router': routerMatcherForConfig,
         'adaptive-virtual-topology': avtMatcher,
         }

   handler = "AvtCliHandler.RouterAdaptiveVirtualTopologyConfigCmd"
   noOrDefaultHandler = "AvtCliHandler.RouterAdaptiveVirtualTopologyConfigDefaultCmd"

# ------------------------------------------------------------------
#  router adaptive-virtual-topology
#     topology role ( edge | pathfinder | ( transit zone | region ) )
# ------------------------------------------------------------------
class AvtRoleConfigCmd( CliCommand.CliCommandClass ):
   if Toggles.BgpCommonToggleLib.toggleEvpnL3DciWithDpsEnabled():
      syntax = ( 'topology role ( pathfinder | '
                 '( edge | ( transit zone | region ) [ gateway vxlan ] ) )' )
   else:
      syntax = ( 'topology role '
                 '( edge | pathfinder | ( transit zone | region ) )' )
   noOrDefaultSyntax = "topology role ..."
   data = {
         'topology': 'Configure role name',
         'role': 'Configure role name',
         'edge': 'Edge router profile',
         'pathfinder': 'Pathfinder router profile',
         'transit': 'Transit router profile',
         'region': 'Inter region transit router profile',
         'zone': 'Inter zone transit router profile',
         'gateway': 'Gateway router profile',
         'vxlan': 'VXLAN gateway router profile',
         }

   handler = "AvtCliHandler.AvtRoleConfigCmdHandler"
   noOrDefaultHandler = "AvtCliHandler.AvtRoleConfigCmdDefaultHandler"

class AvtRegionConfigCmd( CliCommand.CliCommandClass ):
   syntax = '''region REGION_NAME id REGION_ID'''
   noOrDefaultSyntax = syntax.replace( 'REGION_ID', '...' )
   data = {
         'region': 'Configure region name',
         'REGION_NAME': regionConfigNameMatcher,
         'id': 'Configure region id',
         'REGION_ID': regionConfigIdMatcher
         }

   handler = "AvtCliHandler.AvtRegionConfigCmdHandler"
   noOrDefaultHandler = "AvtCliHandler.AvtRegionConfigCmdDefaultHandler"

class AvtZoneConfigCmd( CliCommand.CliCommandClass ):
   syntax = '''zone ZONE_NAME id ZONE_ID'''
   noOrDefaultSyntax = syntax.replace( 'ZONE_ID', '...' )
   data = {
         'zone': 'Configure zone name',
         'ZONE_NAME': zoneConfigNameMatcher,
         'id': 'Configure zone id',
         'ZONE_ID': zoneConfigIdMatcher
         }

   handler = "AvtCliHandler.AvtZoneConfigCmdHandler"
   noOrDefaultHandler = "AvtCliHandler.AvtZoneConfigCmdDefaultHandler"

class AvtSiteConfigCmd( CliCommand.CliCommandClass ):
   syntax = '''site SITE_NAME id SITE_ID'''
   noOrDefaultSyntax = syntax.replace( 'SITE_ID', '...' )
   data = {
         'site': 'Configure site name',
         'SITE_NAME': siteConfigNameMatcher,
         'id': 'Configure site id',
         'SITE_ID': siteConfigIdMatcher
         }

   handler = "AvtCliHandler.AvtSiteConfigCmdHandler"
   noOrDefaultHandler = "AvtCliHandler.AvtSiteConfigCmdDefaultHandler"

class AvtProfileConfigCmd( CliCommand.CliCommandClass ):
   syntax = '''profile AVT_NAME'''
   noOrDefaultSyntax = syntax
   data = {
         'profile': 'Configure AVT name',
         'AVT_NAME': profileConfigNameMatcher,
         }

   handler = "AvtCliHandler.AvtProfileConfigCmdHandler"
   noOrDefaultHandler = "AvtCliHandler.AvtProfileConfigCmdDefaultHandler"

class AvtProfileLoadBalanceConfigCmd( CliCommand.CliCommandClass ):
   syntax = '''path-selection load-balance LB_POLICY'''
   noOrDefaultSyntax = syntax.replace( 'LB_POLICY', '...' )
   data = {
         'path-selection': 'Configure load balance policy',
         'load-balance': 'Configure load balance policy',
         'LB_POLICY': loadBalanceProfileNameMatcher,
         }

   handler = "AvtCliHandler.AvtProfileLoadBalanceConfigCmdHandler"
   noOrDefaultHandler = "AvtCliHandler.AvtProfileLoadBalanceConfigCmdDefaultHandler"

class AvtMetricOrderConfigCmd( CliCommand.CliCommandClass ):
   syntax = ( ' metric order '
              '( loss-rate | latency | jitter | load )' )
   noOrDefaultSyntax = 'metric order...'
   data = {
         'metric': 'Configure metric for paths',
         'order': 'Configure metric order to be used for path comparison',
         'loss-rate': 'Prefer paths with lowest loss-rate',
         'latency': 'Prefer paths with lowest latency',
         'jitter': 'Prefer paths with lowest jitter',
         'load': 'Prefer paths with lowest load',
   }

   handler = "AvtCliHandler.AvtMetricOrderConfigCmdHandler"
   noOrDefaultHandler = "AvtCliHandler.AvtMetricOrderConfigCmdHandler"

class AvtOutlierEnabledConfigCmd( CliCommand.CliCommandClass ):
   syntax = '''path-selection outlier elimination [ disabled ]'''
   noOrDefaultSyntax = syntax
   data = {
         'path-selection': 'Configure load balance policy',
         'outlier': 'Configure avt path outlier',
         'elimination': 'Configure avt path outlier elimination',
         'disabled': 'Disable avt path outlier elimination',
   }

   handler = "AvtCliHandler.AvtOutlierEnabledConfigCmdHandler"
   noOrDefaultHandler = "AvtCliHandler.AvtOutlierEnabledConfigCmdDefaultHandler"

class AvtOutlierLatencyThresholdConfigCmd( CliCommand.CliCommandClass ):
   syntax = ( 'path-selection outlier elimination threshold '
              'latency <latency> milliseconds' )
   noOrDefaultSyntax = 'path-selection outlier elimination threshold latency...'
   data = {
         'path-selection': 'Configure load balance policy',
         'outlier': 'Configure avt path outlier',
         'elimination': 'Configure avt path outlier elimination',
         'threshold': 'Change the threshold values for path comparison',
         'latency': 'Configure latency threshold',
         '<latency>': tokenMilliSecond,
         'milliseconds': 'Configure latency in milliseconds',
   }

   handler = "AvtCliHandler.AvtOutlierLatencyThresholdConfigCmdHandler"
   noOrDefaultHandler = \
      "AvtCliHandler.AvtOutlierLatencyThresholdConfigCmdDefaultHandler"

class AvtOutlierJitterThresholdConfigCmd( CliCommand.CliCommandClass ):
   syntax = ( 'path-selection outlier elimination threshold '
              'jitter <jitter> milliseconds' )
   noOrDefaultSyntax = 'path-selection outlier elimination threshold jitter...'
   data = {
         'path-selection': 'Configure load balance policy',
         'outlier': 'Configure avt path outlier',
         'elimination': 'Configure avt path outlier elimination',
         'threshold': 'Change the threshold values for path comparison',
         'jitter': 'Configure jitter threshold',
         '<jitter>': tokenMilliSecond,
         'milliseconds': 'Configure jitter in milliseconds',
   }

   handler = "AvtCliHandler.AvtOutlierJitterThresholdConfigCmdHandler"
   noOrDefaultHandler = \
      "AvtCliHandler.AvtOutlierJitterThresholdConfigCmdDefaultHandler"

class AvtOutlierLossrateThresholdConfigCmd( CliCommand.CliCommandClass ):
   syntax = ( 'path-selection outlier elimination threshold '
              'loss-rate <percentage> percent' )
   noOrDefaultSyntax = 'path-selection outlier elimination threshold loss-rate...'
   data = {
         'path-selection': 'Configure load balance policy',
         'outlier': 'Configure avt path outlier',
         'elimination': 'Configure avt path outlier elimination',
         'threshold': 'Change the threshold values for path comparison',
         'loss-rate': 'Configure loss-rate threshold',
         '<percentage>': tokenPercentage,
         'percent': 'Configure loss-rate in percent',
   }

   handler = "AvtCliHandler.AvtOutlierLossrateThresholdConfigCmdHandler"
   noOrDefaultHandler = \
      "AvtCliHandler.AvtOutlierLossrateThresholdConfigCmdDefaultHandler"

class AvtOutlierLoadThresholdConfigCmd( CliCommand.CliCommandClass ):
   syntax = ( 'path-selection outlier elimination threshold '
              'load <percentage> percent' )
   noOrDefaultSyntax = 'path-selection outlier elimination threshold load...'
   data = {
         'path-selection': 'Configure load balance policy',
         'outlier': 'Configure avt path outlier',
         'elimination': 'Configure avt path outlier elimination',
         'threshold': 'Change the threshold values for path comparison',
         'load': 'Configure load threshold',
         '<percentage>': tokenPercentage,
         'percent': 'Configure loss-rate in percent',
   }

   handler = "AvtCliHandler.AvtOutlierLoadThresholdConfigCmdHandler"
   noOrDefaultHandler = \
      "AvtCliHandler.AvtOutlierLoadThresholdConfigCmdDefaultHandler"


class AvtProfileInternetExitConfigCmd( CliCommand.CliCommandClass ):
   syntax = '''internet-exit policy IE_POLICY'''
   noOrDefaultSyntax = syntax.replace( 'IE_POLICY', '...' )
   data = {
         'internet-exit': 'Configure internet exit policy',
         'policy': 'Configure internet exit policy',
         'IE_POLICY': profileIePolicyNameMatcher,
         }

   handler = "AvtCliHandler.AvtProfileInternetExitConfigCmdHandler"
   noOrDefaultHandler = "AvtCliHandler.AvtProfileInternetExitConfigCmdDefaultHandler"

class AvtPolicyConfigCmd( CliCommand.CliCommandClass ):
   syntax = '''policy POLICY_NAME'''
   noOrDefaultSyntax = syntax
   data = {
         'policy': 'Configure policy name',
         'POLICY_NAME': policyConfigNameMatcher,
         }

   handler = "AvtCliHandler.AvtPolicyConfigCmdHandler"
   noOrDefaultHandler = "AvtCliHandler.AvtPolicyConfigCmdDefaultHandler"

class AvtPolicyAppProfileConfigCmd( CliCommand.CliCommandClass ):
   syntax = '''match application-profile PROFILE_NAME''' \
            ''' [ORDER PROFILE_NAME_2]'''
   noOrDefaultSyntax = '''match application-profile PROFILE_NAME'''
   data = {
         'match': 'Configure profile name',
         'application-profile': 'Configure profile name',
         'PROFILE_NAME': policyConfigProfileNamematcher,
         'ORDER': policyConfigProfileOrderMatcher,
         'PROFILE_NAME_2': policyConfigProfileName2Matcher,
         }

   handler = "AvtCliHandler.AvtPolicyAppProfileConfigCmdHandler"
   noOrDefaultHandler = "AvtCliHandler.AvtPolicyAppProfileConfigCmdDefaultHandler"

class AvtPolicyAppProfileActionNameConfigCmd( CliCommand.CliCommandClass ):
   syntax = '''avt profile AVT_NAME'''
   noOrDefaultSyntax = syntax.replace( 'AVT_NAME', '...' )
   data = {
         'avt': 'Configure AVT feature',
         'profile': 'Configure AVT profile name',
         'AVT_NAME': policyConfigProfileAvtNameMatcher,
         }

   handler = "AvtCliHandler.AvtPolicyAppProfileActionNameConfigCmdHandler"
   noOrDefaultHandler = \
     "AvtCliHandler.AvtPolicyAppProfileActionNameConfigCmdDefaultHandler"

class AvtPolicyAppProfileTrafficClassConfigCmd( CliCommand.CliCommandClass ):
   syntax = '''traffic-class TRAFFIC_CLASS_VALUE'''
   noOrDefaultSyntax = syntax.replace( 'TRAFFIC_CLASS_VALUE', '...' )
   data = {
         'traffic-class': matcherTrafficClass,
         'TRAFFIC_CLASS_VALUE': matcherTrafficClassNum
         }

   handler = "AvtCliHandler.AvtPolicyAppProfileTrafficClassConfigCmdHandler"
   noOrDefaultHandler = handler

class AvtPolicyAppProfileDscpConfigCmd( CliCommand.CliCommandClass ):
   syntax = '''dscp DSCP_VALUE'''
   noOrDefaultSyntax = syntax.replace( 'DSCP_VALUE', '...' )
   data = {
         'dscp': matcherDscp,
         'DSCP_VALUE': matcherDscpValue
         }

   handler = "AvtCliHandler.AvtPolicyAppProfileDscpConfigCmdHandler"
   noOrDefaultHandler = handler

class AvtVrfConfigCmd( CliCommand.CliCommandClass ):
   syntax = '''vrf VRF_NAME'''
   noOrDefaultSyntax = syntax
   data = {
         'vrf': 'Configure VRF name',
         'VRF_NAME': vrfConfigNameMatcher,
         }

   handler = "AvtCliHandler.AvtVrfConfigCmdHandler"
   noOrDefaultHandler = "AvtCliHandler.AvtVrfConfigCmdDefaultHandler"

class AvtVrfPolicyConfigCmd( CliCommand.CliCommandClass ):
   syntax = '''avt policy AVT_POLICY_NAME'''
   noOrDefaultSyntax = syntax.replace( 'AVT_POLICY_NAME', '...' )
   data = {
         'avt': 'Configure AVT feature',
         'policy': 'Configure AVT policy name',
         'AVT_POLICY_NAME': vrfPolicyNameMatcher,
         }

   handler = "AvtCliHandler.AvtVrfPolicyConfigCmdHandler"
   noOrDefaultHandler = "AvtCliHandler.AvtVrfPolicyConfigCmdDefaultHandler"

class AvtVrfProfileConfigCmd( CliCommand.CliCommandClass ):
   syntax = '''avt profile PROFILE_NAME id ID'''
   noOrDefaultSyntax = syntax.replace( 'ID', '...' )
   data = {
         'avt': 'Configure AVT feature',
         'profile': 'Specify AVT profile name to be mapped to AVT ID',
         'PROFILE_NAME': vrfProfileNameMatcher,
         'id': 'Specify AVT ID to be mapped from AVT profile name',
         'ID': vrfProfileIdMatcher,
         }

   handler = "AvtCliHandler.AvtVrfProfileConfigCmdHandler"
   noOrDefaultHandler = "AvtCliHandler.AvtVrfProfileConfigCmdDefaultHandler"

@configConsistencyCluster( cluster='adaptive-virtual-topology',
                           desc='Check potentially bad configs for AVT' )
def cfCsstAvtCluster( mode, context=None ):
   return dpsSupported()

@configConsistencyCategory( cluster='adaptive-virtual-topology', category='',
                            desc='Check potentially bad configs for AVT' )
def cfCsstAvtCategory( mode, context=None ):
   return dpsSupported()

@configConsistencyCheck( cluster='adaptive-virtual-topology', category='' )
def dpsIntfConfiguredCheck( model, mode, args ):
   builder = ConfigConsistencyMultiTableModelBuilder( model )
   tableName = 'Local DPS Interface Configuration'
   builder.addTable(
         tableName,
         headings=[ 'Category', 'Status', 'Description' ] )
   dpsIntfConfig = gv.dpsIntfConfig.intfConfig.get( 'Dps1' )
   if not dpsIntfConfig:
      builder.addItem( tableName, 'Dps1',
                       status='fail',
                       description="Dps1 interface is not configured" )
      return
   dpsIpIntfConfig = gv.ipConfig.ipIntfConfig.get( 'Dps1' )
   if not dpsIpIntfConfig or dpsIpIntfConfig.addrWithMask.address == '0.0.0.0':
      builder.addItem( tableName, 'Dps1',
                       status='fail',
                       description="Dps1 interface ip not configured" )
      return

# Commands for define AVT general attributes
BasicCli.GlobalConfigMode \
         .addCommandClass( RouterAdaptiveVirtualTopologyConfigCmd )
RouterAdaptiveVirtualTopologyConfigMode.addCommandClass( AvtRegionConfigCmd )
RouterAdaptiveVirtualTopologyConfigMode.addCommandClass( AvtZoneConfigCmd )
RouterAdaptiveVirtualTopologyConfigMode.addCommandClass( AvtSiteConfigCmd )
RouterAdaptiveVirtualTopologyConfigMode.addCommandClass( AvtRoleConfigCmd )

# Commands for define AVT Profile
RouterAdaptiveVirtualTopologyConfigMode.addCommandClass( AvtProfileConfigCmd )
RouterAvtProfileConfigMode.addCommandClass( AvtProfileLoadBalanceConfigCmd )
RouterAvtProfileConfigMode.addCommandClass( AvtProfileInternetExitConfigCmd )

# Commands for define AVT Policy
RouterAdaptiveVirtualTopologyConfigMode.addCommandClass( AvtPolicyConfigCmd )
RouterAvtPolicyConfigMode.addCommandClass( AvtPolicyAppProfileConfigCmd )
RouterAvtPolicyAppProfileConfigMode \
      .addCommandClass( AvtPolicyAppProfileActionNameConfigCmd )
if Toggles.DpsToggleLib.toggleAvtPolicyTrafficClassEnabled():
   RouterAvtPolicyAppProfileConfigMode\
              .addCommandClass( AvtPolicyAppProfileTrafficClassConfigCmd )
   RouterAvtPolicyAppProfileConfigMode\
              .addCommandClass( AvtPolicyAppProfileDscpConfigCmd )

# Comands for define VRT Config
RouterAdaptiveVirtualTopologyConfigMode.addCommandClass( AvtVrfConfigCmd )
RouterAvtVrfConfigMode.addCommandClass( AvtVrfPolicyConfigCmd )
RouterAvtVrfConfigMode.addCommandClass( AvtVrfProfileConfigCmd )

if toggleAvtPathOutlierEliminationEnabled():
   RouterAvtProfileConfigMode.addCommandClass( AvtMetricOrderConfigCmd )

if toggleAvtPathOutlierEliminationEnabled():
   RouterAvtProfileConfigMode.addCommandClass( AvtOutlierEnabledConfigCmd )
   RouterAvtProfileConfigMode.addCommandClass( AvtOutlierLatencyThresholdConfigCmd )
   RouterAvtProfileConfigMode.addCommandClass( AvtOutlierJitterThresholdConfigCmd )
   RouterAvtProfileConfigMode.addCommandClass( AvtOutlierLossrateThresholdConfigCmd )
   if toggleAvtLowestLoadMetricEnabled():
      RouterAvtProfileConfigMode.addCommandClass( AvtOutlierLoadThresholdConfigCmd )

def Plugin( em ):
   gv.avtCliConfig = ConfigMount.mount(
         em, 'avt/input/cli',
         'Avt::AvtCliConfig', 'w' )
   gv.ipConfig = LazyMount.mount( em, "ip/config", "Ip::Config", "r" )
   gv.ipStatus = LazyMount.mount( em, "ip/status", "Ip::Status", "r" )
   gv.dpsIntfConfig = LazyMount.mount( em, "interface/config/dps/intf",
                                       "Interface::DpsIntfConfigDir", "r" )
   gv.dpsIntfStatus = LazyMount.mount( em, "interface/status/dps/intf",
                                       "Interface::DpsIntfStatusDir", "r" )
