#!/usr/bin/env python3
# Copyright (c) 2019 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import BasicCliModes
import CliCommand
import CliGlobal
import CliMatcher
import CliPlugin.AclCli as AclCli # pylint: disable=consider-using-from-import
from CliPlugin.Ip6AddrMatcher import Ip6AddrMatcher
from CliPlugin.IpAddrMatcher import IpAddrMatcher
from CliToken.Bfd import matcherBfdForClear
from CliToken.Clear import clearKwNode
from CliToken.Ip import ipMatcherForClear
from CliToken.Ipv6 import ipv6MatcherForClear
import ConfigMount
import LazyMount
import SmashLazyMount
import Smash
import Tac

# Global variable holder.
gv = CliGlobal.CliGlobal(
   dict(
      bfdHwStatus=None,
      bfdStatusPeer=None,
      bfdPeerHistory=None,
      hwStatusCli=None,
      hwConfigCli=None,
      vrfIdMap=None,
      bfdConfigInfo=None,
      aclStatus=None,
      aclCheckpoint=None,
   )
)

matcherCounters = CliMatcher.KeywordMatcher( 'counters',
      helpdesc='Clear BFD counters and statistics in all sessions' )

# --------------------------------------------------------------------------------
# clear bfd counters [ dest-ip ( DESTIP | DESTIP6 ) ]
# --------------------------------------------------------------------------------
def clearBfdCounters( mode, args ):
   netlinkStub = Tac.newInstance( "BfdPyUtils::BfdNetlinkControlStub" )
   netlinkStub.createFd( False )

   hwCommand = None

   # Extract the destination ip address from the passed args
   destIpAddr = None
   if 'dest-ip' in args:
      if 'DESTIP' in args:
         destIpAddr = args[ 'DESTIP' ]
      elif 'DESTIP6' in args:
         destIpAddr = args[ 'DESTIP6' ]

   if destIpAddr is None:
      # We clear stats for all peers.
      discriminator = 0
      module = True
      if gv.bfdHwStatus.hwAccelerationEnabled:
         hwCommand = Tac.Value( "Bfd::BfdSessionId" )

   else:
      # Only clear for the indicated peer
      module = False
      ip = Tac.Value( 'Arnet::IpGenAddr', destIpAddr )

      peers = [ gv.bfdStatusPeer.peerStatus[ peer ] for peer in
                gv.bfdStatusPeer.peerStatus if peer.ip == ip ]

      if not peers:
         # pylint: disable-next=consider-using-f-string
         print( "%s is an invalid BFD neighbor" % destIpAddr )
         return
      elif len( peers ) > 1:
         # pylint: disable-next=consider-using-f-string
         print( "%s is ambiguous" % destIpAddr )
         return
      else:
         discriminator = peers[ 0 ].localDisc

      if gv.bfdHwStatus.isHwAccelerated( discriminator ):
         for vrfId, entry in gv.vrfIdMap.vrfIdToName.items():
            if peers[ 0 ].peer.vrf == entry.vrfName:
               hwCommand = Tac.newInstance( "Bfd::BfdSessionId",
                                            vrfId,
                                            discriminator )
               break

   netlinkStub.sendGetStats( discriminator, True, True, module )

   # Cleanup previous commands that were processed by PlatformBfd
   if gv.bfdHwStatus.hwAccelerationEnabled:
      statusId = gv.hwStatusCli.clearStatsCommandId
      for commandId in gv.hwConfigCli.clearStatsCommand:
         if 0 < ( statusId - commandId ) & 0xffffffff < 0x7fffffff:
            del gv.hwConfigCli.clearStatsCommand[ commandId ]

   if hwCommand is not None:
      currentId = gv.hwConfigCli.clearStatsCommandId
      gv.hwConfigCli.clearStatsCommand[ currentId ] = hwCommand
      gv.hwConfigCli.clearStatsCommandId = ( currentId + 1 ) & 0xffffffff

class ClearBfdCountersCmd( CliCommand.CliCommandClass ):
   syntax = 'clear bfd counters [ dest-ip ( DESTIP | DESTIP6 ) ]'
   data = {
      'clear': clearKwNode,
      'bfd': matcherBfdForClear,
      'counters': matcherCounters,
      'dest-ip': 'BFD neighbor IP address',
      'DESTIP': IpAddrMatcher( helpdesc='IPv4 Neighbor address' ),
      'DESTIP6': Ip6AddrMatcher( helpdesc='IPv6 Neighbor address' ),
   }
   handler = clearBfdCounters

BasicCliModes.EnableMode.addCommandClass( ClearBfdCountersCmd )

# --------------------------------------------------------------------------------
# clear bfd history
# --------------------------------------------------------------------------------
def clearBfdHistory( mode, args ):
   if gv.bfdPeerHistory.historyEntries:
      gv.bfdConfigInfo.historyGeneration += 1

class ClearBfdHistoryCmd( CliCommand.CliCommandClass ):
   syntax = 'clear bfd history'
   data = {
      'clear': clearKwNode,
      'bfd': matcherBfdForClear,
      'history': 'Clear BFD history',
   }
   handler = clearBfdHistory

BasicCliModes.EnableMode.addCommandClass( ClearBfdHistoryCmd )

# --------------------------------------------------------------------------------
# clear bfd ip/ipv6 access-list counters
# --------------------------------------------------------------------------------
def clearAclCounters( mode, args ):
   aclType = 'ip' if 'ip' in args else 'ipv6'
   AclCli.clearServiceAclCounters( mode, gv.aclStatus, gv.aclCheckpoint, aclType )

class ClearBfdAccessListCountersCmd( CliCommand.CliCommandClass ):
   syntax = 'clear bfd ( ip | ipv6 ) access-list counters'
   data = {
      'clear': clearKwNode,
      'bfd': matcherBfdForClear,
      'ip': ipMatcherForClear,
      'ipv6': ipv6MatcherForClear,
      'access-list': 'Named access-list',
      'counters': matcherCounters,
   }
   handler = clearAclCounters

BasicCliModes.EnableMode.addCommandClass( ClearBfdAccessListCountersCmd )

def Plugin( entityManager ):
   gv.bfdHwStatus = LazyMount.mount( entityManager, 'bfd/hwStatus',
                                     'Hardware::Bfd::Status', 'r' )
   gv.bfdStatusPeer = LazyMount.mount( entityManager, 'bfd/status/peer',
                                       'Bfd::StatusPeer', 'r' )
   gv.bfdPeerHistory = LazyMount.mount( entityManager, 'bfd/status/history',
                                        'Bfd::PeerHistory', 'r' )
   gv.hwStatusCli = LazyMount.mount( entityManager, "bfd/hwStatusCli",
                                     'Bfd::HwStatusCli', 'r' )
   gv.hwConfigCli = ConfigMount.mount( entityManager, 'bfd/hwConfigCli',
                                       'Bfd::HwConfigCli', 'w' )
   gv.vrfIdMap = SmashLazyMount.mount( entityManager, 'vrf/vrfIdMapStatus',
                                       'Vrf::VrfIdMap::Status',
                                       Smash.mountInfo( 'reader' ),
                                       autoUnmount=True )
   gv.bfdConfigInfo = LazyMount.mount( entityManager, 'bfd/config/info',
                                       'Bfd::ConfigInfo', 'w' )
   gv.aclStatus = LazyMount.mount( entityManager, 'acl/status/all',
                                   'Acl::Status', 'r' )
   gv.aclCheckpoint = LazyMount.mount( entityManager, 'acl/checkpoint',
                                       'Acl::CheckpointStatus', 'w' )
