#!/usr/bin/env python3
# Copyright (c) 2019 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# pylint: disable=consider-using-from-import
# pylint: disable=consider-using-f-string

from socket import IPPROTO_UDP
import errno
import functools
import BasicCliModes
import CliCommand
import CliGlobal
import CliMatcher
import AclCliLib
import CliPlugin
import CliPlugin.AclCli as AclCli
import CliPlugin.BfdCli as BfdCli
from CliPlugin.BfdIntfConfigMode import BfdIntervalConfig
from CliPlugin.IpGenAddrMatcher import IpGenAddrMatcher
from CliPlugin.IpAddrMatcher import IpAddrMatcher
from CliPlugin.MacAddr import MacAddrMatcher
from CliPlugin.VlanCli import bridgingSupportedGuard
from CliPlugin.VrfCli import (
      ALL_VRF_NAME,
      DEFAULT_VRF,
      getAllVrfNames,
      vrfExists,
)
import CliPlugin.ConfigConvert
import CliPlugin.VlanIntfCli as VlanIntfCli
import CliPlugin.EthIntfCli as EthIntfCli
import CliPlugin.SubIntfCli as SubIntfCli
import CliPlugin.IraVrfCli as IraVrfCli
import CliPlugin.LoopbackIntfCli as LoopbackIntfCli
from CliPlugin.VirtualIntfRule import IntfMatcher
from CliPlugin import SwitchIntfCli
from CliToken.Bfd import (
      deprecatedData,
      matcherBfd,
      matcherDangerous,
      matcherDangerousStatsInt,
      matcherDebug,
      matcherDebugLevel,
      matcherDiscriminator,
      matcherInitiator,
      matcherInterval,
      matcherMeticulous,
      matcherMinRx,
      matcherMultihop,
      matcherMultiplier,
      matcherMultiplierVal,
      matcherOuterDstPort,
      matcherOuterSrcPort,
      matcherSession,
      matcherSlowTimer,
      matcherStatsInt,
      matcherTxRxIntervalMs,
      matcherVNI,
      matcherVlan,
)
from CliToken.Router import routerMatcherForConfig as matcherRouter
import ConfigMount
import Tac
from TypeFuture import TacLazyType
from CliMode.Bfd import (
      RoutingBfdMode,
      RoutingBfdPeerMode,
      RoutingBfdSbfdMode,
      RoutingBfdVrfMode,
      RoutingBfdVrfPeerMode,
)
import BasicCli
import AclLib
import Ethernet
import Arnet
from BfdLib import (
      VxlanPeerConfig,
      getVxlanValidFlags,
      initiatorConfigDefault,
      reflectorMinRxDefault,
      sbfdLocalIntfDefault,
)
from Toggles.BfdToggleLib import (
   toggleBfdEchoL3SrcAddrEnabled,
   toggleBfdDscpConfigEnabled )

# Global variable holder.
gv = CliGlobal.CliGlobal(
   dict(
      bfdConfigGlobal=None,
      testAppConfigDir=None,
      testVxlanAppConfig=None,
      aclConfig=None,
      aclCpConfig=None,
      bfdConfigPeerCli=None,
   )
)

DscpValue = TacLazyType( 'Arnet::DscpValue' )
authType = Tac.Type( 'Bfd::BfdAuthType' )
sessionType = Tac.Type( "Bfd::SessionType" )
BfdRole = Tac.Type( "Bfd::BfdRole" )
SessionStatsInterval = TacLazyType( 'Bfd::SessionStatsInterval' )
BfdConstants = TacLazyType( 'Bfd::Constants' )

bfdPorts = []
for s in [ 'bfd', 'bfd-echo', 'multihop-bfd', 'micro-bfd' ]:
   bfdPorts.append( AclLib.getServByName( IPPROTO_UDP, s ) )

nodeVlan = CliCommand.Node( matcher=matcherVlan, guard=bridgingSupportedGuard )
intfMatcher = IntfMatcher()
intfMatcher |= VlanIntfCli.VlanIntf.matcher
intfMatcher |= EthIntfCli.EthPhyIntf.ethMatcher
intfMatcher |= SwitchIntfCli.SwitchIntf.matcher
intfMatcher |= SubIntfCli.subMatcher
intfMatcher |= LoopbackIntfCli.LoopbackIntf.matcher
tunnelIdMax = ( 2 ** 64 - 1 )

# -------------------------------------------------------------------------------
# 'router bfd' configuration mode.  A new instance of this mode is created when
# the user enters 'router bfd' at the Cli.
# -------------------------------------------------------------------------------
def routerBfdHandler( mode, args ):
   childMode = mode.childMode( RouterBfdMode )
   mode.session_.gotoChildMode( childMode )
   gv.bfdConfigGlobal.legacyConfig = False

def noRouterBfdHandler( mode, args ):
   gv.bfdConfigGlobal.echoSrcIpIntf = Tac.Value( 'Arnet::IntfId', '' )
   gv.bfdConfigGlobal.hwAccelerationConfig = 'platformDefault'
   gv.bfdConfigGlobal.sessionStatsInterval = SessionStatsInterval.defval
   gv.bfdConfigGlobal.dscpValue = BfdConstants.defaultIpDscp
   # Clean up parameters... set to default
   resetGlobalBfdParams( mode, None )
   # Clean up all Bfd service Acl configuration
   for aclType in ( 'ip', 'ipv6' ):
      serviceAcl = gv.aclCpConfig.cpConfig[ aclType ].serviceAcl
      for vrfName, serviceAclVrfConfig in serviceAcl.items():
         serviceAclConfig = serviceAclVrfConfig.service.get( 'bfd' )
         if serviceAclConfig:
            setServiceAcl( mode, aclType, no=True,
                           aclName=serviceAclConfig.aclName, vrfName=vrfName )
   localAddrNoFunctionCmd( mode, {} )
   resetMultihopGlobalBfdParams( mode, {} )
   noAdminDown( mode, {} )
   resetSlowTimer( mode, {} )
   resetSbfd( mode, {} )

   # Clean up debug commands
   noDebugLevel( mode, {} )
   setDebugSeqNum( mode, noOrDefault=True )

   # Clean up any test static sessions
   testCfgDir = ConfigMount.force( gv.testAppConfigDir )
   for vrf in getAllVrfNames( mode ):
      testCfg = getBfdTestAppVrfAppConfig( testCfgDir, vrf )
      if testCfg is None:
         mode.addError( 'Unable to mount test AppConfig!' )
         return
      else:
         for p in testCfg.peerConfigVer:
            BfdCli.delPeer( p, testCfg )

   vxlanCfg = ConfigMount.force( gv.testVxlanAppConfig )
   if vxlanCfg is None:
      mode.addError( 'Unable to mount test vxlan AppConfig!' )
      return
   else:
      for p in vxlanCfg.peerConfigVer:
         delVxlanPeer( p, vxlanCfg )

class RouterBfdMode( RoutingBfdMode, BasicCli.ConfigModeBase ):
   name = 'BFD global configuration'

   def __init__( self, parent, session ):
      RoutingBfdMode.__init__( self, True )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

class EnterRouterBfdMode( CliCommand.CliCommandClass ):
   syntax = 'router bfd'
   noOrDefaultSyntax = syntax

   data = {
            'router': matcherRouter,
            'bfd': 'Bidirectional Forwarding Detection',
          }
   handler = routerBfdHandler
   noOrDefaultHandler = noRouterBfdHandler

BasicCli.GlobalConfigMode.addCommandClass( EnterRouterBfdMode )

# ------------------------------------------------------------------------------
# Register handler to convert Bfd global config to new 'rotuer bfd' context via
# 'config convert new-syntax' command
# ------------------------------------------------------------------------------
def convertLegacyConfigBfd( mode ):
   gv.bfdConfigGlobal.legacyConfig = False

CliPlugin.ConfigConvert.registerConfigConvertCallback( convertLegacyConfigBfd )

# Helper methods for configuring Bfd peers in router-bfd-peer and router-bfd-vrf-peer
# modes.
def BfdPeerCliConfigKey( ipAddr, vrfName=DEFAULT_VRF ):
   if isinstance( ipAddr, str ):
      ipAddr = Tac.Value( 'Arnet::IpGenAddr', '%s' % ipAddr )
   if vrfName != DEFAULT_VRF:
      vrf = Tac.Value( 'L3::VrfName', vrfName )
   else:
      # Default vrf, which is the default value of the vrfName kwarg.
      vrf = vrfName
   return Tac.Value( 'Bfd::PeerCliConfigKey', ipAddr, vrf )

def createPeerCliConfig( peerAddr, vrfName=DEFAULT_VRF ):
   peerCliConfigKey = BfdPeerCliConfigKey( peerAddr, vrfName )
   return gv.bfdConfigPeerCli.peerCliConfig.newMember( peerCliConfigKey )

# -------------------------------------------------------------------------------
# router bfd
#    vrf VRFNAME
#
# Bfd vrf config sub-mode.  A new instance of this
# mode is created when the user enters the above command in router-bfd mode at the
# Cli.
# -------------------------------------------------------------------------------
class RouterBfdVrfMode( RoutingBfdVrfMode, BasicCli.ConfigModeBase ):
   name = 'BFD VRF Configuration'

   def __init__( self, parent, session, vrfName ):
      self.vrfName = vrfName
      RoutingBfdVrfMode.__init__( self, vrfName )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

class EnterRouterBfdVrfMode( CliCommand.CliCommandClass ):
   syntax = 'vrf VRF'
   noOrDefaultSyntax = syntax

   data = {
      'vrf': 'Specify VRF encapsulating peers',
      'VRF': BfdCli.vrfNameExprFactory,
   }

   @staticmethod
   def handler( mode, args ):
      vrf = args[ 'VRF' ]
      if not vrfExists( vrf ):
         mode.addWarning( "VRF %s not configured." % vrf )

      childMode = mode.childMode( RouterBfdVrfMode,
                                  vrfName=vrf )
      mode.session_.gotoChildMode( childMode )

      # Create an entry for this vrf in configPeerCli.peerCliConfig.  We will use
      # 0.0.0.0 as the placeholder peer Ip for now, as this is not a valid source Ip
      # address.
      createPeerCliConfig( '0.0.0.0', childMode.vrfName )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      # If there are peers configured underneath this vrf, we need to clean them up.
      for p in gv.bfdConfigPeerCli.peerCliConfig:
         if p.vrf == args[ 'VRF' ]:
            del gv.bfdConfigPeerCli.peerCliConfig[ p ]

RouterBfdMode.addCommandClass( EnterRouterBfdVrfMode )

# -------------------------------------------------------------------------------
# router bfd
#    peer ( IPADDR | IP6ADDR )
#
# Bfd peer config sub-mode.  This submode may be entered either from the bfd-config
# context, or the bfd-vrf-config context.
# -------------------------------------------------------------------------------
class RouterBfdPeerMode( RoutingBfdPeerMode, BasicCli.ConfigModeBase ):
   name = 'BFD Peer-Specific Configuration'

   def __init__( self, parent, session, peerAddr ):
      self.peerAddr = peerAddr
      RoutingBfdPeerMode.__init__( self, peerAddr )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

class EnterRouterBfdPeerMode( CliCommand.CliCommandClass ):
   # Peer configuration within the default vrf.
   syntax = 'peer PEERADDR'
   noOrDefaultSyntax = syntax

   data = {
      'peer': 'Specify peer of BFD session',
      'PEERADDR': IpGenAddrMatcher( helpdesc='Peer IP/IPv6 address' ),
   }

   @staticmethod
   def handler( mode, args ):
      address = args[ 'PEERADDR' ]
      childMode = mode.childMode( RouterBfdPeerMode,
                                  peerAddr=address )
      mode.session_.gotoChildMode( childMode )
      createPeerCliConfig( childMode.peerAddr, childMode.vrfName )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      address = args[ 'PEERADDR' ]
      peerCliConfigKey = BfdPeerCliConfigKey( address, DEFAULT_VRF )
      del gv.bfdConfigPeerCli.peerCliConfig[ peerCliConfigKey ]

RouterBfdMode.addCommandClass( EnterRouterBfdPeerMode )

class RouterBfdVrfPeerMode( RoutingBfdVrfPeerMode, BasicCli.ConfigModeBase ):
   name = 'BFD VRF/Peer-Specific Configuration'

   def __init__( self, parent, session, peerAddr, vrfName ):
      self.peerAddr = peerAddr
      self.vrfName = vrfName
      RoutingBfdVrfPeerMode.__init__( self, ( vrfName, peerAddr ) )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

class EnterRouterBfdVrfPeerMode( CliCommand.CliCommandClass ):
   # Peer configuration with a non-default vrf.  The vrf is inherited from the parent
   # mode, RouterBfdVrfMode.
   syntax = 'peer PEERADDR'
   noOrDefaultSyntax = syntax

   data = {
      'peer': 'Specify peer of BFD session',
      'PEERADDR': IpGenAddrMatcher( helpdesc='Peer IP/IPv6 address' ),
   }

   @staticmethod
   def handler( mode, args ):
      address = args[ 'PEERADDR' ]
      childMode = mode.childMode( RouterBfdVrfPeerMode,
                                  peerAddr=address,
                                  vrfName=mode.vrfName )
      mode.session_.gotoChildMode( childMode )

      # Replace the placeholder '0.0.0.0' entry which may have been created by
      # entering the parent vrf mode.
      searchPeer = BfdPeerCliConfigKey( '0.0.0.0', mode.vrfName )
      if searchPeer in gv.bfdConfigPeerCli.peerCliConfig:
         del gv.bfdConfigPeerCli.peerCliConfig[ searchPeer ]

      createPeerCliConfig( childMode.peerAddr, childMode.vrfName )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      address = args[ 'PEERADDR' ]
      peerCliConfigKey = BfdPeerCliConfigKey( address, mode.vrfName )
      del gv.bfdConfigPeerCli.peerCliConfig[ peerCliConfigKey ]

RouterBfdVrfMode.addCommandClass( EnterRouterBfdVrfPeerMode )

# --------------------------------------------------------------------------------
# router bfd
#    sbfd
#
# Bfd sbfd config sub-mode.  A new instance of this mode is created
# when the user enters the above command in router-bfd mode at the Cli.
# --------------------------------------------------------------------------------
def enterMode( mode, args ):
   childMode = mode.childMode( RouterBfdSbfdMode )
   mode.session_.gotoChildMode( childMode )

def resetSbfd( mode, args ):
   # Clean up SBFD related configuration
   gv.bfdConfigGlobal.sbfdReflectorLocalDisc = 0
   gv.bfdConfigGlobal.sbfdReflectorMinRx = reflectorMinRxDefault()
   gv.bfdConfigGlobal.sbfdInitiatorConfig = initiatorConfigDefault()
   gv.bfdConfigGlobal.sbfdLocalIntf = sbfdLocalIntfDefault()
   gv.bfdConfigGlobal.sbfdLocalIntfIp6 = sbfdLocalIntfDefault()
   gv.bfdConfigGlobal.sbfdDiscIsU32 = False
   gv.bfdConfigGlobal.sbfdRttEnabled = False
   gv.bfdConfigGlobal.authType = authType.authNone
   gv.bfdConfigGlobal.secretProfileName = ''
   gv.bfdConfigPeerCli.peerCliConfig.clear()

class RouterBfdSbfdMode( RoutingBfdSbfdMode, BasicCli.ConfigModeBase ):
   name = 'BFD SBfd Configuration'

   def __init__( self, parent, session ):
      RoutingBfdSbfdMode.__init__( self, 'SBfdConfig' )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

class EnterRouterBfdSbfdMode( CliCommand.CliCommandClass ):
   syntax = 'sbfd'
   noOrDefaultSyntax = syntax

   data = {
            'sbfd': 'Seamless BFD',
          }

   handler = enterMode
   noOrDefaultHandler = resetSbfd

RouterBfdMode.addCommandClass( EnterRouterBfdSbfdMode )

# --------------------------------------------------------------------------------
# [ no | default ] bfd debug ( rbfd | ( disc DISC ) ) DEBUGLVL
#
# router bfd
#    debug ( rbfd | ( disc DISC ) ) DEBUGLVL
# --------------------------------------------------------------------------------
def setDebugLevel( mode, debugLevel=1 ):
   gv.bfdConfigGlobal.debugLevel = debugLevel

def setSessionDebug( mode, discriminator, debugLevel=0 ):
   netlinkStub = Tac.newInstance( "BfdPyUtils::BfdNetlinkControlStub" )
   netlinkStub.createFd( False )
   netlinkStub.sendSessionDebugLevel( discriminator, False, debugLevel )
   netlinkStub.bfdRead()

   if netlinkStub.errCode == -errno.ENOSR:
      mode.addError( "Maximum number of session debug levels set." )
      return
   if netlinkStub.errCode != 0:
      mode.addError( "Error during setting session debug: %d" % netlinkStub.errCode )
      return

def noDebugLevel( mode, args ):
   discriminator = args.get( 'DISC' )
   if discriminator is None:
      # clear the global debug
      setDebugLevel( mode, 1 )
      # then clear all the session debugs
      netlinkStub = Tac.newInstance( "BfdPyUtils::BfdNetlinkControlStub" )
      netlinkStub.createFd( False )
      netlinkStub.sendSessionDebugLevel( 0, True, 0 )
   elif discriminator == 0:
      setDebugLevel( mode )
   else:
      setSessionDebug( mode, discriminator, 0 )

def setDebugLevelHandler( mode, args ):
   if 'disc' in args:
      setSessionDebug( mode,
                       args[ 'DISC' ],
                       debugLevel=args[ 'DEBUGLVL' ] )
   else:
      setDebugLevel( mode, debugLevel=args[ 'DEBUGLVL' ] )

class BfdDebugLevelBase( CliCommand.CliCommandClass ):
   syntax = 'debug ( rbfd | ( disc DISC ) ) DEBUGLVL'
   noOrDefaultSyntax = 'debug ( rbfd | ( disc DISC ) ) ...'
   data = {
      'debug': matcherDebug,
      'rbfd': 'Set the debug config for all discriminators',
      'disc': 'Set the debug config for a discriminator',
      'DISC': matcherDiscriminator,
      'DEBUGLVL': matcherDebugLevel,
   }
   handler = setDebugLevelHandler
   noOrDefaultHandler = noDebugLevel
   hidden = True

class BfdDebugLevelCmd( BfdDebugLevelBase ):
   data = BfdDebugLevelBase.data.copy()

RouterBfdMode.addCommandClass( BfdDebugLevelCmd )

class BfdDebugLevelDeprecatedCmd( BfdDebugLevelBase ):
   syntax = 'bfd ' + BfdDebugLevelBase.syntax
   noOrDefaultSyntax = 'bfd ' + BfdDebugLevelBase.noOrDefaultSyntax
   data = BfdDebugLevelCmd.data | { 'bfd': matcherBfd }

BasicCliModes.GlobalConfigMode.addCommandClass( BfdDebugLevelDeprecatedCmd )

# --------------------------------------------------------------------------------
# [ no | default ] bfd debug-seq-num
#
# router bfd
#    [ no | default ] debug seq-num
# --------------------------------------------------------------------------------
def setDebugSeqNum( mode, noOrDefault=False ):
   if gv.bfdConfigGlobal.authType == authType.authDebugSeqNum and noOrDefault:
      gv.bfdConfigGlobal.authType = authType.authNone
   elif gv.bfdConfigGlobal.authType == authType.authNone and not noOrDefault:
      gv.bfdConfigGlobal.authType = authType.authDebugSeqNum
   elif not( gv.bfdConfigGlobal.authType == authType.authNone and noOrDefault ):
      print( 'setDebugSeqNum: Illegal authentication type transition %s to %s' %
              ( gv.bfdConfigGlobal.authType,
                ( 'authNone' if noOrDefault else 'authDebugSeqNum' ) ) )

def debugSeqNumHandler( mode, args ):
   setDebugSeqNum( mode )

def noOrDefaultDebugSeqNumHandler( mode, args ):
   setDebugSeqNum( mode, noOrDefault=True )

class BfdDebugSeqNumBase( CliCommand.CliCommandClass ):
   syntax = 'debug seq-num'
   noOrDefaultSyntax = syntax
   hidden = True
   handler = debugSeqNumHandler
   noOrDefaultHandler = noOrDefaultDebugSeqNumHandler

class BfdDebugSeqNumCmd( BfdDebugSeqNumBase ):
   data = {
      'debug': matcherDebug,
      'seq-num': 'Enable BFD packet sequence number logging',
   }

RouterBfdMode.addCommandClass( BfdDebugSeqNumCmd )

class BfdDebugSeqNumDeprecatedCmd( BfdDebugSeqNumBase ):
   syntax = 'bfd debug-seq-num'
   noOrDefaultSyntax = syntax
   data = {
      'bfd': matcherBfd,
      'debug-seq-num': 'Enable BFD packet sequence number logging',
   }

BasicCliModes.GlobalConfigMode.addCommandClass( BfdDebugSeqNumDeprecatedCmd )

# --------------------------------------------------------------------------------
# [ no | default ] bfd interval INTERVAL min_rx MIN_RX multiplier MULTIPLIER default
#
# router bfd
#    ( no|default ) interval INTERVAL min-rx MIN_RX multiplier MULTIPLIER default
# --------------------------------------------------------------------------------
def setGlobalBfdParams( mode, args ):
   bfdTxInt = args[ 'INTERVAL' ]
   bfdRxInt = args[ 'MIN_RX' ]
   bfdMult = args[ 'MULTIPLIER' ]
   gv.bfdConfigGlobal.globalCfg = BfdIntervalConfig( bfdTxInt, bfdRxInt, bfdMult )

def resetGlobalBfdParams( mode, args ):
   intervalDefault = Tac.Type( 'Bfd::BfdInterval' ).defval
   multDefault = Tac.Type( 'Bfd::BfdMultiplier' ).defval
   gv.bfdConfigGlobal.globalCfg = BfdIntervalConfig( intervalDefault,
                                                     intervalDefault,
                                                     multDefault )

class BfdParamsBase( CliCommand.CliCommandClass ):
   syntax = 'interval INTERVAL min-rx MIN_RX multiplier MULTIPLIER default'
   noOrDefaultSyntax = 'interval ...'
   data = {
      'interval': matcherInterval,
      'INTERVAL': matcherTxRxIntervalMs,
      'min-rx': matcherMinRx,
      'MIN_RX': matcherTxRxIntervalMs,
      'multiplier': matcherMultiplier,
      'MULTIPLIER': matcherMultiplierVal,
      # This keyword doesn't do anything, and exists solely for customer experience
      # parity.
      'default': 'Set a command to its defaults',
   }

   handler = setGlobalBfdParams
   noOrDefaultHandler = resetGlobalBfdParams

class BfdParamsCmd( BfdParamsBase ):
   data = BfdParamsBase.data.copy()

RouterBfdMode.addCommandClass( BfdParamsCmd )

class BfdParamsDeprecatedCmd( BfdParamsBase ):
   syntax, noOrDefaultSyntax = ( 'bfd ' + syntax.replace( 'min-rx', 'min_rx' )
                                 for syntax in ( BfdParamsBase.syntax,
                                                 BfdParamsBase.noOrDefaultSyntax ) )
   data = BfdParamsCmd.data | deprecatedData
   hidden = True

BasicCliModes.GlobalConfigMode.addCommandClass( BfdParamsDeprecatedCmd )

# --------------------------------------------------------------------------------
# [ no | default ] bfd ( ip | ipv6 ) access-group ACL [ vrf VRF ] [ in ]
#
# router bfd
#    [ no | default ] ( ip | ipv6 ) access-group ACL [ vrf VRF ] [ in ]
# --------------------------------------------------------------------------------
def setServiceAcl( mode, aclType, no=False, aclName=None, vrfName=None ):
   vrfName = vrfName or DEFAULT_VRF
   if vrfName == ALL_VRF_NAME:
      vrfNames = getAllVrfNames( mode )
   elif vrfExists( vrfName ):
      vrfNames = [ vrfName ]
   else:
      mode.addError( "VRF %s not configured." % vrfName )
      return

   if no:
      func = functools.partial( AclCliLib.noServiceAcl, mode, 'bfd',
                                gv.aclConfig, gv.aclCpConfig, aclName, aclType )
   else:
      func = functools.partial( AclCliLib.setServiceAcl, mode, 'bfd',
                                IPPROTO_UDP, gv.aclConfig, gv.aclCpConfig,
                                aclName, aclType, port=bfdPorts )
   for vrf in vrfNames:
      func( vrfName=vrf )

def bfdAclHandler( mode, args ):
   aclType = 'ip' if 'ip' in args else 'ipv6'
   aclName = args.get( 'V4ACL' ) or args.get( 'V6ACL' )
   vrfName = args.get( 'VRF' )
   setServiceAcl( mode, aclType, aclName=aclName, vrfName=vrfName )

def clearBfdAclHandler( mode, args ):
   aclType = 'ip' if 'ip' in args else 'ipv6'
   aclName = args.get( 'V4ACL' ) or args.get( 'V6ACL' )
   vrfName = args.get( 'VRF' )
   setServiceAcl( mode, aclType, no=True, aclName=aclName, vrfName=vrfName )

class BfdAccessGroupExpr( CliCommand.CliExpression ):
   expression = '( ( ip access-group V4ACL ) | ( ipv6 access-group V6ACL ) )'

   data = {
      'ip': 'IP config commands',
      'ipv6': 'IPv6 config commands',
      'access-group': AclCli.accessGroupKwMatcher,
      'V4ACL': AclCli.ipAclNameMatcher,
      'V6ACL': AclCli.ip6AclNameMatcher,
   }

class BfdAccessGroupNoOrDefExpr( CliCommand.CliExpression ):
   expression = '( ( ip access-group [ V4ACL ] ) | ( ipv6 access-group [ V6ACL ] ) )'

   data = {
      'ip': 'IP config commands',
      'ipv6': 'IPv6 config commands',
      'access-group': AclCli.accessGroupKwMatcher,
      'V4ACL': AclCli.ipAclNameMatcher,
      'V6ACL': AclCli.ip6AclNameMatcher,
   }

class BfdAccessGroupBase( CliCommand.CliCommandClass ):
   syntax = 'ACL_EXPR [ vrf VRF ] [ in ]'
   noOrDefaultSyntax = 'ACL_NOORDEF_EXPR [ vrf VRF ] [ in ]'
   data = {
      'ACL_EXPR': BfdAccessGroupExpr,
      'ACL_NOORDEF_EXPR': BfdAccessGroupNoOrDefExpr,
      'vrf': 'Configure the VRF in which to apply the access control list',
      'VRF': BfdCli.vrfNameExprFactory,
      'in': 'Inbound packets',
   }
   handler = bfdAclHandler
   noOrDefaultHandler = clearBfdAclHandler

class BfdAccessGroupCmd( BfdAccessGroupBase ):
   data = BfdAccessGroupBase.data.copy()

RouterBfdMode.addCommandClass( BfdAccessGroupCmd )

class BfdAccessGroupDeprecatedCmd( BfdAccessGroupBase ):
   syntax = 'bfd ' + BfdAccessGroupBase.syntax
   noOrDefaultSyntax = 'bfd ' + BfdAccessGroupBase.noOrDefaultSyntax
   data = BfdAccessGroupBase.data | { 'bfd': matcherBfd }
   hidden = True

BasicCliModes.GlobalConfigMode.addCommandClass( BfdAccessGroupDeprecatedCmd )

# --------------------------------------------------------------------------------
# [ no | default ] bfd local-address ( IPADDR | IP6ADDR )
#
# router bfd
#    [ no | default ] local-address ( IPADDR | IP6ADDR )
# --------------------------------------------------------------------------------
def bfdLocalAddrFunctionHelper( mode, no=False, address=None ):
   emptyAddr = Tac.Value( 'Arnet::IpGenAddr', '' )
   if address is None:
      address = emptyAddr

   if not address.isAddrZero:
      addrFamily = address.af
      if addrFamily == 'ipv4':
         if no:
            if address == gv.bfdConfigGlobal.localAddrV4:
               gv.bfdConfigGlobal.localAddrV4 = emptyAddr
            else:
               mode.addWarning( "bfd local-address %s not configured."
                                % address.stringValue )
         else:
            gv.bfdConfigGlobal.localAddrV4 = address
      elif addrFamily == 'ipv6':
         if no:
            if address == gv.bfdConfigGlobal.localAddrV6:
               gv.bfdConfigGlobal.localAddrV6 = emptyAddr
            else:
               mode.addWarning( "bfd local-address %s not configured."
                 % address.stringValue )
         else:
            gv.bfdConfigGlobal.localAddrV6 = address
   else:
      gv.bfdConfigGlobal.localAddrV4 = emptyAddr
      gv.bfdConfigGlobal.localAddrV6 = emptyAddr

def localAddrFunctionCmd( mode, args ):
   address = args[ 'LOCALADDR' ]
   bfdLocalAddrFunctionHelper( mode, address=address )

def localAddrNoFunctionCmd( mode, args ):
   bfdLocalAddrFunctionHelper( mode, no=True, address=args.get( 'LOCALADDR' ) )

class BfdLocalAddressBase( CliCommand.CliCommandClass ):
   syntax = 'local-address LOCALADDR'
   noOrDefaultSyntax = 'local-address [ LOCALADDR ]'
   data = {
      'local-address': 'Configure BFD local IP address',
      'LOCALADDR': IpGenAddrMatcher( helpdesc='Local IP/IPv6 address' ),
   }

   handler = localAddrFunctionCmd
   noOrDefaultHandler = localAddrNoFunctionCmd

class BfdLocalAddressCmd( BfdLocalAddressBase ):
   data = BfdLocalAddressBase.data.copy()

RouterBfdMode.addCommandClass( BfdLocalAddressCmd )

class BfdLocalAddressDeprecatedCmd( BfdLocalAddressBase ):
   syntax = 'bfd ' + BfdLocalAddressBase.syntax
   noOrDefaultSyntax = 'bfd ' + BfdLocalAddressBase.noOrDefaultSyntax
   data = BfdLocalAddressBase.data | { 'bfd': matcherBfd }
   hidden = True

BasicCliModes.GlobalConfigMode.addCommandClass( BfdLocalAddressDeprecatedCmd )

# --------------------------------------------------------------------------------
# [no|default] bfd multihop interval INTERVAL min_rx MIN_RX multiplier MULTIPLIER
#
# router bfd
#   [no|default] multihop interval INTERVAL min-rx MIN_RX multiplier MULTIPLIER
# --------------------------------------------------------------------------------
def setMultihopGlobalBfdParams( mode, args ):
   bfdTxInt = args[ 'INTERVAL' ]
   bfdRxInt = args[ 'MIN_RX' ]
   bfdMult = args[ 'MULTIPLIER' ]

   gv.bfdConfigGlobal.multihopGlobalCfg = BfdIntervalConfig( bfdTxInt,
                                                             bfdRxInt,
                                                             bfdMult )
   mode.addMessage( "It is recommended to configure larger multi-hop interval "
                    "than single-hop interval" )

def resetMultihopGlobalBfdParams( mode, args ):
   gv.bfdConfigGlobal.multihopGlobalCfg = None

class BfdMultihopParamsBase( CliCommand.CliCommandClass ):
   syntax = 'multihop interval INTERVAL min-rx MIN_RX multiplier MULTIPLIER'
   noOrDefaultSyntax = 'multihop ...'
   data = {
      'multihop': matcherMultihop,
      'interval': matcherInterval,
      'INTERVAL': matcherTxRxIntervalMs,
      'min-rx': matcherMinRx,
      'MIN_RX': matcherTxRxIntervalMs,
      'multiplier': matcherMultiplier,
      'MULTIPLIER': matcherMultiplierVal,
   }
   handler = setMultihopGlobalBfdParams
   noOrDefaultHandler = resetMultihopGlobalBfdParams

class BfdMultihopParamsCmd( BfdMultihopParamsBase ):
   data = BfdMultihopParamsBase.data.copy()

RouterBfdMode.addCommandClass( BfdMultihopParamsCmd )

class BfdMultihopParamsDeprecatedCmd( BfdMultihopParamsBase ):
   syntax, noOrDefaultSyntax = ( 'bfd ' + syntax.replace( 'min-rx', 'min_rx' )
                                 for syntax in ( BfdMultihopParamsBase.syntax,
                                        BfdMultihopParamsBase.noOrDefaultSyntax ) )
   data = BfdMultihopParamsBase.data | deprecatedData
   hidden = True

BasicCliModes.GlobalConfigMode.addCommandClass( BfdMultihopParamsDeprecatedCmd )

# --------------------------------------------------------------------------------
# Hidden command used to configure bfd session
# [ no | default ] bfd session ( IPADDR | IP6ADDR ) INTF VRF
#
# router bfd
#    [ no | default ] test session ( IPADDR | IP6ADDR ) INTF VRF
# --------------------------------------------------------------------------------
#
# This hook is invoked by Ira before VRF is deleted. We can cleanup
# corresponding BFD test sessions if any. We are expected to return a tuple: (
# True/False, messages). If we return True, VRF is deleted.
#
def deleteVrfHook( vrfName ):
   testCfgDir = ConfigMount.force( gv.testAppConfigDir )
   if vrfName in testCfgDir.entity:
      testAppVrfDir = testCfgDir.entity[ vrfName ]
      testCfg = testAppVrfDir.entity[ 'test' ]
      for p in testCfg.peerConfigVer:
         BfdCli.delPeer( p, testCfg )
      testAppVrfDir.deleteEntity( 'test' )
      testCfgDir.deleteEntity( vrfName )
   return ( True, None )

def getBfdTestAppVrfAppConfig( testAppDir, vrf ):
   testAppVrfDir = testAppDir.newEntity( 'Tac::Dir', vrf )
   testCfg = testAppVrfDir.newEntity( 'Bfd::AppConfig', 'test' )
   testCfg.appPid = 1
   return testCfg

class BfdStaticSessionBase( CliCommand.CliCommandClass ):
   _baseSyntax = 'session PEERADDR INTF VRF'
   syntax = 'test ' + _baseSyntax
   noOrDefaultSyntax = syntax
   data = {
      'test': 'BFD test harness',
      'session': matcherSession,
      'PEERADDR': IpGenAddrMatcher( helpdesc='Peer IP/IPv6 address' ),
      'INTF': intfMatcher,
      'VRF': BfdCli.vrfNameExprFactory,
   }
   handler = BfdCli.addOrRemoveBfdSession
   noOrDefaultHandler = handler
   hidden = True

class BfdStaticSessionCmd( BfdStaticSessionBase ):
   data = BfdStaticSessionBase.data.copy()

RouterBfdMode.addCommandClass( BfdStaticSessionCmd )

class BfdStaticSessionDeprecatedCmd( BfdStaticSessionBase ):
   syntax = 'bfd ' + BfdStaticSessionBase._baseSyntax
   noOrDefaultSyntax = syntax
   data = BfdStaticSessionBase.data | { 'bfd': matcherBfd }

BasicCliModes.GlobalConfigMode.addCommandClass( BfdStaticSessionDeprecatedCmd )

# --------------------------------------------------------------------------------
# Global hidden command used to configure multihop bfd session
#
# [ no | default ] bfd session multihop ( IPADDR | IP6ADDR )
#    ( V4SRCADDR | V6SRCADDR )  VRF [ INTF ]
#
# router bfd
#    [ no | default ] test session multihop ( IPADDR | IP6ADDR )
#       ( V4SRCADDR | V6SRCADDR )  VRF [ INTF ]
# --------------------------------------------------------------------------------
class BfdStaticSessionMultihopBase( CliCommand.CliCommandClass ):
   _baseSyntax = 'session multihop PEERADDR SRCADDR VRF [ INTF ]'
   syntax = 'test ' + _baseSyntax
   noOrDefaultSyntax = syntax
   data = {
      'test': 'BFD test harness',
      'session': matcherSession,
      'multihop': matcherMultihop,
      'PEERADDR': IpGenAddrMatcher( helpdesc='Peer IP/IPv6 address' ),
      'SRCADDR': IpGenAddrMatcher( helpdesc='Source IP/IPv6 address' ),
      'VRF': BfdCli.vrfNameExprFactory,
      'INTF': intfMatcher,
   }

   handler = BfdCli.addOrRemoveBfdSession
   noOrDefaultHandler = handler
   hidden = True

class BfdStaticSessionMultihopCmd( BfdStaticSessionMultihopBase ):
   data = BfdStaticSessionMultihopBase.data.copy()

RouterBfdMode.addCommandClass( BfdStaticSessionMultihopCmd )

class BfdStaticSessionMultihopDeprecatedCmd( BfdStaticSessionMultihopBase ):
   syntax = 'bfd ' + BfdStaticSessionMultihopBase._baseSyntax
   noOrDefaultSyntax = syntax
   data = BfdStaticSessionMultihopBase.data | { 'bfd': matcherBfd }

BasicCliModes.GlobalConfigMode.addCommandClass(
      BfdStaticSessionMultihopDeprecatedCmd )

# --------------------------------------------------------------------------------
# [ no | default ] bfd shutdown
#
# router bfd
#    [ no | default ] shutdown
# --------------------------------------------------------------------------------
def setAdminDown( mode, args ):
   gv.bfdConfigGlobal.bfdAdminDown = True

def noAdminDown( mode, args ):
   gv.bfdConfigGlobal.bfdAdminDown = False

class BfdShutdownBase( CliCommand.CliCommandClass ):
   syntax = 'shutdown'
   noOrDefaultSyntax = 'shutdown ...'
   data = {
      'shutdown': 'Set BFD admin down globally',
   }
   handler = setAdminDown
   noOrDefaultHandler = noAdminDown

class BfdShutdownCmd( BfdShutdownBase ):
   data = BfdShutdownBase.data.copy()

RouterBfdMode.addCommandClass( BfdShutdownCmd )

class BfdShutdownDeprecatedCmd( BfdShutdownBase ):
   syntax = 'bfd ' + BfdShutdownBase.syntax
   noOrDefaultSyntax = 'bfd ' + BfdShutdownBase.noOrDefaultSyntax
   data = BfdShutdownBase.data | { 'bfd': matcherBfd }
   hidden = True

BasicCliModes.GlobalConfigMode.addCommandClass( BfdShutdownDeprecatedCmd )

# --------------------------------------------------------------------------------
# [ no | default ] bfd slow-timer SLOWTIMER
#
# router bfd
#    [ no | default ] slow-timer
# --------------------------------------------------------------------------------
def _setSlowTimer( slowTimer ):
   gv.bfdConfigGlobal.bfdSlowTimer = Tac.Value( 'Bfd::BfdSlowTimer', slowTimer )

def setSlowTimer( mode, args ):
   _setSlowTimer( args[ 'SLOWTIMER' ] )

def resetSlowTimer( mode, args ):
   _setSlowTimer( Tac.Type( 'Bfd::BfdSlowTimer' ).defval )

class BfdSlowTimerBase( CliCommand.CliCommandClass ):
   syntax = 'slow-timer SLOWTIMER'
   noOrDefaultSyntax = 'slow-timer ...'
   data = {
      'slow-timer': 'Set slow timer rate in milliseconds',
      'SLOWTIMER': matcherSlowTimer,
   }
   handler = setSlowTimer
   noOrDefaultHandler = resetSlowTimer

class BfdSlowTimerCmd( BfdSlowTimerBase ):
   data = BfdSlowTimerBase.data.copy()

RouterBfdMode.addCommandClass( BfdSlowTimerCmd )

class BfdSlowTimerDeprecatedCmd( BfdSlowTimerBase ):
   syntax = 'bfd ' + BfdSlowTimerBase.syntax
   noOrDefaultSyntax = 'bfd ' + BfdSlowTimerBase.noOrDefaultSyntax
   data = BfdSlowTimerBase.data | { 'bfd': matcherBfd }
   hidden = True

BasicCliModes.GlobalConfigMode.addCommandClass( BfdSlowTimerDeprecatedCmd )

# --------------------------------------------------------------------------------
# [ no | default ] bfd vxlan session ( IPADDR | IP6ADDR ) VTEPADDR
#    OUTERDSTPORT OUTERSRCPORT VNI INNERDSTMAC INNERSRCMAC  INTERVAL MIN_RX
#       MULTIPLIER
#
# router bfd
#    [ no | default ] test vxlan session ( IPADDR | IP6ADDR ) VTEPADDR
#       OUTERDSTPORT OUTERSRCPORT VNI INNERDSTMAC INNERSRCMAC  INTERVAL MIN_RX
#          MULTIPLIER
#
# Notes that IPv6 VTEP addresses are not yet supported.
# --------------------------------------------------------------------------------
def addVxlanPeer( config, testCfg ):
   if config.peer not in testCfg.vxlanTunnelConfig:
      testCfg.newVxlanTunnelConfig( config.peer, config.vxlanTunnel,
                                    config.intervalParams )

def delVxlanPeer( config, testCfg ):
   if config in testCfg.vxlanTunnelConfig:
      del testCfg.vxlanTunnelConfig[ config ]

def addVxlanBfdSession( mode, no, peer, vtepAddr, outerDstPort, outerSrcPort, vni,
                        innerDstMac, innerSrcMac, bfdTxInt, bfdRxInt, bfdMult ):
   add = not no
   testCfg = ConfigMount.force( gv.testVxlanAppConfig )
   if testCfg is None:
      mode.addError( 'Unable to mount test VXLAN AppConfig!' )
      return

   peerTac = Tac.Value( 'Bfd::Peer', peer, 'default' )
   peerTac.type = 'vxlanTunnel'
   if add:
      innerDstMac = Ethernet.convertMacAddrToCanonical( innerDstMac )
      innerSrcMac = Ethernet.convertMacAddrToCanonical( innerSrcMac )
      dstIp = '%s' % peer
      srcIp = '%s' % vtepAddr
      vxlanConfig = VxlanPeerConfig( ip=dstIp,
                                     flags=getVxlanValidFlags(),
                                     vni=vni,
                                     innerSrcIp=srcIp,
                                     innerDstMac=innerDstMac,
                                     innerSrcMac=innerSrcMac, outerDstIp=dstIp,
                                     outerSrcIp=srcIp, outerDstPort=outerDstPort,
                                     outerSrcPort=outerSrcPort,
                                     minTx=bfdTxInt, minRx=bfdRxInt,
                                     multiplier=bfdMult )
      addVxlanPeer( vxlanConfig, testCfg )
      BfdCli.addPeer( peerTac, testCfg )
   else:
      BfdCli.delPeer( peerTac, testCfg )
      delVxlanPeer( peerTac, testCfg )

def addStaticBfdVxlanSession( mode, args ):
   addVxlanBfdSession( mode,
                       False,
                       args[ 'PEERADDR' ],
                       args[ 'VTEPADDR' ],
                       args[ 'OUTERDSTPORT' ],
                       args[ 'OUTERSRCPORT' ],
                       args[ 'VNI' ],
                       args[ 'INNERDSTMAC' ],
                       args[ 'INNERSRCMAC' ],
                       args[ 'INTERVAL' ],
                       args[ 'MIN_RX' ],
                       args[ 'MULTIPLIER' ] )

def delStaticBfdVxlanSession( mode, args ):
   addVxlanBfdSession( mode,
                       True,
                       args[ 'PEERADDR' ],
                       args[ 'VTEPADDR' ],
                       args[ 'OUTERDSTPORT' ],
                       args[ 'OUTERSRCPORT' ],
                       args[ 'VNI' ],
                       args[ 'INNERDSTMAC' ],
                       args[ 'INNERSRCMAC' ],
                       args[ 'INTERVAL' ],
                       args[ 'MIN_RX' ],
                       args[ 'MULTIPLIER' ] )

class BfdVxlanSessionBase( CliCommand.CliCommandClass ):
   _baseSyntax = ( 'vxlan session PEERADDR VTEPADDR OUTERDSTPORT '
                   'OUTERSRCPORT VNI INNERDSTMAC INNERSRCMAC INTERVAL MIN_RX '
                   'MULTIPLIER' )
   syntax = 'test ' + _baseSyntax
   noOrDefaultSyntax = syntax
   data = {
      'test': 'BFD test harness',
      'vxlan': 'Create a BFD VXLAN session',
      'session': matcherSession,
      'PEERADDR': IpGenAddrMatcher( helpdesc='Peer IP/IPv6 address' ),
      'VTEPADDR': IpAddrMatcher( helpdesc='IPv4 source VTEP address' ),
      'OUTERDSTPORT': matcherOuterDstPort,
      'OUTERSRCPORT': matcherOuterSrcPort,
      'VNI': matcherVNI,
      'INNERDSTMAC': MacAddrMatcher( helpdesc='Inner destination Ethernet address' ),
      'INNERSRCMAC': MacAddrMatcher( helpdesc='Inner source Ethernet address' ),
      'INTERVAL': matcherTxRxIntervalMs,
      'MIN_RX': matcherTxRxIntervalMs,
      'MULTIPLIER': matcherMultiplierVal,
   }
   handler = addStaticBfdVxlanSession
   noOrDefaultHandler = delStaticBfdVxlanSession
   hidden = True

class BfdVxlanSessionCmd( BfdVxlanSessionBase ):
   data = BfdVxlanSessionBase.data.copy()

RouterBfdMode.addCommandClass( BfdVxlanSessionCmd )

class BfdVxlanSessionDeprecatedCmd( BfdVxlanSessionBase ):
   syntax = 'bfd ' + BfdVxlanSessionBase._baseSyntax
   noOrDefaultSyntax = syntax
   data = BfdVxlanSessionBase.data | { 'bfd': matcherBfd }

BasicCliModes.GlobalConfigMode.addCommandClass( BfdVxlanSessionDeprecatedCmd )

# ------------------------------------------------------------------------------
# router bfd
#    [ no | default ] authentication mode ( simple shared-secret profile PROFILE |
#                                           md5 shared-secret profile PROFILE
#                                               [ meticulous ] |
#                                           sha1 shared-secret profile PROFILE
#                                               [ meticulous ] )
#
# Note that meticulous mode is currently hidden.
# ------------------------------------------------------------------------------
class BfdGlobalAuthModeCmd( CliCommand.CliCommandClass ):
   syntax = ( 'authentication mode ( ( simple shared-secret profile PROFILE ) |'
                                    '( ( sha1 | md5 ) shared-secret profile PROFILE'
                                      '[ meticulous ] ) )' )
   noOrDefaultSyntax = 'authentication ...'
   data = {
            'authentication': 'Configure BFD authentication',
            'mode': 'Specify BFD authentication mode',
            'simple': 'Simple password authentication',
            'md5': 'Keyed MD5 authentication',
            'sha1': 'Keyed Sha1 authentication',
            'shared-secret': 'Specify a shared-secret',
            'profile': 'Shared-secret profile',
            'PROFILE': BfdCli.matcherAuthProfile,
            'meticulous': CliCommand.Node( matcherMeticulous, hidden=True ),
          }

   @staticmethod
   def handler( mode, args ):
      gv.bfdConfigGlobal.authType = BfdCli.getAuthModeFromArgs( args )
      gv.bfdConfigGlobal.secretProfileName = args[ 'PROFILE' ]

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      gv.bfdConfigGlobal.authType = authType.authNone
      gv.bfdConfigGlobal.secretProfileName = ''

RouterBfdMode.addCommandClass( BfdGlobalAuthModeCmd )

# ------------------------------------------------------------------------------
# router bfd
#    peer 1.1.1.1
#       [ no | default ] authentication mode ( disabled |
#                                              simple shared-secret profile PROFILE |
#                                              md5 shared-secret profile PROFILE
#                                                  [ meticulous ] |
#                                              sha1 shared-secret profile PROFILE
#                                                  [ meticulous ] )
#
# router bfd
#    vrf vrfName
#       peer 1.1.1.1
#          [ no | default ] authentication mode ( disabled |
#                                                 simple shared-secret
#                                                    profile PROFILE |
#                                                 md5 shared-secret profile PROFILE
#                                                     [ meticulous ] |
#                                                 sha1 shared-secret profile PROFILE
#                                                     [ meticulous ] )
# ------------------------------------------------------------------------------
class BfdPeerAuthModeCmd( CliCommand.CliCommandClass ):
   syntax = ( 'authentication mode ( disabled |'
                                    '( simple shared-secret profile PROFILE ) |'
                                    '( ( sha1 | md5 ) shared-secret profile PROFILE'
                                      '[ meticulous ] ) )' )
   noOrDefaultSyntax = 'authentication ...'
   data = {
            'authentication': 'Configure BFD authentication',
            'mode': 'Specify BFD authentication mode',
            'disabled': 'Authentication disabled',
            'simple': 'Simple password authentication',
            'md5': 'Keyed MD5 authentication',
            'sha1': 'Keyed Sha1 authentication',
            'shared-secret': 'Specify a shared-secret',
            'profile': 'Shared-secret profile',
            'PROFILE': BfdCli.matcherAuthProfile,
            'meticulous': CliCommand.Node( matcherMeticulous, hidden=True ),
          }

   @staticmethod
   def handler( mode, args ):
      peerCliConfigKey = BfdPeerCliConfigKey( mode.peerAddr, mode.vrfName )
      peerConfigCli = gv.bfdConfigPeerCli.peerCliConfig[ peerCliConfigKey ]
      peerConfigCli.authType = BfdCli.getAuthModeFromArgs( args )

      # If the mode configured is disabled, then there's no need to configure a
      # secret
      peerConfigCli.secretProfileName = args.get( 'PROFILE', '' )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      peerCliConfigKey = BfdPeerCliConfigKey( mode.peerAddr, mode.vrfName )
      peerConfigCli = gv.bfdConfigPeerCli.peerCliConfig[ peerCliConfigKey ]
      peerConfigCli.authType = authType.authInherit
      peerConfigCli.secretProfileName = ''

RouterBfdPeerMode.addCommandClass( BfdPeerAuthModeCmd )
RouterBfdVrfPeerMode.addCommandClass( BfdPeerAuthModeCmd )

# ------------------------------------------------------------------------------
# router bfd
#    [no|default] hardware acceleration disabled
# ------------------------------------------------------------------------------
class BfdHwAccelCmd( CliCommand.CliCommandClass ):
   syntax = 'hardware acceleration disabled'
   noOrDefaultSyntax = syntax
   data = {
      'hardware': 'Configure hardware-specific BFD parameters',
      'acceleration': 'Configure BFD processing in hardware',
      'disabled': 'Disable offloading BFD processing to hardware',
   }

   @staticmethod
   def handler( mode, args ):
      gv.bfdConfigGlobal.hwAccelerationConfig = 'disabledByCli'

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      gv.bfdConfigGlobal.hwAccelerationConfig = 'platformDefault'

RouterBfdMode.addCommandClass( BfdHwAccelCmd )

# --------------------------------------------------------------------------------
# router bfd
#    sbfd
#       [no|default] local-interface <Ethernet|Loopback|...> < ipv4 | ipv6 >
#
# ipv6 option would be added in the future, NOT in phase1
# --------------------------------------------------------------------------------
class BfdSbfdLocalIntfCmd( CliCommand.CliCommandClass ):
   syntax = 'local-interface INTF ipv4'
   noOrDefaultSyntax = 'local-interface ...'
   data = {
            'local-interface': 'Configure SBFD local interface',
            'INTF': intfMatcher,
            'ipv4': 'Configuration on IPv4',
          }

   @staticmethod
   def handler( mode, args ):
      intf = args[ 'INTF' ]
      gv.bfdConfigGlobal.sbfdLocalIntf = Tac.Value( 'Arnet::IntfId', intf.name )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      gv.bfdConfigGlobal.sbfdLocalIntf = sbfdLocalIntfDefault()

class BfdSbfdLocalIntfV4AndV6Cmd( CliCommand.CliCommandClass ):
   syntax = 'local-interface INTF ( ipv4 | ipv6 | ( ipv4 ipv6 ) )'
   noOrDefaultSyntax = 'local-interface [INTF] [ ( ipv4 | ipv6 | ( ipv4 ipv6 ) ) ]'
   data = {
            'local-interface': 'Configure SBFD local interface',
            'INTF': intfMatcher,
            'ipv4': 'Configuration on IPv4',
            'ipv6': 'Configuration on IPv6',
          }

   @staticmethod
   def handler( mode, args ):
      intf = args[ 'INTF' ].name
      if 'ipv4' in args:
         gv.bfdConfigGlobal.sbfdLocalIntf = Tac.Value( 'Arnet::IntfId', intf )
      if 'ipv6' in args:
         gv.bfdConfigGlobal.sbfdLocalIntfIp6 = Tac.Value( 'Arnet::IntfId', intf )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      intf = args.get( 'INTF' )
      allVersion = 'ipv4' not in args and 'ipv6' not in args
      ipv4 = 'ipv4' in args or allVersion
      ipv6 = 'ipv6' in args or allVersion
      if ipv4 and ( not intf or str( intf ) == gv.bfdConfigGlobal.sbfdLocalIntf ):
         gv.bfdConfigGlobal.sbfdLocalIntf = sbfdLocalIntfDefault()
      if ipv6 and ( not intf or str( intf ) == gv.bfdConfigGlobal.sbfdLocalIntfIp6 ):
         gv.bfdConfigGlobal.sbfdLocalIntfIp6 = sbfdLocalIntfDefault()

RouterBfdSbfdMode.addCommandClass( BfdSbfdLocalIntfV4AndV6Cmd )

# --------------------------------------------------------------------------------
# router bfd
#    sbfd
#       [no|default] initiator interval <INTERVAL> multiplier <MULTIPLIER>
# --------------------------------------------------------------------------------
def SbfdInitiatorTimerConfig( minTx, mult ):
   return Tac.Value( 'Bfd::InitiatorTimerConfig', minTx, mult )

def SbfdInitIntervalHandler( mode, args ):
   minTx = args.get( 'INTERVAL', initiatorConfigDefault().minTx )
   mult = args.get( 'MULTIPLIER', initiatorConfigDefault().mult )
   gv.bfdConfigGlobal.sbfdInitiatorConfig = SbfdInitiatorTimerConfig( minTx, mult )

class BfdSbfdInitIntervalCmd( CliCommand.CliCommandClass ):
   syntax = 'initiator interval INTERVAL multiplier MULTIPLIER'
   noOrDefaultSyntax = 'initiator interval ...'
   data = {
            'initiator': matcherInitiator,
            'interval': matcherInterval,
            'INTERVAL': matcherTxRxIntervalMs,
            'multiplier': 'Sets the SBFD multiplier',
            'MULTIPLIER': matcherMultiplierVal,
          }

   handler = SbfdInitIntervalHandler
   noOrDefaultHandler = handler

RouterBfdSbfdMode.addCommandClass( BfdSbfdInitIntervalCmd )

# --------------------------------------------------------------------------------
# router bfd
#    sbfd
#       [no|default] initiator measurement delay round-trip
# set to default value if no|default
# --------------------------------------------------------------------------------
def sbfdInitRttHandler( mode, args ):
   gv.bfdConfigGlobal.sbfdRttEnabled = True

def noSfdInitRttHandler( mode, args ):
   gv.bfdConfigGlobal.sbfdRttEnabled = False

class BfdSbfdInitRttCmd( CliCommand.CliCommandClass ):
   syntax = 'initiator measurement delay round-trip'
   noOrDefaultSyntax = 'initiator measurement ...'
   data = {
            'initiator': matcherInitiator,
            'measurement': 'Enable round-trip delay measurement',
            'delay': 'Enable round-trip delay measurement',
            'round-trip': 'Enable round-trip delay measurement',
          }

   handler = sbfdInitRttHandler
   noOrDefaultHandler = noSfdInitRttHandler

RouterBfdSbfdMode.addCommandClass( BfdSbfdInitRttCmd )

# --------------------------------------------------------------------------------
# router bfd
#    sbfd
#       [no|default] reflector local-discriminator DISC [ proxy ]
# set to 0 if no|default, 0 is invalid
# --------------------------------------------------------------------------------
class SbfdReflectorDiscExpr( CliCommand.CliExpression ):
   expression = "ADDR | U32"
   data = { 'ADDR': IpAddrMatcher(
                    helpdesc='The SBFD discriminator in IPv4 address format' ),
            'U32': CliMatcher.IntegerMatcher( 1, 2**32 - 1,
                   helpdesc='The SBFD discriminator in U32 form' ),
          }

def reflectorAdapter( mode, args, argsList ):
   result = ( 0, False ) # IP address, "in u32 form"
   addr = args.pop( 'ADDR', None )
   if addr:
      result = ( Arnet.IpAddr( addr ).value, False )
   addr = args.pop( 'U32', None )
   if addr:
      result = ( addr, True )

   args[ 'DISC' ] = result

def SbfdReflectorLocalDiscHandler( mode, args ):
   disc, isU32 = args[ 'DISC' ]
   gv.bfdConfigGlobal.sbfdReflectorLocalDisc = disc
   gv.bfdConfigGlobal.sbfdDiscIsU32 = isU32
   gv.bfdConfigGlobal.sbfdProxyEnabled = 'proxy' in args

class BfdSbfdReflectorLocalDiscCmd( CliCommand.CliCommandClass ):
   syntax = 'reflector local-discriminator DISC [ proxy ]'
   noOrDefaultSyntax = 'reflector local-discriminator ...'
   data = {
            'reflector': 'The passive node in a SBFD session.',
            'local-discriminator': 'The local discriminator for reflector',
            'DISC': SbfdReflectorDiscExpr,
            'proxy': ( 'In addition enable support to respond on '
                        'behalf of another endpoint' )
          }

   adapter = reflectorAdapter

   handler = SbfdReflectorLocalDiscHandler
   noOrDefaultHandler = handler

RouterBfdSbfdMode.addCommandClass( BfdSbfdReflectorLocalDiscCmd )

# --------------------------------------------------------------------------------
# Hidden command used to configure sbfd session
#
# router bfd
#   sbfd
#      [ no | default ] test session PEERADDR TUNNEL
#                       [ interval INTERVAL multiplier MULTIPLIER ]
# --------------------------------------------------------------------------------
class BfdSbfdSessionCmd( CliCommand.CliCommandClass ):
   syntax = ( 'test session PEERADDR TUNNEL '
              '[ interval INTERVAL multiplier MULTIPLIER ]'
              '[ rtt-echo-enabled ] [ proxy ]' )
   noOrDefaultSyntax = 'test session PEERADDR TUNNEL ...'
   data = {
            'test': 'BFD test harness',
            'session': matcherSession,
            'PEERADDR': IpAddrMatcher(
               helpdesc='The SBFD discriminator in IPv4 address format' ),
            'TUNNEL': CliMatcher.IntegerMatcher( 1, tunnelIdMax,
                      helpdesc='SrTe Tunnel ID' ),
            'interval': matcherInterval,
            'INTERVAL': matcherTxRxIntervalMs,
            'multiplier': matcherMultiplier,
            'MULTIPLIER': matcherMultiplierVal,
            'rtt-echo-enabled': 'Enable RTT for echo sessions',
            'proxy': 'Enable S-BFD proxy',
          }

   handler = BfdCli.addOrRemoveBfdSession
   noOrDefaultHandler = handler
   hidden = True

RouterBfdSbfdMode.addCommandClass( BfdSbfdSessionCmd )

# --------------------------------------------------------------------------------
# router bfd
#    sbfd
#       [no|default] reflector min-rx <MINRX>
# set to default value if no|default
# --------------------------------------------------------------------------------
def SbfdReflectorMinRxHandler( mode, args ):
   sbfdRxInt = args.get( 'MINRX', reflectorMinRxDefault() )
   gv.bfdConfigGlobal.sbfdReflectorMinRx = sbfdRxInt

class BfdSbfdReflectorMinRxCmd( CliCommand.CliCommandClass ):
   syntax = 'reflector min-rx MINRX'
   noOrDefaultSyntax = 'reflector min-rx ...'
   data = {
            'reflector': 'The passive node in a SBFD session.',
            'min-rx': matcherMinRx,
            'MINRX': matcherTxRxIntervalMs,
          }

   handler = SbfdReflectorMinRxHandler
   noOrDefaultHandler = handler

RouterBfdSbfdMode.addCommandClass( BfdSbfdReflectorMinRxCmd )

# --------------------------------------------------------------------------------
# router bfd
#    ( no|default ) session stats snapshot interval
#       ( ( dangerous DANGER_INT ) | INT )
#
# Session-stats timer intervals less than 10 seconds are prepended by the keyword
# "dangerous" to remind customers that such intervals may be detrimental to normal
# BFD packet processing in some configurations.
# --------------------------------------------------------------------------------
class BfdSessionStatsIntervalCmd( CliCommand.CliCommandClass ):
   syntax = 'session stats snapshot interval ( ( dangerous DANGER_INT ) | INT )'
   noOrDefaultSyntax = 'session stats snapshot interval ...'
   data = {
         'session': matcherSession,
         'stats': 'BFD session statistics',
         'snapshot': 'BFD session statistics snapshots',
         'interval': 'Configure interval of session statistics snapshot timer',
         'dangerous': matcherDangerous,
         'DANGER_INT': matcherDangerousStatsInt,
         'INT': matcherStatsInt,
   }

   @staticmethod
   def adapter( mode, args, argsList ):
      args[ 'INTERVAL' ] = args.get( 'DANGER_INT' ) or args.get( 'INT' )

   @staticmethod
   def handler( mode, args ):
      gv.bfdConfigGlobal.sessionStatsInterval = args[ 'INTERVAL' ]

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      gv.bfdConfigGlobal.sessionStatsInterval = SessionStatsInterval.defval

RouterBfdMode.addCommandClass( BfdSessionStatsIntervalCmd )

# --------------------------------------------------------------------------------
# router bfd
#    [ no | default ] role passive
# --------------------------------------------------------------------------------

class BfdRoleCmd( CliCommand.CliCommandClass ):
   syntax = 'role passive'
   noOrDefaultSyntax = syntax
   data = {
      'role': 'Set BFD role globally',
      'passive': 'Passive role'
   }

   @staticmethod
   def handler( mode, args ):
      gv.bfdConfigGlobal.role = BfdRole.passive

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      gv.bfdConfigGlobal.role = BfdRole.active

RouterBfdMode.addCommandClass( BfdRoleCmd )

# --------------------------------------------------------------------------------
# router bfd
#  [no|default] session echo local-interface <LOOPBACK_INTF_NAME>
# --------------------------------------------------------------------------------
class BfdEchoSrcIpCmd( CliCommand.CliCommandClass ):
   syntax = 'session echo local-interface INTF'
   noOrDefaultSyntax = 'session echo local-interface ...'
   data = {
      'session': matcherSession,
      'echo': 'Enabling echo feature',
      'local-interface': 'Interface we get IPSA from',
      'INTF': LoopbackIntfCli.LoopbackIntf.matcher,
   }

   @staticmethod
   def handler( mode, args ):
      intf = args[ 'INTF' ]
      gv.bfdConfigGlobal.echoSrcIpIntf = Tac.Value( 'Arnet::IntfId', intf.name )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      gv.bfdConfigGlobal.echoSrcIpIntf = Tac.Value( 'Arnet::IntfId', '' )

if toggleBfdEchoL3SrcAddrEnabled():
   RouterBfdMode.addCommandClass( BfdEchoSrcIpCmd )

# --------------------------------------------------------------------------------
# router bfd
#  [no|default] qos dscp <DSCP_VALUE>
# --------------------------------------------------------------------------------
class BfdQosDscp( CliCommand.CliCommandClass ):
   syntax = 'qos dscp DSCP_VALUE'
   noOrDefaultSyntax = 'qos dscp ...'
   data = {
         'qos': 'Set QoS parameters for BFD traffic',
         'dscp': 'Set DSCP value for BFD traffic',
         'DSCP_VALUE': CliMatcher.IntegerMatcher( DscpValue.min,
                                                  DscpValue.max,
                                                  helpdesc='DSCP value' ),
   }

   @staticmethod
   def handler( mode, args ):
      # The DSCP value in the config is a 6 bit value (0-63).
      # We store this value as an 8 bit integer,
      # with the 6 most significant bits being the DSCP configured
      # value
      gv.bfdConfigGlobal.dscpValue = Tac.enumName( "Arnet::IpDscp",
                                                   args[ 'DSCP_VALUE' ] << 2 )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      gv.bfdConfigGlobal.dscpValue = BfdConstants.defaultIpDscp

if toggleBfdDscpConfigEnabled():
   RouterBfdMode.addCommandClass( BfdQosDscp )

def Plugin( entityManager ):
   gv.testAppConfigDir = ConfigMount.mount( entityManager,
                                            'bfd/config/app/test',
                                            'Tac::Dir', 'wic' )
   gv.bfdConfigGlobal = ConfigMount.mount( entityManager, 'bfd/config/global',
                                           'Bfd::ConfigGlobal', 'w' )
   gv.bfdConfigPeerCli = ConfigMount.mount( entityManager, 'bfd/config/peerCli',
                                         'Bfd::ConfigPeerCli', 'w' )
   gv.testVxlanAppConfig = ConfigMount.mount( entityManager,
                                              'bfd/config/vxlanTunnelApp/test',
                                              'Bfd::VxlanTunnelAppConfig', 'w' )
   IraVrfCli.canDeleteVrfHook.addExtension( deleteVrfHook )
   gv.aclConfig = ConfigMount.mount( entityManager, "acl/config/cli",
                                  "Acl::Input::Config", "w" )
   gv.aclCpConfig = ConfigMount.mount( entityManager, "acl/cpconfig/cli",
                                       "Acl::Input::CpConfig", "w" )
