# Copyright (c) 2012 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# pylint: disable=consider-using-f-string
# pylint: disable=bad-string-format-type

import os
from socket import AF_INET
from socket import AF_INET6
from collections import defaultdict
import Ark
import TableOutput
from ArnetModel import IpGenericAddress
from IpLibConsts import DEFAULT_VRF
import Tac
from BfdLib import ( comparePeerKey, compareIpKey, compareHwSessionKey,
                     diagEnumToReason, dispTime, operSessionEnumToType,
                     authEnumNumToText )
from CliModel import Bool, Dict, Enum, Float, Int, List, Model, Str, Submodel
from IntfModels import Interface
import CliPlugin.BfdCli as BfdCli # pylint: disable=consider-using-from-import
from TypeFuture import TacLazyType

OperSessionType = TacLazyType( "Bfd::OperSessionType" )
SegmentType = TacLazyType( "SrTePolicy::SegmentType" )
TunnelIdType = TacLazyType( "Tunnel::TunnelTable::TunnelId" )
TunnelType = TacLazyType( "Tunnel::TunnelTable::TunnelType" )

segmentTypeIntToEnum = {
   0: SegmentType.invalid,
   1: SegmentType.mplsLabel,
   2: SegmentType.ipNexthop,
}

operSessionEnumToStr = {
   OperSessionType.sessionTypeNormal: "normal",
   OperSessionType.sessionTypeLagLegacy: "LAG per-link *",
   OperSessionType.sessionTypeLagRfc7130: "LAG RFC7130 *",
   OperSessionType.sessionTypeMicroLegacy: "micro per-link",
   OperSessionType.sessionTypeMicroRfc7130: "micro RFC7130",
   OperSessionType.sessionTypeVxlanTunnel: "VXLAN",
   OperSessionType.sessionTypeMultihop: "multi-hop",
   OperSessionType.sessionTypeSbfdInitiator: "Initiator",
   OperSessionType.sessionTypeSbfdReflector: "Reflector",
   OperSessionType.sessionTypeNormal + " echo": "normal echo",
   OperSessionType.sessionTypeLagLegacy + " echo": "LAG per-link * echo",
   OperSessionType.sessionTypeLagRfc7130 + " echo": "LAG RFC7130 * echo",
   OperSessionType.sessionTypeMicroLegacy + " echo": "micro per-link echo",
   OperSessionType.sessionTypeMicroRfc7130 + " echo": "micro RFC7130 echo",
   OperSessionType.sessionTypeVxlanTunnel + " echo": "VXLAN echo",
   OperSessionType.sessionTypeMultihop + " echo": "multi-hop echo",
}

def capitalize( x ):
   # x is a string. Capitalize first letter and leave the rest of the string
   # intact
   return x[ 0 ].upper() + x[ 1 : ]

def convertToMs( microseconds ):
   return microseconds // 1000

# timeElapsed can never be negative unless UTC wraps or the system is reset to an
# earlier time
def dispElapsedTime( timeElapsed, state ):
   if state != 'up' or timeElapsed < 0:
      return "NA"

   secondsPerMinute = 60
   secondsPerHour = secondsPerMinute * 60
   secondsPerDay = secondsPerHour * 24

   rem = timeElapsed
   ( days, rem ) = divmod( rem, secondsPerDay )
   ( hrs, rem ) = divmod( rem, secondsPerHour )
   ( mins, rem ) = divmod( rem, secondsPerMinute )
   ( secs, rem ) = divmod( rem, 1 )
   # get first two digits
   centisecs = rem * 100

   if days:
      disp = "%d days, %02d:%02d:%02d.%02d" % ( days, hrs, mins, secs, centisecs )
   elif hrs:
      disp = "%02d:%02d:%02d.%02d" % ( hrs, mins, secs, centisecs )
   elif mins:
      disp = "%02d:%02d.%02d" % ( mins, secs, centisecs )
   else:
      disp = "%02d.%02d" % ( secs, centisecs )

   return disp

def getTunnelIdInfo( tunnelIdVal, detailCmd=True ):
   tunnelId = TunnelIdType( tunnelIdVal )
   tunnelType = tunnelId.tunnelType()
   idx = tunnelId.extractIndexAndAf()
   if tunnelType == TunnelType.eosSdkSbfdSessionTunnel:
      if detailCmd:
         return "Tunnel ID %d(EOS SDK SBFD)" % tunnelIdVal
      else:
         return "EOS SDK SBFD(%d)" % idx
   elif tunnelType == TunnelType.srTeSegmentListTunnel:
      if detailCmd:
         return "Tunnel ID %d(SR), Segment list ID %d" % ( tunnelIdVal, idx )
      else:
         return "SR-Tunnel(%d[%d])" % ( tunnelIdVal, idx )
   else:
      assert False, "Unknown Tunnel Type %s" % tunnelType
      return None

class PeerSnapshot( Model ):
   ip = IpGenericAddress( help="IP address of peer" )
   intf = Interface( help="Local egress interface" )
   vrf = Str( help="Vrf that peer is configured in" )
   status = Enum( values=[ "adminDown", "down", "init", "up" ], help="State of "
      "peer session" )
   prevState = Enum( values=[ "adminDown", "down", "init", "up" ], help="previous "
      "state " )
   localIpAddr = IpGenericAddress( help="Local IP" )
   localDisc = Int( help="Local discriminator of BFD session" )
   remoteDisc = Int( help="Remote discriminator BFD session" )
   eventTime = Float( help="Time of BFD session event in seconds" )
   event = Enum( values=[ "sessionDelete", "sessionDown", "sessionAdminDown",
                          "sessionPeerAdminDown", "sessionNoEvent",
                          "sessionPeerDown" ], help="BFD session event" )
   lastUp = Float( help="Last time peer session went up in seconds, 0.0 if never "
      "up" )
   lastDown = Float( help="Last time peer session went down in seconds, 0.0 if "
      "never down" )
   lastDiag = Enum( values=[ 'diagNone', 'diagCtrlTimeout', 'diagEchoFail',
      'diagNeighDown', 'diagForwardingReset', 'diagPathDown', 'diagConcatPathDown',
      'diagAdminDown', 'diagRevConcatPathDown' ], help="Last diagnostic for reason "
      "of change to Down state" )
   authType = Enum( values=list( authEnumNumToText.values() ),
                    help="Authentication mode" )
   authProfileName = Str( help="Configured authentication profile" )
   txInterval = Int( help="Rate in microseconds at which BFD control packets are "
      "sent" )
   rxInterval = Int( help="Rate in microseconds at which BFD control packets are "
      "received" )
   detectTime = Int( help="Detection timeout in microseconds" )
   rxCount = Int( help="Number of packets received" )
   txCount = Int( help="Number of packets sent" )
   sentWithin1p = Int( help="Number of packets sent within configured period since "
      "last packet" )
   sentWithin2p = Int( help="Number of packets sent later than one period but "
      "before two periods of last packet" )
   sentWithin3p = Int( help="Number of packets sent later than two periods but "
      "before three periods of last packet" )
   sentAfter3p = Int( help="Number of packets sent later than three periods of "
      "last packet" )
   minLateness = Int( help="Minimum number of microseconds a message sent after "
      "its configured period", optional=True )
   maxLateness = Int( help="Max number of microseconds a message sent after its "
      "configured period", optional=True )
   avgLateness = Int( help="Average number of microseconds messages sent after its "
      "configured period", optional=True )
   lastTxTime = Float( help="Time of last successfully transmitted packet in "
      "seconds", optional=True )
   minPeriod = Int( help="Minimum time between received packets in microseconds",
      optional=True )
   maxPeriod = Int( help="Maximum time between received packets in microseconds",
      optional=True )
   avgPeriod = Int( help="Average time between received packets in microseconds",
      optional=True )
   lastRxTime = Float( help="Time of last successfully authenticated and validated "
      "packet received in seconds", optional=True )
   echoOn = Bool( help="Echo turned on" )
   echoTxInterval = Int( help="Rate in microseconds at which BFD echo packets are "
      "sent" )
   echoRxInterval = Int( help="Rate in microseconds at which BFD echo packets are "
      "received" )
   echoDetectTime = Int( help="Detection timeout in microseconds for echo" )
   echoRxCount = Int( help="Number of echo packets received" )
   echoTxCount = Int( help="Number of echo packets sent" )
   echoSentWithin1p = Int( help="Number of echo packets sent within configured "
      "period since last echo packets" )
   echoSentWithin2p = Int( help="Number of echo packets sent later than one "
      "period but before two periods of last echo packet" )
   echoSentWithin3p = Int( help="Number of echo packets sent later than two "
      "periods but before three periods of last echo packet" )
   echoSentAfter3p = Int( help="Number of echo packets sent later than three "
      "periods of last echo packet" )
   tunnelId = Int( help="TunnelID of SBFD Initiator", optional=True )
   echoMinLateness = Int( help="Minimum number of microseconds a message sent after "
      "its configured period for echo", optional=True )
   echoMaxLateness = Int( help="Max number of microseconds a message sent after its "
      "configured period for echo", optional=True )
   echoAvgLateness = Int( help="Average number of microseconds messages sent after "
      "its configured period for echo", optional=True )
   echoLastTxTime = Float( help="Time of last successfully transmitted packet in "
      "seconds for echo", optional=True )
   echoMinPeriod = Int( help="Minimum time between received packets in microseconds"
      " for echo", optional=True )
   echoMaxPeriod = Int( help="Maximum time between received packets in microseconds"
      "for echo", optional=True )
   echoAvgPeriod = Int( help="Average time between received packets in microseconds"
      "for echo", optional=True )
   echoLastRxTime = Float( help="Time of last successfully authenticated and "
      "validated packet received in seconds for echo", optional=True )

class PeerHistory( Model ):
   peerSnapshots = List( valueType=PeerSnapshot, help="List of BFD peer history" )

   def render( self ):
      eventMessage = { 'sessionDown': "Processed BFD session down event at %s",
                       'sessionPeerDown': "Processed Peer BFD down event at %s",
                       'sessionAdminDown': "Processed BFD admin down event at %s",
                       'sessionPeerAdminDown': "Processed Peer BFD admin down event "
                                               "at %s",
                       'sessionDelete': "Processed BFD session delete request from "
                                        "the protocol at %s" }
      for pst in self.peerSnapshots:
         if pst.tunnelId:
            col3 = getTunnelIdInfo( pst.tunnelId )
         else:
            col3 = "Intf %s" % pst.intf.stringValue
         prevState = ""
         prevState = ", Prev State %s" % ( capitalize( pst.prevState ) )
         print( "Peer VRF {}, Addr {}, {}, State {}{}".format(
                pst.vrf, pst.ip, col3, capitalize( pst.status ), prevState ) )
         print( "LAddr %s, LD/RD %s/%s" %
                ( pst.localIpAddr, pst.localDisc, pst.remoteDisc ) )
         if pst.event in eventMessage:
            message = eventMessage[ pst.event ] % dispTime( pst.eventTime )
            if pst.tunnelId:
               message = message.replace( 'BFD', 'SBFD(Initiator)' )
            print( message )
         else:
            print( "Processed UNKNOWN({}) event at {}".format( pst.event,
                  dispTime( pst.eventTime ) ) )
         print( "Last Up %s, Last Down %s"
                % ( dispTime( pst.lastUp ), dispTime( pst.lastDown ) ) )
         print( "Last Diag: %s" % diagEnumToReason[ pst.lastDiag ] )
         print( "Authentication mode: %s" % pst.authType )
         print( "Shared-secret profile: %s" % ( 'None' if pst.authProfileName
                                               == '' else pst.authProfileName ) )
         print( "TxInt: %d ms, RxInt: %d ms, Detect Time: %d ms"
                % ( pst.txInterval // 1000,
                      pst.rxInterval // 1000, pst.detectTime // 1000 ) )
         eventMonoTime = Tac.tacNowToLinuxMonoTime(
               Ark.utcToSwitchTime( pst.eventTime ) )
         printHistoryStats( pst, eventMonoTime )
         print( "SchedDelay: 1*TxInt: %d, 2*TxInt: %d, 3*TxInt: %d, GT 3*TxInt: %d"
             % ( pst.sentWithin1p, pst.sentWithin2p, pst.sentWithin3p,
                 pst.sentAfter3p ) )
         if pst.echoOn:
            print( "Echo TxInt: %d ms, RxInt: %d ms, Detect Time: %d ms"
               % ( pst.echoTxInterval // 1000, pst.echoRxInterval // 1000,
                   pst.echoDetectTime // 1000 ) )
            printHistoryStats( pst, eventMonoTime, pst.echoOn )
            print( "Echo SchedDelay: 1*TxInt: %d, 2*TxInt: %d, 3*TxInt: %d, "
               "GT 3*TxInt: %d"
              % ( pst.echoSentWithin1p, pst.echoSentWithin2p, pst.echoSentWithin3p,
                   pst.echoSentAfter3p ) )
         # Adds newline since print adds one automatically at the end
         print( "" )

class Via( Model ):
   nextHop = IpGenericAddress( help="Next hop IP address" )
   interface = Interface( help="Egress L3 interface of the next hop" )

class TunnelInfo( Model ):
   mplsLabels = List( valueType=int, help="MPLS label stack" )
   mplsExp = Int( help="MPLS EXP value" )
   ipDscp = Int( help="IP DSCP value" )
   via = Submodel( valueType=Via, help="Via next hop", optional=True )
   sbfdReturnPathLabelStack = List( valueType=int,
                          help="MPLS label stack of SBFD return path",
                          optional=True )

class PeerStatsDetail( Model ):
   role = Enum( values=( "active", "passive" ), help="BFD role" )
   apps = List( valueType=str, help="Registered protocols" )
   echoOn = Bool( help="Echo turned on" )
   echoActive = Bool( help="Active echo session with both tx and rx session" )
   echoDisc = Int( help="Discriminator of BFD Echo session" )
   operEchoTxRxInterval = Int( help="Operational rate in microseconds at which Bfd "
      "echo packets are sent and received" )
   operTxInterval = Int( help="Operational rate in microseconds at which BFD "
      "control packets are sent" )
   operRxInterval = Int( help="Operational rate in microseconds at which BFD "
      "control pckets are received" )
   detectMult = Int( help="Number of consecutive BFD packets missed before failure" )
   detectTime = Int( help="Detection timeout in microseconds" )
   remoteMinRxInterval = Int( help="Minimum rate in microseconds at which BFD "
      "packets can be received by the peer" )
   remoteDetectMult = Int( help="Receive multiplier of peer" )
   localIpAddr = IpGenericAddress( help="Local IP" )
   lastVersion = Int( help="Last received Peer BFD version" )
   lastDiag = Int( help="Last received Peer diagnostic code between values 0-8" )
   lastState = Enum( values=[ "adminDown", "down", "init", "up" ], help="Last "
      "received Peer BFD session state" )
   lastDemand = Bool( help="Last received Peer demand bit" )
   lastPoll = Bool( help="Last received Peer poll bit" )
   lastFinal = Bool( help="Last received Peer final bit" )
   lastDetectMult = Int( help="Last received Peer detection multiplier" )
   lastLength = Int( help="Last received packet length" )
   lastMyDiscU32 = Int( help="Last received Peer discriminator" )
   lastYourDiscU32 = Int( help="Last received local discriminator" )
   lastMinTxIntv = Int( help="Last received minimum Tx interval in microseconds" )
   lastMinRxIntv = Int( help="Last received minimum Rx interval in microseconds" )
   lastEchoRxIntv = Int( help="Last received Echo Rx interval in microseconds" )
   segmentType = Enum( values=segmentTypeIntToEnum.values(), help="Segment type" )
   segments = List( valueType=int, help="Proxy segment list" )
   hwAcceleratedStates = Dict( keyType=str, valueType=bool,
         help="Mapping of session type to hw acceleration state" )
   _localSsoReady = Bool( help="Local system ready for SSO",
      default=False, optional=True )
   tunnelInfo = Submodel( valueType=TunnelInfo,
      help="Tunnel information of tunnelId", optional=True )

class RxStats( Model ):
   __revision__ = 4
   numReceived = Int( help="Number of packets received" )
   minPeriod = Int( help="Minimum time between received packets in microseconds",
                     optional=True )
   maxPeriod = Int( help="Maximum time between received packets in microseconds",
                     optional=True )
   avgPeriod = Int( help="Average time between received packets in microseconds",
                     optional=True )
   lastRxTime = Float( help="Time of last successfully authenticated and validated "
      "packet received in seconds", optional=True )
   lastRxStateChg = Float( help="Time of last Rx packet of state change "
                           "received in seconds", optional=True )
   lastRxState = Enum( values=[ "adminDown", "down", "init", "up" ],
                       help="Last Rx state of the peer", optional=True )

   def degrade( self, dictRepr, revision ):
      if revision < 4:
         for key in self.__attributes__:
            if key not in dictRepr:
               if key in ( "lastTxTime", "lastRxTime" ):
                  dictRepr[ key ] = 0.0
               elif key in ( "lastRxState", "lastRxStateChg" ):
                  continue
               else:
                  dictRepr[ key ] = 0
      return dictRepr

class TxStats( Model ):
   __revision__ = 4
   numSent = Int( help="Number of packets sent since last Tx config change" )
   minLateness = Int( help="Minimum number of microseconds a message sent after "
      "its configured period", optional=True )
   maxLateness = Int( help="Max number of microseconds a message sent after its "
      "configured period", optional=True )
   avgLateness = Int( help="Average number of microseconds messages sent after its "
      "configured period", optional=True )
   lastTxTime = Float( help="Time of last successfully transmitted packet in "
      "seconds", optional=True )
   sentWithin1p = Int( help="Number of packets sent within configured period since "
      "last packet", optional=True )
   sentWithin2p = Int( help="Number of packets sent later than one period but "
      "before two periods of last packet", optional=True )
   sentWithin3p = Int( help="Number of packets sent later than two periods but "
      "before three periods of last packet", optional=True )
   sentAfter3p = Int( help="Number of packets sent later than three periods of "
      "last packet", optional=True )

   def degrade( self, dictRepr, revision ):
      if revision < 4:
         for key in self.__attributes__:
            if key not in dictRepr:
               if key in ( "lastTxTime", "lastRxTime" ):
                  dictRepr[ key ] = 0.0
               else:
                  dictRepr[ key ] = 0
      return dictRepr

class RttStats( Model ):
   minRtt = Int( help="Minimum RTT in microseconds" )
   maxRtt = Int( help="Maximum RTT in microseconds" )
   avgRtt = Int( help="Average RTT in microseconds" )
   lastRtt = Int( help="Last RTT in microseconds" )

class BfdSessionBase_( Model ):
   sessType = Enum( values=operSessionEnumToStr, help="BFD session type" )
   localDisc = Int( help="Local discriminator of BFD session" )

class PeerStats( BfdSessionBase_ ):
   status = Enum( values=[ "adminDown", "down", "init", "up" ], help="State of "
      "the peer" )
   remoteDisc = Int( help="Remote discriminator BFD session" )
   lastUp = Float( help="Last time peer session came up in seconds, 0.0 if never "
      "up" )
   lastDown = Float( help="Last time peer session went down in seconds, 0.0 if "
      "never down" )
   lastDiag = Enum( values=[ 'diagNone', 'diagCtrlTimeout', 'diagEchoFail',
      'diagNeighDown', 'diagForwardingReset', 'diagPathDown', 'diagConcatPathDown',
      'diagAdminDown', 'diagRevConcatPathDown' ], help="Last diagnostic for reason "
      "of change to Down state" )
   authType = Enum( values=list( authEnumNumToText.values() ),
                    help="Authentication mode" )
   authProfileName = Str( help="Configured authentication profile" )
   l3intf = Interface( help="Local egress interface" )
   proxy = Bool( help="S-BFD proxy Enabled" )
   kernelIfIndex = Int( help="Kernel interface", optional=True )
   peerStatsDetail = Submodel( valueType=PeerStatsDetail, help="Detailed peer "
      "stats", optional=True )
   rxStats = Submodel( valueType=RxStats, help="Rx statistics for peer",
                       optional=True )
   txStats = Submodel( valueType=TxStats, help="Tx statistics for peer",
                       optional=True )
   echoRxStats = Submodel( valueType=RxStats, help="Echo Rx statistics for peer",
                       optional=True )
   echoTxStats = Submodel( valueType=TxStats, help="Echo Tx statistics for peer",
                       optional=True )
   tunnelId = Int( help="TunnelID of SBFD Initiator", optional=True )
   tunnelType = Enum( values=[ TunnelType.eosSdkSbfdSessionTunnel,
                               TunnelType.srTeSegmentListTunnel ],
                      help="Tunnel type of SBFD Initiator", optional=True )
   tunnelIdx = Int( help="Tunnel index of SBFD Initiator", optional=True )
   destPort = Int( help="Destination Port of SBFD Reflector", optional=True )
   rttStats = Submodel( valueType=RttStats, help="RTT statisics for peer",
                        optional=True )

class BfdNeighborSrcAddr( Model ):
   peerStats = Dict( keyType=IpGenericAddress, valueType=PeerStats,
      help="A mapping between source address and peer stats" )

class BfdNeighborIntfPeer( Model ):
   types = Dict( keyType=str, valueType=BfdNeighborSrcAddr,
      help="A mapping between peer type and source address" )

class BfdNeighborInterface( Model ):
   __revision__ = 3
   peers = Dict( keyType=Interface, valueType=BfdNeighborIntfPeer,
      help="A mapping between interface and peer type" )

   def degrade( self, dictRepr, revision ):
      dictRepr[ 'peerStats' ] = {}
      # pylint: disable=too-many-nested-blocks
      if dictRepr[ 'peers' ]:
         if revision == 1:
            for intf in dictRepr[ 'peers' ]:
               if dictRepr[ 'peers' ][ intf ][ 'types' ]:
                  for typeStr in dictRepr[ 'peers' ][ intf ][ 'types' ]:
                     neighSrcAddr = dictRepr[ 'peers' ][ intf ][ 'types' ][ typeStr ]
                     peerStats = neighSrcAddr[ 'peerStats' ]
                     # Revision 1 can only be used for single hop bfd, in which case
                     # there is only one source address and peer per interface. Thus,
                     # we can map each interface to the only existing PeerStats
                     # object
                     if peerStats:
                        peerStats = next( iter( peerStats.values() ) )
                     else:
                        # It is also possible for CliModelRevisionTest to generate an
                        # empty peerStats dictionary, in which case we can map the
                        # interface to a dummy PeerStats dictionary to satisfy the
                        # test
                        peerStats = PeerStats( peerStatsDetail=PeerStatsDetail(),
                                               rxStats=RxStats(),
                                               txStats=TxStats(),
                                               echoRxStats=RxStats(),
                                               echoTxStats=TxStats() ).__dict__
                        for attr, val in peerStats.items():
                           if issubclass( type( val ), Model ):
                              peerStats[ attr ] = val.__dict__
                     dictRepr[ 'peerStats' ][ intf ] = peerStats
         elif revision == 2:
            # in revision 2, peerStats is keyed by peer intf
            # peerStats = Dict( keyType=Interface, valueType=BfdNeighborSrcAddr,
            # help = "A mapping between interface and source address" )
            for intf in dictRepr[ 'peers' ]:
               for _ in dictRepr[ 'peers' ][ intf ]:
                  if dictRepr[ 'peers' ][ intf ][ 'types' ]:
                     for typeStr in dictRepr[ 'peers' ][ intf ][ 'types' ]:
                        intfTypes = dictRepr[ 'peers' ][ intf ][ 'types' ]
                        dictRepr[ 'peerStats' ][ intf ] = intfTypes[ typeStr ]

      if 'peers' in dictRepr:
         del dictRepr[ 'peers' ]
      return dictRepr

class SbfdInitiatorTunnel( Model ):
   peerStats = Dict( keyType=int, valueType=PeerStats,
      help="A mapping between tunnel ID and peer stats" )

class SbfdReflector( Model ):
   # keyType is long to cover the whole range of remoteDisc (0x0-0xFFFFFFFF)
   peerStats = Dict( keyType=int, valueType=PeerStats,
      help="A mapping between remote discriminator and peer stats" )

class BfdNeighborVrf( Model ):
   ipv4Neighbors = Dict( keyType=IpGenericAddress, valueType=BfdNeighborInterface,
      help="A mapping between IPv4 peer address and interface" )
   ipv6Neighbors = Dict( keyType=IpGenericAddress, valueType=BfdNeighborInterface,
      help="A mapping between IPv6 peer address and interface" )
   ipv4InitiatorNeighbors = Dict( keyType=IpGenericAddress,
                        valueType=SbfdInitiatorTunnel,
                        help="A mapping between IPv4 peer address and tunnel ID" )
   ipv6InitiatorNeighbors = Dict( keyType=IpGenericAddress,
                        valueType=SbfdInitiatorTunnel,
                        help="A mapping between IPv6 peer address and tunnel ID" )
   ipv4ReflectorNeighbors = Dict( keyType=IpGenericAddress, valueType=SbfdReflector,
            help="A mapping between IPv4 peer address and remote discriminators" )
   ipv6ReflectorNeighbors = Dict( keyType=IpGenericAddress, valueType=SbfdReflector,
            help="A mapping between IPv6 peer address and remote discriminators" )

def printHistoryStats( pst, baseline, echoOn=False ):
   # Currently not displaying any last RxSession/last TxSession Time
   # that contains a negative value
   # This is a patch for bug 745887, a later patch will address the issue in
   # BfdAgent.
   if not echoOn:
      if pst.lastRxTime <= baseline:
         print( "Rx Count: %d, Rx Interval (ms) min/max/avg: %d/%d/%d last: %s"
               % ( pst.rxCount, pst.minPeriod // 1000, pst.maxPeriod // 1000,
                  pst.avgPeriod // 1000,
                  calculateMsAgo( pst.lastRxTime, baseline ) ) )
      if pst.lastTxTime <= baseline:
         print( "Tx Count: %d, Tx Interval (ms) min/max/avg: %d/%d/%d last: %s"
               % ( pst.txCount, pst.minLateness // 1000, pst.maxLateness // 1000,
                  pst.avgLateness // 1000,
                  calculateMsAgo( pst.lastTxTime, baseline ) ) )
   else:
      if pst.echoLastRxTime <= baseline:
         print( "Echo Rx Count: %d, Rx Interval (ms)"
               " min/max/avg: %d/%d/%d last: %s"
               % ( pst.echoRxCount, pst.echoMinPeriod // 1000,
               pst.echoMaxPeriod // 1000, pst.echoAvgPeriod // 1000,
                  calculateMsAgo( pst.echoLastRxTime, baseline ) ) )
      if pst.echoLastTxTime <= baseline:
         print( "Echo Tx Count: %d, Tx Interval (ms)"
               " min/max/avg: %d/%d/%d last: %s"
               % ( pst.echoTxCount, pst.echoMinLateness // 1000,
                  pst.echoMaxLateness // 1000, pst.echoAvgLateness // 1000,
                  calculateMsAgo( pst.echoLastTxTime, baseline ) ) )


def calculateMsAgo( sec, baseline ):
   if sec == 0.0:
      return "never"
   return "%d ms ago" % ( ( baseline - sec ) * 1000 )

def printRxStats( rxStats, baseline, statsType=None ):
   linePrefix = "Echo " if statsType == "echo" else ""
   if not rxStats:
      print( linePrefix +
             "Rx Count: 0, Rx Interval (ms) min/max/avg: 0/0/0 last: never" )
   else:
      if rxStats.lastRxTime is None:
         print( linePrefix + "Rx Count: %d" % rxStats.numReceived )
      else:
         print( linePrefix +
            "Rx Count: %d, Rx Interval (ms) min/max/avg: %d/%d/%d last: %s"
             % ( rxStats.numReceived,
                 rxStats.minPeriod // 1000,
                 rxStats.maxPeriod // 1000,
                 rxStats.avgPeriod // 1000,
                 calculateMsAgo( rxStats.lastRxTime, baseline ) ) )

def printTxStats( txStats, peerStatsDetail, baseline, statsType=None ):
   linePrefix = "Echo " if statsType == "echo" else ""

   if not txStats:
      print( linePrefix +
             "Tx Count: 0, Tx Interval (ms) min/max/avg: 0/0/0 last: never" )
      if statsType == "echo":
         print( "Echo Detect Time: %s ms" % ( convertToMs(
                  peerStatsDetail.operEchoTxRxInterval *
                  peerStatsDetail.detectMult ) ) )
      else:
         print( "Detect Time: %s ms" % ( convertToMs(
                  peerStatsDetail.detectTime ) ) )
      print( linePrefix +
             "Sched Delay: 1*TxInt: 0, 2*TxInt: 0, 3*TxInt: 0, GT 3*TxInt: 0" )
   else:
      if txStats.lastTxTime is None:
         print( linePrefix + "Tx Count: %d" % txStats.numSent )
      else:
         print( linePrefix +
            "Tx Count: %d, Tx Interval (ms) min/max/avg: %d/%d/%d last: %s"
               % ( txStats.numSent,
                   txStats.minLateness // 1000,
                   txStats.maxLateness // 1000,
                   txStats.avgLateness // 1000,
                   calculateMsAgo( txStats.lastTxTime, baseline ) ) )
      if statsType == "echo":
         print( "Echo Detect Time: %s ms" %
               ( convertToMs( peerStatsDetail.operEchoTxRxInterval *
                  peerStatsDetail.detectMult ) ) )
      else:
         print( "Detect Time: %s ms" %
                ( convertToMs( peerStatsDetail.detectTime ) ) )
      if txStats.lastTxTime is not None:
         print( linePrefix +
            "Sched Delay: 1*TxInt: %d, 2*TxInt: %d, 3*TxInt: %d, GT 3*TxInt: %d"
             % ( txStats.sentWithin1p,
                 txStats.sentWithin2p,
                 txStats.sentWithin3p,
                 txStats.sentAfter3p ) )

def printRttStats( rttStats, sessType ):
   if rttStats:
      print( "RTT (us) min/max/avg: %d/%d/%d last: %d us"
            % ( rttStats.minRtt, rttStats.maxRtt, rttStats.avgRtt,
                rttStats.lastRtt ) )
   elif ( 'ARTEST_BFD_CLI_TEST' in os.environ and
          sessType == OperSessionType.sessionTypeSbfdInitiator ):
      # For testing purpose only
      print( "RTT (us) min/max/avg: 0/0/0 last: 0 us" )

def printPeerStatsDetail( peerStats, ip, intf, vrf, baseline ):
   peerStatsDetail = peerStats.peerStatsDetail

   if peerStats.sessType == OperSessionType.sessionTypeSbfdInitiator:
      print( "Peer Addr %s, %s, "
             "Type SBFD(initiator), State %s"
            % ( str( ip ), getTunnelIdInfo( peerStats.tunnelId ),
                capitalize( peerStats.status ) ) )
   elif peerStats.sessType == OperSessionType.sessionTypeSbfdReflector:
      print( "Peer Addr %s, Dest Port %d, Type SBFD(reflector), State %s"
             % ( ip, peerStats.destPort, capitalize( peerStats.status ) ) )
   else:
      intf = intf or "NA"
      print( "Peer Addr %s, Intf %s, Type %s, Role %s, State %s"
         % ( str( ip ), intf, operSessionEnumToType[ peerStats.sessType ],
            peerStatsDetail.role, capitalize( peerStats.status ) ) )
   print( "VRF {}, LAddr {}, LD/RD {}/{}".format( vrf, peerStatsDetail.localIpAddr,
      peerStats.localDisc, peerStats.remoteDisc ) )
   if ( peerStats.sessType == OperSessionType.sessionTypeSbfdReflector and
        peerStatsDetail.segmentType == SegmentType.mplsLabel and
        peerStatsDetail.segments ):
      proxyString = "Proxy: MPLS label stack [ "
      for segment in peerStatsDetail.segments:
         proxyString += "%d " % segment
      proxyString += "]"
      print( proxyString )
   sessState = "Session state is %s" % capitalize( peerStats.status )
   echoActive = peerStatsDetail.echoActive
   echoSuffix = ''
   sessState += ' and %susing echo function' \
                  % ( '' if echoActive else 'not ' )
   if peerStats.sessType in [ OperSessionType.sessionTypeLagLegacy,
                              OperSessionType.sessionTypeLagRfc7130 ]:
      print( sessState )

      # pylint: disable-msg=protected-access
      if peerStatsDetail._localSsoReady:
         print( "Local system ready for SSO" )

      if peerStatsDetail.echoOn:
         if not echoActive:
            print( "BFD echo over per-link still inactive on this port-channel. " )
            print( "Peer system MAY NOT be ready for SSO" )
         else:
            print( "BFD echo over per-link has converged on this port-channel. " )
            print( "Peer system ready for SSO if other BFD per-link echo have "
                   "also converged" )
      print( "Last Diag: %s" % diagEnumToReason[ peerStats.lastDiag ] )
      print( "Registered protocols: %s" % ", ".join( peerStatsDetail.apps ) )
      print( "Parent session, please check port channel config for member info\n" )
      return

   if echoActive:
      echoSuffix = ' with %s ms interval. LD: %s' % \
          ( convertToMs( peerStatsDetail.operEchoTxRxInterval ),
            peerStatsDetail.echoDisc )
   sessState += '%s' % echoSuffix

   if peerStats.sessType == OperSessionType.sessionTypeSbfdReflector:
      print( "Session state is %s" % capitalize( peerStats.status ) )
      print( "Last Up %s" % dispTime( peerStats.lastUp ) )
      print( "Last Down %s" % dispTime( peerStats.lastDown ) )
      print( "Last Diag: %s" % diagEnumToReason[ peerStats.lastDiag ] )
      print( "Last Rx state change: %s, old Rx state is %s"
         % ( dispTime( peerStats.rxStats.lastRxStateChg ),
             capitalize( peerStats.rxStats.lastRxState ) ) )
      print( "RxInt: %s ms" % peerStatsDetail.operRxInterval )
      print( "Received TxInt: %s ms, Received RxInt: %s ms, Received Multiplier: %s"
         % ( convertToMs( peerStatsDetail.lastMinTxIntv ),
             convertToMs( peerStatsDetail.lastMinRxIntv ),
             peerStatsDetail.lastDetectMult ) )
      rxStats = peerStats.rxStats
      print( "Rx Count: %d, Rx Interval (ms) min/max/avg: %d/%d/%d last: %s"
         % ( rxStats.numReceived, rxStats.minPeriod // 1000,
             rxStats.maxPeriod // 1000, rxStats.avgPeriod // 1000,
             calculateMsAgo( rxStats.lastRxTime, baseline ) ) )
      txStats = peerStats.txStats
      print( "Tx Count: %d, last: %s"
             % ( peerStats.txStats.numSent,
                 calculateMsAgo( txStats.lastTxTime, baseline ) ) )
   else:
      print( sessState )
      print( "Hardware Acceleration: " + ", ".join(
         "{} {}".format( k, "On" if v else "Off" )
         for ( k, v ) in peerStatsDetail.hwAcceleratedStates.items() ) )
      print( "Last Up %s" % dispTime( peerStats.lastUp ) )
      print( "Last Down %s" % dispTime( peerStats.lastDown ) )
      print( "Last Diag: %s" % diagEnumToReason[ peerStats.lastDiag ] )
      print( "Authentication mode: %s" % peerStats.authType )
      print( "Shared-secret profile: %s" % ( 'None' if peerStats.authProfileName
                                            == '' else peerStats.authProfileName ) )
      print( "TxInt: %s ms, RxInt: %s ms, Multiplier: %s"
         % ( convertToMs( peerStatsDetail.operTxInterval ),
             convertToMs( peerStatsDetail.operRxInterval ),
             peerStatsDetail.detectMult ) )
      print( "Received RxInt: %s ms, Received Multiplier: %s"
         % ( convertToMs( peerStatsDetail.remoteMinRxInterval ),
             peerStatsDetail.remoteDetectMult ) )

      printRxStats( peerStats.rxStats, baseline )
      printTxStats( peerStats.txStats, peerStatsDetail, baseline )
      printRttStats( peerStats.rttStats, peerStats.sessType )
      if echoActive:
         print( "Echo TxInt: %s ms, RxInt: %s ms, Multiplier: %s"
            % ( convertToMs( peerStatsDetail.operEchoTxRxInterval ),
                convertToMs( peerStatsDetail.operEchoTxRxInterval ),
                peerStatsDetail.detectMult ) )
         printRxStats( peerStats.echoRxStats, baseline, statsType="echo" )
         printTxStats( peerStats.echoTxStats, peerStatsDetail, baseline,
                       statsType="echo" )

      print( "Registered protocols: %s" % ", ".join( peerStatsDetail.apps ) )
      print( "Uptime: %s" % dispElapsedTime( Tac.utcNow() - peerStats.lastUp,
                                            peerStats.status ) )

   def printTunnelInfo():
      if peerStatsDetail.tunnelInfo:
         table = TableOutput.TableFormatter()
         f = TableOutput.Format( justify='left' )
         f.padLimitIs( True )
         f.noPadLeftIs( True )
         table.formatColumns( f, f )
         tunnelInfo = peerStatsDetail.tunnelInfo
         tunnelInfoStr = "Tunnel Info: "
         space = " " * len( tunnelInfoStr )
         mplsLabelVal = "[" + " ".join( str( l )
                                        for l in tunnelInfo.mplsLabels ) + "]"
         table.newRow( tunnelInfoStr, "MPLS label stack: %s" % mplsLabelVal )
         table.newRow( space, "MPLS EXP: %d" % tunnelInfo.mplsExp )
         table.newRow( space, "IP DSCP: %d" % tunnelInfo.ipDscp )
         if peerStats.proxy:
            table.newRow( space, "Proxy: Enabled" )
         if tunnelInfo.sbfdReturnPathLabelStack:
            table.newRow( space, "SBFD return path label stack: [" +
                          " ".join( str( l )
                          for l in tunnelInfo.sbfdReturnPathLabelStack ) + "]" )
         if tunnelInfo.via:
            table.newRow( space, "Via: Next hop: {}, Interface: {}".format(
               tunnelInfo.via.nextHop, tunnelInfo.via.interface.stringValue ) )
         print( table.output().rstrip( '\n' ) )

   def printLastPacket():
      table = TableOutput.TableFormatter( tableWidth=74 )
      f = TableOutput.Format( justify='left' )
      f.padLimitIs( True )
      f.noPadLeftIs( True )
      table.formatColumns( f, f, f )
      lastPacket = "Last packet: "
      space = " " * len( lastPacket )
      table.newRow( lastPacket,
                    "Version: %s" % peerStatsDetail.lastVersion,
                    "- Diagnostic: %s" % peerStatsDetail.lastDiag )
      table.newRow( space,
                    "State bit: %s" % capitalize( peerStatsDetail.lastState ),
                    "- Demand bit: %s" % int( peerStatsDetail.lastDemand ) )
      table.newRow( space,
                    "Poll bit: %s" % int( peerStatsDetail.lastPoll ),
                    "- Final bit: %s" % int( peerStatsDetail.lastFinal ) )
      table.newRow( space,
                    "Multiplier: %d" % peerStatsDetail.lastDetectMult,
                    "- Length: %s" % peerStatsDetail.lastLength )
      table.newRow( space,
                    "My Discr.: %s" % peerStatsDetail.lastMyDiscU32,
                    "- Your Discr.: %s" % peerStatsDetail.lastYourDiscU32 )
      table.newRow( space, "Min tx interval: %d" %
                    convertToMs( peerStatsDetail.lastMinTxIntv ),
                    "- Min rx interval: %d" %
                    convertToMs( peerStatsDetail.lastMinRxIntv ) )
      table.newRow( space, "Min Echo interval: %d" %
                    convertToMs( peerStatsDetail.lastEchoRxIntv ), "" )
      if peerStats.sessType in [ OperSessionType.sessionTypeMicroLegacy,
                                 OperSessionType.sessionTypeMicroRfc7130 ]:
         print( table.output(), end=' ' )
         print( "Member session under parent interface %s" %
                peerStats.l3intf.stringValue )
      else:
         print( table.output().rstrip( '\n' ) )

   printTunnelInfo()
   printLastPacket()

   addrFamily = AF_INET if '.' in str( ip ) else AF_INET6

   if addrFamily == AF_INET:
      routingConfigured = BfdCli.gv.routingConfig.routing
      version = ''
   else:
      routingConfigured = BfdCli.gv.routing6Config.routing
      version = '6'

   if peerStatsDetail.echoOn and not peerStatsDetail.echoActive:
      print( "WARNING: Echo packets are not being echoed back to the peer. "
             "Perhaps IP%s routing is not enabled on the peer." % version )

   if peerStats.sessType == OperSessionType.sessionTypeVxlanTunnel and \
      str( peerStats.status ) == 'down' and not routingConfigured:
      print( "WARNING: IP%s routing is not enabled" % version )
   print()

# pylint: disable-next=pointless-string-statement
'''
Capi Model Hierarchy for "show bfd peers [detail]" CLI command
__revision__ 2
Summary: bfd -> vrf -> ipv4/ipv6 -> peerIp -> intf -> srcIp -> peerStats
Models: BfdNeighbors -> BfdNeighborVrf -> BfdNeighborInterface ->
   BfdNeighborSrcAddr -> PeerStats
Json Object:
{
   'vrfs' : {
      'default' : {
         'ipv4Neighbors' : {
            '1.1.1.1' : {
               'peerStats' : {
                  'Ethernet1' : {
                     'peerStats' : {
                        '1.1.1.0' : {
                           'status' : 'up',
                           'sessType' : 'normal',
                           ...
                        }
                     }
                  }

               }
            }
         }
         'ipv6Neighbors' : {
            '1::1' {
               'peerStats' : {
                  'Ethernet1' : {
                     'peerStats' : {
                        '1::0' : {
                           'status' : 'up',
                           'sessType' : 'normal',
                           ...
                        }
                     }
                  }
               }
            }
         }
      }
   }
}

__revision__ 3
Summary: bfd -> vrf -> ipv4/ipv6 -> peerIp -> intf -> peerType -> srcIp -> peerStats
         bfd -> vrf -> ipv4Initiator/ipv6Initiator -> peerIp -> tunnelId -> peerStats
         bfd -> vrf -> ipv4Reflector/ipv6Reflector -> peerIp -> remoteDisc
             -> peerStats
Models: BfdNeighbors -> BfdNeighborVrf -> BfdNeighborInterface -> bfdNeighborIntfPeer
   -> BfdNeighborSrcAddr -> PeerStats
        BfdNeighbors -> BfdNeighborVrf -> SbfdInitiatorTunnel -> PeerStats
        BfdNeighbors -> BfdNeighborVrf -> SbfdReflector -> PeerStats
Json Object:
{
   'vrfs' : {
      'default' : {
         'ipv4Neighbors' : {
            '1.1.1.1' : {
               'peers' : {
                  'Ethernet1' : {
                     'types' : {
                        'normal' : {
                           'peerStats' : {
                              '1.1.1.0' : {
                                 'status' : 'up',
                                 'sessType' : 'normal',
                                 ...
                              }
                           }
                        }
                     }
                  }
               }
            }
         }
         'ipv6Neighbors' : {
            '1::1' {
               'peers' : {
                  'Ethernet1' : {
                     'types' : {
                        'normal' : {
                           'peerStats' : {
                              '1::0' : {
                                 'status' : 'up',
                                 'sessType' : 'normal',
                                 ...
                              }
                           }
                        }
                     }
                  }
               }
            }
         }
         'ipv4Initiator' : {
            '1.1.1.1' {
               'peers' : {
                  '1' : {
                     'peerStats' : {
                        'status' : 'up',
                        'sessType' : 'initiator',
                        ...
                     }
                  }
               }
            }
         }
         'ipv4Reflector' : {
            '2.2.2.2' {
               'peers' : {
                  '100' : {
                     'peerStats' : {
                        'status' : 'up',
                        'sessType' : 'reflector',
                        ...
                     }
                  }
               }
            }
         }
         'ipv6Initiator' : {
            '1::1' {
               'peers' : {
                  '1' : {
                     'peerStats' : {
                        'status' : 'up',
                        'sessType' : 'initiator',
                        ...
                     }
                  }
               }
            }
         }
         'ipv6Reflector' : {
            '2::2' {
               'peers' : {
                  '1' : {
                     'peerStats' : {
                        'status' : 'up',
                        'sessType' : 'reflector',
                        ...
                     }
                  }
               }
            }
         }
      }
   }
}
'''

class BfdNeighbors( Model ):
   __revision__ = 4
   vrfs = Dict( valueType=BfdNeighborVrf, help="A mapping between vrf and peer "
      "stats" )
   _detailed = Bool( help="Detailed show bfd peers command" )

   def render( self ):
      firstVrf = True

      if DEFAULT_VRF in self.vrfs:
         print( "VRF name:", DEFAULT_VRF )
         print( "-----------------" )
         self.printBfd( DEFAULT_VRF,
                        self.vrfs[ DEFAULT_VRF ].ipv4Neighbors,
                        self.vrfs[ DEFAULT_VRF ].ipv4InitiatorNeighbors,
                        self.vrfs[ DEFAULT_VRF ].ipv4ReflectorNeighbors )
         self.printBfd( DEFAULT_VRF,
                        self.vrfs[ DEFAULT_VRF ].ipv6Neighbors,
                        self.vrfs[ DEFAULT_VRF ].ipv6InitiatorNeighbors,
                        self.vrfs[ DEFAULT_VRF ].ipv6ReflectorNeighbors )
         firstVrf = False
      for vrf, vrfModel in sorted( self.vrfs.items() ):
         if vrf == DEFAULT_VRF:
            continue
         if not firstVrf:
            print()
         print( "VRF name:", vrf )
         print( "-----------------" )
         self.printBfd( vrf, vrfModel.ipv4Neighbors,
                        vrfModel.ipv4InitiatorNeighbors,
                        vrfModel.ipv4ReflectorNeighbors )
         self.printBfd( vrf, vrfModel.ipv6Neighbors,
                        vrfModel.ipv6InitiatorNeighbors,
                        vrfModel.ipv6ReflectorNeighbors )
         firstVrf = False

   def sortByIntf( self, bfdNeighborInterface ):
      bfdNeighborIntfPeerList = sorted( bfdNeighborInterface.peers.items(),
                                       key=comparePeerKey )
      # convert bfdNeighborIntfPeerList into dict keyed by l3Intf to list of
      # peer stats
      intfBfdNeighborMap = defaultdict( list )
      for intf, intfPeer in bfdNeighborIntfPeerList:
         # First append normal session
         if 'normal' in intfPeer.types:
            bfdNeighborSrcAddr = intfPeer.types[ 'normal' ]
            intfBfdNeighborMap[ list( bfdNeighborSrcAddr.peerStats.values() )[ 0 ].
                                l3intf.stringValue ].append( ( intf,
                                      bfdNeighborSrcAddr ) )
         # append other session types
         for typeStr, bfdNeighborSrcAddr in intfPeer.types.items():
            if typeStr != "normal":
               intfBfdNeighborMap[ list(
                                   bfdNeighborSrcAddr.peerStats.values() )[ 0 ].
                                   l3intf.stringValue ].append( ( intf,
                                         bfdNeighborSrcAddr ) )
      sortedBfdNeighborSrcAddrList = []
      for intf, intfBfdNeighborList in sorted( intfBfdNeighborMap.items() ):
         sortedBfdNeighborSrcAddrList.extend( intfBfdNeighborList )
      return sortedBfdNeighborSrcAddrList

   def printBfd( self, vrf, bfdNeighbors, sbfdInitiators, sbfdReflectors ):
      baseline = Tac.fastMonoTimeOnly()
      if not bfdNeighbors and not sbfdInitiators and not sbfdReflectors:
         return

      if not self._detailed:
         headings = ( "DstAddr", "MyDisc", "YourDisc", "Interface/Transport", "Type",
                         "LastUp", "LastDown", "LastDiag", "State" )
         formatLeft = TableOutput.Format( justify="left" )
         formatLeft.noPadLeftIs( True )
         formatRight = TableOutput.Format( justify="right" )
         table = TableOutput.createTable( headings )
         table.formatColumns( formatLeft, formatRight, formatRight, formatRight,
            formatRight, formatRight, formatRight, formatRight, formatRight )

         if bfdNeighbors:
            for ip, bfdNeighborInterface in sorted( bfdNeighbors.items(),
                  key=compareIpKey ):
               for intf, bfdNeighborSrcAddr in self.sortByIntf(
                                                            bfdNeighborInterface ):
                  for _, peerStats in sorted( bfdNeighborSrcAddr.peerStats.items(),
                        key=compareIpKey ):
                     interface = intf + ( "(%d)" % peerStats.kernelIfIndex ) \
                                 if intf else "NA"
                     table.newRow( str( ip ), peerStats.localDisc,
                           peerStats.remoteDisc,
                           interface,
                           operSessionEnumToType[ peerStats.sessType ],
                           dispTime( peerStats.lastUp, shortDisp=True ),
                           dispTime( peerStats.lastDown, shortDisp=True ),
                           diagEnumToReason[ peerStats.lastDiag ],
                           capitalize( str( peerStats.status ) ) )
         if sbfdInitiators:
            for ip, sbfdInitiatorTunnel in sorted( sbfdInitiators.items(),
                  key=compareIpKey ):
               for _, peerStats in sorted(
                     sbfdInitiatorTunnel.peerStats.items() ):
                  transport = getTunnelIdInfo( peerStats.tunnelId, detailCmd=False )
                  table.newRow( str( ip ), peerStats.localDisc,
                        peerStats.remoteDisc,
                        transport,
                        operSessionEnumToType[ peerStats.sessType ],
                        dispTime( peerStats.lastUp, shortDisp=True ),
                        dispTime( peerStats.lastDown, shortDisp=True ),
                        diagEnumToReason[ peerStats.lastDiag ],
                        capitalize( str( peerStats.status ) ) )
         if sbfdReflectors:
            for ip, sbfdReflector in \
                  sorted( sbfdReflectors.items(), key=compareIpKey ):
               for _, peerStats in sorted( sbfdReflector.peerStats.items() ):
                  table.newRow( str( ip ), peerStats.localDisc,
                                peerStats.remoteDisc,
                                "NA",
                                operSessionEnumToType[ peerStats.sessType ],
                                dispTime( peerStats.lastUp, shortDisp=True ),
                                dispTime( peerStats.lastDown, shortDisp=True ),
                                diagEnumToReason[ peerStats.lastDiag ],
                                capitalize( str( peerStats.status ) ) )
         print( table.output() )
      else:
         if bfdNeighbors:
            for ip, bfdNeighborInterface in sorted( bfdNeighbors.items(),
                  key=compareIpKey ):
               for intf, bfdNeighborSrcAddr in self.sortByIntf(
                                                            bfdNeighborInterface ):
                  for _, peerStats in sorted( bfdNeighborSrcAddr.peerStats.items(),
                        key=compareIpKey ):
                     printPeerStatsDetail( peerStats, ip, intf, vrf, baseline )
         if sbfdInitiators:
            for ip, sbfdInitiatorTunnel in sorted( sbfdInitiators.items(),
                  key=compareIpKey ):
               for _, peerStats in sorted(
                     sbfdInitiatorTunnel.peerStats.items() ):
                  printPeerStatsDetail( peerStats, ip, '', vrf, baseline )

         if sbfdReflectors:
            for ip, sbfdReflector in \
                  sorted( sbfdReflectors.items(), key=compareIpKey ):
               for _, peerStats in sorted( sbfdReflector.peerStats.items() ):
                  printPeerStatsDetail( peerStats, ip, '', vrf, baseline )

class RbfdStats( Model ):
   rfcDrops = Int( help="Number of packets dropped on receive before processing" )
   rxPacketMissing = Int( help="Number of times packet missing upon Rx trigger" )
   numInvalidSessions = Int( help="Number of packets without session in Rx" )
   numPassupDisc0 = Int( help="Number of packets passed up with discriminator zero" )
   netlinkPacketChangeFail = Int( help="Number of times packet change sent but "
                                       "broadcast failed" )
   netlinkFailureFail = Int( help="Number of times session failure sent but "
                                  "broadcast failed" )
   pskbExpandFail = Int( help="Number of times pskb needed to be expanded but "
                              "failed for packets without a session" )

   def render( self ):
      print( 'rfcDrops: %d' % self.rfcDrops )
      print( 'rxPacketMissing: %d' % self.rxPacketMissing )
      print( 'numInvalidSessions: %d' % self.numInvalidSessions )
      print( 'numPassupDisc0: %d' % self.numPassupDisc0 )
      print( 'netlinkPacketChangeFail: %d' % self.netlinkPacketChangeFail )
      print( 'netlinkFailureFail: %d' % self.netlinkFailureFail )
      print( 'pskbExpandFail: %d' % self.pskbExpandFail )

class SbfdStats( Model ):
   rxValid = Int( help="Number of valid packets received" )
   rxDrop = Int( help="Number of packets dropped on receive" )
   rxTruncated = Int( help="Number of packets truncated on receive" )
   rxInvalidVer = Int( help="Number of packets with invalid version on receive" )
   rxInvalidLen = Int( help="Number of packets with invalid length on receive" )
   rxInvalidSrcPort = Int(
      help="Number of packets with invalid source port on receive" )
   rxDemandUnset = Int( help="Number of packets with demand mode unset on receive" )
   rxInvalidMulti = Int(
      help="Number of packets with invalid multiple detect on receive" )
   rxInvalidMyDisc = Int(
      help="Number of packets with invalid my discriminator on receive" )
   rxYourDiscUnmatch = Int(
      help="Number of packets with unmatched your discriminator on receive" )
   rxInvalidBfdState = Int(
      help="Number of packets with invalid bfd state on receive" )
   rxUnsupported = Int( help="Number of packets unsupported on receive" )
   rxInvalidPktFormat = Int(
      help="Number of packets with invalid packet format on receive" )
   rxInvalidTlv = Int(
      help="Number of packets with invalid TLVs on receive" )
   txSend = Int( help="Number of packets sent" )
   txSendToFail = Int( help="Number of packets failed to send" )
   txInvalidLen = Int( help="Number of packets with invalid length" )
   rxHashMiss = Int( help="Number of session cache lookup miss" )
   rxHashHit = Int( help="Number of session cache lookup hit" )
   rxHashTrim = Int( help="Number of session cache lookup trim" )

   def render( self ):
      print( 'rxValid: %d' % self.rxValid )
      print( 'rxDrop: %d' % self.rxDrop )
      print( 'rxTruncated: %d' % self.rxTruncated )
      print( 'rxInvalidVer: %d' % self.rxInvalidVer )
      print( 'rxInvalidLen: %d' % self.rxInvalidLen )
      print( 'rxInvalidSrcPort: %d' % self.rxInvalidSrcPort )
      print( 'rxDemandUnset: %d' % self.rxDemandUnset )
      print( 'rxInvalidMulti: %d' % self.rxInvalidMulti )
      print( 'rxInvalidMyDisc: %d' % self.rxInvalidMyDisc )
      print( 'rxYourDiscUnmatch: %d' % self.rxYourDiscUnmatch )
      print( 'rxInvalidBfdState: %d' % self.rxInvalidBfdState )
      print( 'rxUnsupported: %d' % self.rxUnsupported )
      print( 'rxInvalidPktFormat: %d' % self.rxInvalidPktFormat )
      print( 'rxInvalidTlv: %d' % self.rxInvalidTlv )
      print( 'txSend: %d' % self.txSend )
      print( 'txSendToFail: %d' % self.txSendToFail )
      print( 'txInvalidLen: %d' % self.txInvalidLen )

class SessionCount( Model ):
   echo = Int( help="Number of echo sessions" )

# In Python3, 'async' is a keyword.
SessionCount.__attributes__[ 'async' ] = Int( help="Number of async sessions" )

class SessionCountByState( Model ):
   states = Dict( keyType=str, valueType=SessionCount,
                  help="Mapping of session state to number of sessions" )

class SessionCountByType( Model ):
   types = Dict( keyType=str, valueType=SessionCountByState,
                 help="Mapping of session type to session count by state" )

def countByState( model, sessionType, state, sbfd=False ):
   stateModel = model.states[ state ]
   async_ = getattr( stateModel, 'async' )
   if sbfd:
      return str( async_ )
   return f'{async_} [{stateModel.echo}]'

def printRow( table, sessionModel, session, sessionType, sbfd=False ):
   typeModel = sessionModel.types[ sessionType ]
   operState = Tac.Type( "Bfd::OperState" )
   table.newRow( session, operSessionEnumToStr.get( sessionType, "All" ),
                 countByState( typeModel, sessionType, operState.up, sbfd ),
                 countByState( typeModel, sessionType, operState.init, sbfd ),
                 countByState( typeModel, sessionType, operState.down, sbfd ),
                 countByState( typeModel, sessionType, operState.adminDown, sbfd ),
               )

class BfdSummary( Model ):
   sessions = Dict( keyType=str, valueType=SessionCountByType,
                    help="Mapping of addressing to session count by type" )
   minTx = Int( help='Operational rate in microseconds at which BFD '
                     'control packets are sent' )
   minRx = Int( help='Operational rate in microseconds at which BFD '
                     'control pckets are received' )
   mult = Int( help='Number of consecutive BFD packets missed before failure' )
   mhMinTx = Int( help='Operational rate in microseconds at which BFD '
                       'control packets are sent on multihop BFD session' )
   mhMinRx = Int( help='Operational rate in microseconds at which BFD '
                       'control pckets are received on multihop BFD session' )
   mhMult = Int( help='Number of consecutive BFD packets missed before failure on'
                      'multihop BFD session' )
   slowTimer = Int( help='Operation rate in microseconds at which BFD control '
                         'packets are received on async sessions when echo is '
                         'enabled.' )
   adminDown = Bool( help='BFD has been shut down administratively' )
   sessionStatsInterval = Int( help='Interval in seconds between unsolicited '
                                    'per-session statistics from BFD acclerators' )
   dscpValue = Int( help='DSCP value' )

   # SBFD related attributes.
   _initiatorOnly = Bool( help='Display initiators only' )
   _reflectorOnly = Bool( help='Display reflectors only' )
   localIntf = Interface( help="SBFD local interface", optional=True )
   localIntfIp6 = Interface( help="SBFD IPv6 local interface", optional=True )
   localIntfIpAdd = IpGenericAddress( help="Local interface IP address" )
   localIntfIp6Add = IpGenericAddress( help="Local interface IPv6 address" )
   initiatorMinTx = Int( help='Operational rate in microseconds at which SBFD '
                              'control packets are sent on initiator' )
   initiatorMult = Int( help='Number of consecutive SBFD packets missed '
                             'before failure' )
   reflectorLocalDisc = Int( help="Local discriminator of SBFD session" )
   reflectorMinRx = Int( help='Operational rate in microseconds at which SBFD '
                              'control pckets are received on reflector' )
   # IPv6 support would be added later

   def renderSbfdSummary( self, table, sbfdApp, sbfdAppAf ):
      printRow( table, self.sessions[ sbfdApp ], sbfdApp,
                "All", sbfd=True )
      sessionModel = self.sessions[ "%sIPv4" % sbfdAppAf ]
      if self._initiatorOnly:
         printRow( table, sessionModel, "    IPv4",
                   OperSessionType.sessionTypeSbfdInitiator, sbfd=True )
      elif self._reflectorOnly:
         printRow( table, sessionModel, "    IPv4",
                   OperSessionType.sessionTypeSbfdReflector, sbfd=True )
      else:
         printRow( table, sessionModel, "    IPv4",
                   OperSessionType.sessionTypeSbfdInitiator, sbfd=True )
         printRow( table, sessionModel, "",
                   OperSessionType.sessionTypeSbfdReflector, sbfd=True )
      sessionModel = self.sessions[ "%sIPv6" % sbfdAppAf ]
      if self._initiatorOnly:
         printRow( table, sessionModel, "    IPv6",
                   OperSessionType.sessionTypeSbfdInitiator, sbfd=True )
      elif self._reflectorOnly:
         printRow( table, sessionModel, "    IPv6",
                   OperSessionType.sessionTypeSbfdReflector, sbfd=True )
      else:
         printRow( table, sessionModel, "    IPv6",
                   OperSessionType.sessionTypeSbfdInitiator, sbfd=True )
         printRow( table, sessionModel, "",
                   OperSessionType.sessionTypeSbfdReflector, sbfd=True )

   def render( self ):
      print( "Global administrative shutdown: ", "Yes" if self.adminDown else "No" )
      print( "Configured session stats snapshot interval: %ds" %
               self.sessionStatsInterval )
      print( "DSCP value: %d" % self.dscpValue )

      print( "BFD:" )
      print( "Global single hop interval %dms min_rx %dms multiplier %d"
            % ( self.minTx, self.minRx, self.mult ) )
      print( "Global multiple hop interval %dms min_rx %dms multiplier %d"
            % ( self.mhMinTx, self.mhMinRx, self.mhMult ) )
      if self.mhMinRx * self.mhMult < self.minRx * self.mult:
         print( "WARNING: Multi-hop detect time should be larger than"
                " single hop detect time" )
      print( "Slow timer: %dms" % self.slowTimer )

      print( "SBFD:" )
      if self.adminDown:
         print( "IPv4 operational state: globally disabled (adminDown state)" )
      elif not self.localIntf:
         print( "IPv4 operational state: globally disabled "
                 "(Local interface is not configured)" )
      elif self.localIntfIpAdd.isAddrZero:
         print( "IPv4 operational state: globally disabled "
                 "(Local interface IP address not configured)" )
      elif self.reflectorLocalDisc == 0:
         print( "IPv4 operational state: initiator enabled, reflector disabled "
                 "(Reflector discriminator not configured)" )
      else:
         print( "IPv4 operational state: enabled" )
      print( "Configured global initiator tx interval %dms multiplier %d"
            % ( self.initiatorMinTx, self.initiatorMult ) )
      print( "Configured reflector rx interval %dms" % self.reflectorMinRx )

      print( "\nLegend:" )
      print( "*: pseudo LAG session (not counted in total sessions)" )
      print( "<N>[<M>]: Number of sessions",
             "[ Number of sessions with echo enabled ]\n" )

      headings = ( "Addressing", "Type", "Up", "Init", "Down", "AdminDown" )
      formatLeft = TableOutput.Format( justify="left" )
      formatLeft.noPadLeftIs( True )
      formatRight = TableOutput.Format( justify="right" )
      table = TableOutput.createTable( headings )
      table.formatColumns( formatLeft, formatLeft, formatRight, formatRight,
         formatRight, formatRight )

      printRow( table, self.sessions[ "All" ], "All", "All" )
      singlehopTypes = [
                   "singlehopAll",
                   OperSessionType.sessionTypeNormal,
                   OperSessionType.sessionTypeLagRfc7130,
                   OperSessionType.sessionTypeMicroRfc7130,
                   OperSessionType.sessionTypeLagLegacy,
                   OperSessionType.sessionTypeMicroLegacy,
                 ]
      if "IPv4" in self.sessions:
         sessionModel = self.sessions[ "IPv4" ]
         printRow( table, sessionModel, "IPv4", sessionType="All" )
         for sessionType in singlehopTypes:
            printRow( table, sessionModel, "    single hop" if
                      sessionType == "singlehopAll" else "", sessionType )
         printRow( table, sessionModel, "    multi-hop",
                   OperSessionType.sessionTypeMultihop )
         sessionModel = self.sessions[ "IPv6" ]
         printRow( table, sessionModel, "IPv6", sessionType="All" )
         for sessionType in singlehopTypes:
            printRow( table, sessionModel, "    single hop" if
                      sessionType == "singlehopAll" else "", sessionType )
         printRow( table, sessionModel, "    multi-hop",
                   OperSessionType.sessionTypeMultihop )
         printRow( table, self.sessions[ "Tunnel" ], "Tunnel",
                   OperSessionType.sessionTypeVxlanTunnel )
         printRow( table, self.sessions[ "L2" ], "L2",
                   OperSessionType.sessionTypeLagRfc7130 )
         printRow( table, self.sessions[ "L2" ], "",
                   OperSessionType.sessionTypeMicroRfc7130 )

      if "SR-TE Tunnel" in self.sessions:
         self.renderSbfdSummary( table, "SR-TE Tunnel", "srte" )
      if "EOS SDK SBFD" in self.sessions:
         self.renderSbfdSummary( table, "EOS SDK SBFD", "eosSdkSbfd" )

      print( table.output() )

class BfdDebug( Model ):
   downSessions = Dict( keyType=str, valueType=int,
                        help="Mapping of diag to number of down sessions" )

   class Issue( Model ):
      issueList = List( help="List of issues", valueType=str )
   issues = Dict( keyType=str, valueType=Issue,
                  help="Mapping of issue type to issue detail" )
   slowTxs = Dict( keyType=str, valueType=int,
                   help="Mapping of peer to number of slow tx" )
   rxIntervals = Dict( keyType=str, valueType=str,
                       help="Mapping of peer to rx interval" )
   slowEchoTxs = Dict( keyType=str, valueType=int,
                       help="Mapping of peer to number of slow echo tx" )
   echoRxIntervals = Dict( keyType=str, valueType=str,
                           help="Mapping of peer to echo rx interval" )

   def addIssue( self, key, issue ):
      if key in self.issues:
         self.issues[ key ].issueList.append( issue )
      else:
         issueModel = BfdDebug.Issue()
         issueModel.issueList.append( issue )
         self.issues[ key ] = issueModel

   def render( self ):
      if self.downSessions:
         print( "------- Number of sessions down --------" )
         for diag, numSessions in self.downSessions.items():
            print( diag, ': ', numSessions )
         print( '\n' )

      for key, issueModel in self.issues.items():
         if issueModel:
            print( key )
            for issue in sorted( issueModel.issueList ):
               print( issue )

      header = [ "VRF", "DstAddr", "Interface", "Type" ]
      formatLeft = TableOutput.Format( justify="left" )
      formatLeft.noPadLeftIs( True )

      def printTable( headings, rows ):
         table = TableOutput.createTable( headings )
         table.formatColumns( formatLeft, formatLeft, formatLeft, formatLeft,
                              formatLeft )
         for peer, number in sorted( rows.items() ):
            tokens = peer.split()
            table.newRow( tokens[ 0 ], tokens[ 1 ], tokens[ 2 ], tokens[ 3 ],
                          number )
         print( table.output() )

      if self.slowTxs:
         printTable( tuple( header + [ "GT 3*TxInt" ] ),
                     self.slowTxs )

      if self.rxIntervals:
         printTable( tuple( header + [ "Rx interval (ms) min/max/avg" ] ),
                     self.rxIntervals )

      if self.slowEchoTxs:
         printTable( tuple( header + [ "GT 3*TxInt" ] ),
                     self.slowEchoTxs )

      if self.echoRxIntervals:
         printTable( tuple( header + [ "Echo rx interval (ms) min/max/avg" ] ),
                     self.echoRxIntervals )

class BfdHwAccel( Model ):
   supported = Bool( help="Hardware acceleration is supported by platform" )
   running = Bool( help="Hardware acceleration is running" )
   reasons = List( help="List of reasons why hardware acceleration is not running",
                   valueType=str )

   def render( self ):
      if not self.supported:
         print( "Hardware acceleration is not supported" )
      elif self.running:
         print( "Hardware acceleration is running" )
      else:
         print( "Hardware acceleration is not running{}".format(
               ": " + ", ".join( self.reasons ) if self.reasons else "" ) )

class BfdHwResourceSession( BfdSessionBase_ ):
   ip = IpGenericAddress( help="IP address of peer" )
   intf = Interface( help="Local egress interface" )
   vrf = Str( help="VRF that peer is configured in" )

class BfdHwResource( Model ):
   maxSessions = Int( help="Maximum number of sessions configured of the chip" )
   numSessions = Int( help="Number of sessions configured on a chip", optional=True )
   ssoSupported = Bool( help="HW BFD SSO is supported" )
   sessions = List( valueType=BfdHwResourceSession,
         help="List of sessions configured on the chip", optional=True )

class BfdHwResourceList( Model ):
   resources = Dict( valueType=BfdHwResource,
         help="Hardware resource information keyed by the resource's name" )
   _detailed = Bool( help="Detailed show bfd hardware utilization command" )
   _sso = Bool( help="SSO is configured" )

   def render( self ):
      formatLeft = TableOutput.Format( justify="left" )
      formatLeft.noPadLeftIs( True )
      formatRight = TableOutput.Format( justify="right" )
      if not self._detailed:
         header = ( "Chip Name", "Number Of HW Sessions*",
               "Maximum Number Of HW Sessions*", "Remarks" )

         table = TableOutput.createTable( header )
         table.formatColumns( formatLeft, formatRight, formatRight, formatLeft )
         for name, resource in sorted( self.resources.items() ):
            remark = "SSO not supported" \
               if self._sso and not resource.ssoSupported else ""
            table.newRow( name, resource.numSessions, resource.maxSessions, remark )
         print( table.output().rstrip( "\n" ) )
         print( '* A BFD session with echo enabled typically consumes two hardware '
                'sessions' )
         print()

      else:
         header = ( "Dst Addr", "My Disc", "Interface", "VRF", "Type" )

         for name, resource in sorted( self.resources.items() ):
            table = TableOutput.createTable( header )
            table.formatColumns( formatLeft, formatRight, formatLeft, formatLeft,
                  formatLeft )
            for session in sorted( resource.sessions, key=compareHwSessionKey ):
               table.newRow( session.ip, session.localDisc, session.intf.stringValue,
                     session.vrf, operSessionEnumToStr[ session.sessType ] )
            print( 'Chip:', name )
            print( table.output() )

class VxlanVtepBfdStatus( Model ):
   vteps = Dict( keyType=IpGenericAddress, valueType=str,
                 help="A mapping of VXLAN VTEP to its BFD status" )

class BfdVxlanMlagPrimary( Model ):
   interfaces = Dict( keyType=Interface, valueType=VxlanVtepBfdStatus,
                      help="Mapping from VXLAN interface to its VTEP addresses" )

   def render( self ):
      table = TableOutput.createTable( ( 'VTEP', 'BFD Status' ) )
      vtepFormat = TableOutput.Format( justify='left' )
      vtepFormat.noPadLeftIs( True )
      operStateFormat = TableOutput.Format( justify='left' )
      table.formatColumns( vtepFormat, operStateFormat )
      for vti, entry in self.interfaces.items():
         print( "Remote VTEPS for %s on MLAG primary:\n" % vti )
         for vtep, operState in entry.vteps.items():
            table.newRow( vtep, operState )
      print( table.output() )
