#!/usr/bin/env python3
# Copyright (c) 2019 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# pylint: disable=consider-using-f-string
# pylint: disable=consider-using-in

import os
import re
import socket
import sys
from collections import namedtuple
from itertools import chain
from struct import ( unpack )
import ArPyUtils
import Cell
import BasicCli
import CliCommand
from CliCommand import guardedKeyword, singleKeyword, singleNode
import CliGlobal
import CliMatcher
from CliMatcher import EnumMatcher
import ShowCommand
import CliParser
import CliPlugin.AclCli as AclCli # pylint: disable=consider-using-from-import
from CliPlugin.AclCliModel import AllAclList
import CliPlugin.BfdModel as BfdModel # pylint: disable=consider-using-from-import
from CliPlugin.BfdModel import (
   BfdDebug,
   BfdHwAccel,
   BfdHwResourceList,
   BfdNeighbors,
   BfdSummary,
   BfdVxlanMlagPrimary,
   PeerHistory,
   RbfdStats,
   SbfdStats,
   SegmentType,
   segmentTypeIntToEnum,
   VxlanVtepBfdStatus,
)
from CliPlugin.Ip6AddrMatcher import Ip6AddrMatcher
from CliPlugin.IpAddrMatcher import IpAddrMatcher
# pylint: disable-next=consider-using-from-import
import CliPlugin.TechSupportCli as TechSupportCli
from CliPlugin.VrfCli import ALL_VRF_NAME, VrfExprFactory, getAllVrfNames, vrfExists
from CliToken.Ip import ipMatcherForShow
from CliToken.Ipv6 import ipv6MatcherForShow
from Intf.IntfRange import intfRangeMatcher
import LazyMount
import Smash
import SmashLazyMount
import Tac
import Tracing
from IpLibConsts import DEFAULT_VRF
from BfdLib import ( diagEnumToReason,
                     authType,
                     authEnumTypeToText,
                     authEnumNumToText,
                     operSessionEnumToType,
                     peerSessionAf )
from DeviceNameLib import eosIntfToKernelIntf
from TypeFuture import TacLazyType

TunnelTableIdentifier = TacLazyType( 'Tunnel::TunnelTable::TunnelTableIdentifier' )
TunnelTableMounter = TacLazyType( 'Tunnel::TunnelTable::TunnelTableMounter' )
TunnelIdType = TacLazyType( "Tunnel::TunnelTable::TunnelId" )
TunnelType = TacLazyType( 'Tunnel::TunnelTable::TunnelType' )
BfdConstants = TacLazyType( 'Bfd::Constants' )
BfdRole = TacLazyType( 'Bfd::BfdRole' )
OperState = TacLazyType( "Bfd::OperState" )
defaultSbfdClientStatus = Tac.newInstance(
   'Bfd::ClientStatus', OperState.init, OperState.up, False )

traceHandle = Tracing.defaultTraceHandle()
t0 = traceHandle.trace0

# Global variable holder.
gv = CliGlobal.CliGlobal(
   dict(
      aclCpConfig=None,
      aclStatus=None,
      aclCheckpoint=None,
      appConfigDir=None,
      bfdConfigGlobal=None,
      bfdHwStatus=None,
      bfdHwConfig=None,
      bfdStatusPeer=None,
      bfdStatusIntfInfo=None,
      bfdPeerHistory=None,
      bfdConfigInfo=None,
      bfdConfigPeer=None,
      bfdOperInfoPeer=None,
      bfdRxStatsSmash=None,
      bfdTxStatsSmash=None,
      configBfdLag=None,
      allIntfStatusDir=None,
      bfdHwResourceConfig=None,
      redundancyConfig=None,
      ipConfig=None,
      ip6Config=None,
      srteSegmentList=None,
      eosSdkSbfdTunnelTable=None,
      mlagVxlanStatus=None,
   )
)

matcherBfd = CliMatcher.KeywordMatcher( 'bfd',
      helpdesc='Status for the BFD protocol' )
matcherDebug = CliMatcher.KeywordMatcher( 'debug',
      helpdesc='Bfd debug information' )
matcherDestIp = CliMatcher.KeywordMatcher( 'dest-ip',
      helpdesc='BFD neighbor IP address' )
matcherNbrDetail = CliMatcher.KeywordMatcher( 'detail',
      helpdesc='Detailed neighbor information' )
matcherHistory = CliMatcher.KeywordMatcher( 'history',
      helpdesc='Neighbor transition history' )
matcherInterface = CliMatcher.KeywordMatcher( 'interface',
      helpdesc='Interface keyword' )
matcherInterval = CliMatcher.KeywordMatcher( 'interval',
      helpdesc='Transmit rate in milliseconds' )
matcherMin_RxDeprecated = singleKeyword( 'min_rx',
      helpdesc='Expected minimum incoming rate in milliseconds',
      hidden=True )
matcherMinRx = CliMatcher.KeywordMatcher( 'min-rx',
      helpdesc='Expected minimum incoming rate in milliseconds' )
matcherMultiplier = CliMatcher.KeywordMatcher( 'multiplier',
      helpdesc='BFD multiplier' )
neighborsDeprecatedNode = CliCommand.Node(
      matcher=CliMatcher.KeywordMatcher( 'neighbors',
                  helpdesc='BFD Neighbor information' ),
      deprecatedByCmd='show bfd peers' )
matcherPeers = CliMatcher.KeywordMatcher( 'peers',
      helpdesc='BFD Neighbor information' )
matcherSrcIp = CliMatcher.KeywordMatcher( 'src-ip',
      helpdesc='BFD source IP address' )
matcherSummary = CliMatcher.KeywordMatcher( 'summary',
      helpdesc='BFD summary' )
nodeDetail = guardedKeyword( 'detail',
      helpdesc='Access list detail',
      guard=AclCli.countersPerChipEnabledGuard )
matcherHardware = CliMatcher.KeywordMatcher( 'hardware',
      helpdesc='Hardware-specific BFD parameters' )
emptyResourceCompletion = [ CliParser.Completion( 'RESOURCE', 'Resource name',
                                                  literal=False ) ]
matcherResource = CliMatcher.DynamicKeywordMatcher(
      lambda mode: [ r for rConfig in gv.bfdHwResourceConfig.values()
         for r in rConfig.hwResourceLimits ],
      emptyTokenCompletion=emptyResourceCompletion )
matcherProtocol = CliMatcher.KeywordMatcher( 'protocol',
      helpdesc='Subscribing protocol' )
matcherSbfd = CliMatcher.KeywordMatcher( 'sbfd',
      helpdesc='Status for the SBFD sessions only', )
matcherInitiators = CliMatcher.KeywordMatcher( 'initiators',
      helpdesc='SBFD initiators' )
matcherReflectors = CliMatcher.KeywordMatcher( 'reflectors',
      helpdesc='SBFD reflectors' )
vrfExprFactory = VrfExprFactory(
      helpdesc='BFD information of VRF',
      maxMatches=1 )

AddressFamily = Tac.Type( 'Arnet::AddressFamily' )
configSessionType = Tac.Type( 'Bfd::SessionType' )
operSessionType = Tac.Type( 'Bfd::OperSessionType' )
historyInfo = Tac.Value( 'Bfd::HistoryInfo' )
operState = Tac.Type( 'Bfd::OperState' )

sbfdGlobalfields = [ 'rxDrop', 'txSend', 'rxValid', 'rxTruncated', 'rxInvalidVer',
     'rxInvalidLen', 'rxInvalidSrcPort', 'rxDemandUnset', 'rxInvalidMulti',
     'rxInvalidMyDisc', 'rxYourDiscUnmatch', 'rxInvalidBfdState',
     'rxUnsupported', 'rxInvalidPktFormat', 'rxInvalidTlv' ]
sbfdGlobalfields += [ 'txSendToFail', 'txInvalidLen', 'rxHashMiss', 'rxHashHit',
      'rxHashTrim', 'padding' ]
sbfdReflectorGlobal = namedtuple( "sbfdReflectorGlobal", sbfdGlobalfields )

sbfdReflectorSessFields = [ 'srcIp', 'disc', 'minTxInt',
     'minRxInt', 'minEchoRxInt', 'minPeriod', 'maxPeriod',
     'lastPktYourDisc', 'srcPort', 'detectMulti', 'bfdState', 'lastBfdState',
     'diag', 'version', 'demand', 'final', 'poll', 'lastPktLen',
     'sentBfdState', 'lastSentBfdState', 'sentDiag' ]
sbfdReflectorSessFields += [ 'segmentType', 'addrFamily', 'seg1', 'seg2',
   'seg3', 'seg4', 'seg5', 'seg6' ]
sbfdReflectorSessFields += [ 'rxNumReceived', 'rxTotalPeriod',
     'rxLastRxTimeSec', 'rxLastRxTimeNsec',
     'rxLastStateChgSec', 'rxLastStateChgNsec',
     'txNumSent',
     'txLastTxTimeSec', 'txLastTxTimeNsec',
     'txLastStateChgSec', 'txLastStateChgNsec' ]
sbfdReflectorSess = namedtuple( "sbfdReflectorSess", sbfdReflectorSessFields )

# Reflector Size constants
globalSize = 88
sessionSize = 148
# TLV error is an addition
globalSize += 8
# Segment List is an addition
sessionSize += 24
sbfdReflectorCacheSize = { 'global': globalSize,
                           'session': sessionSize,
                           'maxSess': 512,
}

def getReflectorStats():
   globalStats = {}
   sessions = {}
   shmName = '/dev/shm/sbfdreflectord'
   nspath = os.getenv( 'NSPATH' )
   if nspath:
      shmName = shmName + nspath.replace( '/', '-' )
   if not os.path.exists( shmName ):
      # sbfdReflectord is not configured and never run
      return globalStats, sessions
   cacheSize = sbfdReflectorCacheSize[ 'global' ] + \
            sbfdReflectorCacheSize[ 'session' ] * sbfdReflectorCacheSize[ 'maxSess' ]
   if os.path.getsize( shmName ) != cacheSize:
      t0( "sbfdReflector statistics are of wrong size, expected %d" % cacheSize )
      return globalStats, sessions
   with open( shmName, 'rb' ) as f:
      buf = f.read( sbfdReflectorCacheSize[ 'global' ] )
      globalStats = sbfdReflectorGlobal( *unpack( '3Q18I', buf ) )
      for _ in range( sbfdReflectorCacheSize[ 'maxSess' ] ):
         buf = f.read( sbfdReflectorCacheSize[ 'session' ] )
         sess = sbfdReflectorSess( *unpack( '=16s7IH14B6I11Q', buf ) )
         if sess.disc:
            ipStr = socket.inet_ntop( socket.AF_INET6, sess.srcIp ) \
               if sess.addrFamily == socket.AF_INET6 \
               else socket.inet_ntop( socket.AF_INET, sess.srcIp[ 0 : 4 ] )
            sessions[ ( ipStr, sess.disc ) ] = sess
   return globalStats, sessions

def getReflectorPeerStatus( sess, localIpAddr=None ):
   ipStr = socket.inet_ntop( socket.AF_INET6, sess.srcIp ) \
               if sess.addrFamily == socket.AF_INET6 \
                  else socket.inet_ntop( socket.AF_INET, sess.srcIp[ 0 : 4 ] )
   ip = Tac.Value( 'Arnet::IpGenAddr', ipStr )
   peer = Tac.Value( 'Bfd::Peer', ip, DEFAULT_VRF )
   peer.type = configSessionType.sbfdReflector
   peerStatus = Tac.newInstance( 'Bfd::PeerStatus', peer,
                                 operSessionType.sessionTypeSbfdReflector,
                                 BfdRole.active,
                                 defaultSbfdClientStatus )
   peerStatus.localDisc = gv.bfdConfigGlobal.sbfdReflectorLocalDisc
   peerStatus.remoteDisc = sess.disc
   peerStatus.status = stateName[ sess.sentBfdState ]
   peerStatus.authType = authType.authNone
   peerStatus.localIp = localIpAddr
   return peerStatus

def showBfdNeighborHistory( vrfNames=None, intfSet=None,
                            destIpAddr=None, srcIpAddr=None,
                            minTx=None, minRx=None, multiplier=None, proto=None,
                            tunnelId=None, af=None, bfdOnly=False, sbfdOnly=False ):
   seq = gv.bfdPeerHistory.currentSeq
   peerHistoryModel = BfdModel.PeerHistory()
   for _ in range( len( gv.bfdPeerHistory.historyEntries ) ):
      # Currently printing most recent entry first. This might need to change.
      if seq == 0:
         seq = historyInfo.maxEntries
      seq -= 1
      peerHistory = gv.bfdPeerHistory.historyEntries.get( seq, None )
      if peerHistory is None:
         continue
      # peerHistory is sorted and the loop starts from the most recent entry,
      # so we can return as soon as genId != historyGeneration
      if peerHistory.genId != gv.bfdConfigInfo.historyGeneration:
         return peerHistoryModel
      pst = peerHistory.peerSnapshot
      if ( peerMatchesFilter( pst.peer, vrfNames=vrfNames, intfSet=intfSet,
                              destIpAddr=destIpAddr, srcIpAddr=srcIpAddr,
                              minTx=minTx, minRx=minRx, multiplier=multiplier,
                              proto=proto, tunnelId=tunnelId, addrFamily=af,
                              bfdOnly=bfdOnly, sbfdOnly=sbfdOnly ) ):
         peerSnapshotModel = printNeighborHistory( pst )
         peerHistoryModel.peerSnapshots.append( peerSnapshotModel )
   return peerHistoryModel

MAX_U64 = 0xFFFFFFFFFFFFFFFF
MAX_U32 = 0xFFFFFFFF

def convertBfdToSecs( sec, nsec ):
   return float( sec + nsec / 1000000000 )

def getTxStats( peerStatus, netlinkStub, echo=False ):
   txStats = None
   txStatsSmash = None
   hwAsync = False
   disc = peerStatus.localDisc
   if echo:
      disc = peerStatus.echoDisc

   if gv.bfdHwStatus.isHwAccelerated( disc ):
      txStatsSmash = gv.bfdTxStatsSmash.txStats.get( disc )
      hwAsync = True

   if txStatsSmash:
      txStats = Tac.newInstance( 'BfdPyUtils::TxStatsEntry', disc )
      txStats.fromSmashStats( txStatsSmash )
   else:
      txStats = netlinkStub.receivedTxStats.get( disc )
   return txStats, hwAsync

def getRxStats( peerStatus, netlinkStub, echo=False ):
   rxStats = None
   rxStatsSmash = None
   hwAsync = False
   disc = peerStatus.localDisc
   if echo:
      disc = peerStatus.echoDisc

   if gv.bfdHwStatus.isHwAccelerated( disc ):
      rxStatsSmash = gv.bfdRxStatsSmash.rxStats.get( disc )
      hwAsync = True

   if rxStatsSmash:
      rxStats = Tac.newInstance( 'BfdPyUtils::RxStatsEntry', disc )
      rxStats.fromSmashStats( rxStatsSmash )
   else:
      rxStats = netlinkStub.receivedRxStats.get( disc )
   return rxStats, hwAsync

def generateTxStatsModel( disc, txStats ):
   txStatsModel = BfdModel.TxStats()
   txStatsModel.numSent = txStats.bfdMsgTxStats.numSent
   if not gv.bfdHwStatus.isHwAccelerated( disc ) or \
         gv.bfdHwConfig.hwAccelerationConfig.lastTxTimeStatSupported:
      txStatsModel.minLateness = txStats.bfdMsgTxStats.minLateness
      txStatsModel.maxLateness = txStats.bfdMsgTxStats.maxLateness
      txStatsModel.avgLateness = txStats.bfdMsgTxStats.avgLateness
      txStatsModel.lastTxTime = convertBfdToSecs(
                                    txStats.bfdMsgTxStats.lastTxTimeSec,
                                    txStats.bfdMsgTxStats.lastTxTimeNsec )
      txStatsModel.sentWithin1p = txStats.bfdMsgTxStats.sentWithin1p
      txStatsModel.sentWithin2p = txStats.bfdMsgTxStats.sentWithin2p
      txStatsModel.sentWithin3p = txStats.bfdMsgTxStats.sentWithin3p
      txStatsModel.sentAfter3p = txStats.bfdMsgTxStats.sentAfter3p
   return txStatsModel

def generateRxStatsModel( disc, rxStats ):
   rxStatsModel = BfdModel.RxStats()
   rxStatsModel.numReceived = rxStats.bfdMsgRxStats.numReceived
   if not gv.bfdHwStatus.isHwAccelerated( disc ) or \
         gv.bfdHwConfig.hwAccelerationConfig.lastRxTimeStatSupported:
      rxStatsModel.minPeriod = rxStats.bfdMsgRxStats.minPeriod
      rxStatsModel.maxPeriod = rxStats.bfdMsgRxStats.maxPeriod
      rxStatsModel.avgPeriod = rxStats.bfdMsgRxStats.avgPeriod
      rxStatsModel.lastRxTime = convertBfdToSecs(
                                    rxStats.bfdMsgRxStats.lastRxTimeSec,
                                    rxStats.bfdMsgRxStats.lastRxTimeNsec )

   return rxStatsModel

def isRttEnabled( rxStats ):
   # If lastRtt is a positive value, then RTT is enabled for this peer
   return bool( rxStats and rxStats.bfdMsgRxStats.lastRtt )

def generateRttStatsModel( rxStats ):
   rttStatsModel = BfdModel.RttStats()
   rttStatsModel.minRtt = rxStats.bfdMsgRxStats.minRtt
   rttStatsModel.maxRtt = rxStats.bfdMsgRxStats.maxRtt
   rttStatsModel.avgRtt = rxStats.bfdMsgRxStats.avgRtt
   rttStatsModel.lastRtt = rxStats.bfdMsgRxStats.lastRtt
   return rttStatsModel

def updatePeerStatsDetailReflector( peer, peerStatus, peerStatsModel,
                                    peerStatsDetailModel, reflectorSess ):
   sess = reflectorSess.get( ( str( peer.ip ), peerStatus.remoteDisc ) )
   if not sess:
      t0( "session (%s, %d) doesn't exist" % ( peer.ip, peerStatus.remoteDisc ) )
      return False
   peerStatsModel.localDisc = peerStatus.localDisc
   peerStatsModel.remoteDisc = peerStatus.remoteDisc
   peerStatsModel.status = peerStatus.status
   peerStatsModel.sessType = peerStatus.type
   peerStatsModel.destPort = sess.srcPort
   peerStatsModel.lastDiag = diagToEnum[ sess.sentDiag ]
   peerStatsModel.authType = authEnumTypeToText[ peerStatus.authType ]
   peerStatsModel.authProfileName = ''
   peerStatsModel.l3intf = ''
   if peerStatus.status == operState.up:
      peerStatsModel.lastUp = float( sess.txLastStateChgSec +
                                     sess.txLastStateChgNsec / 1000000000 )
      peerStatsModel.lastDown = 0.0
   else:
      peerStatsModel.lastUp = 0.0
      peerStatsModel.lastDown = float( sess.txLastStateChgSec +
                                       sess.txLastStateChgNsec / 1000000000 )
   peerStatsModel.rxStats = BfdModel.RxStats()
   peerStatsModel.rxStats.numReceived = sess.rxNumReceived
   peerStatsModel.rxStats.minPeriod = sess.minPeriod
   peerStatsModel.rxStats.maxPeriod = sess.maxPeriod
   if sess.rxNumReceived > 1:
      peerStatsModel.rxStats.avgPeriod = int( sess.rxTotalPeriod //
                                              ( sess.rxNumReceived - 1 ) )
   else:
      peerStatsModel.rxStats.avgPeriod = 0
   peerStatsModel.rxStats.lastRxTime = float( sess.rxLastRxTimeSec +
                                              sess.rxLastRxTimeNsec / 1000000000 )
   peerStatsModel.rxStats.lastRxStateChg = float(
      sess.rxLastStateChgSec + sess.rxLastStateChgNsec / 1000000000 )
   peerStatsModel.rxStats.lastRxState = stateName[ sess.lastBfdState ]
   peerStatsModel.txStats = BfdModel.TxStats()
   peerStatsModel.txStats.numSent = sess.txNumSent
   peerStatsModel.txStats.lastTxTime = float( sess.txLastTxTimeSec +
                                              sess.txLastTxTimeNsec / 1000000000 )
   peerStatsDetailModel.operRxInterval = gv.bfdConfigGlobal.sbfdReflectorMinRx
   peerStatsDetailModel.localIpAddr = str( peerStatus.localIp )

   # Last rx packet info
   peerStatsDetailModel.lastVersion = sess.version
   peerStatsDetailModel.lastDiag = sess.diag
   peerStatsDetailModel.lastState = stateName[ sess.bfdState ]
   peerStatsDetailModel.lastDemand = bool( sess.demand )
   peerStatsDetailModel.lastPoll = bool( sess.poll )
   peerStatsDetailModel.lastFinal = bool( sess.final )
   peerStatsDetailModel.lastDetectMult = sess.detectMulti
   peerStatsDetailModel.lastLength = sess.lastPktLen
   peerStatsDetailModel.lastMyDiscU32 = sess.disc
   peerStatsDetailModel.lastYourDiscU32 = sess.lastPktYourDisc
   peerStatsDetailModel.lastMinTxIntv = sess.minTxInt
   peerStatsDetailModel.lastMinRxIntv = sess.minRxInt
   peerStatsDetailModel.lastEchoRxIntv = sess.minEchoRxInt
   peerStatsDetailModel.segmentType = segmentTypeIntToEnum[ sess.segmentType ]
   segmentList = []
   if peerStatsDetailModel.segmentType == SegmentType.mplsLabel:
      segmentList += ( [ sess.seg1 ] +
                       ( [ sess.seg2 ] if sess.seg2 else [] ) +
                       ( [ sess.seg3 ] if sess.seg3 else [] ) +
                       ( [ sess.seg4 ] if sess.seg4 else [] ) +
                       ( [ sess.seg5 ] if sess.seg5 else [] ) +
                       ( [ sess.seg6 ] if sess.seg6 else [] ) )
   peerStatsDetailModel.segments = segmentList
   peerStatsModel.peerStatsDetail = peerStatsDetailModel
   return True

def updateTunnelInfoSrte( tunnelId, peerStatsDetailModel ):
   slId = tunnelId.extractIndexAndAf()
   sl = gv.srteSegmentList.segmentList.get( slId )
   if not sl:
      return
   tunnelInfo = BfdModel.TunnelInfo()
   # 0th index in segment list has top label
   tunnelInfo.mplsLabels = [ sl.segment.get( i ).mplsLabel
                             for i in range( sl.size ) ]
   tunnelInfo.mplsExp = BfdConstants.sbfdDefaultMplsExp
   tunnelInfo.ipDscp = Tac.enumValue( 'Arnet::IpDscp', gv.bfdConfigGlobal.dscpValue )
   tunnelInfo.sbfdReturnPathLabelStack = \
      [ sl.sbfdReturnPath.labelStack( i )
        for i in range( sl.sbfdReturnPath.stackSize ) ]
   peerStatsDetailModel.tunnelInfo = tunnelInfo

def updateTunnelInfoEosSdkSbfd( tunnelId, peerStatsDetailModel ):
   entry = gv.eosSdkSbfdTunnelTable.entry.get( tunnelId )
   if not entry:
      return
   tunnelInfo = BfdModel.TunnelInfo()
   via = entry.via
   # 0th index in labels has bottom label
   tunnelInfo.mplsLabels = [ via.labels.label( i )
                             for i in range( via.labels.stackSize - 1, -1, -1 ) ]
   tunnelInfo.mplsExp = via.mplsExp
   tunnelInfo.ipDscp = Tac.enumValue( 'Arnet::IpDscp', via.ipDscp )
   tunnelInfo.via = BfdModel.Via()
   tunnelInfo.via.nextHop = via.nexthop
   tunnelInfo.via.interface = via.intfId
   peerStatsDetailModel.tunnelInfo = tunnelInfo

def updateTunnelInfo( tunnelId, peerStatsDetailModel ):
   tunnelId = TunnelIdType( tunnelId )
   tunnelType = tunnelId.tunnelType()
   if tunnelType == TunnelType.eosSdkSbfdSessionTunnel:
      updateTunnelInfoEosSdkSbfd( tunnelId, peerStatsDetailModel )
   elif tunnelType == TunnelType.srTeSegmentListTunnel:
      updateTunnelInfoSrte( tunnelId, peerStatsDetailModel )
   else:
      assert False, "Unknown Tunnel Type %s" % tunnelType

def getStatsFromAccelerator( mode, netlinkStub, disc,
      clear=False, noSend=False, module=False ):
   netlinkStub.sendGetStats( disc, clear, noSend, module )
   netlinkStub.bfdRead()
   if netlinkStub.errCode != 0: # errCode 0 is a netlink ACK
      mode.addError(
            'Fetch of stats from BFD accelerator failed due to: {err}.'
            '\nStats may be incomplete.'.format(
                  err=os.strerror( netlinkStub.errCode ) )
      )

def updatePeerStatsDetail( mode, peer, peerStatus, peerStatsModel,
                           peerStatsDetailModel, operInfo, netlinkStub, apps,
                           sessType, destIpAddr=None, reflectorSess=None ):
   if peerStatus.type == operSessionType.sessionTypeSbfdReflector:
      return updatePeerStatsDetailReflector( peer, peerStatus, peerStatsModel,
                                             peerStatsDetailModel, reflectorSess )

   l3intf = peer.intf
   if sessType == operSessionType.sessionTypeMicroLegacy or\
      sessType == operSessionType.sessionTypeMicroRfc7130:
      l3intf = lagIntfIdForIntfId( peer.intf )
      if l3intf is None:
         # unable to look up which port-channel this physical port is
         # configured in. The safe thing to do here to return without printing
         # anything.
         return False

   peerStatsModel.status = str( peerStatus.status )
   peerStatsModel.sessType = sessType
   peerStatsModel.localDisc = peerStatus.localDisc
   peerStatsModel.remoteDisc = peerStatus.remoteDisc
   peerStatsModel.lastUp = operInfo.lastUp
   peerStatsModel.lastDown = operInfo.lastDown
   peerStatsModel.lastDiag = diagToEnum[ operInfo.lastDiag ]
   peerStatsModel.l3intf = l3intf
   peerStatsModel.tunnelId = peer.tunnelId
   peerStatsModel.authType = authEnumTypeToText[ peerStatus.authType ]
   peerStatsModel.authProfileName = peerStatus.secretProfileName
   peerStatsDetailModel.role = peerStatus.role
   peerStatsDetailModel.localIpAddr = str( peerStatus.localIp )
   peerStatsDetailModel.apps = sorted(
         mapAppName( a, showDetail=True ) for a in apps )
   peerStatsDetailModel.echoOn = peerStatus.echoOn
   peerStatsDetailModel.echoActive = peerStatus.echoActive
   peerStatsDetailModel.echoDisc = peerStatus.echoDisc
   peerStatsDetailModel.operEchoTxRxInterval = \
      peerStatus.operEchoTxRxIntervalInMicrosecond
   peerStatsDetailModel.operTxInterval = peerStatus.operTxIntervalInMicrosecond
   peerStatsDetailModel.operRxInterval = peerStatus.operRxIntervalInMicrosecond
   peerStatsDetailModel.detectMult = peerStatus.detectMult
   peerStatsDetailModel.remoteMinRxInterval = \
      peerStatus.remoteMinRxIntervalInMicrosecond
   peerStatsDetailModel.remoteDetectMult = peerStatus.remoteDetectMult
   peerStatsDetailModel.detectTime = peerStatus.detectTimeInMicrosecond
   peerStatsModel.proxy = peerStatus.proxy

   tunnelId = TunnelIdType( peer.tunnelId )
   tunnelType = tunnelId.tunnelType()
   if ( tunnelType == TunnelType.eosSdkSbfdSessionTunnel or
         tunnelType == TunnelType.srTeSegmentListTunnel ):
      peerStatsModel.tunnelType = tunnelType
      peerStatsModel.tunnelIdx = tunnelId.extractIndexAndAf()

   # Get stats for this particular peer, if destIpAddr is not None
   if destIpAddr:
      getStatsFromAccelerator( mode, netlinkStub, peerStatus.localDisc )
   txStats, hwAccelTxAsync = getTxStats( peerStatus, netlinkStub, echo=False )
   rxStats, hwAccelRxAsync = getRxStats( peerStatus, netlinkStub, echo=False )
   hwAccelAsync = hwAccelTxAsync or hwAccelRxAsync

   peerStatsDetailModel.hwAcceleratedStates[ "Async" ] = hwAccelAsync

   if rxStats:
      peerStatsModel.rxStats = generateRxStatsModel( peerStatus.localDisc, rxStats )

   if txStats:
      peerStatsModel.txStats = generateTxStatsModel( peerStatus.localDisc, txStats )

   if isRttEnabled( rxStats ):
      peerStatsModel.rttStats = generateRttStatsModel( rxStats )

   # Get echo stats for this particular peer, if destIpAddr is not None
   if destIpAddr:
      getStatsFromAccelerator( mode, netlinkStub, peerStatus.echoDisc )
   txStats, hwAccelTxEcho = getTxStats( peerStatus, netlinkStub, echo=True )
   rxStats, hwAccelRxEcho = getRxStats( peerStatus, netlinkStub, echo=True )
   hwAccelEcho = hwAccelTxEcho or hwAccelRxEcho
   peerStatsDetailModel.hwAcceleratedStates[ "Echo" ] = hwAccelEcho

   if rxStats:
      peerStatsModel.echoRxStats = \
            generateRxStatsModel( peerStatus.localDisc, rxStats )

   if txStats:
      peerStatsModel.echoTxStats = \
            generateTxStatsModel( peerStatus.localDisc, txStats )

   peerStatsDetailModel.lastVersion = peerStatus.lastVersion
   peerStatsDetailModel.lastDiag = peerStatus.lastDiag
   peerStatsDetailModel.lastState = stateName[ peerStatus.lastState ]
   peerStatsDetailModel.lastDemand = peerStatus.lastDemand
   peerStatsDetailModel.lastPoll = peerStatus.lastPoll
   peerStatsDetailModel.lastFinal = peerStatus.lastFinal
   peerStatsDetailModel.lastDetectMult = peerStatus.lastDetectMult
   peerStatsDetailModel.lastLength = peerStatus.lastLength
   peerStatsDetailModel.lastMyDiscU32 = peerStatus.lastMyDiscU32
   peerStatsDetailModel.lastYourDiscU32 = peerStatus.lastYourDiscU32
   peerStatsDetailModel.lastMinTxIntv = peerStatus.lastMinTxIntvU32
   peerStatsDetailModel.lastMinRxIntv = peerStatus.lastMinRxIntvU32
   peerStatsDetailModel.lastEchoRxIntv = peerStatus.lastEchoRxIntvU32

   if peerStatus.microBfdPeer:
      localSsoReady = True
      for microPeer in peerStatus.microBfdPeer:
         microPeerStatus = gv.bfdStatusPeer.peerStatus.get( microPeer )
         if not microPeerStatus:
            continue
         if not gv.bfdHwStatus.isHwAccelerated( microPeerStatus.localDisc ):
            localSsoReady = False
            break

      # pylint: disable-msg=protected-access
      peerStatsDetailModel._localSsoReady = localSsoReady

   if sessType == operSessionType.sessionTypeSbfdInitiator:
      updateTunnelInfo( peer.tunnelId, peerStatsDetailModel )

   peerStatsModel.peerStatsDetail = peerStatsDetailModel

   return True

def updateNeighborsModel( peer, peerStatus, bfdNeighborsModel, peerStatsModel ):
   bfdNeighborVrfModel = bfdNeighborsModel.vrfs.setdefault( peer.vrf,
                         BfdModel.BfdNeighborVrf() )
   if peerStatus.type == operSessionType.sessionTypeSbfdInitiator:
      if peerStatus.localIp.af == 'ipv4':
         neighbors = bfdNeighborVrfModel.ipv4InitiatorNeighbors
      else:
         neighbors = bfdNeighborVrfModel.ipv6InitiatorNeighbors
      sbfdInitiatorTunnelModel = neighbors.setdefault( str( peer.ip ),
            BfdModel.SbfdInitiatorTunnel() )
      sbfdInitiatorTunnelModel.peerStats[ int( peer.tunnelId ) ] = \
            peerStatsModel
   elif peerStatus.type == operSessionType.sessionTypeSbfdReflector:
      if peerStatus.localIp.af == 'ipv4':
         neighbors = bfdNeighborVrfModel.ipv4ReflectorNeighbors
      else:
         neighbors = bfdNeighborVrfModel.ipv6ReflectorNeighbors
      sbfdReflectorModel = neighbors.setdefault( str( peer.ip ),
            BfdModel.SbfdReflector() )
      sbfdReflectorModel.peerStats[ int( peerStatus.remoteDisc ) ] = peerStatsModel
   else:
      if peer.ip.af == 'ipv4':
         neighbors = bfdNeighborVrfModel.ipv4Neighbors
      else:
         neighbors = bfdNeighborVrfModel.ipv6Neighbors
      bfdNeighborInterfaceModel = neighbors.setdefault( str( peer.ip ),
            BfdModel.BfdNeighborInterface() )
      bfdNeighborIntfPeerModel = bfdNeighborInterfaceModel.peers.setdefault(
            peer.intf, BfdModel.BfdNeighborIntfPeer() )
      bfdNeighborSrcAddrModel = bfdNeighborIntfPeerModel.types.setdefault(
            operSessionEnumToType[ peerStatus.type ], BfdModel.BfdNeighborSrcAddr() )
      bfdNeighborSrcAddrModel.peerStats[ str( peerStatus.localIp ) ] = \
            peerStatsModel

def printBfdNeighborDetailEntry( mode, peerStatus, bfdNeighborsModel, netlinkStub,
                                 destIpAddr=None, reflectorSess=None ):
   try:
      peer = peerStatus.peer
      peerStatsModel = BfdModel.PeerStats()
      peerStatsDetailModel = BfdModel.PeerStatsDetail()
      if peerStatus.type == operSessionType.sessionTypeSbfdReflector:
         operInfo = None
         apps = None
      else:
         peerOperInfo = gv.bfdOperInfoPeer.peerOperInfo[ peer ]
         operInfo = peerOperInfo.info
         apps = getApp( peer )
         if apps is None:
            return
   except KeyError:
      # This peer has been removed during the show command, so skip it.
      return
   sessType = peerStatus.type
   if not updatePeerStatsDetail( mode, peer, peerStatus, peerStatsModel,
                                 peerStatsDetailModel, operInfo, netlinkStub,
                                 apps, sessType, destIpAddr, reflectorSess ):
      # failed to update peerStatsModel, so skip it
      return

   updateNeighborsModel( peer, peerStatus, bfdNeighborsModel, peerStatsModel )

def showBfdNeighborHelper( mode, bfdNeighborsModel, displayType, addrFamily=None,
                           vrfNames=None, intfSet=None, destIpAddr=None,
                           srcIpAddr=None, minTx=None, minRx=None, multiplier=None,
                           resource=None, proto=None, tunnelId=None,
                           bfdOnly=False, sbfdOnly=False,
                           sbfdInitiatorOnly=False, sbfdReflectorOnly=False,
                           reflectorSess=None ):
   # pylint: disable=too-many-nested-blocks
   if not sbfdReflectorOnly and gv.bfdStatusPeer.peerStatus:
      if displayType == 'detail':
         netlinkStub = Tac.newInstance( "BfdPyUtils::BfdNetlinkControlStub" )
         netlinkStub.createFd( False )
         if destIpAddr is None:
            # We get stats for all peers.
            getStatsFromAccelerator( mode, netlinkStub, 0 )

      for peerStatus in gv.bfdStatusPeer.peerStatus.values():
         if ( peerMatchesFilter( peerStatus.peer, addrFamily=addrFamily,
                                 vrfNames=vrfNames, intfSet=intfSet,
                                 destIpAddr=destIpAddr, srcIpAddr=srcIpAddr,
                                 minTx=minTx, minRx=minRx, multiplier=multiplier,
                                 resource=resource, proto=proto, tunnelId=tunnelId,
                                 bfdOnly=bfdOnly, sbfdOnly=sbfdOnly ) ):
            if displayType == 'detail':
               printBfdNeighborDetailEntry( mode, peerStatus, bfdNeighborsModel,
                                            netlinkStub, destIpAddr,
                                            reflectorSess=reflectorSess )
               continue

            try:
               peer = peerStatus.peer
               intfToPeerMap = gv.bfdStatusIntfInfo.intfToPeerMap.get( peer.intf )
               peerOperInfo = gv.bfdOperInfoPeer.peerOperInfo[ peer ].info

               peerStatsModel = BfdModel.PeerStats()
               peerStatsModel.localDisc = peerStatus.localDisc
               peerStatsModel.remoteDisc = peerStatus.remoteDisc
               peerStatsModel.status = peerStatus.status
               peerStatsModel.sessType = peerStatus.type
               peerStatsModel.lastUp = peerOperInfo.lastUp
               peerStatsModel.lastDown = peerOperInfo.lastDown
               peerStatsModel.lastDiag = diagToEnum[ peerOperInfo.lastDiag ]
               peerStatsModel.kernelIfIndex = ( intfToPeerMap.kernelIfIndex
                                                if intfToPeerMap else 0 )
               peerStatsModel.authType = authEnumTypeToText[ peerStatus.authType ]
               peerStatsModel.authProfileName = peerStatus.secretProfileName
               l3intf = peer.intf
               if ( peerStatus.type == operSessionType.sessionTypeMicroLegacy or
                    peerStatus.type == operSessionType.sessionTypeMicroRfc7130 ):
                  l3intf = lagIntfIdForIntfId( peer.intf )
                  if not l3intf:
                     # unable to look up which port-channel this physical port is
                     # configured in. The safe thing to do here to return without
                     # printing anything.
                     return
               peerStatsModel.l3intf = l3intf
               peerStatsModel.tunnelId = peer.tunnelId

               updateNeighborsModel( peer, peerStatus, bfdNeighborsModel,
                                     peerStatsModel )

            except KeyError:
               # This peer has been removed during the show command, so skip it.
               continue

   if ( not bfdOnly and not sbfdInitiatorOnly and
        addrFamily and AddressFamily.ipv4 in addrFamily and
        vrfNames and DEFAULT_VRF in vrfNames and
        not minTx and not multiplier and not resource and not proto ):

      if ( minRx and minRx != gv.bfdConfigGlobal.sbfdReflectorMinRx ):
         return
      for sess in reflectorSess.values():
         if sess.addrFamily == socket.AF_INET:
            ipIntfConfig =\
               gv.ipConfig.ipIntfConfig.get( gv.bfdConfigGlobal.sbfdLocalIntf )
            localIpStr = ipIntfConfig.addrWithMask.address\
                           if ipIntfConfig else '0.0.0.0'
         else:
            ip6IntfConfig =\
               gv.ip6Config.intf.get( gv.bfdConfigGlobal.sbfdLocalIntfIp6 )
            localIpStr = "::"
            if ip6IntfConfig:
               for addr6 in ip6IntfConfig.addr:
                  if not addr6.address.isLinkLocal and not addr6.address.isLoopback\
                                                and not addr6.address.isUnspecified:
                     localIpStr = addr6.stringValue.split( "/" )[ 0 ]
                     break
         localIpAddr = Tac.Value( 'Arnet::IpGenAddr', localIpStr )

         if ( srcIpAddr and srcIpAddr != localIpStr ):
            return

         ipStr = socket.inet_ntop( socket.AF_INET6, sess.srcIp ) \
               if sess.addrFamily == socket.AF_INET6 \
                  else socket.inet_ntop( socket.AF_INET, sess.srcIp[ 0 : 4 ] )
         if destIpAddr and destIpAddr != ipStr:
            continue
         peerStatus = getReflectorPeerStatus( sess, localIpAddr )
         if displayType == 'detail':
            printBfdNeighborDetailEntry( mode, peerStatus, bfdNeighborsModel,
                                         None, destIpAddr,
                                         reflectorSess=reflectorSess )
            continue

         p = peerStatus.peer
         peerStatsModel = BfdModel.PeerStats()
         peerStatsModel.localDisc = peerStatus.localDisc
         peerStatsModel.remoteDisc = peerStatus.remoteDisc
         peerStatsModel.status = peerStatus.status
         peerStatsModel.sessType = peerStatus.type
         sess = reflectorSess.get( ( str( p.ip ), peerStatus.remoteDisc ) )
         peerStatsModel.destPort = sess.srcPort
         peerStatsModel.lastDiag = diagToEnum[ sess.sentDiag ]
         peerStatsModel.authType = authEnumTypeToText[ peerStatus.authType ]
         peerStatsModel.authProfileName = ''
         peerStatsModel.l3intf = ''
         if peerStatus.status == operState.up:
            peerStatsModel.lastUp = float( sess.txLastStateChgSec +
                                           sess.txLastStateChgNsec / 1000000000 )
            peerStatsModel.lastDown = 0.0
         else:
            peerStatsModel.lastUp = 0.0
            peerStatsModel.lastDown = float( sess.txLastStateChgSec +
                                             sess.txLastStateChgNsec / 1000000000 )
         updateNeighborsModel( p, peerStatus, bfdNeighborsModel,
                               peerStatsModel )

def discoverLagIntfs( intfs ):
   lagMembers = set()
   for intf in intfs:
      # Assuming that we won't find non-PortChannel intfs in there.
      if config := gv.configBfdLag.bfdLagConfig.get( intf ):
         lagMembers.update( config.bfdReqPort )
   return lagMembers

def showBfdNeighbor( mode, vrfName=None, intfs=None, destIpAddr=None, srcIpAddr=None,
                     minTx=None, minRx=None, multiplier=None, proto=None,
                     tunnelId=None, displayType=None, addrFamilies=None,
                     bfdOnly=False, sbfdOnly=False, sbfdInitiatorOnly=False,
                     sbfdReflectorOnly=False ):

   if vrfName == ALL_VRF_NAME:
      vrfNames = set( getAllVrfNames( mode ) )
   elif vrfExists( vrfName ):
      vrfNames = [ vrfName ]
   else:
      mode.addError( "VRF %s not configured." % vrfName )
      if displayType == 'summary':
         return showBfdSummary( mode, [ vrfName ] )
      elif displayType == 'debug':
         return BfdModel.BfdDebug()
      elif displayType == 'history':
         return BfdModel.PeerHistory()
      elif displayType == 'detail':
         return BfdModel.BfdNeighbors( _detailed=True )
      else:
         # If request is not for bfd neighbor history|detail|summary|debug
         # then assumption is that it is for just bfd neighbor.
         return BfdModel.BfdNeighbors( _detailed=False )

   intfSet = set( intfs )
   # Include member intfs if filtered by LAG intf
   intfSet.update( discoverLagIntfs( intfs ) )

   if destIpAddr is not None and not isinstance( destIpAddr, str ):
      destIpAddr = destIpAddr.stringValue
   if srcIpAddr is not None and not isinstance( srcIpAddr, str ):
      srcIpAddr = srcIpAddr.stringValue

   if addrFamilies is None:
      addrFamilies = [ 'ipv4', 'ipv6' ]
   else:
      addrFamilies = [ addrFamilies ]

   # pylint: disable=too-many-function-args

   if ( bfdOnly or sbfdInitiatorOnly ):
      reflectorSess = {}
   else:
      _, reflectorSess = getReflectorStats()

   if displayType == 'summary':
      return showBfdSummary( mode, vrfNames, intfSet, destIpAddr, srcIpAddr,
                             minTx, minRx, multiplier, proto, tunnelId=tunnelId,
                             af=addrFamilies,
                             bfdOnly=bfdOnly, sbfdOnly=sbfdOnly,
                             sbfdInitiatorOnly=sbfdInitiatorOnly,
                             sbfdReflectorOnly=sbfdReflectorOnly,
                             reflectorSess=reflectorSess )
   elif displayType == 'debug':
      return showBfdDebug( mode, vrfNames, intfSet, destIpAddr, srcIpAddr,
                           minTx, minRx, multiplier, proto, addrFamilies,
                           tunnelId=tunnelId, bfdOnly=bfdOnly, sbfdOnly=sbfdOnly )
   # pylint: disable=too-many-nested-blocks
   elif displayType == 'history':
      return showBfdNeighborHistory( vrfNames, intfSet, destIpAddr, srcIpAddr,
                                     minTx, minRx, multiplier, proto,
                                     tunnelId=tunnelId, af=addrFamilies,
                                     bfdOnly=bfdOnly, sbfdOnly=sbfdOnly )
   else:

      if displayType == 'detail':
         bfdNeighborsModel = BfdModel.BfdNeighbors( _detailed=True )
      else:
         bfdNeighborsModel = BfdModel.BfdNeighbors( _detailed=False )

      showBfdNeighborHelper( mode,
                             bfdNeighborsModel,
                             displayType,
                             addrFamily=addrFamilies,
                             vrfNames=vrfNames,
                             intfSet=intfSet,
                             destIpAddr=destIpAddr,
                             srcIpAddr=srcIpAddr,
                             minTx=minTx, minRx=minRx,
                             multiplier=multiplier,
                             proto=proto,
                             tunnelId=tunnelId,
                             bfdOnly=bfdOnly,
                             sbfdOnly=sbfdOnly,
                             sbfdInitiatorOnly=sbfdInitiatorOnly,
                             sbfdReflectorOnly=sbfdReflectorOnly,
                             reflectorSess=reflectorSess )

      return bfdNeighborsModel

helpDescBfdShow = 'Status for the BFD protocol'

def incrementCount( typeModel, sessType, peerStatus ):
   countByType = typeModel.types[ sessType ]
   countByState = countByType.states[ peerStatus.status ]
   async_ = getattr( countByState, 'async' )
   setattr( countByState, 'async', async_ + 1 )
   if peerStatus.echoActive:
      countByState.echo += 1

def getSbfdInitiatorSummarySessionApp( peerStatus ):
   tunnelType = TunnelIdType( peerStatus.peer.tunnelId ).tunnelType()
   if tunnelType == TunnelType.eosSdkSbfdSessionTunnel:
      return ( "EOS SDK SBFD", "eosSdkSbfd" )
   elif tunnelType == TunnelType.srTeSegmentListTunnel:
      return ( "SR-TE Tunnel", "srte" )
   else:
      assert False, "Unknown Tunnel Type %s" % tunnelType
      return None

def showBfdSummary( mode, vrfNames=None, intfSet=None, destIpAddr=None,
                    srcIpAddr=None, minTx=None, minRx=None, multiplier=None,
                    proto=None, tunnelId=None, af=None, bfdOnly=False,
                    sbfdOnly=False, sbfdInitiatorOnly=False, sbfdReflectorOnly=False,
                    reflectorSess=None ):
   bfdSummaryModel = BfdModel.BfdSummary()
   bfdSummaryModel.minTx = gv.bfdConfigGlobal.globalCfg.minTx
   bfdSummaryModel.minRx = gv.bfdConfigGlobal.globalCfg.minRx
   bfdSummaryModel.mult = gv.bfdConfigGlobal.globalCfg.mult
   if gv.bfdConfigGlobal.multihopGlobalCfg is not None:
      bfdSummaryModel.mhMinTx = gv.bfdConfigGlobal.multihopGlobalCfg.minTx
      bfdSummaryModel.mhMinRx = gv.bfdConfigGlobal.multihopGlobalCfg.minRx
      bfdSummaryModel.mhMult = gv.bfdConfigGlobal.multihopGlobalCfg.mult
   else:
      bfdSummaryModel.mhMinTx = gv.bfdConfigGlobal.globalCfg.minTx
      bfdSummaryModel.mhMinRx = gv.bfdConfigGlobal.globalCfg.minRx
      bfdSummaryModel.mhMult = gv.bfdConfigGlobal.globalCfg.mult
   bfdSummaryModel.slowTimer = gv.bfdConfigGlobal.bfdSlowTimer
   bfdSummaryModel.adminDown = gv.bfdConfigGlobal.bfdAdminDown
   bfdSummaryModel.sessionStatsInterval = gv.bfdConfigGlobal.sessionStatsInterval
   bfdSummaryModel.dscpValue = Tac.enumValue( 'Arnet::IpDscp',
                                              gv.bfdConfigGlobal.dscpValue ) >> 2

   bfdSummaryModel.localIntf = gv.bfdConfigGlobal.sbfdLocalIntf
   bfdSummaryModel.localIntfIp6 = gv.bfdConfigGlobal.sbfdLocalIntfIp6
   bfdSummaryModel.initiatorMinTx = gv.bfdConfigGlobal.sbfdInitiatorConfig.minTx
   bfdSummaryModel.initiatorMult = gv.bfdConfigGlobal.sbfdInitiatorConfig.mult
   bfdSummaryModel.reflectorLocalDisc = gv.bfdConfigGlobal.sbfdReflectorLocalDisc
   bfdSummaryModel.reflectorMinRx = gv.bfdConfigGlobal.sbfdReflectorMinRx
   # pylint: disable=protected-access
   bfdSummaryModel._initiatorOnly = sbfdInitiatorOnly
   bfdSummaryModel._reflectorOnly = sbfdReflectorOnly

   ipIntfConfig = gv.ipConfig.ipIntfConfig.get( bfdSummaryModel.localIntf )
   if ipIntfConfig:
      bfdSummaryModel.localIntfIpAdd = ipIntfConfig.addrWithMask.address
   else:
      bfdSummaryModel.localIntfIpAdd = Tac.Value( 'Arnet::IpGenAddr', '0.0.0.0' )

   ip6IntfConfig = gv.ip6Config.intf.get( bfdSummaryModel.localIntfIp6 )
   if ip6IntfConfig:
      for addr6 in ip6IntfConfig.addr:
         if not addr6.address.isLinkLocal and not addr6.address.isLoopback\
                                          and not addr6.address.isUnspecified:
            localIpStr = addr6.stringValue.split( "/" )[ 0 ]
            bfdSummaryModel.localIntfIp6Add =\
               Tac.Value( 'Arnet::IpGenAddr', localIpStr )
            break
   else:
      bfdSummaryModel.localIntfIp6Add = Tac.Value( 'Arnet::IpGenAddr', "::" )

   showBoth = not bfdOnly and not sbfdOnly

   types = set( operSessionType.attributes )
   sessions = []
   if bfdOnly:
      sessions = [ "All", "IPv4", "IPv6", "Tunnel", "L2" ]
      types.update( [ "All", "singlehopAll" ] )
   elif sbfdOnly:
      sessions = [ "All", "SR-TE Tunnel", "srteIPv4", "srteIPv6" ]
      sessions.extend( [ "EOS SDK SBFD", "eosSdkSbfdIPv4", "eosSdkSbfdIPv6" ] )
      types.update( [ "All" ] )
   else:
      sessions = [ "All", "IPv4", "IPv6", "Tunnel", "L2", "SR-TE Tunnel",
                   "srteIPv4", "srteIPv6" ]
      sessions.extend( [ "EOS SDK SBFD", "eosSdkSbfdIPv4", "eosSdkSbfdIPv6" ] )
      types.update( [ "All", "singlehopAll" ] )

   for session in sessions:
      bfdSummaryModel.sessions[ session ] = BfdModel.SessionCountByType()
      for sessType in types:
         model = BfdModel.SessionCountByState()
         bfdSummaryModel.sessions[ session ].types[ sessType ] = model
         for state in operState.attributes:
            model.states[ state ] = BfdModel.SessionCount()
            setattr( model.states[ state ], 'async', 0 )
            model.states[ state ].echo = 0

   allPeerStatus = filterPeerStatus( vrfNames=vrfNames, intfSet=intfSet,
                                     destIpAddr=destIpAddr, srcIpAddr=srcIpAddr,
                                     minTx=minTx, minRx=minRx,
                                     multiplier=multiplier, proto=proto,
                                     tunnelId=tunnelId, addrFamily=af,
                                     bfdOnly=bfdOnly, sbfdOnly=sbfdOnly,
                                     sbfdInitiatorOnly=sbfdInitiatorOnly,
                                     sbfdReflectorOnly=sbfdReflectorOnly,
                                     reflectorSess=reflectorSess )
   if not allPeerStatus:
      return bfdSummaryModel

   if bfdOnly or showBoth:
      for peerStatus in allPeerStatus:
         typeModel = None
         sessType = peerStatus.type
         if ( sessType == operSessionType.sessionTypeSbfdInitiator or
              sessType == operSessionType.sessionTypeSbfdReflector ):
            continue
         if sessType == operSessionType.sessionTypeMicroLegacy or \
            sessType == operSessionType.sessionTypeMicroRfc7130:
            lagIntf = lagIntfIdForIntfId( peerStatus.peer.intf )
            if lagIntf is None:
               # unable to look up which port-channel this physical port is
               # configured in. The safe thing to do here to return without printing
               # anything.
               continue
            intfStatus = gv.allIntfStatusDir.intfStatus[ lagIntf ]
            if intfStatus and intfStatus.forwardingModel != \
                                                      'intfForwardingModelRouted':
               typeModel = bfdSummaryModel.sessions[ 'L2' ]
         elif sessType == operSessionType.sessionTypeLagRfc7130:
            intfStatus = gv.allIntfStatusDir.intfStatus[ peerStatus.peer.intf ]
            if intfStatus and intfStatus.forwardingModel != \
                                                      'intfForwardingModelRouted':
               typeModel = bfdSummaryModel.sessions[ 'L2' ]

         if not typeModel:
            if sessType == operSessionType.sessionTypeVxlanTunnel:
               typeModel = bfdSummaryModel.sessions[ 'Tunnel' ]
            elif '.' in peerStatus.peer.ip.stringValue:
               typeModel = bfdSummaryModel.sessions[ 'IPv4' ]
            else:
               typeModel = bfdSummaryModel.sessions[ 'IPv6' ]

         incrementCount( typeModel, sessType, peerStatus )
         if sessType != operSessionType.sessionTypeLagLegacy and \
            sessType != operSessionType.sessionTypeLagRfc7130:
            incrementCount( typeModel, "All", peerStatus )
            if sessType != operSessionType.sessionTypeMultihop:
               incrementCount( typeModel, "singlehopAll", peerStatus )
            incrementCount( bfdSummaryModel.sessions[ "All" ], "All", peerStatus )

   if sbfdOnly or showBoth:
      for peerStatus in allPeerStatus:
         typeModel = None
         sessType = peerStatus.type
         if ( sessType == operSessionType.sessionTypeSbfdInitiator
              and not sbfdReflectorOnly ):
            sbfdApp, sbfdAppAf = getSbfdInitiatorSummarySessionApp( peerStatus )
            if peerStatus.localIp.af == AddressFamily.ipv6:
               typeModel = bfdSummaryModel.sessions[ '%sIPv6' % sbfdAppAf ]
            else:
               typeModel = bfdSummaryModel.sessions[ '%sIPv4' % sbfdAppAf ]

            incrementCount( typeModel, sessType, peerStatus )
            incrementCount( bfdSummaryModel.sessions[ sbfdApp ], "All",
                            peerStatus )
            incrementCount( bfdSummaryModel.sessions[ "All" ], "All", peerStatus )

         if ( sessType == operSessionType.sessionTypeSbfdReflector
              and not sbfdInitiatorOnly ):
            typeModel = bfdSummaryModel.sessions[ 'srteIPv4' ]
            incrementCount( typeModel, sessType, peerStatus )
            incrementCount( bfdSummaryModel.sessions[ "SR-TE Tunnel" ], "All",
                            peerStatus )
            incrementCount( bfdSummaryModel.sessions[ "All" ], "All", peerStatus )

   return bfdSummaryModel

def lagIntfIdForIntfId( intfId ):
   for key in gv.configBfdLag.bfdLagConfig:
      bfdLagConfig = gv.configBfdLag.bfdLagConfig[ key ]
      if bfdLagConfig.bfdReqPort.get( intfId, None ):
         return bfdLagConfig.intf
   return None

diagToEnum = { 0: 'diagNone',
               1: 'diagCtrlTimeout',
               2: 'diagEchoFail',
               3: 'diagNeighDown',
               4: 'diagForwardingReset',
               5: 'diagPathDown',
               6: 'diagConcatPathDown',
               7: 'diagAdminDown',
               8: 'diagRevConcatPathDown'
             }

stateName = { 0: 'adminDown',
              1: 'down',
              2: 'init',
              3: 'up',
            }

def printNeighborHistory( pst ):
   peerSnapshotModel = BfdModel.PeerSnapshot()

   peerSnapshotModel.ip = pst.peer.ip
   if pst.peer.type == 'sbfdInitiator':
      peerSnapshotModel.intf = ''
   else:
      peerSnapshotModel.intf = pst.peer.intf
   peerSnapshotModel.vrf = pst.peer.vrf
   peerSnapshotModel.status = pst.status
   peerSnapshotModel.prevState = pst.prevState
   peerSnapshotModel.eventTime = pst.eventTime
   peerSnapshotModel.event = pst.event
   peerSnapshotModel.lastUp = pst.lastUp
   peerSnapshotModel.lastDown = pst.lastDown
   peerSnapshotModel.localIpAddr = pst.localIpAddr
   peerSnapshotModel.localDisc = pst.localDisc
   peerSnapshotModel.remoteDisc = pst.remoteDisc
   peerSnapshotModel.lastDiag = diagToEnum[ pst.lastDiag ]
   peerSnapshotModel.authType = authEnumNumToText[ pst.authType ]
   peerSnapshotModel.authProfileName = pst.profileName
   peerSnapshotModel.txInterval = pst.txInterval
   peerSnapshotModel.rxInterval = pst.rxInterval
   peerSnapshotModel.detectTime = pst.detectTime
   peerSnapshotModel.rxCount = pst.rxCount
   peerSnapshotModel.txCount = pst.txCount
   peerSnapshotModel.minLateness = pst.minLateness
   peerSnapshotModel.avgLateness = pst.avgLateness
   peerSnapshotModel.maxLateness = pst.maxLateness
   peerSnapshotModel.minPeriod = pst.minPeriod
   peerSnapshotModel.avgPeriod = pst.avgPeriod
   peerSnapshotModel.maxPeriod = pst.maxPeriod
   peerSnapshotModel.lastTxTime = convertBfdToSecs( pst.lastTxTimeSec,
         pst.lastTxTimeNsec )
   peerSnapshotModel.lastRxTime = convertBfdToSecs( pst.lastRxTimeSec,
         pst.lastRxTimeNsec )
   peerSnapshotModel.sentWithin1p = pst.sentWithin1p
   peerSnapshotModel.sentWithin2p = pst.sentWithin2p
   peerSnapshotModel.sentWithin3p = pst.sentWithin3p
   peerSnapshotModel.sentAfter3p = pst.sentAfter3p
   peerSnapshotModel.echoOn = pst.echoOn
   peerSnapshotModel.echoTxInterval = pst.echoTxInterval
   peerSnapshotModel.echoRxInterval = pst.echoRxInterval
   peerSnapshotModel.echoDetectTime = pst.echoDetectTime
   peerSnapshotModel.echoRxCount = pst.echoRxCount
   peerSnapshotModel.echoTxCount = pst.echoTxCount
   peerSnapshotModel.echoSentWithin1p = pst.echoSentWithin1p
   peerSnapshotModel.echoSentWithin2p = pst.echoSentWithin2p
   peerSnapshotModel.echoSentWithin3p = pst.echoSentWithin3p
   peerSnapshotModel.echoSentAfter3p = pst.echoSentAfter3p
   peerSnapshotModel.echoMinLateness = pst.echoMinLateness
   peerSnapshotModel.echoAvgLateness = pst.echoAvgLateness
   peerSnapshotModel.echoMaxLateness = pst.echoMaxLateness
   peerSnapshotModel.echoMinPeriod = pst.echoMinPeriod
   peerSnapshotModel.echoAvgPeriod = pst.echoAvgPeriod
   peerSnapshotModel.echoMaxPeriod = pst.echoMaxPeriod
   peerSnapshotModel.echoLastTxTime = convertBfdToSecs( pst.echoLastTxTimeSec,
         pst.echoLastTxTimeNsec )
   peerSnapshotModel.echoLastRxTime = convertBfdToSecs( pst.echoLastRxTimeSec,
         pst.echoLastRxTimeNsec )
   peerSnapshotModel.tunnelId = pst.peer.tunnelId

   return peerSnapshotModel

def intfMatchesFilter( intf, resource ):
   for resourceConfig in gv.bfdHwResourceConfig.values():
      if resourceConfig.hwResource.get( intf ) == resource:
         return True
      elif resourceConfig.hwResourceForVirtualInterface == resource:
         return True
   return False

def peerMatchesFilter( peer, addrFamily=None, vrfNames=None, intfSet=None,
                       destIpAddr=None, srcIpAddr=None,
                       minTx=None, minRx=None, multiplier=None, proto=None,
                       tunnelId=None, resource=None, bfdOnly=False, sbfdOnly=False ):
   if ( bfdOnly and peer.type == 'sbfdInitiator' ) or \
         ( sbfdOnly and peer.type != 'sbfdInitiator' ):
      return False
   if not vrfNames:
      vrfNames = [ DEFAULT_VRF ]
   peerStatus = gv.bfdStatusPeer.peerStatus.get( peer )
   hwSessionStatus = None
   echoHwSessionStatus = None
   if peerStatus:
      hwSessionStatus = gv.bfdHwStatus.sessionStatus.get( peerStatus.localDisc )
      if peerStatus.echoOn:
         echoHwSessionStatus = gv.bfdHwStatus.sessionStatus.get(
            peerStatus.echoDisc )

   peerApps = getApp( peer )

   return ( ( not addrFamily or peerSessionAf( peer, gv.bfdConfigGlobal )
         in addrFamily )
            and peer.vrf in vrfNames
            and ( not destIpAddr or str( peer.ip ) == destIpAddr )
            and ( not srcIpAddr or peerStatus and
                                   str( peerStatus.localIp ) == srcIpAddr )
            and ( not intfSet or peer.intf in intfSet )
            and ( not minTx or peerStatus and
                               peerStatus.intervalConfig.minTx == minTx )
            and ( not minRx or peerStatus and
                               peerStatus.intervalConfig.minRx == minRx )
            and ( not multiplier or peerStatus and
                                    peerStatus.intervalConfig.mult == multiplier )
            and ( not resource or
               ( hwSessionStatus and hwSessionStatus.hwAccelerated and
                 intfMatchesFilter( hwSessionStatus.hwAccelIntf, resource ) ) or
               ( echoHwSessionStatus and echoHwSessionStatus.hwAccelerated and
                 intfMatchesFilter( echoHwSessionStatus.hwAccelIntf, resource ) ) )
            and ( not proto or
                  ( peerApps and
                    proto in { mapAppName( a ) for a in peerApps } ) )
            and ( tunnelId is None or peer.tunnelId == tunnelId )
   )

def filterPeerStatus( addrFamily=None, vrfNames=None, intfSet=None,
                      destIpAddr=None, srcIpAddr=None,
                      minTx=None, minRx=None, multiplier=None,
                      resource=None, proto=None, tunnelId=None,
                      bfdOnly=False, sbfdOnly=False,
                      sbfdInitiatorOnly=False, sbfdReflectorOnly=False,
                      reflectorSess=None ):

   if not sbfdReflectorOnly:
      peers = [ p for p in gv.bfdStatusPeer.peerStatus.values()
                if ( peerMatchesFilter( p.peer, addrFamily=addrFamily,
                                        vrfNames=vrfNames, intfSet=intfSet,
                                        destIpAddr=destIpAddr, srcIpAddr=srcIpAddr,
                                        minTx=minTx, minRx=minRx,
                                        multiplier=multiplier, resource=resource,
                                        proto=proto, tunnelId=tunnelId,
                                        bfdOnly=bfdOnly, sbfdOnly=sbfdOnly ) ) ]
   else:
      peers = []

   if ( sbfdInitiatorOnly or bfdOnly or
        not addrFamily or AddressFamily.ipv4 not in addrFamily or
        not vrfNames or DEFAULT_VRF not in vrfNames or not reflectorSess or
        minTx or multiplier or resource or proto ):
      return peers

   if minRx and minRx != gv.bfdConfigGlobal.sbfdReflectorMinRx:
      return peers

   for sess in reflectorSess.values():
      if sess.addrFamily == socket.AF_INET:
         ipIntfConfig =\
            gv.ipConfig.ipIntfConfig.get( gv.bfdConfigGlobal.sbfdLocalIntf )
         localIpStr = ipIntfConfig.addrWithMask.address\
                        if ipIntfConfig else '0.0.0.0'
      else:
         ip6IntfConfig =\
               gv.ip6Config.intf.get( gv.bfdConfigGlobal.sbfdLocalIntfIp6 )
         localIpStr = "::"
         if ip6IntfConfig:
            for addr6 in ip6IntfConfig.addr:
               if not addr6.address.isLinkLocal and not addr6.address.isLoopback\
                                                and not addr6.address.isUnspecified:
                  localIpStr = addr6.stringValue.split( "/" )[ 0 ]
                  break
      localIpAddr = Tac.Value( 'Arnet::IpGenAddr', localIpStr )

      if ( srcIpAddr and srcIpAddr != localIpStr ):
         return peers

      ipStr = socket.inet_ntop( socket.AF_INET6, sess.srcIp ) \
               if sess.addrFamily == socket.AF_INET6 \
                  else socket.inet_ntop( socket.AF_INET, sess.srcIp[ 0 : 4 ] )
      if destIpAddr and destIpAddr != ipStr:
         continue
      peerStatus = getReflectorPeerStatus( sess, localIpAddr )
      peers.append( peerStatus )
   return peers

# ------------------------------------------------------------
# check for rfc-7130 specific issues:
# rfc-7130 mac address not registred in kernel
# ipv6 is disabled at /proc/sys/net/ipv6/conf/ for ipv6 peer
# local route not installed for po interface
# ------------------------------------------------------------
def checkRfc7130( allPeerStatus, bfdDebugModel ):
   routesByVrf = {}
   key1 = '--------- rfc-7130 mac address 01:00:5e:90:00:01 not registered ' \
          'in kernel---------'
   key2 = '--------- ipv6 disabled at /proc/sys/net/ipv6/conf/ ---------'
   for peerStatus in allPeerStatus:
      peer = peerStatus.peer
      if peerStatus.type == operSessionType.sessionTypeMicroRfc7130 and \
         peerStatus.localIp:
         if peer.vrf not in routesByVrf:
            routesByVrf[ peer.vrf ] = set()
         routesByVrf[ peer.vrf ].add( peerStatus.localIp.stringValue )

      elif peerStatus.type == operSessionType.sessionTypeLagRfc7130:
         # check whether rfc-7130 mac address is registered in kernel
         vrfCmd = 'ip netns exec ns-%s' % peer.vrf \
                  if peer.vrf != DEFAULT_VRF else ''
         devName = eosIntfToKernelIntf( peer.intf )

         cmd = f'{vrfCmd} ip maddr show dev {devName}'
         output = Tac.run( cmd.split(), stdout=Tac.CAPTURE, asRoot=True,
                           stderr=Tac.CAPTURE, ignoreReturnCode=True )
         if '01:00:5e:90:00:01' not in output:
            bfdDebugModel.addIssue( key1, output )

         # for IPv6 peer check whether disable_ipv6 is 0
         if ':' in peer.ip.stringValue:
            cmd = 'cat /proc/sys/net/ipv6/conf/%s/disable_ipv6' % devName
            output = Tac.run( cmd.split(), stdout=Tac.CAPTURE, asRoot=True,
                              stderr=Tac.CAPTURE, ignoreReturnCode=True )
            if output.strip() != '0':
               bfdDebugModel.addIssue( key2, peer.intf )

   key3 = '--------- missing rfc-7130 required routes ------------'
   for vrf, routes in routesByVrf.items():
      kernelRoutes = set()
      vrfCmd = 'ip netns exec ns-%s' % vrf if vrf != DEFAULT_VRF else ''
      for cmd in [ '%s ip route show table local' % vrfCmd,
                   '%s ip -6 route show table local' % vrfCmd ]:
         output = Tac.run( cmd.split(), stdout=Tac.CAPTURE, asRoot=True,
                           stderr=Tac.CAPTURE, ignoreReturnCode=True )
         lines = output.split( '\n' )
         for line in lines:
            tokens = line.split()
            if len( tokens ) > 2:
               kernelRoutes.add( tokens[ 1 ] )
      for route in routes:
         if route not in kernelRoutes:
            bfdDebugModel.addIssue( key3, vrf + ' ' + route )

# ------------------------------------------------------------
# check peerStatus for following:
# peer with slow tx: tx delay > 3 *TxInt
# peer with minRx < 5ms or maxRx > detectTime
# number of sessions down and session diag respectively
# ------------------------------------------------------------
def checkBfdPeerStatus( mode, allPeerStatus, bfdDebugModel ):
   if not allPeerStatus:
      return
   netlinkStub = Tac.newInstance( "BfdPyUtils::BfdNetlinkControlStub" )
   netlinkStub.createFd( False )
   getStatsFromAccelerator( mode, netlinkStub, 0 )
   minrx_threshold = 5

   # sessions with slow tx and bursty/slow rx - search "show bfd neighbor detail"
   for peerStatus in allPeerStatus:
      try:
         peer = peerStatus.peer
         peerOperInfo = gv.bfdOperInfoPeer.peerOperInfo[ peer ]
         operInfo = peerOperInfo.info
      except KeyError:
         # This peer has been removed during the show command, so skip it.
         continue

      # down session count
      if peerStatus.status != 'up':
         diag = diagEnumToReason[ diagToEnum[ operInfo.lastDiag ] ]
         if diag not in bfdDebugModel.downSessions:
            bfdDebugModel.downSessions[ diag ] = 0
         bfdDebugModel.downSessions[ diag ] += 1

      txStats, _ = getTxStats( peerStatus, netlinkStub, echo=False )
      rxStats, _ = getRxStats( peerStatus, netlinkStub, echo=False )

      peerStr = ( peer.vrf + ' ' + peer.ip.stringValue + ' ' +
                  ( peer.intf if peer.intf else 'NA' ) + ' ' + peer.type )
      if rxStats:
         minPeriod = rxStats.bfdMsgRxStats.minPeriod / 1000
         maxPeriod = rxStats.bfdMsgRxStats.maxPeriod / 1000
         avgPeriod = rxStats.bfdMsgRxStats.avgPeriod / 1000
         if minPeriod < minrx_threshold or \
            maxPeriod > peerStatus.detectTimeInMicrosecond:
            bfdDebugModel.rxIntervals[ peerStr ] = str( minPeriod ) + '/' \
                                       + str( maxPeriod ) + '/' + str( avgPeriod )

      if txStats and txStats.bfdMsgTxStats.sentAfter3p:
         bfdDebugModel.slowTxs[ peerStr ] = txStats.bfdMsgTxStats.sentAfter3p

      # Get echo session stats
      if peerStatus.echoActive:
         txStats, _ = getTxStats( peerStatus, netlinkStub, echo=True )
         rxStats, _ = getRxStats( peerStatus, netlinkStub, echo=True )

         if rxStats and rxStats.bfdMsgRxStats.minPeriod:
            minPeriod = rxStats.bfdMsgRxStats.minPeriod / 1000
            maxPeriod = rxStats.bfdMsgRxStats.maxPeriod / 1000
            avgPeriod = rxStats.bfdMsgRxStats.avgPeriod / 1000
            if minPeriod < minrx_threshold \
               or maxPeriod > peerStatus.detectTimeInMicrosecond:
               bfdDebugModel.echoRxIntervals[ peerStr ] = str( minPeriod ) + '/' \
                                       + str( maxPeriod ) + '/' + str( avgPeriod )

         if txStats and txStats.bfdMsgTxStats.sentAfter3p:
            bfdDebugModel.slowEchoTxs[ peerStr ] = txStats.bfdMsgTxStats.sentAfter3p

# ------------------------------------------------------------
# return the value after specified token, i.e. for 'rxbfd_nolagdev 3'
# return '3'
# ------------------------------------------------------------
def findTokenValue( output, token ):
   match = re.search( r'(\s|\()%s\s\w*' % token, output )
   if match:
      return match.group().split()[ 1 ]
   return ''

# ------------------------------------------------------------
# check in dma and record following issues:
# rxbfd_nolagdev > 0
# fdevtx_locktime > 1000us
# ------------------------------------------------------------
def checkDma( bfdDebugModel ):
   tokens = [ 'rxbfd_nolagdev', 'fdevtx_locktime' ]
   key = '---------- show platform pkt -----------'
   locktime_threshold = 1000 # 1000us
   try: # pylint: disable-msg=R1702, too-many-nested-blocks
      output = Tac.run( [ '/usr/bin/fab', 'dump' ], stdout=Tac.CAPTURE, asRoot=True )
      for token in tokens:
         value = findTokenValue( output, token )
         if value:
            try:
               if token == 'rxbfd_nolagdev' and int( value ):
                  # rxbfd_nolagdev > 0
                  bfdDebugModel.addIssue( key, token + ' ' + value )
               if token == 'fdevtx_locktime':
                  # fdevtx_locktime > 1000us
                  if int( value.split( 'us' )[ 0 ] ) > locktime_threshold:
                     bfdDebugModel.addIssue( key, token + ' ' + value )
            except ( ValueError, IndexError ):
               pass
   except: # pylint: disable=bare-except
      pass

# ------------------------------------------------------------
# check for intf counter discards
# ------------------------------------------------------------
def checkIntfCounter( mode, bfdDebugModel ):
   with ArPyUtils.FileHandleInterceptor( [ sys.stdout.fileno() ] ) as out:
      mode.session_.runCmd( "show interfaces counters discards | nz", aaa=False )
   output = out.contents().strip( '\n' )
   if 'Totals' in output:
      key = "---------- interfaces counters discards -----------"
      bfdDebugModel.addIssue( key, output )

# ------------------------------------------------------------
# check for cpu counters drop
# ------------------------------------------------------------
def checkCpuCounter( mode, bfdDebugModel ):
   try:
      with ArPyUtils.FileHandleInterceptor( [ sys.stdout.fileno() ] ) as out:
         mode.session_.runCmd( "show cpu counters queue | nz", aaa=False )
      lines = out.contents().split( '\n' )
      key = "---------- cpu counters drop -----------"
      # print any queue that has drops
      for line in lines:
         tokens = line.split()
         if tokens:
            try:
               # the last value in line is drop counter
               if int( tokens[ -1 ] ):
                  # if drop counter is non-zero
                  bfdDebugModel.addIssue( key, line )
            except ValueError:
               pass
   except: # pylint: disable=bare-except
      # "show cpu counters queue" command failed on this platform
      pass

# ------------------------------------------------------------
# check for kernel qdisc drop
# ------------------------------------------------------------
def checkQdisc( bfdDebugModel ):
   key = '---------- kernel qdisc drop -----------------'
   output = Tac.run( [ 'tc', '-s', 'qdisc', 'show' ], stdout=Tac.CAPTURE,
                     asRoot=True, stderr=Tac.CAPTURE, ignoreReturnCode=True )
   lines = output.split( '\n' )
   while len( lines ) > 3:
      devName = findTokenValue( lines[ 0 ], 'dev' )
      # record qdisc drop on interfaces
      if devName:
         # output for each dev takes 3 lines
         dropped = findTokenValue( lines[ 1 ], 'dropped' )
         try:
            if int( dropped ):
               bfdDebugModel.addIssue( key, '\n'.join( lines[ : 3 ] ) )
         except ( ValueError, IndexError ):
            pass
      lines = lines[ 3 : ]

# ------------------------------------------------------------
# check for iptables drop
# ------------------------------------------------------------
def checkIptablesDrop( allPeerStatus, bfdDebugModel, vrfNames ):
   key = '-------- iptables drop in default vrf --------'
   timeoutCmd = 'timeout -s SIGKILL 2'
   cmd = '%s iptables -vnL' % timeoutCmd
   output = Tac.run( cmd.split(), stdout=Tac.CAPTURE,
                     asRoot=True, stderr=Tac.CAPTURE, ignoreReturnCode=True )
   lines = output.split( '\n' )
   for line in lines:
      if 'DROP' in line:
         tokens = line.split()
         try:
            if int( tokens[ 0 ] ) or int( tokens[ 1 ] ):
               bfdDebugModel.addIssue( key, line )
         except ( ValueError, IndexError ):
            pass

def showBfdDebug( mode, vrfNames, intfSet, destIpAddr, srcIpAddr,
                  minTx, minRx, multiplier, proto, addrFamily,
                  tunnelId=None, bfdOnly=False, sbfdOnly=False ):
   bfdDebugModel = BfdModel.BfdDebug()
   allPeerStatus = filterPeerStatus( vrfNames=vrfNames, intfSet=intfSet,
                                     destIpAddr=destIpAddr, srcIpAddr=srcIpAddr,
                                     minTx=minTx, minRx=minRx,
                                     multiplier=multiplier, proto=proto,
                                     tunnelId=tunnelId, addrFamily=addrFamily,
                                     bfdOnly=bfdOnly, sbfdOnly=sbfdOnly )
   if not allPeerStatus:
      return bfdDebugModel
   checkRfc7130( allPeerStatus, bfdDebugModel )
   checkBfdPeerStatus( mode, allPeerStatus, bfdDebugModel )

   checkCpuCounter( mode, bfdDebugModel )
   checkIntfCounter( mode, bfdDebugModel )
   checkIptablesDrop( allPeerStatus, bfdDebugModel, vrfNames )
   checkDma( bfdDebugModel )
   checkQdisc( bfdDebugModel )

   return bfdDebugModel

def getApp( peer ):
   peerEntry = gv.bfdConfigPeer.peerConfig.get( peer )
   return peerEntry.registeredApp if peerEntry else None

def showBfdHwUtilization( mode, args ):
   detail = 'detail' in args
   bfdHwResourceListModel = BfdModel.BfdHwResourceList(
         _detailed=detail, _sso=gv.redundancyConfig.protocol == 'sso' )

   if resourceFilter := args.get( 'RESOURCE' ):
      for resourceConfig in gv.bfdHwResourceConfig.values():
         if resourceFilter in resourceConfig.hwResourceLimits:
            break
      else:
         mode.addError( '"%s" is not a valid resource name' % resourceFilter )
         return None

   intfSet = set( args.get( 'INTFS', () ) )
   vrfNames = set( getAllVrfNames( mode ) )

   if detail:
      # Include member intfs if filtered by LAG intf
      intfSet.update( discoverLagIntfs( intfSet ) )

      if vrfName := args.get( 'VRF' ):
         if not vrfExists( vrfName ):
            mode.addErrorAndStop( "VRF %s not configured." % vrfName )
         vrfNames = [ vrfName ]

   if destIp := args.get( 'DESTIP' ) or args.get( 'DESTIP6' ):
      destIp = str( destIp )

   for resourceConfig in gv.bfdHwResourceConfig.values():
      for resource, resourceLimits in \
            resourceConfig.hwResourceLimits.items():
         if resourceFilter and resource != resourceFilter:
            continue
         resourceModel = \
            BfdModel.BfdHwResource( maxSessions=resourceLimits.maxHwSessions,
                                    ssoSupported=resourceLimits.ssoSupported )
         bfdHwResourceListModel.resources[ resource ] = resourceModel

         allPeerStatus = filterPeerStatus( vrfNames=vrfNames,
                                           intfSet=intfSet,
                                           destIpAddr=destIp,
                                           resource=resource )

         if not detail:
            resourceModel.numSessions = 0
            resourceModel.sessions = None

         for peerStatus in allPeerStatus:
            if gv.bfdHwStatus.isHwAccelerated( peerStatus.localDisc ):
               if detail:
                  p = peerStatus.peer
                  sessionModel = BfdModel.BfdHwResourceSession( ip=p.ip, intf=p.intf,
                        vrf=p.vrf, sessType=peerStatus.type,
                        localDisc=peerStatus.localDisc )
                  resourceModel.sessions.append( sessionModel )
               else:
                  resourceModel.numSessions += 1

            if ( peerStatus.echoOn and
                 gv.bfdHwStatus.isHwAccelerated( peerStatus.echoDisc ) ):
               if detail:
                  p = peerStatus.peer
                  sessionModel = BfdModel.BfdHwResourceSession( ip=p.ip,
                        intf=p.intf, vrf=p.vrf, sessType=peerStatus.type + " echo",
                        localDisc=peerStatus.echoDisc )
                  resourceModel.sessions.append( sessionModel )
               else:
                  resourceModel.numSessions += 1

   return bfdHwResourceListModel

# --------------------------------------------------------------------------------
# show bfd ( ( ip access-list [ IPACL ] ) | ( ipv6 access-list [ IP6ACL ] ) )
# --------------------------------------------------------------------------------
def showAcl( mode, args ):
   aclType = 'ip' if 'ip' in args else 'ipv6'
   name = args[ '<aclNameExpr>' ]
   return AclCli.showServiceAcl( mode,
                                 gv.aclCpConfig,
                                 gv.aclStatus,
                                 gv.aclCheckpoint,
                                 aclType,
                                 name,
                                 serviceName='bfd' )

class BfdIpAccessListCmd( ShowCommand.ShowCliCommandClass ):
   syntax = ( 'show bfd ( ( ip access-list [ IPACL ] ) | '
            '( ipv6 access-list [ IP6ACL ] ) )' )
   data = {
      'bfd': matcherBfd,
      'ip': ipMatcherForShow,
      'ipv6': ipv6MatcherForShow,
      'access-list': AclCli.accessListKwMatcherForServiceAcl,
      'IPACL': AclCli.ipAclNameExpression,
      'IP6ACL': AclCli.ip6AclNameExpression,
   }
   handler = showAcl
   cliModel = AllAclList

BasicCli.addShowCommandClass( BfdIpAccessListCmd )

# --------------------------------------------------------------------------------
# Deprecated syntax:
# show bfd neighbors [ bfd | ( sbfd [initiators | reflectors] ) ]
#    [ vrf ( default | all | VRFNAME ) ] [ interface INTFS ]
#    [ dest-ip ( DESTIP | DESTIP6 ) ] [ src-ip ( SRCIP | SRCIP6 ) ]
#    [ interval INTERVAL ] [ min_rx MIN_RX ] [ multiplier MULTIPLIER ] [ detail ]
#
# show bfd peers [ bfd | ( sbfd [initiators | reflectors] ) ]
#    [ ipv4 | ipv6 ] [ vrf ( default | all | VRFNAME ) ] [ interface INTFS ]
#    [ dest-ip ( DESTIP | DESTIP6 ) ] [ src-ip ( SRCIP | SRCIP6 ) ]
#    [ interval INTERVAL ] [ min_rx MIN_RX ] [ multiplier MULTIPLIER ]
#    [ protocol PROTO ] [ tunnel-id TUNNELID] [ detail ]
# sbfd reflectors filter would be added in the future.
# --------------------------------------------------------------------------------
def showBfdNeighborHandler( mode, args ):
   bfdOnly = 'bfdonly' in args
   sbfdOnly = 'sbfd' in args
   sbfdInitiatorOnly = 'initiators' in args
   sbfdReflectorOnly = 'reflectors' in args
   # As reflector stats is not supported yet, just blindly ignore arg
   # 'initiators' and always print initiators stats when sbfd peers are
   # needed for show.
   vrfName = args.get( 'VRF', ALL_VRF_NAME )
   intfs = args.get( 'INTFS', () )
   destIp = None
   srcIp = None
   if 'dest-ip' in args:
      if 'DESTIP' in args:
         destIp = args[ 'DESTIP' ]
      elif 'DESTIP6' in args:
         destIp = args[ 'DESTIP6' ]
   if 'src-ip' in args:
      if 'SRCIP' in args:
         srcIp = args[ 'SRCIP' ]
      elif 'SRCIP6' in args:
         srcIp = args[ 'SRCIP6' ]
   interval = args.get( 'INTERVAL' )
   minRx = args.get( 'MIN_RX' )
   mult = args.get( 'MULTIPLIER' )
   proto = args.get( 'PROTO' )
   af = args.get( 'AF' )
   tunnelId = args.get( 'TUNNELID' )
   displayType = None
   if 'summary' in args:
      displayType = 'summary'
   if 'debug' in args:
      displayType = 'debug'
   if 'history' in args:
      displayType = 'history'
   if 'detail' in args:
      displayType = 'detail'

   return showBfdNeighbor( mode, vrfName=vrfName, intfs=intfs,
                           destIpAddr=destIp, srcIpAddr=srcIp, minTx=interval,
                           minRx=minRx, multiplier=mult, proto=proto,
                           tunnelId=tunnelId,
                           displayType=displayType, addrFamilies=af,
                           bfdOnly=bfdOnly, sbfdOnly=sbfdOnly,
                           sbfdInitiatorOnly=sbfdInitiatorOnly,
                           sbfdReflectorOnly=sbfdReflectorOnly )

def mapAppName( name, showDetail=False ):
   mapping = {
         'ospfv3_ipv4': 'ospfv3', # simplify ospfv3 tokens
         'ospfv3_ipv6': 'ospfv3',
         'pim6': 'pim', # simplify pim tokens
         'fhrp': 'vrrp', # fhrp encompasses vrrp and varp-- only vrrp cares about bfd
         'static': 'static-bfd', # clarity
         # make this a nicer token for cli
         'srtepolicy': 'sr-te policy' if showDetail else 'sr-te',
   }
   return mapping.get( name, name )

def getProtocolNames( mode ):
   # Vxlan peer config lives in a different dir in Sysdb
   apps = { a for a in chain( gv.appConfigDir, [ 'vxlan' ] ) if a != 'test' }
   return { mapAppName( a ): '' for a in apps }

class BfdSbfdFilterExpression( CliCommand.CliExpression ):

   expression = 'bfdonly | ( sbfd [ initiators | reflectors ] )'

   data = {
      'bfdonly': CliMatcher.KeywordMatcher( 'bfd',
                                 helpdesc='Status for BFD sessions only' ),
      'sbfd': matcherSbfd,
      'initiators': matcherInitiators,
      'reflectors': matcherReflectors,
   }

class BfdShowPeersCmdArgsExpression( CliCommand.CliExpression ):

   expression = '''{ AF
                   | VRF
                   | ( interface INTFS )
                   | ( dest-ip ( DESTIP | DESTIP6 ) )
                   | ( src-ip ( SRCIP | SRCIP6 ) )
                   | ( interval INTERVAL )
                   | ( ( min-rx | min_rx ) MIN_RX )
                   | ( multiplier MULTIPLIER )
                   | ( protocol PROTO )
                   | ( tunnel-id TUNNELID )
   }'''

   data = {
      'AF': singleNode( matcher=EnumMatcher( {
         'ipv4': 'Sessions over IPv4',
         'ipv6': 'Sessions over IPv6',
      } ) ),
      'VRF': vrfExprFactory,
      'interface': singleNode( matcherInterface ),
      'INTFS': singleNode( intfRangeMatcher ),
      'dest-ip': singleNode( matcherDestIp ),
      'DESTIP': singleNode( IpAddrMatcher( helpdesc='IPv4 Neighbor address' ) ),
      'DESTIP6': singleNode( Ip6AddrMatcher( helpdesc='IPv6 Neighbor address' ) ),
      'src-ip': singleNode( matcherSrcIp ),
      'SRCIP': singleNode( IpAddrMatcher( helpdesc='IPv4 Source address' ) ),
      'SRCIP6': singleNode( Ip6AddrMatcher( helpdesc='IPv6 Source address' ) ),
      'interval': singleNode( matcherInterval ),
      'INTERVAL': singleNode( CliMatcher.IntegerMatcher( 50, 60000,
                     helpdesc='Rate in milliseconds between 50-60000' ) ),
      'min-rx': singleNode( matcherMinRx ),
      'min_rx': matcherMin_RxDeprecated,
      'MIN_RX': singleNode( CliMatcher.IntegerMatcher( 50, 60000,
                     helpdesc='Rate in milliseconds between 50-60000' ) ),
      'multiplier': singleNode( matcherMultiplier ),
      'MULTIPLIER': singleNode( CliMatcher.IntegerMatcher( 3, 50,
                     helpdesc='Range is 3-50' ) ),
      'protocol': singleNode( matcherProtocol ),
      'PROTO': singleNode( CliMatcher.DynamicKeywordMatcher( getProtocolNames ) ),
      'tunnel-id': 'Tunnel interface ID',
      'TUNNELID': singleNode(
         CliMatcher.IntegerMatcher( TunnelIdType.minValue, TunnelIdType.maxValue,
                                    helpdesc='Tunnel interface ID' )
      ),
   }

class BfdPeersCmd( ShowCommand.ShowCliCommandClass ):

   syntax = ( 'show bfd ( neighbors | peers ) [ BFD_SBFD_FILTER ] '
              '[ SHOW_CMD_EXPR ] [ detail ]' )
   data = {
      'bfd': matcherBfd,
      'neighbors': neighborsDeprecatedNode,
      'peers': matcherPeers,
      'BFD_SBFD_FILTER': BfdSbfdFilterExpression,
      'SHOW_CMD_EXPR': BfdShowPeersCmdArgsExpression,
      'detail': matcherNbrDetail,
   }

   handler = showBfdNeighborHandler
   cliModel = BfdNeighbors

BasicCli.addShowCommandClass( BfdPeersCmd )

# --------------------------------------------------------------------------------
# Deprecated syntax:
# show bfd neighbors [ vrf ( default | all | VRFNAME ) ] [ interface INTFS ]
#    [ dest-ip ( DESTIP | DESTIP6 ) ] [ src-ip ( SRCIP | SRCIP6 ) ]
#    [ interval INTERVAL ] [ min_rx MIN_RX ] [ multiplier MULTIPLIER ] debug
#
# show bfd peers [ ipv4 | ipv6 ] [ vrf ( default | all | VRFNAME ) ]
#    [ interface INTFS ] [ dest-ip ( DESTIP | DESTIP6 ) ]
#    [ src-ip ( SRCIP | SRCIP6 ) ] [ interval INTERVAL ] [ min-rx MIN_RX ]
#    [ multiplier MULTIPLIER ] [ protocol PROTO ] [ tunnel-id TUNNELID ] debug
# --------------------------------------------------------------------------------
class BfdPeersDebugCmd( ShowCommand.ShowCliCommandClass ):

   syntax = ( 'show bfd ( neighbors | peers ) [ BFD_SBFD_FILTER ] '
              '[ SHOW_CMD_EXPR ] debug' )
   data = {
      'bfd': matcherBfd,
      'neighbors': neighborsDeprecatedNode,
      'peers': matcherPeers,
      'BFD_SBFD_FILTER': BfdSbfdFilterExpression,
      'SHOW_CMD_EXPR': BfdShowPeersCmdArgsExpression,
      'debug': matcherDebug,
   }

   handler = showBfdNeighborHandler
   cliModel = BfdDebug

BasicCli.addShowCommandClass( BfdPeersDebugCmd )

# --------------------------------------------------------------------------------
# Deprecated syntax:
# show bfd neighbors [ bfd | ( sbfd [initiators | reflectors] ) ]
#    [ vrf ( default | all | VRFNAME ) ] [ interface INTFS ]
#    [ dest-ip ( DESTIP | DESTIP6 ) ] [ src-ip ( SRCIP | SRCIP6 ) ]
#    [ interval INTERVAL ] [ min_rx MIN_RX ] [ multiplier MULTIPLIER ] history
#
# show bfd peers [ bfd | ( sbfd [initiators | reflectors] ) ]
#    [ ipv4 | ipv6 ] [ vrf ( default | all | VRFNAME ) ] [ interface INTFS ]
#    [ dest-ip ( DESTIP | DESTIP6 ) ] [ src-ip ( SRCIP | SRCIP6 ) ]
#    [ interval INTERVAL ] [ min_rx MIN_RX ] [ multiplier MULTIPLIER ]
#    [ protocol PROTO ] [ tunnel-id TUNNELID ] history
# sbfd reflectors filter would be added in the future.
# --------------------------------------------------------------------------------
class BfdPeersHistoryCmd( ShowCommand.ShowCliCommandClass ):

   syntax = ( 'show bfd ( neighbors | peers ) [ BFD_SBFD_FILTER ] '
              '[ SHOW_CMD_EXPR ] history' )
   data = {
      'bfd': matcherBfd,
      'neighbors': neighborsDeprecatedNode,
      'peers': matcherPeers,
      'BFD_SBFD_FILTER': BfdSbfdFilterExpression,
      'SHOW_CMD_EXPR': BfdShowPeersCmdArgsExpression,
      'history': matcherHistory,
   }

   handler = showBfdNeighborHandler
   cliModel = PeerHistory

BasicCli.addShowCommandClass( BfdPeersHistoryCmd )

# --------------------------------------------------------------------------------
# Deprecated syntax:
# show bfd neighbors [ bfd | ( sbfd [initiators | reflectors] ) ]
#    [ vrf ( default | all | VRFNAME ) ] [ interface INTFS ]
#    [ dest-ip ( DESTIP | DESTIP6 ) ] [ src-ip ( SRCIP | SRCIP6 ) ]
#    [ interval INTERVAL ] [ min_rx MIN_RX ] [ multiplier MULTIPLIER ] summary
#
# show bfd peers [ bfd | ( sbfd [initiators | reflectors] ) ]
#    [ ipv4 | ipv6 ] [ vrf ( default | all | VRFNAME ) ] [ interface INTFS ]
#    [ dest-ip ( DESTIP | DESTIP6 ) ] [ src-ip ( SRCIP | SRCIP6 ) ]
#    [ interval INTERVAL ] [ min_rx MIN_RX ] [ multiplier MULTIPLIER ]
#    [ protocol PROTO ] [ tunnel-id TUNNELID ] summary
# sbfd reflectors filter would be added in the future, but global state for
# reflector is available now.
# --------------------------------------------------------------------------------
class BfdPeersSummaryCmd( ShowCommand.ShowCliCommandClass ):

   syntax = ( 'show bfd ( neighbors | peers ) [ BFD_SBFD_FILTER ] '
              '[ SHOW_CMD_EXPR ] summary' )
   data = {
      'bfd': matcherBfd,
      'neighbors': neighborsDeprecatedNode,
      'peers': matcherPeers,
      'BFD_SBFD_FILTER': BfdSbfdFilterExpression,
      'SHOW_CMD_EXPR': BfdShowPeersCmdArgsExpression,
      'summary': matcherSummary,
   }

   handler = showBfdNeighborHandler
   cliModel = BfdSummary

BasicCli.addShowCommandClass( BfdPeersSummaryCmd )

# --------------------------------------------------------------------------------
#  This command displays synced BFD state for VXLAN BFD from the MLAG primary if
#  invoked on an MLAG secondary device. Otherwise, it displays the local BFD state.
# --------------------------------------------------------------------------------
class BfdVxlanMlagPrimaryCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show bfd peers protocol vxlan mlag primary'
   data = {
      'bfd': matcherBfd,
      'peers': matcherPeers,
      'protocol': matcherProtocol,
      'vxlan': '',
      'mlag': 'MLAG status',
      'primary': 'BFD session state synced from MLAG primary',
   }
   cliModel = BfdVxlanMlagPrimary

   @staticmethod
   def handler( mode, args ):
      model = BfdVxlanMlagPrimary()
      # Display synced BFD state if available, or default to local VXLAN BFD status.
      if gv.mlagVxlanStatus.allVxlanBfdPeerStatus:
         for vtiVtep, status in gv.mlagVxlanStatus.allVxlanBfdPeerStatus.items():
            vtepBfdStatus = model.interfaces.get( vtiVtep.vti, VxlanVtepBfdStatus() )
            vtepBfdStatus.vteps[ vtiVtep.vtep ] = status.operState
            model.interfaces[ vtiVtep.vti ] = vtepBfdStatus
      else:
         vtepBfdStatus = VxlanVtepBfdStatus()
         for peer, status in gv.bfdStatusPeer.peerStatus.items():
            if status.type == operSessionType.sessionTypeVxlanTunnel:
               vtepBfdStatus.vteps[ peer.ip ] = status.status
         # The synced status is organized by VTI, but until there is multi-VTI
         # support assume all VXLAN BFD sessions are from Vxlan1.
         model.interfaces[ 'Vxlan1' ] = vtepBfdStatus
      return model

BasicCli.addShowCommandClass( BfdVxlanMlagPrimaryCmd )

# --------------------------------------------------------------------------------
# show bfd rbfd-stats
# --------------------------------------------------------------------------------
def showRbfdStats( mode, args ):
   netlinkStub = Tac.newInstance( "BfdPyUtils::BfdNetlinkControlStub" )
   netlinkStub.createFd( False )
   # We get stats for all VRFs
   getStatsFromAccelerator( mode, netlinkStub, 0, module=True )
   rbfdStatsModel = RbfdStats()
   rbfdStatsModel.rfcDrops = netlinkStub.rfcDrops
   rbfdStatsModel.rxPacketMissing = netlinkStub.rxPacketMissing
   rbfdStatsModel.numInvalidSessions = netlinkStub.numInvalidSessions
   rbfdStatsModel.numPassupDisc0 = netlinkStub.numPassupDisc0
   rbfdStatsModel.netlinkPacketChangeFail = netlinkStub.netlinkPacketChangeFail
   rbfdStatsModel.netlinkFailureFail = netlinkStub.netlinkFailureFail
   rbfdStatsModel.pskbExpandFail = netlinkStub.pskbExpandFail
   return rbfdStatsModel

class BfdRbfdStatsCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show bfd rbfd-stats'
   data = {
      'bfd': matcherBfd,
      'rbfd-stats': 'Show aggregated rbfd module statistics',
   }
   handler = showRbfdStats
   cliModel = RbfdStats
   hidden = True

BasicCli.addShowCommandClass( BfdRbfdStatsCmd )

# --------------------------------------------------------------------------------
# show bfd sbfd-stats reflectors
# --------------------------------------------------------------------------------
def showSbfdStats( mode, args ):
   globalStats, _ = getReflectorStats()
   stats = {} if globalStats == {} else globalStats._asdict()
   sbfdStatsModel = SbfdStats()
   for attr in sbfdReflectorGlobal._fields:
      if attr != 'padding':
         setattr( sbfdStatsModel, attr, stats.get( attr, 0 ) )
   return sbfdStatsModel

class BfdSbfdStatsCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show bfd sbfd-stats reflectors'
   data = {
      'bfd': matcherBfd,
      'sbfd-stats': 'Show sbfd global statistics',
      'reflectors': 'Show sbfd reflectors global statistics',
   }
   handler = showSbfdStats
   cliModel = SbfdStats
   hidden = True

BasicCli.addShowCommandClass( BfdSbfdStatsCmd )

# ------------------------------------------------------------------------------
# show bfd hardware acceleration
# ------------------------------------------------------------------------------
class ShowBfdHwAccelCmd( ShowCommand.ShowCliCommandClass ):
   syntax = '''show bfd hardware acceleration'''
   data = { 'bfd': matcherBfd,
            'hardware': matcherHardware,
            'acceleration': 'BFD processing in hardware',
            }
   cliModel = BfdHwAccel

   @staticmethod
   def handler( mode, args ):
      reasons = []
      if gv.bfdConfigGlobal.hwAccelerationConfig == 'disabledByCli':
         reasons.append( "user disabled" )
      if not gv.bfdHwStatus.hwSessionsInUse:
         reasons.append( "no eligible sessions" )
      if ( not gv.bfdHwConfig.hwAccelerationConfig.hwbfdSsoSupported and
           Cell.cellType() == 'supervisor' and
           gv.redundancyConfig.protocol == 'sso' ):
         reasons.append( "not supported with SSO" )
      return BfdModel.BfdHwAccel(
            supported=gv.bfdHwConfig.hwAccelerationConfig.hwAccelerationTxSupported,
            running=gv.bfdHwStatus.hwAccelerationEnabled,
            reasons=reasons )

BasicCli.addShowCommandClass( ShowBfdHwAccelCmd )

# --------------------------------------------------------------------------------
# show bfd hardware utilization [ RESOURCE ] [ detail [ { ( vrf VRF ) |
#    ( interface INTFS ) | ( dest-ip ( DESTIP | DESTIP6 ) ) } ] ]
# --------------------------------------------------------------------------------
class ShowBfdHwUtilizationCmd( ShowCommand.ShowCliCommandClass ):
   syntax = '''show bfd hardware utilization [ RESOURCE ]
               [ detail [ { ( VRF )
                          | ( interface INTFS )
                          | ( dest-ip ( DESTIP | DESTIP6 ) ) } ] ]'''
   data = { 'bfd': matcherBfd,
            'hardware': matcherHardware,
            'utilization': 'Resource utilization information',
            'RESOURCE': matcherResource,
            'detail': 'Detailed list of sessions per resource',
            'VRF': vrfExprFactory,
            'interface': singleNode( matcherInterface ),
            'INTFS': singleNode( intfRangeMatcher ),
            'dest-ip': singleNode( matcherDestIp ),
            'DESTIP': singleNode(
               IpAddrMatcher( helpdesc='IPv4 Neighbor address' ) ),
            'DESTIP6': singleNode(
               Ip6AddrMatcher( helpdesc='IPv6 Neighbor address' ) ),
            }
   cliModel = BfdHwResourceList
   handler = showBfdHwUtilization

BasicCli.addShowCommandClass( ShowBfdHwUtilizationCmd )

# -------------------------------------------------------------------------------
# BFD commands in "show tech-support"
# -------------------------------------------------------------------------------
# These commands were in the original "show tech" command list from the EOS
# package.  To avoid having them move around in the "show tech" output,
# we use a made up timestamp.
TechSupportCli.registerShowTechSupportCmd(
   '2010-09-30 00:00:00',
   cmds=[ 'show bfd peers summary',
          'show bfd peers history',
          'show bfd peers detail',
          'show bfd rbfd-stats',
          'show bfd peers debug',
          'show bfd hardware utilization detail',
          'bash cat /proc/net/stat/arp_cache',
          'bash cat /proc/net/stat/ndisc_cache' ],
   summaryCmds=[ 'show bfd peers summary' ] )

def Plugin( entityManager ):
   gv.aclStatus = LazyMount.mount( entityManager, 'acl/status/all',
                                   'Acl::Status', 'r' )
   gv.aclCpConfig = LazyMount.mount( entityManager, 'acl/cpconfig/cli',
                                     'Acl::Input::CpConfig', 'r' )
   gv.aclCheckpoint = LazyMount.mount( entityManager, 'acl/checkpoint',
                                       'Acl::CheckpointStatus', 'r' )
   gv.appConfigDir = LazyMount.mount( entityManager, 'bfd/config/app',
                                      'Tac::Dir', 'ri' )
   gv.bfdConfigGlobal = LazyMount.mount( entityManager, 'bfd/config/global',
                                         'Bfd::ConfigGlobal', 'r' )
   gv.bfdHwConfig = LazyMount.mount( entityManager, 'bfd/hwConfig',
                                     'Hardware::Bfd::Config', 'r' )
   gv.bfdHwStatus = LazyMount.mount( entityManager, 'bfd/hwStatus',
                                     'Hardware::Bfd::Status', 'r' )
   gv.bfdStatusPeer = LazyMount.mount( entityManager, 'bfd/status/peer',
                                       'Bfd::StatusPeer', 'r' )
   gv.bfdStatusIntfInfo = LazyMount.mount( entityManager, 'bfd/status/intfInfo',
                                       'Bfd::StatusIntfInfo', 'r' )
   gv.bfdPeerHistory = LazyMount.mount( entityManager, 'bfd/status/history',
                                        'Bfd::PeerHistory', 'r' )
   gv.bfdConfigInfo = LazyMount.mount( entityManager, 'bfd/config/info',
                                       'Bfd::ConfigInfo', 'r' )
   gv.bfdConfigPeer = LazyMount.mount( entityManager, 'bfd/config/peer',
                                       'Bfd::ConfigPeer', 'r' )
   gv.bfdOperInfoPeer = LazyMount.mount( entityManager,
                                         'bfd/status/peerOper',
                                         'Bfd::OperInfoPeer', 'r' )
   gv.bfdTxStatsSmash = SmashLazyMount.mount( entityManager, 'bfd/hwTxStats',
                                            'Bfd::HwBfdSmashTxStats',
                                            Smash.mountInfo( 'reader' ) )
   gv.bfdRxStatsSmash = SmashLazyMount.mount( entityManager, 'bfd/hwRxStats',
                                            'Bfd::HwBfdSmashRxStats',
                                            Smash.mountInfo( 'reader' ) )
   gv.configBfdLag = LazyMount.mount( entityManager, 'bfd/config/lag',
                                      'Bfd::ConfigBfdLag', 'r' )
   gv.allIntfStatusDir = LazyMount.mount( entityManager, 'interface/status/all',
                                          'Interface::AllIntfStatusDir', 'r' )
   gv.bfdHwResourceConfig = LazyMount.mount( entityManager, 'bfd/hwResourceConfig',
                                             'Tac::Dir', 'ri' )
   gv.redundancyConfig = LazyMount.mount( entityManager, 'redundancy/config',
                                          'Redundancy::RedundancyConfig', 'r' )
   gv.ipConfig = LazyMount.mount( entityManager, "ip/config", "Ip::Config", "r" )
   gv.ip6Config = LazyMount.mount( entityManager, "ip6/config", "Ip6::Config", "r" )
   gv.srteSegmentList = LazyMount.mount( entityManager,
      'te/segmentrouting/srtepolicy/allsegmentlist',
      'SrTePolicy::Export::SegmentListCollection', 'r' )
   tableInfo = TunnelTableMounter.getMountInfo(
      TunnelTableIdentifier.eosSdkSbfdSessionTunnelTable ).tableInfo
   gv.eosSdkSbfdTunnelTable = SmashLazyMount.mount(
      entityManager, tableInfo.mountPath, tableInfo.tableType,
      Smash.mountInfo( 'reader' ) )
   gv.mlagVxlanStatus = LazyMount.mount( entityManager, 'mlag/vxlan/status',
                                         'Mlag::VxlanStatus', 'r' )
