#!/usr/bin/env python3
# Copyright (c) 2018 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import CliCommand
import CliMatcher
from CliPlugin.RoutingBgpCli import (
   RouterBgpSharedModelet,
   afModeExtensionHook,
   RouterBgpAfUnsupportedPrefixListSharedModelet,
   RouterBgpBaseMode,
)
from CliPlugin.RoutingBgpInstanceCli import (
   MissingPolicyAfSharedCmd,
   SkipRibBestpathCommand,
)
from CliPlugin.RoutingBgpNeighborCli import ( RouterBgpAfActivateCommand,
                                              SetNeighborMaxAcceptedRoutesCmd,
                                              SetNeighborMaxAdvRoutesCmd,
                                              SetNeighborMaxRoutesCmd,
                                              SetNeighborMissingPolicyAfCmd,
                                              SetNeighborRcfCmd )
from CliPlugin.ArBgpCliCommon import getBgpExtCommParserContainerSm
from CliPlugin.RcfCliLib import rcfFunctionMatcher
from CliPlugin.FlowspecConfigCli import flowspecPolicyNameMatcher
from CliMode.BgpCommon import RoutingBgpBaseAfMode, RoutingBgpVrfAfMode
import CliToken.RoutingBgp as bgpTokens

import BasicCli
import CliParser
from IpLibConsts import DEFAULT_VRF
from BgpLib import updateConfigAttrsAfMap
import Tac
from Toggles import (
   BgpCommonToggleLib,
   RcfLibToggleLib,
   RoutingLibToggleLib,
   FlowspecToggleLib,
)

# pkgdeps: rpm BgpFlowspec-lib

updateConfigAttrsAfMap( { 'flow-spec ipv4' : [ 'V4Flowspec', 'AfV4Flowspec' ],
                          'flow-spec ipv6' : [ 'V6Flowspec', 'AfV6Flowspec' ] },
                          True )

if BgpCommonToggleLib.toggleFlowspecVpnRxTxEnabled():
   updateConfigAttrsAfMap(
         { 'flow-spec vpn-ipv4' : [ 'V4FlowspecVpn', 'AfV4FlowspecVpn' ],
           'flow-spec vpn-ipv6' : [ 'V6FlowspecVpn', 'AfV6FlowspecVpn' ] },
           # Flowspec VPN address families are only supported in default VRF
           vrfSupported=False )

class RouterBgpBaseAfFlowspecMode( RoutingBgpBaseAfMode, BasicCli.ConfigModeBase ):
   name = 'BGP address family Flow Specification configuration'

   def __init__( self, parent, session, addrFamily ):
      self.vrfName = DEFAULT_VRF
      RoutingBgpBaseAfMode.__init__( self, addrFamily,
                                     modeKey='router-bgp-af-flow-spec' )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

class RouterBgpVrfAfFlowspecMode( RoutingBgpVrfAfMode, BasicCli.ConfigModeBase ):
   name = 'BGP VRF address family Flow Specification configuration'

   def __init__( self, parent, session, addrFamily, vrfName ):
      self.vrfName = vrfName
      RoutingBgpVrfAfMode.__init__( self, addrFamily, vrfName )
      # set cli prompt in vrf af config mode. TODO: get it cli-reviewed
      self.longModeKey = f'router-bgp-vrf-{vrfName}-af-flow-spec'
      BasicCli.ConfigModeBase.__init__( self, parent, session )

class RouterBgpAfFlowspecModelet( CliParser.Modelet ):
   pass

RouterBgpBaseAfFlowspecMode.addModelet( RouterBgpAfFlowspecModelet )
RouterBgpBaseAfFlowspecMode.addModelet(
   RouterBgpAfUnsupportedPrefixListSharedModelet )
RouterBgpVrfAfFlowspecMode.addModelet( RouterBgpAfFlowspecModelet )
RouterBgpVrfAfFlowspecMode.addModelet(
   RouterBgpAfUnsupportedPrefixListSharedModelet )

#-------------------------------------------------------------------------------
# "[ no | default ] address-family flow-spec ( ipv4 | ipv6 )"
#-------------------------------------------------------------------------------
flowspecKwMatcher = CliMatcher.KeywordMatcher(
   'flow-spec',
   helpdesc='Flow Specification' )

class EnterBgpFlowspecAfMode( CliCommand.CliCommandClass ):
   syntax = 'address-family flow-spec AF'
   noOrDefaultSyntax = syntax

   data = { 'address-family' : bgpTokens.addrFamily,
            'flow-spec' : flowspecKwMatcher,
            'AF' : CliMatcher.EnumMatcher( {
               'ipv4' : 'IPv4 Flow Specification',
               'ipv6' : 'IPv6 Flow Specification',
            } ) }
   handler = "BgpFlowspecCliHandler.EnterBgpFlowspecAfMode_handler"
   noOrDefaultHandler = \
      "BgpFlowspecCliHandler.EnterBgpFlowspecAfMode_noOrDefaultHandler"

RouterBgpSharedModelet.addCommandClass( EnterBgpFlowspecAfMode )

#-------------------------------------------------------------------------------
# "[ no | default ] address-family flow-spec ( vpn-ipv4 | vpn-ipv6 )"
#-------------------------------------------------------------------------------
class RouterBgpBaseAfFlowspecVpnMode( RoutingBgpBaseAfMode,
                                      BasicCli.ConfigModeBase ):
   name = 'BGP address family Flow Specification VPN configuration'

   def __init__( self, parent, session, addrFamily ):
      self.vrfName = DEFAULT_VRF
      RoutingBgpBaseAfMode.__init__( self, addrFamily,
                                     modeKey='router-bgp-af-flow-spec' )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

RouterBgpBaseAfFlowspecVpnMode.addModelet(
   RouterBgpAfUnsupportedPrefixListSharedModelet )

class EnterBgpFlowspecVpnAfMode( CliCommand.CliCommandClass ):
   syntax = 'address-family flow-spec VPN_AF'
   noOrDefaultSyntax = syntax

   data = { 'address-family' : bgpTokens.addrFamily,
            'flow-spec' : flowspecKwMatcher,
            'VPN_AF' : CliMatcher.EnumMatcher( {
               'vpn-ipv4' : 'VPN IPv4 Flow Specification',
               'vpn-ipv6' : 'VPN IPv6 Flow Specification',
            } ) }
   handler = "BgpFlowspecCliHandler.EnterBgpFlowspecVpnAfMode_handler"
   noOrDefaultHandler = \
      "BgpFlowspecCliHandler.EnterBgpFlowspecVpnAfMode_noOrDefaultHandler"

if BgpCommonToggleLib.toggleFlowspecVpnRxTxEnabled():
   RouterBgpBaseMode.addCommandClass( EnterBgpFlowspecVpnAfMode )

# -------------------------------------------------------------------------------
# "[no|default] bgp skip rib-install [ bestpath-selection ] " command, in
# IPv4/v6 Flowspec AfiSafi modes
# -------------------------------------------------------------------------------
RouterBgpAfFlowspecModelet.addCommandClass( SkipRibBestpathCommand )

# -------------------------------------------------------------------------------
# "[no] redistribute flow-spec POLICY_NAME [ rcf FUNCTION ]" command in
# IPv4/v6 Flowspec AfiSafi modes
# -------------------------------------------------------------------------------
class RedistributeFlowspecPolicyCommand( CliCommand.CliCommandClass ):
   syntax = 'redistribute flow-spec policy POLICY_NAME [ rcf FUNCTION ]'
   noOrDefaultSyntax = 'redistribute flow-spec policy POLICY_NAME ...'
   data = {
      'redistribute' : bgpTokens.redistribute,
      'flow-spec' : 'Redistribute flow-spec',
      'policy' : 'Flow-spec policy name',
      'POLICY_NAME' : flowspecPolicyNameMatcher,
      'rcf' : bgpTokens.rcf,
      'FUNCTION' : rcfFunctionMatcher,
   }

   handler = "BgpFlowspecCliHandler.redistributeFlowspecPolicy_handler"
   noOrDefaultHandler = \
      "BgpFlowspecCliHandler.redistributeFlowspecPolicy_noOrDefaultHandler"

if RoutingLibToggleLib.toggleBgpFlowspecStaticPolicyEnabled():
   RouterBgpAfFlowspecModelet.addCommandClass( RedistributeFlowspecPolicyCommand )

#-------------------------------------------------------------------------------
# "[no|default] neighbor <ipaddr|peerGroup> activate"
#-------------------------------------------------------------------------------
RouterBgpAfFlowspecModelet.addCommandClass( RouterBgpAfActivateCommand )
if BgpCommonToggleLib.toggleFlowspecVpnRxTxEnabled():
   RouterBgpBaseAfFlowspecVpnMode.addCommandClass( RouterBgpAfActivateCommand )

#-------------------------------------------------------------------------------
# "[no|default] bgp missing-policy [include sub-route-map] direction [ in | out ]
# action" command, in BgpFlowspec mode.
#-------------------------------------------------------------------------------
RouterBgpAfFlowspecModelet.addCommandClass( MissingPolicyAfSharedCmd )
RouterBgpAfFlowspecModelet.addCommandClass( SetNeighborMissingPolicyAfCmd )
if BgpCommonToggleLib.toggleFlowspecVpnRxTxEnabled():
   RouterBgpBaseAfFlowspecVpnMode.addCommandClass( MissingPolicyAfSharedCmd )
   RouterBgpBaseAfFlowspecVpnMode.addCommandClass( SetNeighborMissingPolicyAfCmd )

# --------------------------------------------------------------------------------
# ( no | default ) neighbor ( ADDR | V6ADDR | LLV6ADDR | PEERGROUPNAME )
# rcf DIRECTION ( ( FUNCTION [ disabled ] ) | disabled )
# --------------------------------------------------------------------------------
RouterBgpAfFlowspecModelet.addCommandClass( SetNeighborRcfCmd )
if RcfLibToggleLib.toggleRcfFlowspecVpnPoaEnabled():
   RouterBgpBaseAfFlowspecVpnMode.addCommandClass( SetNeighborRcfCmd )

# -------------------------------------------------------------------------------
# "[no|default] neighbor GROUP | ADDR maximum-routes NUM_ROUTES
#  [ warning-limit THRESHOLD percent ] [ warning-only ]" command, in BgpFlowspec mode
# -------------------------------------------------------------------------------
RouterBgpAfFlowspecModelet.addCommandClass( SetNeighborMaxRoutesCmd )
RouterBgpBaseAfFlowspecVpnMode.addCommandClass( SetNeighborMaxRoutesCmd )

# -----------------------------------------------------------------------------
# "[no|default] neighbor GROUP | ADDR maximum-accepted-routes NUM_ACCEPTED_ROUTES
#    [warning-limit THRESHOLD]" command, in BgpFlowspec mode
# -----------------------------------------------------------------------------
RouterBgpAfFlowspecModelet.addCommandClass( SetNeighborMaxAcceptedRoutesCmd )
RouterBgpBaseAfFlowspecVpnMode.addCommandClass( SetNeighborMaxAcceptedRoutesCmd )

# -----------------------------------------------------------------------------
# "[no|default] neighbor GROUP | ADDR maximum-advertised-routes NUM_ADV_ROUTES
#    [warning-limit THRESHOLD [percent] ]", in BgpFlowspec mode
# -----------------------------------------------------------------------------
RouterBgpAfFlowspecModelet.addCommandClass( SetNeighborMaxAdvRoutesCmd )
RouterBgpBaseAfFlowspecVpnMode.addCommandClass( SetNeighborMaxAdvRoutesCmd )

# -------------------------------------------------------------------------------
# "[no|default] route-target redirect vrf lpm" command in
# IPv4/v6 Flowspec AfiSafi modes
# -------------------------------------------------------------------------------
class RouteTargetRedirectVrfLpmCommand( CliCommand.CliCommandClass ):
   syntax = 'route-target redirect vrf lpm'
   noOrDefaultSyntax = 'route-target redirect vrf lpm'
   data = {
      'route-target' : 'Route Target',
      'redirect' : 'Redirect packet flow',
      'vrf' : 'Vrf',
      'lpm' : 'Longest prefix match route lookup',
   }

   handler = "BgpFlowspecCliHandler.routeTargetRedirectVrfLpm_handler"
   noOrDefaultHandler = \
      "BgpFlowspecCliHandler.routeTargetRedirectVrfLpm_noOrDefaultHandler"

if FlowspecToggleLib.toggleFlowspecVrfSelectionEnabled():
   RouterBgpAfFlowspecModelet.addCommandClass( RouteTargetRedirectVrfLpmCommand )

def Plugin( entityManager ):
   for af in [ 'flow-spec ipv4', 'flow-spec ipv6' ]:
      afModeExtensionHook.addAfModeExtension( af, RouterBgpBaseAfFlowspecMode )
      afModeExtensionHook.addVrfAfModeExtension( af, RouterBgpVrfAfFlowspecMode )
   for af in [ 'flow-spec vpn-ipv4', 'flow-spec vpn-ipv6' ]:
      # Flowspec VPN address families are only supported in default VRF
      afModeExtensionHook.addAfModeExtension( af, RouterBgpBaseAfFlowspecVpnMode )
   # We need to register our BgpFlowspec specific parsing subrules for extended
   # communities, because they are reparsed in the Cli client to generate the
   # user facing string representations.
   # initialize the global ext comm parser state machinery, or get the existing one
   bgpExtCommParserContainerSm = getBgpExtCommParserContainerSm()
   # create Flowspec-specific ext comm parsing SM and add it to the container
   parseExtCommFlowspecSm = Tac.newInstance(
      "Routing::Bgp::FlowspecPlugin::FlowspecParseExtCommSm",
      bgpExtCommParserContainerSm.parseExtCommRuleCollections )
   bgpExtCommParserContainerSm.extCommParserSms.enq( parseExtCommFlowspecSm )
