#!/usr/bin/env python3
# Copyright (c) 2018 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import BasicCli
from BasicCli import addShowCommandClass
from CliCommand import CliExpression
# pylint:disable=W0611
import CliPlugin.BgpCliHelperCli
from CliPlugin.RoutingBgpCli import V4V6PeerKeyCliExpression
from CliPlugin.RoutingBgpShowCli import (
   allVrfExprFactory,
   summaryVrfModel,
)
from CliPlugin.ArBgpCli import ShowBgpDebugPolicyBase
from CliPlugin.FlowspecShowCli import flowSpecMatcherForShow
from CliPlugin.FlowspecMatcher import ruleStringMatcher
from CliPlugin.FlowspecConfigCli import flowspecPolicyNameMatcher
from CliToken.RoutingBgpShowCliTokens import (
   advertised,
   bgpAfterShow,
   detail,
   neighbors,
   received,
   receivedAll,
   summary,
   routeTypeMatcher,
)
from CliToken.Flowspec import (
   ruleMatcher,
   identifierMatcher,
   ruleIdMatcher,
   destinationMatcher,
   sourceMatcher,
   ipv4AddrMatcherForShow,
   ipv6AddrMatcherForShow,
   ipv4FlowspecVpnMatcherForShow,
   ipv6FlowspecVpnMatcherForShow,
   RouteDistinguisherCliExpression,
   interfaceSetMatcherForShow,
   interfaceSetIdMatcher,
   policyMatcherForShow
)
from CliToken.Ipv4 import ipv4MatcherForShow
from CliToken.Ipv6 import ipv6MatcherForShow

from ShowCommand import ShowCliCommandClass
from Toggles.BgpCommonToggleLib import toggleFlowspecVpnRxTxEnabled
from Toggles.RoutingLibToggleLib import toggleBgpFlowspecInterfaceSetEnabled
from Toggles.RoutingLibToggleLib import toggleBgpFlowspecStaticPolicyEnabled

from Toggles import RcfLibToggleLib

class RouteTypeAdvPolicyMatcher( CliExpression ):
   # Matcher (similar to routeTypeMatcher) but allows an optional policy parameter
   # for advertised-routes. This is to specify the flowspec policy for redistributed
   # routes.
   expression = "( routes | received-routes | "
   if toggleBgpFlowspecStaticPolicyEnabled():
      expression += "( advertised-routes [ policy POLICY_NAME ] ) )"
   else:
      expression += "advertised-routes )"

   data = {
         "routes" : received,
         "advertised-routes" : advertised,
         "received-routes" : receivedAll,
         "policy" : policyMatcherForShow,
         "POLICY_NAME" : flowspecPolicyNameMatcher,
         }

   @staticmethod
   def adapter( mode, args, argList ):
      if args.pop( "routes", None ):
         args[ "ROUTE_TYPE" ] = "routes"
      elif args.pop( "advertised-routes", None ):
         args[ "ROUTE_TYPE" ] = "advertised-routes"
      elif args.pop( "received-routes", None ):
         args[ "ROUTE_TYPE" ] = "received-routes"
      else:
         assert False

#-------------------------------------------------------------------------------
# "show bgp flow-spec ..."
#-------------------------------------------------------------------------------
# Define common syntax for the ipv4 and ipv6 versions of these commands.
# We can have a common handler, but need to register separate command classes
# because the address-family of the command affects which matchers should be used
# for matching prefix args.
# Define syntax and data in this base class, the derived classes need to update the
# syntax and data to add address-family specific matchers.
#
class ShowBgpFlowspecBaseClass:
   @staticmethod
   def showFlowspecSyntax( af ):
      command = ( f'show bgp flow-spec {af} '
                  '[ ( rule ( STRING | ( identifier RULEID ) ) ) ] '
                  '[ destination DSTPREFIX ] [ source SRCPREFIX ] '
                  '[ interface-set SETID ] '
                  '[ policy POLICY_NAME ] '
                  '[ detail ] [ VRF ]' )
      if not toggleBgpFlowspecInterfaceSetEnabled():
         command = command.replace( '[ interface-set SETID ] ', '' )
      if not toggleBgpFlowspecStaticPolicyEnabled():
         command = command.replace( '[ policy POLICY_NAME ] ', '' )
      return command

   @staticmethod
   def showFlowspecNeighborSyntax( af ):
      command = ( f'show bgp neighbors PEER_ADDR flow-spec {af} '
                  '( ROUTE_TYPE ) '
                  '[ ( rule ( STRING | ( identifier RULEID ) ) ) ] '
                  '[ destination DSTPREFIX ] [ source SRCPREFIX ] '
                  '[ interface-set SETID ] '
                  '[ detail ] [ VRF ]' )
      if not toggleBgpFlowspecInterfaceSetEnabled():
         command = command.replace( '[ interface-set SETID ] ', '' )
      return command

   data = {
      'bgp' : bgpAfterShow,
      'flow-spec' : flowSpecMatcherForShow,
      # af : needs to be updated in each specific show command,
      'rule' : ruleMatcher,
      'STRING' : ruleStringMatcher,
      'identifier' : identifierMatcher,
      'RULEID' : ruleIdMatcher,
      'source' : sourceMatcher,
      'destination' : destinationMatcher,
      # 'SRCPREFIX' : needs to be updated in each specific show command,
      # 'DSTPREFIX' : needs to be updated in each specific show command,
      'interface-set' : interfaceSetMatcherForShow,
      'SETID' : interfaceSetIdMatcher,
      'policy' : policyMatcherForShow,
      'POLICY_NAME' : flowspecPolicyNameMatcher,
      'detail' : detail,
      'VRF' : allVrfExprFactory,
   }
   if not toggleBgpFlowspecInterfaceSetEnabled():
      for kw in [ 'interface-set', 'SETID' ]:
         data.pop( kw )
   if not toggleBgpFlowspecStaticPolicyEnabled():
      for kw in [ 'policy', 'POLICY_NAME' ]:
         data.pop( kw )

   # neighbor data for afiSafi's not supporting redistributed routes (e.g. VPN)
   neighborData = data.copy()
   neighborData.update( {
      'neighbors' : neighbors,
      'PEER_ADDR' : V4V6PeerKeyCliExpression,
      'ROUTE_TYPE' : routeTypeMatcher,
   } )
   # neighbor data for afiSafi's supporting redistributed routes (e.g. ipv4/v6)
   neighborDataWithPolicy = data.copy()
   neighborDataWithPolicy.update( {
      'neighbors' : neighbors,
      'PEER_ADDR' : V4V6PeerKeyCliExpression,
      'ROUTE_TYPE' : RouteTypeAdvPolicyMatcher,
   } )

   handler = "BgpFlowspecShowCliHandler.ShowBgpFlowspecBaseClass_handler"

#-------------------------------------------------------------------------------
# "show bgp flow-spec ipv4 [ rule ( string | identifier id ) ]
#                          [ destination <ipv4-prefix> ]
#                          [ source <ipv4-prefix> ]
#                          [ interface-set < id > ]
#                          [ policy < policy-name > ]
#                          [ detail ] [vrf <vrfName/all]"
#-------------------------------------------------------------------------------
# Show BGP flow-spec rules, optionally filtering by rule, rule id,
# source, destination prefix or static flowspec policy name.
class ShowBgpFlowspecV4Cmd( ShowCliCommandClass,
                            ShowBgpFlowspecBaseClass ):
   _af = 'ipv4'
   syntax = ShowBgpFlowspecBaseClass.showFlowspecSyntax( _af )
   data = ShowBgpFlowspecBaseClass.data.copy()
   data.update( {
      _af : ipv4MatcherForShow,
      'SRCPREFIX' : ipv4AddrMatcherForShow,
      'DSTPREFIX' : ipv4AddrMatcherForShow,
   } )

   cliModel = "BgpFlowspecCliModels.bgpFlowspecRulesVrfModel"

BasicCli.addShowCommandClass( ShowBgpFlowspecV4Cmd )

#-------------------------------------------------------------------------------
# "show bgp flow-spec ipv6 [ rule ( string | identifier id ) ]
#                          [ destination <ipv6-prefix> ]
#                          [ source <ipv6-prefix> ]
#                          [ policy < policy-name > ]
#                          [ detail ] [vrf <vrfName/all]"
#-------------------------------------------------------------------------------
class ShowBgpFlowspecV6Cmd( ShowCliCommandClass,
                            ShowBgpFlowspecBaseClass ):
   _af = 'ipv6'
   syntax = ShowBgpFlowspecBaseClass.showFlowspecSyntax( _af )
   data = ShowBgpFlowspecBaseClass.data.copy()
   data.update( {
      _af : ipv6MatcherForShow,
      'SRCPREFIX' : ipv6AddrMatcherForShow,
      'DSTPREFIX' : ipv6AddrMatcherForShow,
   } )

   cliModel = "BgpFlowspecCliModels.bgpFlowspecRulesVrfModel"

BasicCli.addShowCommandClass( ShowBgpFlowspecV6Cmd )

class ShowBgpFlowspecVpnBaseClass ( ShowBgpFlowspecBaseClass ):
   @staticmethod
   def showFlowspecSyntax( af ):
      command = ( f'show bgp flow-spec {af} '
                  '[ ( rule ( STRING | ( identifier RULEID ) ) ) ] '
                  '[ destination DSTPREFIX ] [ source SRCPREFIX ] '
                  '[ interface-set SETID ] '
                  '[ RD ] '
                  '[ detail ] [ VRF ]' )
      if not toggleBgpFlowspecInterfaceSetEnabled():
         command = command.replace( '[ interface-set SETID ] ', '' )
      return command

   @staticmethod
   def showFlowspecNeighborSyntax( af ):
      command = ( f'show bgp neighbors PEER_ADDR flow-spec {af} '
                  '( ROUTE_TYPE ) '
                  '[ ( rule ( STRING | ( identifier RULEID ) ) ) ] '
                  '[ destination DSTPREFIX ] [ source SRCPREFIX ] '
                  '[ interface-set SETID ] '
                  '[ RD ] '
                  '[ detail ] [ VRF ]' )
      if not toggleBgpFlowspecInterfaceSetEnabled():
         command = command.replace( '[ interface-set SETID ] ', '' )
      return command

   data = ShowBgpFlowspecBaseClass.data.copy()
   data.update( {
      'RD' : RouteDistinguisherCliExpression
   } )
   neighborData = ShowBgpFlowspecBaseClass.neighborData.copy()
   neighborData.update( {
      'RD' : RouteDistinguisherCliExpression
   } )

   handler = "BgpFlowspecShowCliHandler.ShowBgpFlowspecVpnBaseClass_handler"

# -------------------------------------------------------------------------------
# "show bgp flow-spec vpn-ipv4 [ rule ( string | identifier id ) ]
#                              [ destination <ipv6-prefix> ]
#                              [ source <ipv6-prefix> ]
#                              [ rd <route-distinguisher> ]
#                              [ interface-set <id> ]
#                              [ detail ] [vrf <vrfName/all]"
# -------------------------------------------------------------------------------
class ShowBgpFlowspecVpnV4Cmd( ShowCliCommandClass, ShowBgpFlowspecVpnBaseClass ):
   _af = 'vpn-ipv4'
   syntax = ShowBgpFlowspecVpnBaseClass.showFlowspecSyntax( _af )
   data = ShowBgpFlowspecVpnBaseClass.data.copy()
   data.update( {
      _af : ipv4FlowspecVpnMatcherForShow,
      'SRCPREFIX' : ipv4AddrMatcherForShow,
      'DSTPREFIX' : ipv4AddrMatcherForShow,
   } )

   cliModel = "BgpFlowspecCliModels.bgpFlowspecRulesVrfModel"

BasicCli.addShowCommandClass( ShowBgpFlowspecVpnV4Cmd )

# -------------------------------------------------------------------------------
# "show bgp flow-spec vpn-ipv6 [ rule ( string | identifier id ) ]
#                              [ destination <ipv6-prefix> ]
#                              [ source <ipv6-prefix> ]
#                              [ interface-set <id> ]
#                              [ rd <route-distinguisher> ]
#                              [ detail ] [vrf <vrfName/all]"
# -------------------------------------------------------------------------------
class ShowBgpFlowspecVpnV6Cmd( ShowCliCommandClass, ShowBgpFlowspecVpnBaseClass ):
   _af = 'vpn-ipv6'
   syntax = ShowBgpFlowspecVpnBaseClass.showFlowspecSyntax( _af )
   data = ShowBgpFlowspecVpnBaseClass.data.copy()
   data.update( {
      _af : ipv6FlowspecVpnMatcherForShow,
      'SRCPREFIX' : ipv6AddrMatcherForShow,
      'DSTPREFIX' : ipv6AddrMatcherForShow,
   } )

   cliModel = "BgpFlowspecCliModels.bgpFlowspecRulesVrfModel"

BasicCli.addShowCommandClass( ShowBgpFlowspecVpnV6Cmd )

# -------------------------------------------------------------------------------
# "show bgp neighbors <peerAddr> flow-spec ipv4
#                          ( routes | received-routes | advertised-routes )
#                          [ rule ] [ rule identifier ]
#                          [ destination <ipv4-prefix> ]
#                          [ source <ipv4-prefix> ]
#                          [ policy POLICY_NAME ]
#                          [ detail ] [vrf <vrfName/all]"
# -------------------------------------------------------------------------------
# Show BGP flow-spec ipv4 rules received or advertised to a particular peer,
# optionally filtering by rule, rule id, source or destination prefix.
# Advertised-routes can also be filtered by policy name (for redistributed routes)
class ShowBgpNeighborsFlowspecV4Cmd( ShowCliCommandClass,
                                     ShowBgpFlowspecBaseClass ):
   syntax = ShowBgpFlowspecBaseClass.showFlowspecNeighborSyntax( 'ipv4' )
   data = ShowBgpFlowspecBaseClass.neighborDataWithPolicy.copy()
   data.update( {
      'ipv4' : ipv4MatcherForShow,
      'SRCPREFIX' : ipv4AddrMatcherForShow,
      'DSTPREFIX' : ipv4AddrMatcherForShow,
   } )

   cliModel = "BgpFlowspecCliModels.bgpFlowspecRulesVrfModel"

BasicCli.addShowCommandClass( ShowBgpNeighborsFlowspecV4Cmd )

# -------------------------------------------------------------------------------
# "show bgp neighbors <ip> flow-spec ipv6
#                          ( routes | received-routes | advertised-routes )
#                          [ rule ] [ rule identifier ]
#                          [ destination <ipv6-prefix> ]
#                          [ source <ipv6-prefix> ]
#                          [ policy POLICY_NAME ]
#                          [ detail ] [vrf <vrfName/all]"
# -------------------------------------------------------------------------------
# Show BGP flow-spec ipv6 rules received or advertised to a particular peer,
# optionally filtering by rule, rule id, source or destination prefix.
# Advertised-routes can also be filtered by policy name (for redistributed routes)
class ShowBgpNeighborsFlowspecV6Cmd( ShowCliCommandClass,
                                     ShowBgpFlowspecBaseClass ):
   syntax = ShowBgpFlowspecBaseClass.showFlowspecNeighborSyntax( 'ipv6' )
   data = ShowBgpFlowspecBaseClass.neighborDataWithPolicy.copy()
   data.update( {
      'ipv6' : ipv6MatcherForShow,
      'SRCPREFIX' : ipv6AddrMatcherForShow,
      'DSTPREFIX' : ipv6AddrMatcherForShow,
   } )

   cliModel = "BgpFlowspecCliModels.bgpFlowspecRulesVrfModel"

BasicCli.addShowCommandClass( ShowBgpNeighborsFlowspecV6Cmd )

# -------------------------------------------------------------------------------
# "show bgp neighbors <ip> flow-spec vpn-ipv4
#                          ( routes | received-routes | advertised-routes )
#                          [ rule ] [ rule identifier ]
#                          [ destination <ipv6-prefix> ]
#                          [ source <ipv6-prefix> ]
#                          [ rd <route-distinguisher> ]
#                          [ detail ] [vrf <vrfName/all]"
# -------------------------------------------------------------------------------
# Show BGP flow-spec vpn-ipv4 rules received or advertised to a particular peer,
# optionally filtering by rule, rule id, source or destination prefix.
class ShowBgpNeighborsFlowspecVpnV4Cmd( ShowCliCommandClass,
                                        ShowBgpFlowspecVpnBaseClass ):
   syntax = ShowBgpFlowspecVpnBaseClass.showFlowspecNeighborSyntax( 'vpn-ipv4' )
   data = ShowBgpFlowspecVpnBaseClass.neighborData.copy()
   data.update( {
      'vpn-ipv4' : ipv4FlowspecVpnMatcherForShow,
      'SRCPREFIX' : ipv4AddrMatcherForShow,
      'DSTPREFIX' : ipv4AddrMatcherForShow,
   } )

   cliModel = "BgpFlowspecCliModels.bgpFlowspecRulesVrfModel"

if toggleFlowspecVpnRxTxEnabled():
   BasicCli.addShowCommandClass( ShowBgpNeighborsFlowspecVpnV4Cmd )

# -------------------------------------------------------------------------------
# "show bgp neighbors <ip> flow-spec vpn-ipv6
#                          ( routes | received-routes | advertised-routes )
#                          [ rule ] [ rule identifier ]
#                          [ destination <ipv6-prefix> ]
#                          [ source <ipv6-prefix> ]
#                          [ rd <route-distinguisher> ]
#                          [ detail ] [vrf <vrfName/all]"
# -------------------------------------------------------------------------------
# Show BGP flow-spec vpn-ipv6 rules received or advertised to a particular peer,
# optionally filtering by rule, rule id, source or destination prefix.
class ShowBgpNeighborsFlowspecVpnV6Cmd( ShowCliCommandClass,
                                        ShowBgpFlowspecVpnBaseClass ):
   syntax = ShowBgpFlowspecVpnBaseClass.showFlowspecNeighborSyntax( 'vpn-ipv6' )
   data = ShowBgpFlowspecVpnBaseClass.neighborData.copy()
   data.update( {
      'vpn-ipv6' : ipv6FlowspecVpnMatcherForShow,
      'SRCPREFIX' : ipv6AddrMatcherForShow,
      'DSTPREFIX' : ipv6AddrMatcherForShow,
   } )

   cliModel = "BgpFlowspecCliModels.bgpFlowspecRulesVrfModel"

if toggleFlowspecVpnRxTxEnabled():
   BasicCli.addShowCommandClass( ShowBgpNeighborsFlowspecVpnV6Cmd )

#------------------------------------------------------
# "show bgp flow-spec ipv4 summary [vrf <vrfName/all]"
#------------------------------------------------------
class ShowBgpFlowspecSummaryCmd( ShowCliCommandClass ):
   syntax = 'show bgp flow-spec ( ipv4 | ipv6 | vpn-ipv4 | vpn-ipv6 ) summary \
               [ VRF ]'

   data = {
      'bgp' : bgpAfterShow,
      'flow-spec' : flowSpecMatcherForShow,
      'ipv4' : ipv4MatcherForShow,
      'ipv6' : ipv6MatcherForShow,
      'vpn-ipv4' : ipv4FlowspecVpnMatcherForShow,
      'vpn-ipv6' : ipv6FlowspecVpnMatcherForShow,
      'summary' : summary,
      'VRF' : allVrfExprFactory,
   }
   cliModel = summaryVrfModel
   handler = "BgpFlowspecShowCliHandler.ShowBgpFlowspecSummaryCmd_handler"

addShowCommandClass( ShowBgpFlowspecSummaryCmd )

# --------------------------------------------------------------------------------
# show bgp debug policy ( inbound | outbound ) neighbor ( ADDR | ADDR6 | all )
# flow-spec ( ipv4 | ipv6 ) [ VRF ] [ rcf RCF_FUNC_NAME() ] rule FLOWSPEC_RULE
# --------------------------------------------------------------------------------
class ShowBgpDebugPolicyFlowspecBase( ShowBgpDebugPolicyBase ):
   data = ShowBgpDebugPolicyBase.data.copy()
   data.update( { 'flow-spec' : flowSpecMatcherForShow,
                  'rule' : ruleMatcher,
                  'FLOWSPEC_RULE' : ruleStringMatcher } )
   handler = "BgpFlowspecShowCliHandler.policyDebugFlowspecHandler"

class ShowBgpDebugPolicyFlowspecInbound( ShowBgpDebugPolicyFlowspecBase ):
   syntax = 'show bgp debug policy ' \
            'inbound neighbor ( ADDR | ADDR6 | all ) ' \
            'flow-spec ( ipv4 | ipv6 ) [ VRF ] [ rcf RCF_FUNC_NAME ] ' \
            'rule FLOWSPEC_RULE '
   data = ShowBgpDebugPolicyFlowspecBase.data.copy()

BasicCli.addShowCommandClass( ShowBgpDebugPolicyFlowspecInbound )

class ShowBgpDebugPolicyFlowspecOutbound( ShowBgpDebugPolicyFlowspecBase ):
   syntax = 'show bgp debug policy ' \
            'outbound neighbor ( ADDR | ADDR6 ) ' \
            'flow-spec ( ipv4 | ipv6 ) [ VRF ] [ rcf RCF_FUNC_NAME ] ' \
            'rule FLOWSPEC_RULE '
   data = ShowBgpDebugPolicyFlowspecBase.data.copy()

BasicCli.addShowCommandClass( ShowBgpDebugPolicyFlowspecOutbound )

# --------------------------------------------------------------------------------
# show bgp debug policy ( inbound | outbound ) neighbor ( ADDR | ADDR6 | all )
# flow-spec ( vpn-ipv4 | vpn-ipv6 ) [ rcf RCF_FUNC_NAME() ] rule FLOWSPEC_RULE
#
# Note: There is no optional VRF parameter since only the default VRF is
#       applicable to Flowspec VPN.
# --------------------------------------------------------------------------------
class ShowBgpDebugPolicyFlowspecVpnBase( ShowBgpDebugPolicyFlowspecBase ):
   data = ShowBgpDebugPolicyFlowspecBase.data.copy()
   data.update( { 'vpn-ipv4' : 'VPN-IPv4 related',
                  'vpn-ipv6' : 'VPN-IPv6 related' } )

class ShowBgpDebugPolicyFlowspecVpnInbound( ShowBgpDebugPolicyFlowspecVpnBase ):
   syntax = 'show bgp debug policy ' \
            'inbound neighbor ( ADDR | ADDR6 | all ) ' \
            'flow-spec ( vpn-ipv4 | vpn-ipv6 ) [ rcf RCF_FUNC_NAME ] ' \
            'rule FLOWSPEC_RULE '
   data = ShowBgpDebugPolicyFlowspecVpnBase.data.copy()

if RcfLibToggleLib.toggleShowBgpDebugPolicyRcfFlowspecVpnEnabled():
   BasicCli.addShowCommandClass( ShowBgpDebugPolicyFlowspecVpnInbound )

class ShowBgpDebugPolicyFlowspecVpnOutbound( ShowBgpDebugPolicyFlowspecVpnBase ):
   syntax = 'show bgp debug policy ' \
            'outbound neighbor ( ADDR | ADDR6 ) ' \
            'flow-spec ( vpn-ipv4 | vpn-ipv6 ) [ rcf RCF_FUNC_NAME ] ' \
            'rule FLOWSPEC_RULE '
   data = ShowBgpDebugPolicyFlowspecVpnBase.data.copy()

if RcfLibToggleLib.toggleShowBgpDebugPolicyRcfFlowspecVpnEnabled():
   BasicCli.addShowCommandClass( ShowBgpDebugPolicyFlowspecVpnOutbound )

# --------------------------------------------------------------------------------
# show bgp debug policy redistribute flow-spec ( ipv4 | ipv6 )
#    [ VRF ] [ rcf RCF_FUNC_NAME() ] rule FLOWSPEC_RULE
# --------------------------------------------------------------------------------
class ShowBgpDebugPolicyFlowspecRedistribute( ShowBgpDebugPolicyFlowspecBase ):
   syntax = 'show bgp debug policy redistribute ' \
            'flow-spec ( ipv4 | ipv6 ) [ VRF ] [ rcf RCF_FUNC_NAME ] ' \
            'rule FLOWSPEC_RULE '
   data = ShowBgpDebugPolicyFlowspecBase.data.copy()

if RcfLibToggleLib.toggleShowBgpDebugPolicyRcfRedistFlowspecEnabled():
   BasicCli.addShowCommandClass( ShowBgpDebugPolicyFlowspecRedistribute )
