# Copyright (c) 2014 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from __future__ import absolute_import, division, print_function

from itertools import chain
import BasicCli
import ConfigMount
import Tac
import BasicCliModes
import CliCommand
from CliMode.BgpGroup import BgpGroupMode, BgpBuiltinGroupMode
from CliPlugin.GroupLib import GroupConfigMode
from CliPlugin.IraVrfCli import vrfNamePattern
from CliPlugin.MaintenanceMode import MaintenanceUnitConfigMode
from CliPlugin.RoutingBgpCli import PeerCliExpression
from CliPlugin import VrfCli
from CliPlugin.MaintenanceCliLib import (
   bgpNeighVrfBuiltinGroupPrefix,
   groupNameMatcher,
)
from CliPlugin.VrfCli import vrfMatcher

globalBgpGroupConfigDir = None

class BgpBuiltinGroupConfigMode( GroupConfigMode, BgpBuiltinGroupMode,
                                 BasicCli.ConfigModeBase ):
   name = "Bgp Builtin Group configuration"

   def __init__( self, parent, session, bgpBuiltinGroup ):
      GroupConfigMode.__init__( self, session, bgpBuiltinGroup )
      BgpBuiltinGroupMode.__init__( self, bgpBuiltinGroup.name() )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

class BgpGroupConfigMode( GroupConfigMode, BgpGroupMode,
                          BasicCli.ConfigModeBase ):
   name = "Bgp Group configuration"

   def __init__( self, parent, session, bgpGroup ):
      GroupConfigMode.__init__( self, session, bgpGroup )
      BgpGroupMode.__init__( self, bgpGroup.name() )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

bgpGroupNameMatcher = groupNameMatcher( lambda mode:
                                        globalBgpGroupConfigDir.config )

bgpGroupNameWithBuiltinMatcher = groupNameMatcher(
   lambda mode: chain( globalBgpGroupConfigDir.config,
                       ( bgpNeighVrfBuiltinGroupPrefix + vrf for vrf in
                         VrfCli.getAllVrfNames( mode ) ) ),
   pattern=vrfNamePattern )

#--------------------------------------------------------------------------------
# [ no | default ] group bgp BGP_GROUP_NAME in global config mode
#--------------------------------------------------------------------------------
class GroupBgpCmd( CliCommand.CliCommandClass ):
   syntax = 'group bgp BGP_GROUP_NAME'
   noOrDefaultSyntax = syntax
   data = {
      'group': 'Configure Group',
      'bgp': 'Configure BGP Group',
      'BGP_GROUP_NAME': bgpGroupNameWithBuiltinMatcher,
   }

   handler = "BgpGroupCliHandler.GroupBgpCmd_handler"
   noOrDefaultHandler = "BgpGroupCliHandler.GroupBgpCmd_noOrDefaultHandler"

BasicCliModes.GlobalConfigMode.addCommandClass( GroupBgpCmd )

#--------------------------------------------------------------------------------
# [ no | default ] neighbor PEER
#--------------------------------------------------------------------------------
class BgpGroupNeighborCmd( CliCommand.CliCommandClass ):
   syntax = 'neighbor PEER'
   noOrDefaultSyntax = syntax
   data = {
      'neighbor': 'Configure member',
      'PEER': PeerCliExpression,
   }

   handler = "BgpGroupCliHandler.BgpGroupNeighborCmd_handler"
   noOrDefaultHandler = "BgpGroupCliHandler.BgpGroupNeighborCmd_noOrDefaultHandler"

BgpGroupConfigMode.addCommandClass( BgpGroupNeighborCmd )

#--------------------------------------------------------------------------------
# [ no | default ] vrf [ VRFNAME ]
#--------------------------------------------------------------------------------
class BgpGroupVrfCmd( CliCommand.CliCommandClass ):
   syntax = 'vrf VRF_NAME'
   noOrDefaultSyntax = 'vrf [ VRF_NAME ]'
   data = {
      'vrf': 'VRF name',
      'VRF_NAME': vrfMatcher,
   }

   handler = "BgpGroupCliHandler.BgpGroupVrfCmd_handler"
   noOrDefaultHandler = "BgpGroupCliHandler.BgpGroupVrfCmd_noOrDefaultHandler"

BgpGroupConfigMode.addCommandClass( BgpGroupVrfCmd )

#--------------------------------------------------------------------------------
# [ no | default ] group bgp GROUPNAME in maintenance-unit mode
#--------------------------------------------------------------------------------
class MaintenanceUnitGroupBgpCmd( CliCommand.CliCommandClass ):
   syntax = 'group bgp BGP_GROUP_NAME'
   noOrDefaultSyntax = syntax
   data = {
      'group': 'Configure Group',
      'bgp': 'Configure BGP Group',
      'BGP_GROUP_NAME': bgpGroupNameWithBuiltinMatcher,
   }

   handler = "BgpGroupCliHandler.MaintenanceUnitGroupBgpCmd_handler"
   noOrDefaultHandler = "BgpGroupCliHandler." \
      "MaintenanceUnitGroupBgpCmd_noOrDefaultHandler"

MaintenanceUnitConfigMode.addCommandClass( MaintenanceUnitGroupBgpCmd )

def Plugin( entityManager ):
   global globalBgpGroupConfigDir
   globalBgpGroupConfigDir = ConfigMount.mount( entityManager,
      'group/config/bgp', 'BgpGroup::ConfigDir', 'w' )
