#!/usr/bin/env python3
# Copyright (c) 2017 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# pylint: disable=consider-using-f-string
from __future__ import absolute_import, division, print_function
from Toggles.TunnelToggleLib import toggleDsfPhase1Enabled
import Arnet
from ArnetModel import (
   Ip4Address,
   IpGenericAddress
)
import Tac
from CliModel import (
   Bool,
   Enum,
   GeneratorList,
   Int,
   Float,
   List,
   Dict,
   Model,
   Str,
   Submodel
)
from CliPlugin.BgpCliModels import (
   BgpOrrPosition,
   TcpInfo,
   _degradeToIpGenAddress,
   generateAfiSafiModel,
   peerKeyFromData,
   trimIntfName,
)
from CliPlugin import BgpNeighborsGatedConsts as GatedConstants
from BgpLib import PeerConfigKey
from HumanReadable import formatTimeInterval
from DeviceNameLib import intfLongName
import six
from six.moves import zip

# This file contains the CAPI model structure for the 'show ip bgp neighbors'
# command for both ArBgp and gated. The ArBgp implementation is present in
# '/src/ArBgp/BgpPeerCommandCallback.tin', both the cli text and CAPI output
# are carried out via this c++ streaming. The ArBgp neighbors command is loosely
# coupled to these models, with only tests verifying the c++ output matches this
# model structure.
# Gated directly depends on the models in this file, and utilises the processData
# functions via RibCapiLib to manipulate the data into more meaningful attribute
# names and values. The gated implementation also uses the render functions for the
# text output.
# The CAPI documentation for both ArBgp and Gated is generated via this
# file. It is important to ensure the ArBgp attribute names and structure
# conforms to these models, as they are not inherently verified via the
# cli CAPI infra.

# -------------------------------------------------------------------------------
# "show ip[v6] bgp neighbors [<ip>] [bfd] [vrf <vrfName>]"
# -------------------------------------------------------------------------------

removeOrReplacePrivateAsEgressHelp = '''From outbound updates to this neighbor:
   removePrivateAsEgress - Remove private AS numbers when there are only private 
   AS numbers in inbound/outbound AS path lists.
   removePrivateAsEgressAll - Always remove private AS numbers in outbound 
   AS path lists. 
   removePrivateAsEgressReplace - Replace private AS numbers with the local 
   AS number in outbound AS path lists.'''

removeOrReplacePrivateAsIngressHelp = '''From inbound updates to this neighbor:
   removePrivateAsIngress - Always remove private AS numbers in 
   inbound AS path lists.
   removePrivateAsIngressReplace - Replace private AS numbers with the local 
   AS number in inbound AS path lists.'''

class TcpConnOptions( Model ):
   timestampsEnabled = Bool( help='Timestamps are enabled' )
   selectiveAcknowledgementsEnabled = Bool( help='Selective Acknowledgments '
                                                 'are enabled' )
   windowScaleEnabled = Bool( help='Window Scale is enabled' )
   ecnEnabled = Bool( help='Explicit Congestion Notification (ECN) is enabled' )

class BgpPeerTcpInfo( TcpInfo ):
   __revision__ = 2

   def getMioAttrId( self ):
      return GatedConstants.MioDgetTcpInfo.MIO_DGET_BGP_PEER_TCPINFO

   def degrade( self, dictRepr, revision ):
      if revision < 2:
         # See BUG279484 for details.
         renamedAttributes = {
               'inqLen': 'inputQueueLength',
               'inqMaxLen': 'inputMaxQueueLength',
               'outqLen': 'outputQueueLength',
               'outqMaxLen': 'outputMaxQueueLength',
               'slowStartThr': 'slowStartThreshold'
               }
         for before, after in six.iteritems( renamedAttributes ):
            dictRepr[ before ] = dictRepr.pop( after )
      return dictRepr

   options = Int( help='TCP socket options' )
   connectionOptions = Submodel( valueType=TcpConnOptions, optional=True,
                                 help='TCP socket optional settings' )
   totalRetrans = Int( help='Total number of TCP retransmissions' )
   slowStartThreshold = Int( help='TCP send slow start size threshold' )
   rcvRtt = Int( help='TCP receive round-trip time (microseconds)' )
   outputQueueLength = Int( help='TCP output queue length' )
   outputMaxQueueLength = Int( help='TCP output queue max length' )
   inputQueueLength = Int( help='TCP input queue length' )
   inputMaxQueueLength = Int( help='TCP input queue max length' )

   def processData( self, data ):
      if 'tcpiState' in data:
         # pylint: disable=W0201,attribute-defined-outside-init
         self.state = data.pop( 'tcpiState' )
      if 'tcpiOptions' in data:
         self.options = data.pop( 'tcpiOptions' )
         self.connectionOptions = TcpConnOptions()
         self.connectionOptions.timestampsEnabled = \
               bool( self.options &
                     GatedConstants.DabitBgpPeerTcpInfo.
                           DABIT_BGP_PEER_TCPINFO_OPT_TIMESTAMPS )
         self.connectionOptions.selectiveAcknowledgementsEnabled = \
               bool( self.options &
                     GatedConstants.DabitBgpPeerTcpInfo.
                           DABIT_BGP_PEER_TCPINFO_OPT_SACK )
         self.connectionOptions.windowScaleEnabled = \
               bool( self.options &
                     GatedConstants.DabitBgpPeerTcpInfo.
                           DABIT_BGP_PEER_TCPINFO_OPT_WSCALE )
         self.connectionOptions.ecnEnabled = \
               bool( self.options &
                     GatedConstants.DabitBgpPeerTcpInfo.
                           DABIT_BGP_PEER_TCPINFO_OPT_ECN )
      if 'tcpiSndWscale' in data:
         self.sendWindowScale = data.pop( 'tcpiSndWscale' )
      if 'tcpiRcvWscale' in data:
         self.rcvWindowScale = data.pop( 'tcpiRcvWscale' )
      if 'tcpiRto' in data:
         self.retransTimeout = data.pop( 'tcpiRto' )
      if 'tcpiAto' in data:
         self.delayedAckTimeout = data.pop( 'tcpiAto' )
      if 'tcpiSndMss' in data:
         self.maxSegmentSize = data.pop( 'tcpiSndMss' )
      if 'tcpiRtt' in data:
         self.sndRtt = data.pop( 'tcpiRtt' )
      if 'tcpiRttvar' in data:
         self.sndRttVariance = data.pop( 'tcpiRttvar' )
      if 'tcpiSndSsthresh' in data:
         self.slowStartThreshold = data.pop( 'tcpiSndSsthresh' )
      if 'tcpiSndCwnd' in data:
         self.congestionWindow = data.pop( 'tcpiSndCwnd' )
      if 'tcpiRcvRtt' in data:
         self.rcvRtt = data.pop( 'tcpiRcvRtt' )
      if 'tcpiRcvSpace' in data:
         self.rcvWindow = data.pop( 'tcpiRcvSpace' )
      if 'tcpiTotalRetrans' in data:
         self.totalRetrans = data.pop( 'tcpiTotalRetrans' )
      if 'tcpiOutqLen' in data:
         self.outputQueueLength = data.pop( 'tcpiOutqLen' )
      if 'tcpiOutqMaxLen' in data:
         self.outputMaxQueueLength = data.pop( 'tcpiOutqMaxLen' )
      if 'tcpiInqLen' in data:
         self.inputQueueLength = data.pop( 'tcpiInqLen' )
      if 'tcpiInqMaxLen' in data:
         self.inputMaxQueueLength = data.pop( 'tcpiInqMaxLen' )

   def render( self ):
      if self.state is None:
         return

      inputQueueLength = self.inputQueueLength if self.inputQueueLength is not None \
               else "not available"
      inqMax = self.inputMaxQueueLength if self.inputMaxQueueLength is not None \
               else "not available"
      outputQueueLength = self.outputQueueLength if self.outputQueueLength \
                          is not None \
                else "not available"
      outqMax = self.outputMaxQueueLength if self.outputMaxQueueLength is not None \
                else "not available"

      print( "TCP Socket Information:" )
      print( "  TCP state is %s" % ( self.state ) )
      print( "  Recv-Q: %s/%s" % ( inputQueueLength, inqMax ) )
      print( "  Send-Q: %s/%s" % ( outputQueueLength, outqMax ) )
      print( "  Outgoing Maximum Segment Size (MSS): %s" % ( self.maxSegmentSize ) )
      print( "  Total Number of TCP retransmissions: %s" % ( self.totalRetrans ) )

      print( "  Options:" )
      print( "    Timestamps enabled: %s" %
             ( "yes" if self.connectionOptions.timestampsEnabled else "no" ) )
      print( "    Selective Acknowledgments enabled: %s" %
             ( "yes" if self.connectionOptions.selectiveAcknowledgementsEnabled
               else "no" ) )
      print( "    Window Scale enabled: %s" %
             ( "yes" if self.connectionOptions.windowScaleEnabled else "no" ) )
      print( "    Explicit Congestion Notification (ECN) enabled: %s" %
             ( "yes" if self.connectionOptions.ecnEnabled else "no" ) )
      print( "  Socket Statistics:" )
      print( "    Window Scale (wscale): %s,%s" %
             ( self.sendWindowScale, self.rcvWindowScale ) )
      print( "    Retransmission Timeout (rto): %.1fms" %
             ( float( self.retransTimeout ) / 1000 ) )
      print( "    Round-trip Time (rtt/rtvar): %.1fms/%.1fms" %
             ( float( self.sndRtt ) / 1000, float( self.sndRttVariance ) / 1000 ) )
      print( "    Delayed Ack Timeout (ato): %.1fms" %
             ( float( self.delayedAckTimeout ) / 1000 ) )
      print( "    Congestion Window (cwnd): %s" % ( self.congestionWindow ) )

      if self.slowStartThreshold < 65535:
         print( "    Slow-start Threshold (ssthresh): %s" %
                ( self.slowStartThreshold ) )
      if self.sndRtt > 0 and self.maxSegmentSize and self.congestionWindow:
         print( "    TCP Throughput: %.2f Mbps" %
                ( float( self.congestionWindow ) *
                  ( float( self.maxSegmentSize ) * 8. / float( self.sndRtt ) ) ) )

      if self.rcvRtt:
         print( "    Recv Round-trip Time (rcv_rtt): %.1fms" %
                ( float( self.rcvRtt ) / 1000 ) )
      if self.rcvWindow:
         print( "    Advertised Recv Window (rcv_space): %s" % ( self.rcvWindow ) )

      print( " " )

BITS_PER_PEERFLAG = 32

bfdJsonToStateName = {
   'adminDown': 'AdminDown',
   'init': 'Init',
   'up': 'Up',
   'down': 'Down',
}

class ErrorTimeInfo( Model ):
   time = Int( optional=True,
         help='Time at which last error occurred' )
   firstTime = Int( optional=True,
         help='Time at which last error occurred for the first time' )
   repeats = Int( optional=True,
         help='Number of times last error repeated' )

# used for input and output message statistics
class MessageStatInfo( Model ):
   updates = Int( optional=True, help='Updates' )
   keepalives = Int( optional=True, help='Keepalives' )
   opens = Int( optional=True, help='Opens' )
   rtRefreshes = Int( optional=True, help='Route Refreshes' )
   notifications = Int( optional=True, help='Notifications' )
   queueDepth = Int( optional=True, help='Queue Depth' )
   beginOfRtRefreshes = Int( optional=True, help='Beginning of Route Refreshes' )
   endOfRtRefreshes = Int( optional=True, help='End of Route Refreshes' )

class RouteFilterInfo( Model ):
   filterType = Enum( help='Type of route filter',
         values=( 'PrefixList', 'RCF', 'RouteMap' ) )
   inboundIpv4Uni = Str( optional=True, help='Inbound IPv4 Unicast' )
   inboundIpv4UniDefined = Bool( help='Inbound IPv4 Unicast RCF is defined',
                                    optional=True )
   outboundIpv4Uni = Str( optional=True, help='Outbound IPv4 Unicast' )
   outboundIpv4UniDefined = Bool( help='Outbound IPv4 Unicast RCF is defined',
                                     optional=True )
   inboundIpv6Uni = Str( optional=True, help='Inbound IPv6 Unicast' )
   inboundIpv6UniDefined = Bool( help='Inbound IPv6 Unicast RCF is defined',
                                    optional=True )
   outboundIpv6Uni = Str( optional=True, help='Outbound IPv6 Unicast' )
   outboundIpv6UniDefined = Bool( help='Outbound IPv6 Unicast RCF is defined',
                                     optional=True )
   # SrTe attributes unused? Only v4|v6 unicast inbound|outbound populated in
   # BgpPeerCommandCallback.tin
   inboundIpv4SrTe = Str( optional=True, help='Inbound IPv4 SR-TE' )
   inboundIpv4SrTeDefined = Bool( help='Inbound IPv4 SR-TE RCF is defined',
                                     optional=True )
   inboundIpv6SrTe = Str( optional=True, help='Inbound IPv6 SR-TE' )
   inboundIpv6SrTeDefined = Bool( help='Inbound IPv6 SR-TE RCF is defined',
                                     optional=True )
   outboundIpv4SrTe = Str( optional=True, help='Outbound IPv4 SR-TE' )
   outboundIpv4SrTeDefined = Bool( help='Outbound IPv4 SR-TE RCF is defined',
                                  optional=True )
   outboundIpv6SrTe = Str( optional=True, help='Outbound IPv6 SR-TE' )
   outboundIpv6SrTeDefined = Bool( help='Outbound IPv6 SR-TE RCF is defined',
                                  optional=True )

class RouteMapFilterInfo( Model ):
   ''' This model got split from the previous "RouteFilterInfo" model to
   provide specific help.
   Keeping the filterType attribute to avoid incompatible changes.
   '''
   filterType = Enum( help='Type of route filter', values=( 'RouteMap', ) )
   inboundIpv4Uni = Str( optional=True,
         help='Inbound IPv4 Unicast (overridden if RCF is configured)' )
   inboundIpv4UniDefined = Bool( optional=True,
         help='Inbound IPv4 Unicast route-map is defined' )
   outboundIpv4Uni = Str( optional=True,
         help='Outbound IPv4 Unicast (overridden if RCF is configured)' )
   outboundIpv4UniDefined = Bool( optional=True,
         help='Outbound IPv4 Unicast route-map is defined' )
   inboundIpv6Uni = Str( optional=True,
         help='Inbound IPv6 Unicast (overridden if RCF is configured)' )
   inboundIpv6UniDefined = Bool( optional=True,
         help='Inbound IPv6 Unicast route-map is defined' )
   outboundIpv6Uni = Str( optional=True,
         help='Outbound IPv6 Unicast (overridden if RCF is configured)' )
   outboundIpv6UniDefined = Bool( optional=True,
         help='Outbound IPv6 Unicast route-map is defined' )
   inboundIpv4SrTe = Str( optional=True, help='Inbound IPv4 SR-TE' )
   inboundIpv4SrTeDefined = Bool( help='Inbound IPv4 SR-TE route-map is defined',
                                  optional=True )
   inboundIpv6SrTe = Str( optional=True, help='Inbound IPv6 SR-TE' )
   inboundIpv6SrTeDefined = Bool( help='Inbound IPv6 SR-TE route-map is defined',
                                  optional=True )
   outboundIpv4SrTe = Str( optional=True, help='Outbound IPv4 SR-TE' )
   outboundIpv4SrTeDefined = Bool( help='Outbound IPv4 SR-TE route-map is defined',
                                  optional=True )
   outboundIpv6SrTe = Str( optional=True, help='Outbound IPv6 SR-TE' )
   outboundIpv6SrTeDefined = Bool( help='Outbound IPv6 SR-TE route-map is defined',
                                  optional=True )

class RfdPolicyInfo( Model ):
   ipv4Unicast = Str( help="Rfd policy for IPv4 unicast", optional=True )
   ipv6Unicast = Str( help="Rfd policy for IPv6 unicast", optional=True )

class PeerDropStats( Model ):
   inDropAsloop = Int( optional=True, help='In packets dropped due to Asloop' )
   inDropAsSetConfedSetV4Uni = Int( optional=True,
                               help='In IPv4 Unicast packets dropped due to AS_SET'
                                    ' or AS_CONFED_SET in AS path attribute' )
   inDropAsSetConfedSetV6Uni = Int( optional=True,
                               help='In IPv6 Unicast packets dropped due to AS_SET'
                                    ' or AS_CONFED_SET in AS path attribute' )
   inDropClusterIdLoop = Int( optional=True,
         help='In packets dropped due to Cluster ID loop' )
   inDropEnforceFirstAs = Int( optional=True,
         help='In packets dropped due to Enforce First As' )
   inDropMalformedMpbgp = Int( optional=True,
         help='In packets dropped due to Malformed MpBgp' )
   inDropOrigId = Int( optional=True,
         help='In packets dropped due to Orig Id' )
   if toggleDsfPhase1Enabled():
      inDropVOQEndpointSystemId = Int( optional=True,
         help='In packets dropped due to VOQ endpoint in '
              'tunnel encapsulation attribute matches local system ID' )
   inDropNhLocal = Int( optional=True,
         help='In packets dropped due to next hop local' )
   inDropNhInvalid = Int( optional=True,
         help='In packets dropped due to next hop invalid' )
   inDropNhAfV6 = Int( optional=True,
         help='In packets dropped due to next hop AF V6' )
   outDropV4LocalAddr = Int( optional=True,
         help='Out packets dropped due to V4 Local Address' )
   outDropV6LocalAddr = Int( optional=True,
         help='Out packets dropped due to V6 Local Address' )
   prefixLuDroppedV4 = Int( optional=True,
         help='Packets dropped due to V4 Prefix LU ' )
   prefixLuDroppedV6 = Int( optional=True,
         help='Packets dropped due to V6 Prefix LU ' )
   prefixVpnIpv4DroppedImportMatchFailure = Int( optional=True,
         help='VPN-IPv4 NLRIs dropped due to route import match failure' )
   prefixVpnIpv4DroppedMaxRouteLimitViolated = Int( optional=True,
         help='VPN-IPv4 NLRIs dropped due to maximum route limit violation' )
   prefixVpnIpv6DroppedImportMatchFailure = Int( optional=True,
         help='VPN-IPv6 NLRIs dropped due to route import match failure' )
   prefixVpnIpv6DroppedMaxRouteLimitViolated = Int( optional=True,
         help='VPN-IPv6 NLRIs dropped due to maximum route limit violation' )
   prefixEvpnDroppedImportMatchFailure = Int( optional=True,
         help='L2VPN EVPN NLRIs dropped due to route import match failure' )
   prefixDroppedMartianV4 = Int( optional=True,
         help='IPv4 NLRIs dropped due to Martian prefix' )
   prefixDroppedMaxRouteLimitViolatedV4 = Int( optional=True,
         help='IPv4 NLRIs dropped due to maximum route limit violation' )
   prefixDroppedMartianV6 = Int( optional=True,
         help='IPv6 NLRIs dropped due to Martian prefix' )
   prefixDroppedMaxRouteLimitViolatedV6 = Int( optional=True,
         help='IPv6 NLRIs dropped due to maximum route limit violation' )
   prefixLuDroppedMartianV4 = Int( optional=True,
         help='IPv4 LU NLRIs dropped due to Martian prefix' )
   prefixLuDroppedMaxRouteLimitViolatedV4 = Int( optional=True,
         help='IPv4 LU NLRIs dropped due to maximum route limit violation' )
   prefixLuDroppedMartianV6 = Int( optional=True,
         help='IPv6 LU NLRIs dropped due to Martian prefix' )
   prefixLuDroppedMaxRouteLimitViolatedV6 = Int( optional=True,
         help='IPv6 LU NLRIs dropped due to maximum route limit violation' )
   prefixEvpnDroppedUnsupportedRouteType = Int( optional=True,
         help='L2VPN EVPN NLRIs dropped due to unsupported route type' )
   prefixEvpnDroppedMaxRouteLimitViolated = Int( optional=True,
         help='L2VPN EVPN NLRIs dropped due to maximum route limit violation' )
   prefixBgpLsDroppedReceptionUnsupported = Int( optional=True,
         help='Link-state NLRIs dropped due to unsupported route type' )
   prefixRtMembershipDroppedLocalAsReject = Int( optional=True,
         help='RT Membership NLRIs dropped due to local origin ASN '
              'received from external peer' )
   prefixRtMembershipDroppedMaxRouteLimitViolated = Int( optional=True,
         help='RT Membership NLRIs dropped due to maximum route limit violation' )

class TcpAoInfo( Model ):
   sharedSecretProfile = Str( help='TCP-AO shared secret profile' )
   sharedSecretProfileState = Enum( values=( 'valid',
                                             'notExist',
                                             'noValidSecrets' ),
                                    help="State of the Shared Secret Profile" )
   macAlgorithm = Enum( values=( 'hmacSha196',
                                 'aes128Cmac96',
                                 'hmacSha256',
                                 'none' ),
                        help="MAC computation algorithm" )
   currentKeyId = Int( optional=True, help="Key ID of the current TCP-AO key" )
   receiveNextKeyId = Int( optional=True,
                           help="Key ID of the receive next TCP-AO key" )
   activeKeyIds = List( optional=True,
                        valueType=int, help="All active key IDs" )

class BgpPeerInUpdateErrors( Model ):

   def getMioAttrId( self ):
      # We cannot depend on gated. So we must hardcode the value here to match
      # the one in gated/gated-ctk/src/mioagt/bgp_api.h. In fact having "ribd"
      # specific element in BGP Cli Model is violation of modularity. Make sure
      # that the value here matches MIO_DGET_BGP_PEER_IN_UPDATE_ATTR_ERRORS
      return 3

   inUpdErrWithdraw = Int( optional=True, help='in updates treated as withdraw' )
   inUpdErrIgnore = Int( optional=True, help='in updates ignored attributes' )
   inUpdErrDisableAfiSafi = Int( optional=True,
         help='in updates disabling AFI/SAFI' )
   disabledAfiSafi = Str( optional=True, help='disabled AFI/SAFI' )
   withdrawAttr = Str( optional=True, help='last attribute resulting in withdraw' )
   if withdrawAttr is None:
      withdrawAttr = ''
   ignoreAttr = Str( optional=True, help='last attribute resulting in ignore' )
   if ignoreAttr is None:
      ignoreAttr = ''
   disableAfiSafiAttr = Str( optional=True,
         help='last attribute resulting in AFI/SAFI disable' )
   if disableAfiSafiAttr is None:
      disableAfiSafiAttr = ''
   lastUpdErrTime = Int( optional=True, help='last time of Update inbound error' )

   def render( self ):
      print( "  Inbound updates with attribute errors:" )
      print( "    Resulting in removal of all paths in update "
             "(treat-as-withdraw): %d" % ( self.inUpdErrWithdraw ) )
      print( "    Resulting in AFI/SAFI disable: %d"
             % ( self.inUpdErrDisableAfiSafi ) )
      print( "    Resulting in attribute ignore: %d" % ( self.inUpdErrIgnore ) )
      if self.disabledAfiSafi is not None:
         print( "    Disabled AFI/SAFI: %s" % ( self.disabledAfiSafi ) )
      if self.withdrawAttr is not None:
         print( "    Last treat-as-withdraw attribute error: %s"
                % ( self.withdrawAttr ) )
      if self.ignoreAttr is not None:
         print( "    Last ignored attribute error: %s" % ( self.ignoreAttr ) )
      if self.disableAfiSafiAttr is not None:
         print( "    Last disable AFI/SAFI attribute error: %s"
                % ( self.disableAfiSafiAttr ) )
      lastRcvdErrStr = "    Last update with error received at: "
      if self.lastUpdErrTime:
         lastRcvdErrStr += "%8s" % formatTimeInterval( self.lastUpdErrTime )
         print( lastRcvdErrStr )

class SendingAddPaths( Model ):
   sendCapabilityAdvertised = Bool( help="Advertised capability to send "
                                         "additional paths to peer" )
   receiveCapabilityReceived = Bool( help="Received capability to receive "
                                          "additional paths from peer" )
   enabled = Bool( help="Sending of additional paths to peer is enabled" )

class ReceivingAddPaths( Model ):
   receiveCapabilityAdvertised = Bool( help="Advertised capability to receive "
                                            "additional paths from peer" )
   sendCapabilityReceived = Bool( help="Received capability to send "
                                       "additional paths from peer" )
   enabled = Bool( help="Receipt of additional paths from peer is enabled" )

class AddPathsCapability( Model ):
   sendingAddPaths = Submodel( valueType=SendingAddPaths,
                               help="Negotiation state of "
                                    "sending additional paths to peer",
                               optional=True )
   receivingAddPaths = Submodel( valueType=ReceivingAddPaths,
                                 help="Negotiation state of receiving "
                                      "additional paths from peer",
                                 optional=True )

# A breakdown of each additional paths capability based on
# whether an add-paths capability is enabled and if not, who is not capable.
# Sending: Capable of sending add-paths to the peer and
#          whether the peer is capable of receiving add-paths.
# Receiving: Capable of receiving add-paths from the peer and
#            whether the peer is capable of sending add-paths.
AddPathsCapabilities = generateAfiSafiModel( "AddPathsCapabilities", Submodel,
      "%s additional paths capability", valueType=AddPathsCapability, optional=True )

class Capability( Model ):
   advertised = Bool( help='Capability advertised to peer' )
   received = Bool( help='Capability received from peer' )
   enabled = Bool( help='Capability is enabled' )

# The presence of an AfiSafi denotes that
# the device can preserve forwarding state for the AfiSafi and also if it
# has been preserved for the most recent restart.
ForwardingStates = generateAfiSafiModel( "ForwardingStates", Bool,
      "%s forwarding state is preserved", optional=True )

class GrCapability( Model ):
   restartTime = Float( help='Restart Time' )
   bgpRestarted = Bool( help='BGP restarted' )
   gracefulNotificationSupported = Bool( help='Graceful notification is supported' )
   forwardingStatesPreserved = Submodel( valueType=ForwardingStates,
                                         help='Graceful restart capabilities '
                                              'forwarding states are preserved',
                                         optional=True )

# Model for the separation between received and advertised GR caps
class GrCapabilities( Model ):
   received = Submodel( valueType=GrCapability,
                        help="Graceful Restart capability received from peer",
                        optional=True )
   advertised = Submodel( valueType=GrCapability,
                          help="Graceful Restart capability advertised to peer",
                          optional=True )
   ipv4VpnCap = Submodel( valueType=Capability,
                          help="Graceful Restart IPv4 VPN capability",
                          optional=True )

class LlgrCapabilityInfo( Model ):
   forwardingStatePreserved = Bool( help='Forwarding State is preserved',
                                    optional=True )
   staleTime = Int( help='Stale time (Seconds)' )

LgrCapabilityBase_ = generateAfiSafiModel( "LgrCapabilityBase_", Submodel,
      "%s capability", valueType=LlgrCapabilityInfo, optional=True )

class LlgrCapability( LgrCapabilityBase_ ):
   helperOnly = Bool( help='Helper only', optional=True )

class LlgrCapabilities( Model ):
   received = Submodel( valueType=LlgrCapability,
                        help='Long Lived Graceful Restart received capabilities',
                        optional=True )
   advertised = Submodel( valueType=LlgrCapability,
                          help='Long Lived Graceful Restart advertised capabilities',
                          optional=True )

# Representing capabilities that allow advertising IPv4
# with an IPv6 Next Hop address
class ExtendedNexthopCapabilities( Model ):
   ipv4UnicastOverIpv6 = Submodel( valueType=Capability,
                                   help="IPv4 Unicast over an "
                                        "IPv6 backbone capability",
                                   optional=True )
   ipv4LocalNlriOverIpv6 = Bool( help="Local IPv4 NLRIs advertised over an "
                                      "IPv6 backbone",
                                 optional=True )
   v4MplsVpnOverIpv6 = Submodel( valueType=Capability,
                                 help="IPv4 MPLS VPN over an "
                                      "IPv6 backbone capability",
                                 optional=True )

# Multiple Labels Capability implemented in ArBgp
class MultipleLabelsCapability( Model ):
   advertised = Int( help="Advertised count",
                     optional=True )
   received = Int( help="Received count",
                   optional=True )

class MultipleLabelsCapabilities( Model ):
   ipv4MplsLabels = Submodel( valueType=MultipleLabelsCapability,
                             help="IPv4 MPLS Label",
                             optional=True )
   ipv6MplsLabels = Submodel( valueType=MultipleLabelsCapability,
                             help="IPv6 MPLS Label",
                             optional=True )

MultiprotocolCapabilities = generateAfiSafiModel( "MultiprotocolCapabilities",
      Submodel, "%s capability", valueType=Capability, optional=True )

class EndOfRibInfo( Model ):
   status = Enum( values=( 'pending', 'timeout', 'received' ),
                  help='Status of the EOR' )
   eorReceivedTime = Float( optional=True, help='UTC time at which EOR '
                            'was received' )
   stalePathsDeleted = Int( optional=True, help='Number of stale '
                            'paths deleted after graceful restart' )
   pathsReceivedBeforeEor = Int( optional=True, help='Number of paths '
                                 'received before EOR' )

EndOfRib = generateAfiSafiModel( 'EndOfRib', Submodel, '%s EOR Information',
                                         valueType=EndOfRibInfo, optional=True )

class EndOfRibs( Model ):
   incoming = Submodel( valueType=EndOfRib,
                        help="Incoming EOR Information",
                        optional=True )

class NeighborCapabilities( Model ):
   negotiationDisabled = Bool( help='Neighbor capability negotiation is disabled',
                               optional=True )
   # Set to True when the peer has advertised that they do not
   # support a required capability that is supported by this speaker.
   unsupportedCapReceived = Bool( help='Neighbor does not support all '
                                       'required capabilities',
                                  optional=True )
   grCaps = Submodel( valueType=GrCapabilities,
                      help='Graceful Restart capabilities',
                      optional=True )
   longlivedGrCaps = Submodel( valueType=LlgrCapabilities,
                               help='Long Lived Graceful Restart capabilities',
                               optional=True )
   addPathsCaps = Submodel( valueType=AddPathsCapabilities,
                            help='Additional paths capabilities',
                            optional=True )
   extendedNextHopCaps = Submodel( valueType=ExtendedNexthopCapabilities,
                                   help="Extended Next-Hop capabilities",
                                   optional=True )
   multiprotocolCaps = Submodel( valueType=MultiprotocolCapabilities,
                                 help="Multiprotocol capabilities",
                                 optional=True )
   # Capabilities that are not per afi safi
   fourOctetAsnCap = Submodel( valueType=Capability,
                               help="Four Octet ASN capability",
                               optional=True )
   routeRefreshCap = Submodel( valueType=Capability,
                               help="Route Refresh capability",
                               optional=True )
   enhancedRouteRefreshCap = Submodel( valueType=Capability,
                                       help="Enhanced Route Refresh capability",
                                       optional=True )
   sendEorMessages = Submodel( valueType=Capability,
                               help="Send End-of-RIB messages",
                               optional=True )
   dynamicCap = Submodel( valueType=Capability,
                          help="Dynamic capabilities",
                          optional=True )
   multipleLabelsCap = Submodel( valueType=MultipleLabelsCapabilities,
                                 help="Multiple labels capability",
                                 optional=True )

class MissingPolicyDefaultDeny( Model ):
   importActive = Bool(
         help='Missing policy/default deny import action is active',
         optional=True )
   exportActive = Bool(
         help='Missing policy/default deny export action is active',
         optional=True )

missingPolicyDenyTypes = ( 'permit',
                           'noRouteMapConfigured',
                           'namedRouteMapMissing',
                           'routeMapMisconfiguredByOtherAf',
                           'inboundRouteMapMisconfigured',
                           'outboundRouteMapMisconfigured',
                           'misconfiguredSubRouteMap',
                           'misconfiguredPrefixList',
                           'unspecified' )

class MissingPolicyState( Model ):
   action = Enum( values=( 'permit', 'deny' ),
                  help='Missing policy action' )
   status = Enum( values=missingPolicyDenyTypes,
                  help="Missing policy status" )

MissingPolicy = generateAfiSafiModel( "MissingPolicy", Submodel,
      "%s missing policy action", valueType=MissingPolicyState, optional=True )

class MissingPolicyTypes( Model ):
   importPolicy = Submodel( valueType=MissingPolicy,
                        help="AFI/SAFI export missing policy actions",
                        optional=True )
   exportPolicy = Submodel( valueType=MissingPolicy,
                        help="AFI/SAFI import missing policy actions",
                        optional=True )

class RtMembershipNlriHandlingGroup( Model ):
   min = Int( help='Minimum NLRI length' )
   max = Int( help='Maximum NLRI length' )
   # `as` is a Python reserved word, so setting it with setattr below
   # as = Int( optional=True,
   #   help='Origin AS: local or other (default: any)' )

# `as` is a Python reserved word, so cannot do it within the class
setattr( RtMembershipNlriHandlingGroup, 'as',
      Int( optional=True, help='Origin AS: local or other (default: any)' ) )

class RtMembershipNlriInputHandling( Model ):
   key = Submodel( valueType=RtMembershipNlriHandlingGroup,
                   help='Conditions for NLRI to be included in this handling group' )
   action = Enum( values=( 'accept', 'translate' ), help='Action taken on NLRI' )
   to = Int( optional=True, help='New length of NLRI after translation' )

class RtMembershipNlriOutputHandling( Model ):
   key = Submodel( valueType=RtMembershipNlriHandlingGroup,
                   help='Conditions for NLRI to be included in this handling group' )
   action = Enum( values=( 'advertise', 'translate', 'drop' ),
                  help='Action taken on NLRI' )
   to = Int( optional=True, help='New length of NLRI after translation' )

class PerAfiSafiInfo( Model ):
   localNexthop = Str( optional=True,
                       help='Local next hop address for next hop self' )
   maxTotalRoutes = Int( optional=True,
                  help='Maximum number of received routes (0 means unlimited)' )
   maxTotalAcceptedRoutes = Int( optional=True,
                  help='Maximum number of accepted routes (0 means unlimited)' )
   maxTotalAdvertisedRoutes = Int( optional=True,
                  help='Maximum number of advertised routes (0 means unlimited)' )
   totalRoutesWarnLimit = Int( optional=True, help='Number of received routes after '
                     'which a warning is issued (0 means never warn)' )
   totalAcceptedRoutesWarnLimit = Int( optional=True, help='Number of accepted '
                     'routes after which a warning is issued (0 means never warn)' )
   totalAdvertisedRoutesWarnLimit = Int( optional=True, help='Number of advertised '
                     'routes after which a warning is issued (0 means never warn)' )
   apSendMode = Str( optional=True,
                     help='Additional-path send mode' )
   apSendLimit = Int( optional=True,
                      help='Additional-path send limit' )
   apSendPrefixList = Str( optional=True,
                           help='Additional-path send prefix-list' )
   nlriInputLengths = Dict( optional=True, keyType=str,
                            valueType=RtMembershipNlriInputHandling,
                            help='A dictionary of NLRI input handling groups '
                                 'for RT Membership AFI/SAFI' )
   nlriOutputLengths = Dict( optional=True, keyType=str,
                             valueType=RtMembershipNlriOutputHandling,
                             help='A dictionary of NLRI output handling groups '
                                  'for RT Membership AFI/SAFI' )

class LldpNeighbor( Model ):
   device = Str( help='LLDP Neighbor system name' )
   description = Str( optional=True, help='LLDP Neighbor system description' )

class CommunityTypesAdv( Model ):
   standard = Bool( help="Standard community advertised" )
   extended = Bool( help="Extended community advertised" )
   large = Bool( help="Large community advertised" )

class GrHelperStalePolicy( Model ):
   staleRmName = Str( optional=True, help='AFI/SAFI stale policy route map' )
   grOptional = Bool( optional=True,
                      help='Graceful Restart negotiation is optional' )

class GrHelperStalePolicies( Model ):
   ipv4Unicast = Submodel( valueType=GrHelperStalePolicy,
                           help="IPv4 Unicast stale policy information",
                           optional=True )
   ipv6Unicast = Submodel( valueType=GrHelperStalePolicy,
                           help="IPv6 Unicast stale policy information",
                           optional=True )
   ipv4MplsLabels = Submodel( valueType=GrHelperStalePolicy,
                              help="IPv4 Labeled Unicast stale policy information",
                              optional=True )
   ipv6MplsLabels = Submodel( valueType=GrHelperStalePolicy,
                              help="IPv6 Labeled Unicast stale policy information",
                              optional=True )
   dps = Submodel( valueType=GrHelperStalePolicy,
                   help="DPS stale policy information",
                   optional=True )
   ipv4SrTe = Submodel( valueType=GrHelperStalePolicy,
                        help="IPv4 SrTe stale policy information",
                        optional=True )
   l2VpnEvpn = Submodel( valueType=GrHelperStalePolicy,
                         help="EVPN stale policy information",
                         optional=True )
   linkState = Submodel( valueType=GrHelperStalePolicy,
                         help="Link State stale policy information",
                         optional=True )

# A link-state EPE segment (see AID11677).
class LinkStatePeerSegment( Model ):
   sid = Int( help='The SID value' )

class BgpPeer( Model ):
   __revision__ = 2

   def degrade( self, dictRepr, revision ):
      if revision < 2:
         # See BgpCommon/CliPlugin/BgpCliModels.py:_degradePeersDict for details
         _degradeToIpGenAddress( dictRepr[ 'peerAddress' ] )
      return dictRepr

   peerAddress = Str( help='Peer address' )
   peerTcpInfo = Submodel( valueType=BgpPeerTcpInfo, optional=True,
                    help='Peer TCP information' )
   peerInUpdateErrors = Submodel( valueType=BgpPeerInUpdateErrors, optional=True,
                    help='Peer inbound update errors' )
   asn = Str( help='Autonomous system number' )
   routerId = Ip4Address( help='BGP router identity' )
   updateSource = IpGenericAddress( optional=True, help='Local TCP address' )
   localAddressV4 = IpGenericAddress( optional=True, help='Local IPv4 address' )
   localAddressV6 = IpGenericAddress( optional=True, help='Local IPv6 address' )
   localAddressPort = Int( optional=True, help='Local address port' )
   autoLocalAddress = Str( optional=True, help='Auto local address' )
   localAsn = Str( help='Local autonomous system number' )
   localRouterId = Ip4Address( help='Local router ID' )
   linkType = Str( help='Link type' )
   lldpNeighbors = List( optional=True, valueType=LldpNeighbor,
                         help='LLDP Neighbors list' )
   maxTotalRoutes = Int( optional=True,
                  help='Maximum number of received routes (0 means unlimited)' )
   maxTotalAcceptedRoutes = Int( optional=True,
                  help='Maximum number of accepted routes (0 means unlimited)' )
   totalRoutesWarnLimit = Int( optional=True, help='Number of received routes after '
                           'which a warning is issued (0 means never warn)' )
   totalAcceptedRoutesWarnLimit = Int( optional=True, help='Number of accepted '
                     'routes after which a warning is issued (0 means never warn)' )

   version = Int( help='BGP version' )
   negotiatedVersion = Int( help='Negotiated BGP version' )
   updateGroupIndex = Int( help='Update group index' )
   ttl = Int( help='TTL value for peer connections' )
   maxTtlHops = Int( help='Maximum hops allowed for BGP neighbor' )
   ttlConfig = Enum( values=( 'gtsm',
                              'ebgpMultiHop' ),
                     help='TTL max hops configuration mode',
                     optional=True )
   holdTime = Int( optional=True, help='Hold time' )
   holdTimeLeft = Int( optional=True, help='Hold time left' )
   confHoldTime = Int( optional=True, help='Configured hold time' )
   # Attributes with (multi-agent mode) in the help description, are attributes
   # specific to ArBgp. These may be misspelled as they were (re)named in the C++
   # streaming implementation. We include these duplicate attribute names as this
   # ensures compatibility for early ArBgp adopters without the need to
   # increment the model version.
   configHoldTime = Int( optional=True, help='Configured hold time '
                                             '(multi-agent mode)' )
   minHoldTime = Int( optional=True, help='Minimum hold time in seconds' )
   configMinHoldTime = Int( optional=True, help='Configured minimum hold time in '
                                                'seconds' )
   sendFailureHoldTime = Int( optional=True,
                              help='Send failure hold time in seconds' )
   connectFailed = Int( optional=True, help='Failed connection attempts' )
   connectInterval = Int( optional=True, help='Connection interval' )
   connectTimeLeft = Int( optional=True, help='Connection time left' )
   advertisedRestartTime = Int( optional=True, help='Advertised restart time' )
   restart = Int( optional=True, help='Received restart time' )
   configRestartTime = Int( optional=True, help='Configured restart time '
                                                '(multi-agent mode)' )
   restartTimeLeft = Int( optional=True, help='Restart time left' )
   grRestartTimeLeft = Int( optional=True, help='Restart time left '
                                                '(multi-agent mode)' )
   llgrTimers = Dict( optional=True,
                      valueType=int,
                      help='Long lived graceful restart '
                           'stale path timers per received Afi/Safi' )
   grStalePolicies = Submodel( optional=True, valueType=GrHelperStalePolicies,
                               help="Graceful Restart stale policy information "
                                    "per Afi/Safi" )
   endOfRibTimeLeft = Int( optional=True, help='End of rib time left' )
   configIdleHoldTime = Int( optional=True, help='Configured idle hold time' )
   idleHoldTimeLeft = Int( optional=True, help='Idle hold time left' )
   stalePathRemovalTimer = Dict( optional=True,
                                 valueType=int,
                                 help='Inbound enhanced route refresh '
                                      'stale path removal timer per negotiated '
                                      'Afi/Safi' )
   enRRefreshstalePathRemoval = Bool( optional=True,
      help="Enhanced route refresh stale path removal enabled" )
   outboundEnRRefreshEnabled = Bool( optional=True,
      help="Outbound enhanced route refresh enabled" )
   state = Str( optional=True, help='BGP state' )
   idleReason = Str( optional=True, help='Idle reason if session state is idle' )
   lastState = Str( optional=True, help='Last BGP state' )
   lastEvent = Str( optional=True, help='Last BGP event' )
   lastErrorCode = Str( optional=True, help='Last send error code' )
   lastErrorSubcode = Str( optional=True, help='Last send error subcode' )
   lastErrorTimeInfo = Submodel( optional=True, valueType=ErrorTimeInfo,
         help='Last send error time information' )
   lastErrorData = Str( optional=True, help='Last send error data' )
   lastErrorRcvdCode = Str( optional=True, help='Last receive error code' )
   lastErrorRcvdSubcode = Str( optional=True, help='Last receive error subcode' )
   lastErrorRcvdTimeInfo = Submodel( optional=True,
         valueType=ErrorTimeInfo,
         help='Last receive error time information' )
   lastErrorRcvdData = Str( optional=True, help='Last receive error data' )
   lastSocketOutErrorMsg = Str( optional=True,
         help='Last socket out error message' )
   lastSocketOutErrorTimeInfo = Submodel( optional=True,
         valueType=ErrorTimeInfo,
         help='Last socket out error time information' )
   lastSocketInErrorMsg = Str( optional=True,
         help='Last socket in error message' )
   lastSocketInErrorTimeInfo = Submodel( optional=True,
         valueType=ErrorTimeInfo,
         help='Last socket in error time information' )
   if toggleDsfPhase1Enabled():
      voqClusterPeer = Bool( optional=True,
                             help='The peer is part of a configured VOQ cluster' )
   advertisedCommunities = Submodel( optional=True,
         valueType=CommunityTypesAdv,
         help='Types of communities advertised' )
   bgpPeerHiscaps = Int( optional=True, help='BGP capabilities received from peer' )
   bgpPeerCaps = Int( optional=True, help='BGP capabilities sent by us' )
   bgpPeerOptions = Int( optional=True, help='BGP peer options' )
   neighborCapabilities = Submodel( valueType=NeighborCapabilities,
                                    help='BGP neighbor capabilities',
                                    optional=True )
   bgpPeerOptions2 = Int( optional=True, help='BGP peer options 2' )
   bgpPeerOptions3 = Int( optional=True, help='BGP peer options 3' )
   bgpPeerFlags = List( valueType=int, optional=True,
                        help='BGP peer flag list' )
   miscFlags = Int( optional=True, help='BGP miscellaneous flags' )

   connectTimerActive = Bool( help='Connect timer is active', optional=True )
   idleRestartTimerActive = Bool( help='Idle restart timer active', optional=True )
   routeReflectorClient = Bool( help='Neighbor is a route reflector client',
                                optional=True )
   clientReflectionMode = Enum( values=( 'clientToClientDisabled',
                                         'meshed' ),
                                help='Route reflector mode',
                                optional=True )
   usesGlobalClusterId = Bool( help="This peer is using the global cluster ID.",
         optional=True )
   clusterId = Ip4Address( help="Cluster ID in use by this peer",
         optional=True )
   gracefulRestartTimerActive = Bool( help='Graceful Restart timer is active',
                                      optional=True )
   eorTimerActive = Bool( help='End-of-Rib timer is active', optional=True )
   invalidNextHopDirectEbgp = Bool( help='Nexthop invalid for single hop EBGP',
                                    optional=True )
   missingPolicyDefaultDeny = Submodel( valueType=MissingPolicyDefaultDeny,
                                        help="Missing policy/default deny status",
                                        optional=True )
   bfdSessionState = Enum( values=( 'init',
                                    'up',
                                    'down',
                                    'adminDown',
                                    'unknown' ),
                           help='BFD session state when BFD is enabled',
                           optional=True )
   passiveConnEnabled = Bool( help='Passive TCP connection mode is enabled',
                              optional=True )

   removeOrReplacePrivateAsEgress = Enum( values=( 'removePrivateAsEgress',
                                                   'removePrivateAsEgressAll',
                                                   'removePrivateAsEgressReplace' ),
                                          help=removeOrReplacePrivateAsEgressHelp,
                                          optional=True )
   removeOrReplacePrivateAsIngress = Enum( values=(
                                                'removePrivateAsIngress',
                                                'removePrivateAsIngressReplace' ),
                                           help=removeOrReplacePrivateAsIngressHelp,
                                           optional=True )
   md5AuthEnabled = Bool( help='MD5 authentication is enabled', optional=True )
   tcpAoInfo = Submodel( valueType=TcpAoInfo, optional=True,
                    help='TCP-AO information' )

   missingPolicy = Submodel( valueType=MissingPolicyTypes,
                               help='Import/Export missing policies per AFI/SAFI',
                               optional=True )
   v4gateway = IpGenericAddress( optional=True, help='IPv4 gateway address' )
   v6gateway = IpGenericAddress( optional=True, help='IPv6 gateway address' )
   remoteAddressV4 = IpGenericAddress( optional=True, help='Remote IPv4 address' )
   remoteAddressV6 = IpGenericAddress( optional=True, help='Remote IPv6 address' )
   remoteAddrPort = Int( optional=True, help='Remote address port' )
   sentUpdates = Int( optional=True, help='Sent updates' )
   receivedUpdates = Int( optional=True, help='Received updates' )
   sentMessages = Int( optional=True, help='Sent messages' )
   receivedMessages = Int( optional=True, help='Received messages' )
   establishedTransitions = Int( optional=True, help='Established transitions' )
   establishedTime = Int( optional=True, help='Established time' )
   lastSent = Int( optional=True, help='Last sent' )
   lastRcvd = Int( optional=True, help='Last received' )
   keepaliveTime = Int( optional=True, help='Keepalive time' )
   keepaliveTimeLeft = Int( optional=True, help='Keepalive time left' )
   configuredKeepaliveTime = Int( optional=True, help='Configured keepalive time' )
   configKeepaliveTime = Int( optional=True,
                              help='Configured keepalive time '
                                   '(multi-agent mode)' )
   inMessageStats = Submodel( optional=True,
         valueType=MessageStatInfo,
         help='Input message statistics' )
   outMessageStats = Submodel( optional=True,
         valueType=MessageStatInfo,
         help='Output message statistics' )
   localPort = Int( optional=True, help='Local port' )
   remotePort = Int( optional=True, help='Remote port' )
   confRemotePort = Int( optional=True, help='Configured remote port' )
   description = Str( optional=True, help='Description' )
   routeMapInbound = Str( optional=True,
         help='Route map inbound (overridden if RCF is configured)' )
   routeMapOutbound = Str( optional=True,
         help='Route map outbound (overridden if RCF is configured)' )
   routeMapInfo = Submodel( optional=True,
         valueType=RouteMapFilterInfo,
         help='Neighbor route map information' )
   routeMapInboundDefined = Bool( help='Inbound route-map is defined',
                                  optional=True )
   routeMapOutboundDefined = Bool( help='Outbound route-map is defined',
                                   optional=True )
   rcfInbound = Str( optional=True,
         help='Rcf inbound' )
   rcfOutbound = Str( optional=True,
         help='Rcf outbound' )
   rfdPolicyInfo = Submodel( valueType=RfdPolicyInfo, help='Neighbor route flap '
                        'damping information for AFI/SAFI', optional=True )
   prefixListInfo = Submodel( optional=True,
         valueType=RouteFilterInfo,
         help='Neighbor prefix list information' )
   rcfInfo = Submodel( optional=True,
         valueType=RouteFilterInfo,
         help='Neighbor RCF information (supersedes route map)' )
   bfdState = Int( optional=True, help='BFD state' )
   bfdDampingTime = Int( help='BFD current flap damping time',
         optional=True )
   bfdDampingMaxTime = Int( help='BFD flap damping max time',
         optional=True )
   bfdDampingTimeLeft = Float( help='BFD flap damping time left',
         optional=True )
   bfdMinTx = Int( help='BFD transmit rate in milliseconds', optional=True )
   bfdMinRx = Int( help='BFD receive rate in milliseconds', optional=True )
   bfdMult = Int( help='BFD multiplier value', optional=True )
   vrf = Str( help='Vrf' )
   peerGroup = Str( optional=True, help='Peer group name' )
   peerGroupName = Str( optional=True,
                        help='Peer group name '
                             '(multi-agent mode)' )
   ancestorPeerGroups = List( optional=True, valueType=str,
                              help='Ancestor peer group list' )
   peerFilter = Str( optional=True,
                     help='Peer filter name' )
   prefixesSent = Int( optional=True, help='IPv4 unicast prefixes sent' )
   prefixesReceived = Int( optional=True, help='IPv4 unicast prefixes received' )
   v4BestPaths = Int( optional=True,
         help='IPv4 best paths received' )
   v4BestEcmpPaths = Int( optional=True,
         help='IPv4 best ECMP paths received' )
   v6PrefixesSent = Int( optional=True, help='IPv6 Unicast prefixes sent' )
   v6PrefixesReceived = Int( optional=True, help='IPv6 Unicast prefixes received' )
   v6BestPaths = Int( optional=True,
         help='IPv6 best paths received' )
   v6BestEcmpPaths = Int( optional=True,
         help='IPv6 best ECMP paths received' )
   v6SixPePrefixesSent = Int( optional=True,
         help='IPv6 provider edge (6PE) prefixes sent' )
   v6SixPePrefixesReceived = Int( optional=True,
         help='IPv6 provider edge (6PE) prefixes received' )
   v6SixPeBestPaths = Int( optional=True,
         help='IPv6 provider edge (6PE) best paths received' )
   v6SixPeBestEcmpPaths = Int( optional=True,
         help='IPv6 provider edge (6PE) best ECMP paths received' )
   v4SrTePrefixesSent = Int( optional=True,
         help='IPv4 segment routing traffic engineering policies sent' )
   v4SrTePrefixesReceived = Int( optional=True,
         help='IPv4 segment routing traffic engineering policies received' )
   v4SrTeBestPaths = Int( optional=True,
         help='IPv4 segment routing traffic engineering best paths received' )
   v4SrTeBestEcmpPaths = Int( optional=True,
         help='IPv4 segment routing traffic engineering best ECMP paths received' )
   v6SrTePrefixesSent = Int( optional=True,
         help='IPv6 segment routing traffic engineering policies sent' )
   v6SrTePrefixesReceived = Int( optional=True,
         help='IPv6 segment routing traffic engineering policies received' )
   v6SrTeBestPaths = Int( optional=True,
         help='IPv6 segment routing traffic engineering best paths received' )
   v6SrTeBestEcmpPaths = Int( optional=True,
         help='IPv6 segment routing traffic engineering best ECMP paths received' )
   vplsSent = Int( optional=True, help='VPLS prefixes sent' )
   vplsReceived = Int( optional=True, help='VPLS prefixes received' )
   vplsBestPaths = Int( optional=True, help='VPLS best paths received' )
   vplsBestEcmpPaths = Int( optional=True, help='VPLS best ECMP paths received' )
   evpnSent = Int( optional=True,
         help='Ethernet VPN prefixes sent' )
   evpnReceived = Int( optional=True,
         help='Ethernet VPN prefixes received' )
   evpnBestPaths = Int( optional=True,
         help='Evpn best paths received' )
   evpnBestEcmpPaths = Int( optional=True,
         help='Evpn best ECMP paths received' )
   dpsPrefixesSent = Int( optional=True,
         help='Dynamic Path Selection prefixes sent' )
   dpsPrefixesReceived = Int( optional=True,
         help='Dynamic Path Selection prefixes received' )
   dpsBestPaths = Int( optional=True,
         help='Dynamic Path Selection best paths received' )
   dpsBestEcmpPaths = Int( optional=True,
         help='Dynamic Path Selection best ECMP paths received' )
   linkStateSent = Int( optional=True,
         help="Link State NLRIs sent" )
   linkStateReceived = Int( optional=True,
         help="Link State NLRIs received" )
   linkStateBestPaths = Int( optional=True,
         help="Link State best paths received" )
   linkStateBestEcmpPaths = Int( optional=True,
         help='Link State best ECMP paths' )
   rtMembershipPrefixesSent = Int( optional=True,
         help='RT Membership prefixes sent' )
   rtMembershipPrefixesReceived = Int( optional=True,
         help='RT Membership prefixes received' )
   rtMembershipBestPaths = Int( optional=True,
         help='RT Membership best paths' )
   rtMembershipBestEcmpPaths = Int( optional=True,
         help='RT Membership best ECMP paths' )
   v4MplsVpnPrefixesSent = Int( optional=True,
         help='IPv4 MPLS VPN prefixes sent' )
   v4MplsVpnPrefixesReceived = Int( optional=True,
         help='IPv4 MPLS VPN prefixes received' )
   v4MplsVpnBestPaths = Int( optional=True,
         help='IPv4 MPLS VPN best paths received' )
   v4MplsVpnBestEcmpPaths = Int( optional=True,
         help='IPv4 MPLS VPN best ECMP paths received' )
   v6MplsVpnPrefixesSent = Int( optional=True,
         help='IPv6 MPLS VPN prefixes sent' )
   v6MplsVpnPrefixesReceived = Int( optional=True,
         help='IPv6 MPLS VPN prefixes received' )
   v6MplsVpnBestPaths = Int( optional=True,
         help='IPv6 MPLS VPN best paths received' )
   v6MplsVpnBestEcmpPaths = Int( optional=True,
         help='IPv6 MPLS VPN best ECMP paths received' )
   v4FlowspecSent = Int( optional=True,
         help='IPv4 Flow Specification rules sent' )
   v4FlowspecReceived = Int( optional=True,
         help='IPv4 Flow Specification rules received' )
   v4FlowspecBestPaths = Int( optional=True,
         help='IPv4 Flow Specification best paths received' )
   v4FlowspecBestEcmpPaths = Int( optional=True,
         help='IPv4 Flow Specification best ECMP paths received' )
   v6FlowspecSent = Int( optional=True,
         help='IPv6 Flow Specification rules sent' )
   v6FlowspecReceived = Int( optional=True,
         help='IPv6 Flow Specification rules received' )
   v6FlowspecBestPaths = Int( optional=True,
         help='IPv6 Flow Specification best paths received' )
   v6FlowspecBestEcmpPaths = Int( optional=True,
         help='IPv6 Flow Specification best ECMP paths received' )
   v4FlowspecVpnSent = Int( optional=True,
         help='VPN-IPv4 Flow Specification rules sent' )
   v4FlowspecVpnReceived = Int( optional=True,
         help='VPN-IPv4 Flow Specification rules received' )
   v4FlowspecVpnBestPaths = Int( optional=True,
         help='VPN-IPv4 Flow Specification best paths received' )
   v4FlowspecVpnBestEcmpPaths = Int( optional=True,
         help='VPN-IPv4 Flow Specification best ECMP paths received' )
   v6FlowspecVpnSent = Int( optional=True,
         help='VPN-IPv6 Flow Specification rules sent' )
   v6FlowspecVpnReceived = Int( optional=True,
         help='VPN-IPv6 Flow Specification rules received' )
   v6FlowspecVpnBestPaths = Int( optional=True,
         help='VPN-IPv6 Flow Specification best paths received' )
   v6FlowspecVpnBestEcmpPaths = Int( optional=True,
         help='VPN-IPv6 Flow Specification best ECMP paths received' )
   v4MulticastSent = Int( optional=True,
         help='IPv4 Multicast prefixes sent' )
   v4MulticastReceived = Int( optional=True,
         help='IPv4 Multicast prefixes received' )
   v4MulticastBestPaths = Int( optional=True,
         help='IPv4 Multicast best paths received' )
   v4MulticastBestEcmpPaths = Int( optional=True,
         help='IPv4 Multicast best ECMP paths received' )
   v6MulticastSent = Int( optional=True,
         help='IPv6 Multicast prefixes sent' )
   v6MulticastReceived = Int( optional=True,
         help='IPv6 Multicast prefixes received' )
   v6MulticastBestPaths = Int( optional=True,
         help='IPv6 Multicast best paths received' )
   v6MulticastBestEcmpPaths = Int( optional=True,
         help='IPv6 Multicast best ECMP paths received' )
   v4LuPrefixesSent = Int( optional=True,
         help='IPv4 LU prefixes sent' )
   v4LuPrefixesReceived = Int( optional=True,
         help='IPv4 LU prefixes received' )
   v4LuBestPaths = Int( optional=True,
         help='IPv4 LU best paths received' )
   v4LuBestEcmpPaths = Int( optional=True,
         help='IPv4 LU best ECMP paths received' )
   v6LuPrefixesSent = Int( optional=True,
         help='IPv6 LU prefixes sent' )
   v6LuPrefixesReceived = Int( optional=True,
         help='IPv6 LU prefixes received' )
   v6LuBestPaths = Int( optional=True,
         help='IPv6 LU best paths received' )
   v6LuBestEcmpPaths = Int( optional=True,
         help='IPv6 LU best ECMP paths received' )
   updownTime = Int( optional=True, help='Up down time' )
   ifName = Str( optional=True, help='Interface name' )
   maintenance = Bool( optional=True, help='Peering under maintenance' )
   maintenanceRm = Str( optional=True, help='Maintenance mode route map' )
   maintenanceRmIn = Str( optional=True, help='Maintenance mode inbound route map' )
   maintenanceRmOut = Str( optional=True,
                           help='Maintenance mode outbound route map' )
   noComms = Bool( optional=True, help='Send no communities ' )
   noComs = Bool( optional=True, help='Send no communities'
                                      '(multi-agent mode)' )
   as4 = Int( optional=True, help='AS' )
   prependOwnDisabled = Bool( optional=True, help='Do not prepend own AS number' )
   establishFailHint = Str( optional=True,
         help='Establish failure hint' )
   rejectAsSetConfedSetV4Uni = Bool( optional=True,
         help='Treat-as-withdraw IPv4 Unicast AS path with AS_SET or AS_CONFED_SET' )
   rejectAsSetConfedSetV6Uni = Bool( optional=True,
         help='Treat-as-withdraw IPv6 Unicast AS path with AS_SET or AS_CONFED_SET' )
   dropStats = Submodel( optional=True,
         valueType=PeerDropStats,
         help='Prefix drop statistics' )
   bgpSoftReconfigInbound = Enum(
         help='Which inbound routes that are not accepted to \
               retain for soft reconfig',
         values=( 'Default', 'All', 'None' ) )
   rpkiOriginValidationMethod = Enum( optional=True,
         help='Method used for validating prefix origin AS',
         values=( 'disabled', 'local', 'community', 'preferCommunity' ) )
   rpkiOriginValidationSendExtComm = Bool( optional=True,
         help='Attach an extended community carrying the prefix origin validity' )
   rpkiOriginValidationRouteMap = Str( optional=True,
         help='Route map to filter which prefixes should be validated' )
   rpkiIpv4OvValidPaths = Int( optional=True,
         help='Paths with valid origin validation for IPv4 Unicast' )
   rpkiIpv4OvInvalidPaths = Int( optional=True,
         help='Paths with invalid origin validation for IPv4 Unicast' )
   rpkiIpv4OvUnknownPaths = Int( optional=True,
         help='Paths with unknown origin validation for IPv4 Unicast' )
   rpkiIpv6OvValidPaths = Int( optional=True,
         help='Paths with valid origin validation for IPv6 Unicast' )
   rpkiIpv6OvInvalidPaths = Int( optional=True,
         help='Paths with invalid origin validation for IPv6 Unicast' )
   rpkiIpv6OvUnknownPaths = Int( optional=True,
         help='Paths with unknown origin validation for IPv6 Unicast' )
   aigpSessionIpv4Uni = Bool( optional=True, help='AIGP session for IPv4 Unicast' )
   aigpSessionIpv6Uni = Bool( optional=True, help='AIGP session for IPv6 Unicast' )
   aigpSessionIpv4LabeledUni = Bool( optional=True,
         help='AIGP session for IPv4 Labelled Unicast' )
   aigpSessionIpv6LabeledUni = Bool( optional=True,
         help='AIGP session for IPv6 Labelled Unicast' )
   fwdFailoverTriggerSessionIpv4Uni = Bool(
      optional=True, help='Forwarding Failover Trigger Session for IPv4 Unicast' )
   fwdFailoverTriggerSessionIpv6Uni = Bool(
      optional=True, help='Forwarding Failover Trigger Session for IPv6 Unicast' )
   nexthopLuOriginateIpv4Uni = Bool( optional=True,
         help='Next hop labeled unicast origination for IPv4 Unicast' )
   nexthopLuOriginateIpv6Uni = Bool( optional=True,
         help='Next hop labeled unicast origination for IPv6 Unicast' )
   thirdPartyNexthopLuOriginateIpv4Uni = Bool( optional=True,
         help='Next hop labeled unicast origination for received'
              ' IPv4 Unicast nexthops' )
   thirdPartyNexthopLuOriginateIpv6Uni = Bool( optional=True,
         help='Next hop labeled unicast origination for received IPv6'
              ' Unicast nexthops' )
   nexthopLuOriginateLfibBackupIpForwardIpv4Uni = Bool( optional=True,
         help='Next hop labeled unicast origination LFIB backup IP forwarding '
              'for IPv4 Unicast' )
   nexthopLuOriginateLfibBackupIpForwardIpv6Uni = Bool( optional=True,
         help='Next hop labeled unicast origination LFIB backup IP forwarding '
              'for IPv6 Unicast' )
   orrPosition = Dict( valueType=BgpOrrPosition,
                       help='BGP optimal route reflection position per '
                            'address-family',
                       optional=True )
   afiSafiInfo = Dict( keyType=str, valueType=PerAfiSafiInfo,
                       help='A dictionary of address families',
                       optional=True )
   v4MappedV6NextHopTranslationIPv6LabeledUni = Bool( optional=True,
         help='Next hop resolution translates IPv4 mapped IPv6 nexthop '
            'addresses to IPv4 addresses for IPv6 with MPLS Labels' )
   endOfRibs = Submodel( valueType=EndOfRibs,
                       help='EOR Information for all negotiated AfiSafi(s)',
                       optional=True )
   v4RfdPathsSuppressed = Int( optional=True,
         help='IPv4 unicast paths currently suppressed due to route flap damping' )
   v4RfdPathSuppressionsLast24Hours = Int( optional=True,
         help='IPv4 unicast path suppression events due to route flap damping '
              'in the last 24 hours' )
   v4RfdMaxPathsSuppressedLast24Hours = Int( optional=True,
         help='Peak IPv4 unicast paths suppressed due to route flap damping'
              ' in the last 24 hours' )
   v4RfdSuppressionsSinceTracking = Int( optional=True,
         help='Lifetime IPv4 unicast path suppression events due to '
              'route flap damping' )
   v6RfdPathsSuppressed = Int( optional=True,
         help='IPv6 unicast paths currently suppressed due to route flap damping' )
   v6RfdPathSuppressionsLast24Hours = Int( optional=True,
         help='IPv6 unicast path suppression events due to route flap damping '
              'in the last 24 hours' )
   v6RfdMaxPathsSuppressedLast24Hours = Int( optional=True,
         help='Peak IPv6 unicast paths suppressed due to route flap damping'
              ' in the last 24 hours' )
   v6RfdSuppressionsSinceTracking = Int( optional=True,
         help='Lifetime IPv6 unicast path suppression events due to '
              'route flap damping' )
   inboundUpdateProcessingDelay = Int( optional=True,
      help='Inbound update processing delay in seconds' )
   inboundUpdateProcessingDelayState = Str( optional=True,
      help='Inbound update processing delay state' )
   inboundUpdateProcessingDelayLeft = Int( optional=True,
      help='Inbound update processing delay time left in seconds' )
   linkStatePeerNodeSegment = Submodel(
      optional=True,
      valueType=LinkStatePeerSegment,
      help="BGP link-state peer node segment" )

   def render( self ):
      print( "BGP neighbor is %s, remote AS %s, %s link" %
             ( self.peerAddress, self.asn, self.linkType ) )
      if self.description:
         print( "  Description: %s" % self.description )

      print( "  BGP version %s, remote router ID %s, VRF %s" %
             ( self.version, self.routerId, self.vrf ) )

      if self.peerGroup:
         print( "  Inherits configuration from and member of peer-group %s" %
                ( self.peerGroup ) )
      print( "  Negotiated BGP version %s" % ( self.negotiatedVersion ) )

      if self.updateGroupIndex:
         print( "  Member of update group %s" % ( self.updateGroupIndex ) )

      lastReadWriteStr = "  Last read "
      if self.lastRcvd:
         lastReadWriteStr += "%8s, " % formatTimeInterval( self.lastRcvd )
      else:

         lastReadWriteStr += "never, "

      lastReadWriteStr += "last write "
      if self.lastSent:
         lastReadWriteStr += "%8s" % formatTimeInterval( self.lastSent )
      else:
         lastReadWriteStr += "never"

      print( lastReadWriteStr )

      print( "  Hold time is %s, keepalive interval is %s seconds" %
             ( self.holdTime, self.keepaliveTime ) )

      print( "  Configured hold time is %s, keepalive interval is %s seconds" %
             ( self.confHoldTime, self.configuredKeepaliveTime ) )

      connectTimerStr = "  Connect timer is %s" % \
            ( 'active' if self.connectTimerActive else 'inactive' )
      if self.connectTimeLeft is not None:
         connectTimerStr += ", time left: %s" % \
               formatTimeInterval( self.connectTimeLeft )
      print( connectTimerStr )

      if self.connectInterval:
         print( "  Connection interval is %s seconds" %
                ( self.connectInterval ) )
      if self.connectFailed:
         print( "  Failed connection attempts is %s" %
                ( self.connectFailed ) )

      if self.configIdleHoldTime:
         print( "  Configured idle-restart time is %s seconds" %
                ( self.configIdleHoldTime ) )

      idleRestartStr = "  Idle-restart timer is %s" % \
            ( 'active' if self.idleRestartTimerActive else 'inactive' )

      if self.idleHoldTimeLeft:
         idleRestartStr += ", time left: %s" % \
               formatTimeInterval( self.idleHoldTimeLeft )
      print( idleRestartStr )

      if self.maintenance:
         print( "  Session is under maintenance" )

      bgpStateStr = "  BGP state is %s" % self.state
      if self.establishedTime is not None and int( self.establishedTime ) >= 0:
         bgpStateStr += ", up for %8s" % \
               ( formatTimeInterval( self.establishedTime ) )
      elif self.idleReason:
         bgpStateStr += ", %s" % self.idleReason
      print( bgpStateStr )

      if self.establishedTime is None and self.establishFailHint:
         print( "  Peering failure hint: %s" % ( self.establishFailHint ) )

      print( "  Number of transitions to established: %s" %
             ( self.establishedTransitions ) )
      print( "  Last state was %s" % ( self.lastState ) )
      print( "  Last event was %s" % ( self.lastEvent ) )

      if self.lastErrorCode is not None:
         lastErrorStr = "  Last sent notification:%s/%s, Last time %8s" % \
               ( self.lastErrorCode, self.lastErrorSubcode,
                  formatTimeInterval( self.lastErrorTimeInfo.time ) )
         if self.lastErrorTimeInfo.repeats is not None:
            lastErrorStr += ", First time %8s, Repeats %s" % \
               ( formatTimeInterval( self.lastErrorTimeInfo.firstTime ),
                  self.lastErrorTimeInfo.repeats )
         print( lastErrorStr )
      if self.lastErrorData is not None:
         print( "    Sent data: %s" % ( self.lastErrorData ) )
      if self.lastErrorRcvdCode is not None:
         lastErrorRcvdStr = "  Last rcvd notification:%s/%s, Last time %8s" % \
               ( self.lastErrorRcvdCode, self.lastErrorRcvdSubcode,
                  formatTimeInterval( self.lastErrorRcvdTimeInfo.time ) )
         if self.lastErrorRcvdTimeInfo.repeats is not None:
            lastErrorRcvdStr += ", First time %8s, Repeats %s" % \
               ( formatTimeInterval( self.lastErrorRcvdTimeInfo.firstTime ),
                  self.lastErrorRcvdTimeInfo.repeats )
         print( lastErrorRcvdStr )
      if self.lastErrorRcvdData is not None:
         print( "    Rcvd data: %s" % ( self.lastErrorRcvdData ) )
      if self.lastSocketOutErrorMsg is not None:
         lastSocketOutStr = "  Last sent socket-error:%s, Last time %8s" % \
               ( self.lastSocketOutErrorMsg,
                  formatTimeInterval( self.lastSocketOutErrorTimeInfo.time ) )
         if self.lastSocketOutErrorTimeInfo.repeats is not None:
            lastSocketOutStr += ", First time %8s, Repeats %s" % \
               ( formatTimeInterval( self.lastSocketOutErrorTimeInfo.firstTime ),
                  self.lastSocketOutErrorTimeInfo.repeats )
         print( lastSocketOutStr )
      if self.lastSocketInErrorMsg is not None:
         lastSocketInStr = "  Last rcvd socket-error:%s, Last time %8s" % \
               ( self.lastSocketInErrorMsg,
                  formatTimeInterval( self.lastSocketInErrorTimeInfo.time ) )
         if self.lastSocketInErrorTimeInfo.repeats is not None:
            lastSocketInStr += ", First time %8s, Repeats %s" % \
               ( formatTimeInterval( self.lastSocketInErrorTimeInfo.firstTime ),
                  self.lastSocketInErrorTimeInfo.repeats )
         print( lastSocketInStr )

      if self.routeReflectorClient:
         rrStr = "  Neighbor is a route reflector client"
         if self.clientReflectionMode == 'clientToClientDisabled':
            rrStr += " (client-to-client reflection disabled)"
         elif self.clientReflectionMode == 'meshed':
            rrStr += " (meshed)"
         print( rrStr )

      print( "  Neighbor Capabilities:" )

      def renderCapabilities():
         neighborCaps = self.neighborCapabilities
         # Print Neighbor Capabilities

         def printCapability( capability, description ):
            capabilitiesStr = "    %s: " % description
            if capability.advertised or capability.received:
               if capability.advertised:
                  capabilitiesStr += "advertised"
                  if capability.received:
                     capabilitiesStr += " and received"
               else:
                  capabilitiesStr += "received"
               if capability.enabled:
                  capabilitiesStr += " and negotiated"
               print( capabilitiesStr )
         if neighborCaps.multiprotocolCaps:
            mpCaps = neighborCaps.multiprotocolCaps
            if mpCaps.ipv4Unicast:
               printCapability( mpCaps.ipv4Unicast,
                                "Multiprotocol IPv4 Unicast" )
            if mpCaps.ipv4MplsLabels:
               printCapability( mpCaps.ipv4MplsLabels,
                                "Multiprotocol IPv4 Labeled Unicast" )
            if mpCaps.ipv4SrTe:
               printCapability( mpCaps.ipv4SrTe,
                                "Multiprotocol IPv4 SR-TE" )
            if mpCaps.ipv6Unicast:
               printCapability( mpCaps.ipv6Unicast,
                                "Multiprotocol IPv6 Unicast" )
            if mpCaps.ipv6MplsLabels:
               printCapability( mpCaps.ipv6MplsLabels,
                                "Multiprotocol IPv6 Labeled Unicast" )
            if mpCaps.ipv6SrTe:
               printCapability( mpCaps.ipv6SrTe,
                                "Multiprotocol IPv6 SR-TE" )
            if mpCaps.ipv4MplsVpn:
               printCapability( mpCaps.ipv4MplsVpn,
                                "Multiprotocol IPv4 VPN" )

         if neighborCaps.grCaps:
            if neighborCaps.grCaps.ipv4VpnCap:
               printCapability( neighborCaps.grCaps.ipv4VpnCap,
                                "Graceful Restart IPv4 VPN" )

         if neighborCaps.extendedNextHopCaps:
            extendedNHs = neighborCaps.extendedNextHopCaps
            if extendedNHs.ipv4UnicastOverIpv6:
               print( "    Extended Next-Hop Capability:" )
               printCapability( extendedNHs.ipv4UnicastOverIpv6,
                                "  IPv4 Unicast" )
         if neighborCaps.fourOctetAsnCap:
            printCapability( neighborCaps.fourOctetAsnCap, "Four Octet ASN" )
         if neighborCaps.routeRefreshCap:
            printCapability( neighborCaps.routeRefreshCap, "Route Refresh" )
         if neighborCaps.sendEorMessages:
            printCapability( neighborCaps.sendEorMessages,
                             "Send End-of-RIB messages" )
         if neighborCaps.dynamicCap:
            printCapability( neighborCaps.dynamicCap, "Dynamic Capabilities" )

         def printReceiveState( label, cap ):
            capabilitiesStr = "    %s: " % label
            state = None
            if cap.enabled:
               state = "negotiated"
            elif cap.receiveCapabilityAdvertised:
               state = "advertised"
            elif cap.sendCapabilityReceived:
               state = "received"
            if state:
               print( capabilitiesStr + state )

         def printSendState( label, cap ):
            capabilitiesStr = "    %s: " % label
            state = None
            if cap.enabled:
               state = "negotiated"
            elif cap.sendCapabilityAdvertised:
               state = "advertised"
            elif cap.receiveCapabilityReceived:
               state = "received"
            if state:
               print( capabilitiesStr + state )

         # Used to only print the add-paths header once for both send and recv
         def printAddPathsHeader( printHeader, direction ):
            if printHeader:
               print( "    Additional-paths %s capability:" % ( direction ) )
               printHeader = False
            return printHeader

         if neighborCaps.addPathsCaps:
            addPathsCaps = neighborCaps.addPathsCaps
            # Add-paths recv cap
            printHeader = True
            if addPathsCaps.ipv4Unicast:
               label = "  IPv4 Unicast"
               if addPathsCaps.ipv4Unicast.receivingAddPaths:
                  printHeader = printAddPathsHeader( printHeader, "recv" )
                  cap = addPathsCaps.ipv4Unicast.receivingAddPaths
                  printReceiveState( label, cap )

            if addPathsCaps.ipv6Unicast:
               label = "  IPv6 Unicast"
               if addPathsCaps.ipv6Unicast.receivingAddPaths:
                  printHeader = printAddPathsHeader( printHeader, "recv" )
                  cap = addPathsCaps.ipv6Unicast.receivingAddPaths
                  printReceiveState( label, cap )

            if addPathsCaps.ipv4MplsLabels:
               label = "  IPv4 Labeled Unicast"
               if addPathsCaps.ipv4MplsLabels.receivingAddPaths:
                  printHeader = printAddPathsHeader( printHeader, "recv" )
                  cap = addPathsCaps.ipv4MplsLabels.receivingAddPaths
                  printReceiveState( label, cap )

            if addPathsCaps.ipv6MplsLabels:
               label = "  IPv6 Labeled Unicast"
               if addPathsCaps.ipv6MplsLabels.receivingAddPaths:
                  printHeader = printAddPathsHeader( printHeader, "recv" )
                  cap = addPathsCaps.ipv6MplsLabels.receivingAddPaths
                  printReceiveState( label, cap )

            # Add-paths send cap
            printHeader = True
            if addPathsCaps.ipv4Unicast:
               label = "  IPv4 Unicast"
               if addPathsCaps.ipv4Unicast.sendingAddPaths:
                  printHeader = printAddPathsHeader( printHeader, "send" )
                  cap = addPathsCaps.ipv4Unicast.sendingAddPaths
                  printSendState( label, cap )

            if addPathsCaps.ipv6Unicast:
               label = "  IPv6 Unicast"
               if addPathsCaps.ipv6Unicast.sendingAddPaths:
                  printHeader = printAddPathsHeader( printHeader, "send" )
                  cap = addPathsCaps.ipv6Unicast.sendingAddPaths
                  printSendState( label, cap )

            if addPathsCaps.ipv4MplsLabels:
               label = "  IPv4 Labeled Unicast"
               if addPathsCaps.ipv4MplsLabels.sendingAddPaths:
                  printHeader = printAddPathsHeader( printHeader, "send" )
                  cap = addPathsCaps.ipv4MplsLabels.sendingAddPaths
                  printSendState( label, cap )

            if addPathsCaps.ipv6MplsLabels:
               label = "  IPv6 Labeled Unicast"
               if addPathsCaps.ipv6MplsLabels.sendingAddPaths:
                  printHeader = printAddPathsHeader( printHeader, "send" )
                  cap = addPathsCaps.ipv6MplsLabels.sendingAddPaths
                  printSendState( label, cap )

         # Print Graceful Restart Capabilities
         def printGrCapabilities( capability, description ):
            print( description )
            if capability.restartTime:
               print( "       Restart-time is %s" % int( capability.restartTime ) )
               print( "       Restart-State bit: %s" %
                      ( "yes" if capability.bgpRestarted else "no" ) )
               print( "       Graceful notification: %s" %
                      ( "yes" if capability.gracefulNotificationSupported
                        else "no" ) )
            if capability.forwardingStatesPreserved:
               forwardingStates = capability.forwardingStatesPreserved
               if forwardingStates.ipv4Unicast is not None:
                  print( "       IPv4 Unicast is enabled, Forwarding State is "
                         "%s preserved" % ( "" if forwardingStates.ipv4Unicast
                                            else "not" ) )
               if forwardingStates.ipv6Unicast is not None:
                  print( "       IPv6 Unicast is enabled, Forwarding State is "
                         "%s preserved" % ( "" if forwardingStates.ipv6Unicast
                                            else "not" ) )
         if neighborCaps.grCaps:
            grCaps = neighborCaps.grCaps
            if grCaps.advertised:
               printGrCapabilities( grCaps.advertised,
                                    "    Graceful Restart advertised:" )
            if grCaps.received:
               printGrCapabilities( grCaps.received,
                                    "    Graceful Restart received:" )

      if( self.neighborCapabilities.negotiationDisabled or
            self.neighborCapabilities.unsupportedCapReceived ):
         capabilityStr = "    Capability Negotiation: "
         if self.neighborCapabilities.negotiationDisabled:
            capabilityStr += "disabled"
         if self.neighborCapabilities.unsupportedCapReceived:
            if self.neighborCapabilities.negotiationDisabled:
               capabilityStr += " and "
            capabilityStr += "not capable"
         print( capabilityStr )
      else:
         renderCapabilities()

      restartTimerStr = "  Restart timer is %s" % \
            ( 'active' if self.gracefulRestartTimerActive else 'inactive' )
      if self.restartTimeLeft:
         restartTimerStr += ", time left: %s" % \
               formatTimeInterval( self.restartTimeLeft )
      print( restartTimerStr )

      eorTimerStr = "  End of rib timer is %s" % \
            ( 'active' if self.eorTimerActive else 'inactive' )
      if self.endOfRibTimeLeft:
         eorTimerStr += ", time left: %s" % \
               formatTimeInterval( self.endOfRibTimeLeft )
      print( eorTimerStr )

      msgStatFormat = "    %-15s%10s%10s"
      msgStatFormat2 = "    %-15s%10s%10s%15s%20s"
      print( "  Message Statistics:" )
      print( "    InQ depth is %s" % ( self.inMessageStats.queueDepth ) )
      print( "    OutQ depth is %s" % ( self.outMessageStats.queueDepth ) )
      print( msgStatFormat % ( "", "Sent", "Rcvd" ) )
      print( msgStatFormat % ( "Opens:", self.outMessageStats.opens,
                               self.inMessageStats.opens ) )
      print( msgStatFormat % ( "Notifications:",
                               self.outMessageStats.notifications,
                               self.inMessageStats.notifications ) )
      print( msgStatFormat % ( "Updates:", self.outMessageStats.updates,
                               self.inMessageStats.updates ) )
      print( msgStatFormat % ( "Keepalives:",
                               self.outMessageStats.keepalives,
                               self.inMessageStats.keepalives ) )
      print( msgStatFormat % ( "Route-Refresh:",
                               self.outMessageStats.rtRefreshes,
                               self.inMessageStats.rtRefreshes ) )
      print( msgStatFormat % ( "Total messages:",
                               self.sentMessages, self.receivedMessages ) )
      print( "  Prefix Statistics:" )
      print( msgStatFormat2 % ( "", "Sent", "Rcvd", "Best Paths",
                                "Best ECMP Paths" ) )
      print( msgStatFormat2 % ( "IPv4 Unicast:",
                                self.prefixesSent, self.prefixesReceived,
                                self.v4BestPaths, self.v4BestEcmpPaths ) )
      if self.v6SixPePrefixesReceived is not None or \
         self.v6SixPePrefixesSent is not None:
         print( msgStatFormat2 % ( "IPv6 6PE(MPLS Label):",
                                   self.v6SixPePrefixesSent,
                                   self.v6SixPePrefixesReceived, '-', '-' ) )
      elif self.v6PrefixesReceived is not None:
         print( msgStatFormat2 % ( "IPv6 Unicast:",
                                   self.v6PrefixesSent, self.v6PrefixesReceived,
                                   self.v6BestPaths, self.v6BestEcmpPaths ) )
      if self.v4SrTePrefixesSent is not None:
         print( msgStatFormat2 % ( "IPv4 SR-TE:  ",
                                   self.v4SrTePrefixesSent,
                                   self.v4SrTePrefixesReceived,
                                   self.v4SrTeBestPaths, self.v4SrTeBestEcmpPaths ) )
      if self.v6SrTePrefixesSent is not None:
         print( msgStatFormat2 % ( "IPv6 SR-TE:  ",
                                   self.v6SrTePrefixesSent,
                                   self.v6SrTePrefixesReceived,
                                   self.v6SrTeBestPaths, self.v6SrTeBestEcmpPaths ) )
      print( "  Inbound updates dropped by reason:" )
      print( "    AS path loop detection: %d" % ( self.dropStats.inDropAsloop ) )
      print( "    Enforced First AS: %d" % ( self.dropStats.inDropEnforceFirstAs ) )
      print( "    Originator ID matches local router ID: %d" %
             ( self.dropStats.inDropOrigId ) )
      print( "    Nexthop matches local IP address: %d" %
             ( self.dropStats.inDropNhLocal ) )
      print( "    Unexpected IPv6 nexthop for IPv4 routes: %d" %
             ( self.dropStats.inDropNhAfV6 ) )

      if self.invalidNextHopDirectEbgp:
         print( "    Nexthop invalid for single hop eBGP: %d" %
                ( self.dropStats.inDropNhInvalid ) )

      if self.peerInUpdateErrors is not None:
         self.peerInUpdateErrors.render()

      print( "  Inbound paths dropped by reason:" )
      print( "    IPv4 labeled-unicast NLRIs dropped due to excessive labels: %d" %
             ( self.dropStats.prefixLuDroppedV4 ) )
      print( "    IPv6 labeled-unicast NLRIs dropped due to excessive labels: %d" %
             ( self.dropStats.prefixLuDroppedV6 ) )
      print( "  Outbound paths dropped by reason:" )
      print( "    IPv4 local address not available: %d" %
             ( self.dropStats.outDropV4LocalAddr ) )
      print( "    IPv6 local address not available: %d" %
             ( self.dropStats.outDropV6LocalAddr ) )
      if self.missingPolicyDefaultDeny:
         if self.missingPolicyDefaultDeny.importActive:
            print( "  Missing policy/default deny import action is active" )
         if self.missingPolicyDefaultDeny.exportActive:
            print( "  Missing policy/default deny export action is active" )
      if self.routeMapInbound:
         print( "  Inbound route map is %s" % ( self.routeMapInbound ) )
      if self.routeMapOutbound:
         print( "  Outbound route map is %s" % ( self.routeMapOutbound ) )
      if self.routeMapInfo.inboundIpv4Uni:
         print( "  Inbound route map for IPv4 unicast is %s" %
                ( self.routeMapInfo.inboundIpv4Uni ) )
      if self.routeMapInfo.inboundIpv6Uni:
         print( "  Inbound route map for IPv6 unicast is %s" %
                ( self.routeMapInfo.inboundIpv6Uni ) )
      if self.routeMapInfo.inboundIpv4SrTe:
         print( "  Inbound route map for IPv4 SR-TE is %s" %
                ( self.routeMapInfo.inboundIpv4SrTe ) )
      if self.routeMapInfo.inboundIpv6SrTe:
         print( "  Inbound route map for IPv6 SR-TE is %s" %
                ( self.routeMapInfo.inboundIpv6SrTe ) )
      if self.routeMapInfo.outboundIpv4Uni:
         print( "  Outbound route map for IPv4 unicast is %s" %
                ( self.routeMapInfo.outboundIpv4Uni ) )
      if self.routeMapInfo.outboundIpv6Uni:
         print( "  Outbound route map for IPv6 unicast is %s" %
                ( self.routeMapInfo.outboundIpv6Uni ) )
      if self.prefixListInfo.inboundIpv4Uni:
         print( "  Inbound prefix list for IPv4 unicast is %s" %
                ( self.prefixListInfo.inboundIpv4Uni ) )
      if self.prefixListInfo.inboundIpv6Uni:
         print( "  Inbound prefix list for IPv6 unicast is %s" %
                ( self.prefixListInfo.inboundIpv6Uni ) )
      if self.prefixListInfo.outboundIpv4Uni:
         print( "  Outbound prefix list for IPv4 unicast is %s" %
                ( self.prefixListInfo.outboundIpv4Uni ) )
      if self.prefixListInfo.outboundIpv6Uni:
         print( "  Outbound prefix list for IPv6 unicast is %s" %
                ( self.prefixListInfo.outboundIpv6Uni ) )
      if self.neighborCapabilities:
         extNhCaps = self.neighborCapabilities.extendedNextHopCaps
         if extNhCaps and extNhCaps.ipv4LocalNlriOverIpv6:
            print( "  Local IPv4 NLRIs are advertised with IPv6 next-hops" )

      if self.maintenance:
         print( "  Maintenance-mode:" )
         if self.noComms:
            print( "    send-community not enabled" )

      rmStr = "      Route map is "
      print( "    Inbound policy" )
      if self.maintenanceRmIn:
         print( "%s%s" % ( rmStr, self.maintenanceRmIn ) )
      print( "    Outbound policy" )
      if self.maintenanceRmOut:
         print( "%s%s" % ( rmStr, self.maintenanceRmOut ) )

      if self.bgpSoftReconfigInbound != 'Default':
         print( '  Soft reconfiguration inbound is "%s"' %
                self.bgpSoftReconfigInbound )

      print( "Local AS is %s, local router ID %s" %
             ( self.localAsn, self.localRouterId ) )

      ttlStr = "TTL is %s" % self.ttl
      if self.ttlConfig == 'gtsm':
         ttlStr += ", BGP neighbor may be up to %s hops away" % \
                       ( self.maxTtlHops )
      elif self.ttlConfig == 'ebgpMultiHop':
         ttlStr += ", external peer can be %s hops away" % ( self.maxTtlHops )
      print( ttlStr )

      if self.updateSource:
         localTcpStr = "Local TCP address is %s" % ( self.updateSource )
         if self.localPort:
            localTcpStr += ", local port is %s" % ( self.localPort )
         print( localTcpStr )

      if self.peerAddress:
         remoteTcpStr = "Remote TCP address is %s" % ( self.peerAddress )
         if self.remotePort:
            remoteTcpStr += ", remote port is %s" % ( self.remotePort )
         print( remoteTcpStr )

      if self.confRemotePort:
         print( "Configured remote port is %s" % ( self.confRemotePort ) )

      if self.localAddressV4:
         localAddrStr = "Local IPv4 address is %s" % ( self.localAddressV4 )
         if self.localAddressPort:
            localAddrStr += ", local port is %s" % ( self.localAddressPort )
         print( localAddrStr )

      if self.localAddressV6:
         localAddrStr = "Local IPv6 address is %s" % ( self.localAddressV6 )
         if self.localAddressPort:
            localAddrStr += ", local port is %s" % ( self.localAddressPort )
         print( localAddrStr )

      if self.remoteAddressV4:
         remoteAddrStr = "Remote IPv4 address is %s" % ( self.remoteAddressV4 )
         if self.remoteAddrPort:
            remoteAddrStr += ", remote port is %s" % ( self.remoteAddrPort )
         print( remoteAddrStr )

      if self.remoteAddressV6:
         remoteAddrStr = "Remote IPv6 address is %s" % ( self.remoteAddressV6 )
         if self.remoteAddrPort:
            remoteAddrStr += ", remote port is %s" % ( self.remoteAddrPort )
         print( remoteAddrStr )

      if self.autoLocalAddress:
         print( "Auto-Local-Addr is %s" % ( self.autoLocalAddress ) )

      if self.bfdSessionState:
         print( "Bfd is enabled and state is %s"
                % ( bfdJsonToStateName.get( self.bfdSessionState, 'Unknown' ) ) )

      if self.passiveConnEnabled:
         print( "Passive TCP connection-mode is enabled" )

      if self.md5AuthEnabled:
         print( "MD5 authentication is enabled" )

      if self.removeOrReplacePrivateAsEgress == 'removePrivateAsEgressReplace':
         print( "Private AS numbers always replaced with the local AS "
                "in outbound updates to this neighbor" )
      elif self.removeOrReplacePrivateAsEgress == 'removePrivateAsEgressAll':
         print( "Private AS numbers always removed from "
                "outbound updates to this neighbor" )
      elif self.removeOrReplacePrivateAsEgress == 'removePrivateAsEgress':
         print( "Private AS numbers removed from outbound updates to this "
                "neighbor if only private AS numbers are present" )

      if self.prependOwnDisabled:
         print( "Not prepending own AS Number in outbound updates to this neighbor" )

      if self.peerTcpInfo is not None:
         self.peerTcpInfo.render()
      # Add newline after printing output for each neighbor
      print( "" )

   def renderBfd( self ):
      bfdFlags = 'N'
      if self.bfdSessionState:
         if self.bfdSessionState == 'up':
            bfdFlags = 'U'
         elif self.bfdSessionState == 'down':
            bfdFlags = 'D'
         elif self.bfdSessionState == 'init':
            bfdFlags = 'I'
         else:
            bfdFlags = 'N'

      print( "%-18s %-18s %-10s %-11s %-5c" %
             ( self.peerAddress,
               intfLongName( self.ifName )
               if self.ifName is not None else "none",
               formatTimeInterval( self.updownTime )
                  if self.updownTime is not None else "--:--",
               self.state, bfdFlags ) )

   def processData( self, data ):
      self.peerAddress = peerKeyFromData( data )

      if 'updateSource6' in data or 'updateSource4' in data:
         if 'updateSource6' in data and data[ 'updateSource6' ]:
            ipAddr = data[ 'updateSource6' ]
            del data[ 'updateSource6' ]
         elif 'updateSource4' in data and data[ 'updateSource4' ]:
            ipAddr = data[ 'updateSource4' ]
            del data[ 'updateSource4' ]
         self.updateSource = Arnet.IpGenAddr( trimIntfName( ipAddr ) )

      flagNum = 0
      peerFlagName = 'bgp_peer_flags%d' % flagNum
      peerFlags = []
      while peerFlagName in data:
         self.bgpPeerFlags.append( data[ peerFlagName ] )
         peerFlags.append( data[ peerFlagName ] )
         del data[ peerFlagName ]
         flagNum += 1
         peerFlagName = 'bgp_peer_flags%d' % flagNum

      if data[ 'maintenance' ]:
         self.maintenance = ord( data[ 'maintenance' ] ) != 0
         del data[ 'maintenance' ]
      if data[ 'noComms' ]:
         self.noComms = ord( data[ 'noComms' ] ) != 0
         del data[ 'noComms' ]

      # for backward compatibility when single 'inout' route-map was supported
      if 'maintenanceRmIn' in data and 'maintenanceRmOut' in data and \
         ( data[ 'maintenanceRmIn' ] == data[ 'maintenanceRmOut' ] ):
         self.maintenanceRm = data[ 'maintenanceRmOut' ]

      disabled = data.pop( 'prependOwnDisabled', None )
      if disabled is not None:
         self.prependOwnDisabled = ord( disabled ) != 0

      if data[ 'bgpPeerOptions' ] & GatedConstants.BgpoFlag.BGPO_KEEPALL:
         self.bgpSoftReconfigInbound = "All"
      elif data[ 'bgpPeerOptions' ] & GatedConstants.BgpoFlag.BGPO_KEEPNONE:
         self.bgpSoftReconfigInbound = "None"
      else:
         self.bgpSoftReconfigInbound = "Default"

      # the DGET has a large number of attributes with common prefixes
      # processFields is used to map those names to BgpPeer submodels
      # to reduce clutter
      def processFields( data, model, prefixStr ):

         fields = list( model.__attributes__ )

         # input field suffixes have camel case words
         if prefixStr:
            inputFields = [ prefixStr + i[ 0 ].upper() + i[ 1 : ] for i in fields ]
         else:
            inputFields = fields

         for field, inField in zip( fields, inputFields ):
            if inField in data:
               setattr( model, field, data[ inField ] )
               del data[ inField ]

         return model

      if 'lastErrorCode' in data:
         self.lastErrorTimeInfo = processFields( data, ErrorTimeInfo(),
                                                 'lastError' )
      if 'lastErrorRcvdCode' in data:
         self.lastErrorRcvdTimeInfo = processFields( data, ErrorTimeInfo(),
                                                     'lastErrorRcvd' )
      if 'lastSocketOutErrorMsg' in data:
         self.lastSocketOutErrorTimeInfo = processFields( data, ErrorTimeInfo(),
                                                          'lastSocketOutError' )
      if 'lastSocketInErrorMsg' in data:
         self.lastSocketInErrorTimeInfo = processFields( data, ErrorTimeInfo(),
                                                         'lastSocketInError' )

      self.inMessageStats = processFields( data, MessageStatInfo(), 'in' )
      self.outMessageStats = processFields( data, MessageStatInfo(), 'out' )

      self.routeMapInfo = processFields( data, RouteMapFilterInfo(), 'routeMap' )
      self.routeMapInfo.filterType = 'RouteMap'

      self.prefixListInfo = processFields( data, RouteFilterInfo(), 'prefixList' )
      self.prefixListInfo.filterType = 'PrefixList'

      self.dropStats = processFields( data, PeerDropStats(), '' )

      bgpPeerCaps = data[ 'bgpPeerCaps' ]
      miscFlags = data[ 'miscFlags' ]

      # MiscFlags state
      if miscFlags & GatedConstants.DabitBnpEntryMF.DABIT_BNP_ENTRY_MF_MD5:
         data[ 'md5AuthEnabled' ] = True
      if miscFlags & GatedConstants.DabitBnpEntryMF.\
            DABIT_BNP_ENTRY_MF_CONNECT_TIMER_STATE:
         data[ 'connectTimerActive' ] = True
      if miscFlags & GatedConstants.DabitBnpEntryMF.\
            DABIT_BNP_ENTRY_MF_IDLEHOLD_TIMER_STATE:
         data[ 'idleRestartTimerActive' ] = True

      if miscFlags & GatedConstants.DabitBnpEntryMF.\
            DABIT_BNP_ENTRY_MF_RR_CLIENT:
         data[ 'routeReflectorClient' ] = True
         if miscFlags & GatedConstants.DabitBnpEntryMF.\
               DABIT_BNP_ENTRY_MF_NO_CLIENT_REFLECT:
            data[ 'clientReflectionMode' ] = 'clientToClientDisabled'
      elif miscFlags & GatedConstants.DabitBnpEntryMF.\
               DABIT_BNP_ENTRY_MF_RR_MESHED:
         data[ 'routeReflectorClient' ] = True
         data[ 'clientReflectionMode' ] = 'meshed'

      if miscFlags & GatedConstants.DabitBnpEntryMF.\
            DABIT_BNP_ENTRY_MF_RESTART_TIMER_STATE:
         data[ 'gracefulRestartTimerActive' ] = True
      if miscFlags & GatedConstants.DabitBnpEntryMF.\
            DABIT_BNP_ENTRY_MF_EOR_TIMER_STATE:
         data[ 'eorTimerActive' ] = True

      missingPolicies = MissingPolicyDefaultDeny()
      if miscFlags & GatedConstants.DabitBnpEntryMF.\
            DABIT_BNP_ENTRY_MF_IN_DEFAULT_DENY_POLICY:
         missingPolicies.importActive = True
      if miscFlags & GatedConstants.DabitBnpEntryMF.\
            DABIT_BNP_ENTRY_MF_OUT_DEFAULT_DENY_POLICY:
         missingPolicies.exportActive = True

      if missingPolicies.importActive or missingPolicies.exportActive:
         data[ 'missingPolicyDefaultDeny' ] = missingPolicies

      bgpPeerHiscaps = data[ 'bgpPeerHiscaps' ]

      # Capabilities
      def renderCapabilities():
         neighborCaps = NeighborCapabilities()
         # Graceful Restart Capabilities
         grPoaBits = {
              'advertised': {
                 'restartStateBit': GatedConstants.DabitBnpEntryMF.
                 DABIT_BNP_ENTRY_MF_LOCAL_RESTART_STATE,
                 'grNotification': GatedConstants.DabitBnpEntryMF.
                 DABIT_BNP_ENTRY_MF_LOCAL_GR_NOTIFICATION,
                 'forwardV4Preserved': GatedConstants.DabitBnpEntryMF.
                 DABIT_BNP_ENTRY_MF_LOCAL_GR_V4_FWD_PRESERVE,
                 'forwardV6Preserved': GatedConstants.DabitBnpEntryMF.
                 DABIT_BNP_ENTRY_MF_LOCAL_GR_V6_FWD_PRESERVE },
               'received': {
                  'restartStateBit': GatedConstants.DabitBnpEntryMF.
                 DABIT_BNP_ENTRY_MF_RESTART_STATE,
                  'grNotification': GatedConstants.DabitBnpEntryMF.
                 DABIT_BNP_ENTRY_MF_GR_NOTIFICATION,
                  'forwardV4Preserved': GatedConstants.DabitBnpEntryMF.
                 DABIT_BNP_ENTRY_MF_REMOTE_GR_V4_FWD_PRESERVE,
                  'forwardV6Preserved': GatedConstants.DabitBnpEntryMF.
                 DABIT_BNP_ENTRY_MF_REMOTE_GR_V6_FWD_PRESERVE }
               }

         def processGrCap( restartTime, bitmask, miscFlags, advertised=True ):
            grBits = ( grPoaBits[ 'advertised' ] if advertised else
                       grPoaBits[ 'received' ] )
            bgpGrCapModel = GrCapability()
            bgpGrCapModel.restartTime = float( restartTime )
            bgpGrCapModel.bgpRestarted =\
                       bool( miscFlags &
                       grBits[ 'restartStateBit' ] )
            bgpGrCapModel.gracefulNotificationSupported =\
                       bool( miscFlags &
                       grBits[ 'grNotification' ] )
            if ( ( bitmask & GatedConstants.BgpcFlag.BGPC_V4_UNI_RESTART ) or
                 ( bitmask & GatedConstants.BgpcFlag.BGPC_V6_UNI_RESTART and
                   bitmask & GatedConstants.BgpcFlag.BGPC_MP_V6_UNI ) ):
               forwardingStatesPreserved = ForwardingStates()
               if bitmask & GatedConstants.BgpcFlag.BGPC_V4_UNI_RESTART:
                  forwardingStatesPreserved.ipv4Unicast = \
                        bool( miscFlags &
                        grBits[ 'forwardV4Preserved' ] )
               if ( bitmask & GatedConstants.BgpcFlag.BGPC_V6_UNI_RESTART and
                      bitmask & GatedConstants.BgpcFlag.BGPC_MP_V6_UNI ):
                  forwardingStatesPreserved.ipv6Unicast =\
                        bool( miscFlags &
                        grBits[ 'forwardV6Preserved' ] )
               bgpGrCapModel.forwardingStatesPreserved = forwardingStatesPreserved
            return bgpGrCapModel

         grCaps = GrCapabilities()
         if ( bgpPeerCaps & GatedConstants.BgpcFlag.BGPC_V4_UNI_RESTART or
                  bgpPeerCaps & GatedConstants.BgpcFlag.BGPC_V6_UNI_RESTART ):
            grCaps.advertised = processGrCap(
                                       data.get( 'advertisedRestartTime', 0 ),
                                       bgpPeerCaps, miscFlags )

         if ( bgpPeerHiscaps & GatedConstants.BgpcFlag.BGPC_V4_UNI_RESTART or
                  bgpPeerHiscaps & GatedConstants.BgpcFlag.BGPC_V6_UNI_RESTART ):
            grCaps.received = processGrCap(
                                       data.get( 'restart', 0 ),
                                       bgpPeerHiscaps, miscFlags, advertised=False )

         neighborCaps.grCaps = grCaps

         # BGP Peer Capabilities
         sendMy = bgpPeerCaps & GatedConstants.BgpcApFlags.BGPC_AP_SEND_ANY
         recvMy = bgpPeerCaps & GatedConstants.BgpcApFlags.BGPC_AP_RECV_ANY
         sendHis = bgpPeerHiscaps & GatedConstants.BgpcApFlags.BGPC_AP_SEND_ANY
         recvHis = bgpPeerHiscaps & GatedConstants.BgpcApFlags.BGPC_AP_RECV_ANY

         # Check if the peerFlag is set in self.bgpPeerFlags list
         def peerFlagTest( peerFlag ):
            flagNum = peerFlag // BITS_PER_PEERFLAG
            flagMask = ( 1 << ( peerFlag % BITS_PER_PEERFLAG ) )
            return peerFlags[ flagNum ] & flagMask

         def populateCapability( capFlag, peerFlag ):
            capabilityModel = Capability()
            sentCaps = bgpPeerCaps & capFlag
            rcvdCaps = bgpPeerHiscaps & capFlag
            negotiatedCaps = peerFlagTest( peerFlag )
            if sentCaps or rcvdCaps or negotiatedCaps:
               capabilityModel = Capability()
               capabilityModel.advertised = bool( sentCaps )
               capabilityModel.received = bool( rcvdCaps )
               capabilityModel.enabled = bool( negotiatedCaps )
               return capabilityModel
            return None

         multiprotocolCaps = MultiprotocolCapabilities()

         ipv4Unicast = populateCapability(
                             GatedConstants.BgpcFlag.BGPC_MP_V4_UNI,
                             GatedConstants.BgpPeerFlag.BGPPF_V4_UNI )
         if ipv4Unicast:
            multiprotocolCaps.ipv4Unicast = ipv4Unicast

         ipv4MplsLabels = populateCapability(
                             GatedConstants.BgpcFlag.BGPC_MP_V4_LABELS,
                             GatedConstants.BgpPeerFlag.BGPPF_V4_LABELS )
         if ipv4MplsLabels:
            multiprotocolCaps.ipv4MplsLabels = ipv4MplsLabels

         ipv4SrTe = populateCapability(
                             GatedConstants.BgpcFlag.BGPC_MP_V4_SR_TE,
                             GatedConstants.BgpPeerFlag.BGPPF_V4_SR_TE )
         if ipv4SrTe:
            multiprotocolCaps.ipv4SrTe = ipv4SrTe

         ipv6Unicast = populateCapability(
                             GatedConstants.BgpcFlag.BGPC_MP_V6_UNI,
                             GatedConstants.BgpPeerFlag.BGPPF_V6_UNI )
         if ipv6Unicast:
            multiprotocolCaps.ipv6Unicast = ipv6Unicast

         ipv6MplsLabels = populateCapability(
                             GatedConstants.BgpcFlag.BGPC_MP_V6_LABELS,
                             GatedConstants.BgpPeerFlag.BGPPF_V6_LABELS )
         if ipv6MplsLabels:
            multiprotocolCaps.ipv6MplsLabels = ipv6MplsLabels

         ipv6SrTe = populateCapability(
                             GatedConstants.BgpcFlag.BGPC_MP_V6_SR_TE,
                             GatedConstants.BgpPeerFlag.BGPPF_V6_SR_TE )
         if ipv6SrTe:
            multiprotocolCaps.ipv6SrTe = ipv6SrTe

         ipv4MplsVpn = populateCapability(
                             GatedConstants.BgpcFlag.BGPC_MP_V4_VPNIPV4,
                             GatedConstants.BgpPeerFlag.BGPPF_V4_VPNIPV4 )
         if ipv4MplsVpn:
            multiprotocolCaps.ipv4MplsVpn = ipv4MplsVpn

         neighborCaps.multiprotocolCaps = multiprotocolCaps

         grBgpV4VpnCap = populateCapability(
                             GatedConstants.BgpcFlag.BGPC_V4_VPN_RESTART,
                             GatedConstants.BgpPeerFlag.BGPPF_V4_VPN_RESTART )
         if grBgpV4VpnCap:
            if not neighborCaps.grCaps:
               neighborCaps.grCaps = GrCapabilities()
            neighborCaps.grCaps.ipv4VpnCap = grBgpV4VpnCap

         ipv4OverIpv6 = populateCapability(
                             GatedConstants.BgpcFlag.BGPC_EXT_NEXTHOP,
                             GatedConstants.BgpPeerFlag.BGPPF_EXT_NEXTHOP )
         if ipv4OverIpv6:
            neighborCaps.extendedNextHopCaps = ExtendedNexthopCapabilities()
            neighborCaps.extendedNextHopCaps.ipv4UnicastOverIpv6 = ipv4OverIpv6

         # Allow sending of IPv6 next hops for locally sourced IPv4 NLRIs
         # to peers capable of exchanging IPv4+v6NH
         if bgpPeerCaps & GatedConstants.BgpcFlag.BGPC_EXT_NEXTHOP_ORIGINATE:
            if not neighborCaps.extendedNextHopCaps:
               neighborCaps.extendedNextHopCaps = ExtendedNexthopCapabilities()
            neighborCaps.extendedNextHopCaps.ipv4LocalNlriOverIpv6 = True

         fourOctetAsn = populateCapability(
                             GatedConstants.BgpcFlag.BGPC_AS4,
                             GatedConstants.BgpPeerFlag.BGPPF_AS4 )
         if fourOctetAsn:
            neighborCaps.fourOctetAsnCap = fourOctetAsn

         routeRefreshCap = populateCapability(
                             GatedConstants.BgpcFlag.BGPC_RT_REFRESH,
                             GatedConstants.BgpPeerFlag.BGPPF_RT_REFRESH )
         if routeRefreshCap:
            neighborCaps.routeRefreshCap = routeRefreshCap

         sendEorMessages = populateCapability(
                             GatedConstants.BgpcFlag.BGPC_RESTART_ANY,
                             GatedConstants.BgpPeerFlag.BGPPF_RESTART )
         if sendEorMessages:
            neighborCaps.sendEorMessages = sendEorMessages

         dynamicCap = populateCapability(
                             GatedConstants.BgpcFlag.BGPC_DYN_CAPS,
                             GatedConstants.BgpPeerFlag.BGPPF_DYN_CAPS )
         if dynamicCap:
            neighborCaps.dynamicCap = dynamicCap

         def populateAddPathsCapability( rxCapFlag, rxHisCapFlag, rxPeerFlag,
                                         txCapFlag, txHisCapFlag, txPeerFlag ):
            model = AddPathsCapability()
            # ReceivingAddPaths
            rxCapTx = bool( bgpPeerCaps & rxCapFlag )
            txCapRx = bool( bgpPeerHiscaps & rxHisCapFlag )
            rxEnabled = bool( peerFlagTest( rxPeerFlag ) )
            if rxCapTx or txCapRx or rxEnabled:
               recvCap = ReceivingAddPaths()
               recvCap.receiveCapabilityAdvertised = rxCapTx
               recvCap.sendCapabilityReceived = txCapRx
               recvCap.enabled = rxEnabled
               model.receivingAddPaths = recvCap
            # SendingAddPaths
            txCapTx = bool( bgpPeerCaps & txCapFlag )
            rxCapRx = bool( bgpPeerHiscaps & txHisCapFlag )
            enabled = bool( peerFlagTest( txPeerFlag ) )
            if txCapTx or rxCapRx or enabled:
               sendCap = SendingAddPaths()
               sendCap.sendCapabilityAdvertised = bool( bgpPeerCaps & txCapFlag )
               sendCap.receiveCapabilityReceived = bool( bgpPeerHiscaps
                                                         & txHisCapFlag )
               sendCap.enabled = bool( peerFlagTest( txPeerFlag ) )
               model.sendingAddPaths = sendCap

            if model.receivingAddPaths or model.sendingAddPaths:
               return model
            return None

         rxEnabled = bool( recvMy or sendHis )
         txEnabled = bool( sendMy or recvHis )

         if rxEnabled or txEnabled:
            addPathsCaps = AddPathsCapabilities()

            v4Unicast = populateAddPathsCapability(
                              GatedConstants.BgpcFlag.BGPC_V4_UNI_AP_RECV,
                              GatedConstants.BgpcFlag.BGPC_V4_UNI_AP_SEND,
                              GatedConstants.BgpPeerFlag.BGPPF_AP_RECV_IPV4_UNI,
                              GatedConstants.BgpcFlag.BGPC_V4_UNI_AP_SEND,
                              GatedConstants.BgpcFlag.BGPC_V4_UNI_AP_RECV,
                              GatedConstants.BgpPeerFlag.BGPPF_AP_SEND_IPV4_UNI )
            if v4Unicast:
               addPathsCaps.ipv4Unicast = v4Unicast

            v6Unicast = populateAddPathsCapability(
                              GatedConstants.BgpcFlag.BGPC_V6_UNI_AP_RECV,
                              GatedConstants.BgpcFlag.BGPC_V6_UNI_AP_SEND,
                              GatedConstants.BgpPeerFlag.BGPPF_AP_RECV_IPV6_UNI,
                              GatedConstants.BgpcFlag.BGPC_V6_UNI_AP_SEND,
                              GatedConstants.BgpcFlag.BGPC_V6_UNI_AP_RECV,
                              GatedConstants.BgpPeerFlag.BGPPF_AP_SEND_IPV6_UNI )
            if v6Unicast:
               addPathsCaps.ipv6Unicast = v6Unicast

            v4LabeledUnicast = populateAddPathsCapability(
                              GatedConstants.BgpcFlag.BGPC_V4_LABELED_UNI_AP_RECV,
                              GatedConstants.BgpcFlag.BGPC_V4_LABELED_UNI_AP_SEND,
                              GatedConstants.BgpPeerFlag.
                                    BGPPF_AP_RECV_IPV4_LABELED_UNI,
                              GatedConstants.BgpcFlag.BGPC_V4_LABELED_UNI_AP_SEND,
                              GatedConstants.BgpcFlag.BGPC_V4_LABELED_UNI_AP_RECV,
                              GatedConstants.BgpPeerFlag.
                                    BGPPF_AP_SEND_IPV4_LABELED_UNI )
            if v4LabeledUnicast:
               addPathsCaps.ipv4MplsLabels = v4LabeledUnicast

            v6LabeledUnicast = populateAddPathsCapability(
                              GatedConstants.BgpcFlag.BGPC_V6_LABELED_UNI_AP_RECV,
                              GatedConstants.BgpcFlag.BGPC_V6_LABELED_UNI_AP_RECV,
                              GatedConstants.BgpPeerFlag.
                                    BGPPF_AP_RECV_IPV6_LABELED_UNI,
                              GatedConstants.BgpcFlag.BGPC_V6_LABELED_UNI_AP_SEND,
                              GatedConstants.BgpcFlag.BGPC_V6_LABELED_UNI_AP_SEND,
                              GatedConstants.BgpPeerFlag.
                                    BGPPF_AP_SEND_IPV6_LABELED_UNI )
            if v6LabeledUnicast:
               addPathsCaps.ipv6MplsLabels = v6LabeledUnicast

            neighborCaps.addPathsCaps = addPathsCaps

         return neighborCaps

      bgpPeerOptions = data[ 'bgpPeerOptions' ]
      bgpPeerOptions2 = data[ 'bgpPeerOptions2' ]
      peerOptionsNoCaps = bgpPeerOptions & GatedConstants.BgpoFlag.BGPO_NOCAPS
      hisCapsNoCaps = bgpPeerHiscaps & GatedConstants.BgpcFlag.BGPC_NO_CAPS

      if( peerOptionsNoCaps or hisCapsNoCaps ):
         capsModel = NeighborCapabilities()
         if peerOptionsNoCaps:
            capsModel.negotiationDisabled = bool( peerOptionsNoCaps )
         if hisCapsNoCaps:
            capsModel.unsupportedCapReceived = True
         data[ 'neighborCapabilities' ] = capsModel
      else:
         data[ 'neighborCapabilities' ] = renderCapabilities()

      # Others
      if data[ 'asn' ] != data[ 'localAsn' ] and \
         not bgpPeerOptions & GatedConstants.BgpoFlag.BGPO_CONFED \
            and not bgpPeerOptions2 & \
            GatedConstants.Bgpo2Flag.BGPO2_EBGP_MULTIHOP:
         data[ 'invalidNextHopDirectEbgp' ] = True

      if bgpPeerOptions & GatedConstants.BgpoFlag.BGPO_TTLSEC_MAXHOP:
         data[ 'ttlConfig' ] = 'gtsm'
      elif bgpPeerOptions2 & GatedConstants.Bgpo2Flag.BGPO2_EBGP_MULTIHOP:
         data[ 'ttlConfig' ] = 'ebgpMultiHop'
         data[ 'maxTtlHops' ] = data[ 'ttl' ]

      bfdJsonStates = {
            GatedConstants.BfdStateFlag.BFD_ADMIN_DOWN: 'adminDown',
            GatedConstants.BfdStateFlag.BFD_INIT: 'init',
            GatedConstants.BfdStateFlag.BFD_UP: 'up',
            GatedConstants.BfdStateFlag.BFD_DOWN: 'down',
            }

      if bgpPeerOptions2 & GatedConstants.Bgpo2Flag.BGPO2_BFD_ENABLE:
         data[ 'bfdSessionState' ] = bfdJsonStates.get( data.get( 'bfdState' ),
                                                        'unknown' )

      if bgpPeerOptions & GatedConstants.BgpoFlag.BGPO_PASSIVE:
         data[ 'passiveConnEnabled' ] = True

      if bgpPeerOptions2 & GatedConstants.Bgpo2Flag.BGPO2_REPLACE_AS:
         data[ 'removeOrReplacePrivateAsEgress' ] = 'removePrivateAsEgressReplace'
      elif bgpPeerOptions2 & GatedConstants.Bgpo2Flag.BGPO2_REMOVE_PRIVATE_AL:
         data[ 'removeOrReplacePrivateAsEgress' ] = 'removePrivateAsEgressAll'
      elif bgpPeerOptions & GatedConstants.BgpoFlag.BGPO_REMOVE_PRIVATE:
         data[ 'removeOrReplacePrivateAsEgress' ] = 'removePrivateAsEgress'

      return data

# pylint: disable=unnecessary-comprehension
class BgpPeerList( Model ):
   peerList = GeneratorList( valueType=BgpPeer,
                             help='Bgp Peer List ' )
   _showBfd = Bool( help='Bfd output is requested' )

   def renderBfdHeader( self ):
      print( "BGP BFD Neighbor Table" )
      print( "Flags: U - BFD is enabled for BGP neighbor "
             "and BFD session state is UP" )
      print( "       I - BFD is enabled for BGP neighbor "
             "and BFD session state is INIT" )
      print( "       D - BFD is enabled for BGP neighbor "
             "and BFD session state is DOWN" )
      print( "       N - BFD is not enabled for BGP neighbor" )
      print( "%-18s %-18s %-10s %-11s %-5s" % ( "Neighbor", "Interface",
                                                "Up/Down", "State", "Flags" ) )

   def render( self ):
      if self._showBfd:
         self.renderBfdHeader()

      peers = sorted( [ peer for peer in self.peerList ],
                      key=lambda p: PeerConfigKey( p.peerAddress ) )
      for peer in peers:
         if self._showBfd:
            peer.renderBfd()
         else:
            peer.render()
