#!/usr/bin/env python3
# Copyright (c) 2020 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import os

import Cell
import Fru
import LazyMount
import Tac
from RedSupPyClient import peerRedSupPyClient

#--------------------------------------------------------------------------------
# Sysdb helpers
#--------------------------------------------------------------------------------

# Entity MIB
def getEntityMibRoot( entityManager ):
   return LazyMount.mount( entityManager,
                           'hardware/entmib',
                           'EntityMib::Status', 'r' ).root

def isModular( entityMibRoot ):
   return "Chassis" in entityMibRoot.tag

def getActiveSupervisorCard( entityMibRoot, standby=False ):
   assert isModular( entityMibRoot )

   slotId = Fru.peerSupeSlotId() if standby else Fru.slotId()
   card = entityMibRoot.cardSlot[ slotId ].card
   return card

def getSysdbAbootVersion( entityMibRoot ):
   if isModular( entityMibRoot ):
      card = getActiveSupervisorCard( entityMibRoot )
      return card.firmwareRev
   return entityMibRoot.firmwareRev

# Redundancy status
def getRedundancyStatus( entityManager ):
   return LazyMount.mount( entityManager,
                           Cell.path( 'redundancy/status' ),
                           'Redundancy::RedundancyStatus',
                           'r' )

def populateBiosDeprecated( mode, model, standby=False ):
   sbOutdatedMsg = ( 'Secure boot will not work after May 2027 '
                     'due to certificate expiration.' )
   abootDeprecateStatus = getAbootDeprecateStatus( mode.entityManager,
                                                   standby=standby )

   if abootDeprecateStatus.sbOutdated:
      model.biosDeprecated = sbOutdatedMsg

def isActive( redundancyStatus ):
   return redundancyStatus.mode == 'active'

def isPeerStandby( redundancyStatus ):
   return redundancyStatus.peerMode == 'standby'

def hasStandby( entityMibRoot, redundancyStatus ):
   return isModular( entityMibRoot ) and isPeerStandby( redundancyStatus )

def getSecurebootSupport( entityManager, standby=False ):
   if standby and 'SIMULATION_VMID' not in os.environ:
      pc = peerRedSupPyClient()
      peerRoot = pc.root()[ pc.sysname_ ][ "Sysdb" ]
      tpmConf = \
         peerRoot[ "hardware" ][ "cell" ][ str( Cell.peerCellId() ) ]\
                 [ "tpm" ][ "config" ]
   else:
      cellId = Cell.peerCellId() if standby else Cell.cellId()
      tpmConf = LazyMount.mount( entityManager,
                              f"hardware/cell/{cellId}/tpm/config",
                              "Hardware::TpmLib::Config", "r" )
   return tpmConf.securebootSupported

# Secureboot
def getSecurebootStatus( entityManager, standby=False ):
   if standby and 'SIMULATION_VMID' not in os.environ:
      pc = peerRedSupPyClient()
      peerRoot = pc.root()[ pc.sysname_ ][ "Sysdb" ]
      sbStatus = \
         Cell.root( peerRoot, cell=Cell.peerCellId() )[ "aboot" ][ "sb" ][ "status" ]
   else:
      sbStatus = LazyMount.mount( entityManager,
                              Cell.path( "aboot/sb/status",
                              cell=Cell.peerCellId() if standby else None ),
                              "Aboot::Secureboot::Status", "r" )

   if getSecurebootSupport( entityManager, standby=standby ):
      # supported is set by the TpmLib.SecureBootSm once TPM-stored
      # configuration is loaded.
      # loadFailed is set once SecureBootSm stops trying to load configuration,
      # after a few failed tries.
      Tac.waitFor( lambda: sbStatus.loadFailed or sbStatus.supported,
                   warnAfter=None )

   return sbStatus

def isSecurebootSupported( sbStatus ):
   return sbStatus.supported

def isSPIUpdateEnabled( sbStatus ):
   return sbStatus.enableSPIUpdate == 1

def isSecurebootEnabled( sbStatus ):
   return isSecurebootSupported( sbStatus ) and not sbStatus.securebootDisabled

def isSpiLocked( sbStatus ):
   return isSecurebootSupported( sbStatus ) and not sbStatus.unlockSPI

def getAbootDeprecateStatus( entityManager, standby=False ):
   if standby and 'SIMULATION_VMID' not in os.environ:
      pc = peerRedSupPyClient()
      peerRoot = pc.root()[ pc.sysname_ ][ "Sysdb" ]
      cell = Cell.root( peerRoot, cell=Cell.peerCellId() )
      abootDeprecateStatus = cell[ "platformSecurity" ][ "abootDeprecateStatus" ]
   else:
      cellId = Cell.peerCellId() if standby else None
      cell = Cell.path( "platformSecurity/abootDeprecateStatus", cell=cellId )
      abootDeprecateStatus = LazyMount.mount(
         entityManager, cell,
         "PlatformSecurity::AbootDeprecateStatus",
         "r" )

   return abootDeprecateStatus
