#!/usr/bin/env python3
# Copyright (c) 2020 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import datetime

import BiosLib

from AbootVersion import AbootVersion, parseVersion
from BiosInstallLib import HistoryCode, historyError
from CliModel import Bool, Enum, Int, List, Model, Str, Submodel
from CliPlugin import BiosCliLib
from TableOutput import createTable, Format
from Toggles.AbootEosToggleLib import toggleShowQueuedUpdatesEnabled

flashReadErr = "Failed to read version from flash"

historyPayloadTypeNames = {
   "Aboot": "aboot",
   "CertificateBundle": "certificateBundle",
   "Other": "other",
}

def getRunningVersionAndSysName( mode, standby=False ):
   running = None
   sysName = None
   entityMibRoot = BiosCliLib.getEntityMibRoot( mode.entityManager )
   if entityMibRoot is None:
      pass
   elif BiosCliLib.isModular( entityMibRoot ):
      card = BiosCliLib.getActiveSupervisorCard( entityMibRoot, standby=standby )
      if card:
         label = int( card.label )
         sysName = "%s%d" % ( card.tag, label )

      # The firmware revision isn't always in sysdb, but if it is use it rather than
      # reading /proc/cmdline directly
      if card.firmwareRev:
         running = AbootVersion( card.firmwareRev )
      else:
         running = AbootVersion(
            BiosLib.getRunningAbootVersion( standby=standby ) )
   else:
      running = AbootVersion( entityMibRoot.firmwareRev )
      sysName = "FixedSystem"
   return ( running, sysName )

class BiosVersion( Model ):
   line = Str( help="Aboot line (eg. norcal9)" )
   version = Str( help="Version string (eg. 7.2.1)" )
   pcieLaneConfig = Str( help="PCIe lane configuration", optional=True )
   core = Str( help="Core string", optional=True )
   changeNumber = Str( help="Change number" )
   versionString = Str( help="Complete Aboot version string" )

   def toModel( self, abootVersion ):
      self.line = abootVersion.norcal
      self.version = abootVersion.version

      if abootVersion.pcieLaneConfig:
         self.pcieLaneConfig = 'pcie%s' % ( abootVersion.pcieLaneConfig )

      if abootVersion.core:
         self.core = '%dcore' % ( abootVersion.core )

      self.changeNumber = abootVersion.changeNumber

      self.versionString = str( abootVersion )

      return self

class BiosVersions( Model ):
   __revision__ = 2

   sysName = Str( help="Name of the system" )
   runningVersion = Submodel( valueType=BiosVersion,
                              help="BIOS version currently running on the switch" )
   programmedVersion = Submodel( valueType=BiosVersion,
                                 help="BIOS version programmed on the switch",
                                 optional=True )
   fallbackVersion = Submodel( valueType=BiosVersion,
                               help="BIOS fallback version programmed on the switch",
                               optional=True )
   if toggleShowQueuedUpdatesEnabled():
      queuedVersions = List( valueType=str,
                             help="Pending bios version updates(s) for the switch",
                             optional=True )

   biosDeprecated = Str( help="Information about deprecated BIOS",
                         optional=True )

   def degrade( self, dictRepr, revision ):
      if revision == 1 and 'programmedVersion' not in dictRepr:
         # Originally programmedVersion was non-optional, but it contained
         # all-optional fields. On error when reading the programmed version, an
         # optional "error" field was filled in. Instead, the version fields
         # should have been non-optional, the programmed version should be optional,
         # and we should print a normal error when we can't parse it from the flash -
         # this is what happens in revision 2.
         #
         # Nevertheless we must maintain backwards compatibility, so create an empty
         # Aboot version.
         dictRepr[ 'programmedVersion' ] = {}
      return dictRepr

   def render( self ):
      if not self.sysName:
         print( "System not yet initialized" )
         return

      if self.biosDeprecated:
         print( f"! {self.sysName} BIOS deprecation warning: {self.biosDeprecated}" )

      print( "%s BIOS versions" % self.sysName )
      t = createTable( ( "Location", "Version" ), indent=2 )
      f = Format( justify="left" )
      f.noPadLeftIs( True )
      f.padLimitIs( True )
      t.formatColumns( f, f )
      t.newRow( "Running", self.runningVersion.versionString )
      if self.programmedVersion:
         t.newRow( "Programmed", self.programmedVersion.versionString )
      if self.fallbackVersion:
         t.newRow( "Fallback", self.fallbackVersion.versionString )

      if toggleShowQueuedUpdatesEnabled():
         if not self.queuedVersions:
            t.newRow( "Queued", "There are no updates queued" )
         else:
            for count, queuedVersion in enumerate( self.queuedVersions ):
               if count == 0:
                  t.newRow( "Queued", queuedVersion )
               else:
                  t.newRow( "", queuedVersion )

      print( t.output() )

class DualSupBiosVersions( BiosVersions ):
   __revision__ = 2

   standbyVersions = Submodel( valueType=BiosVersions,
                               help="Versions on standby supervisor",
                               optional=True )

   def render( self ):
      super().render()

      if self.standbyVersions:
         self.standbyVersions.render()

class BiosHistory( Model ):
   __revision__ = 2

   class HistoryEntry( Model ):
      __revision__ = 2

      epoch = Int( help="Epoch of update" )
      name = Str( help="Name of AUF used" )
      payloadType = Enum( help="Type of AUF used",
                          values=tuple( historyPayloadTypeNames.values() ) )
      checksum = Str( help="Checksum of AUF used" )
      error = Int( help="Error code" )

      def getTimestamp( self ):
         return datetime.datetime.fromtimestamp( self.epoch ).strftime( '%c' )

      def assessType( self ):
         ver = parseVersion( self.name )
         return historyPayloadTypeNames.get( ver.product,
                                             historyPayloadTypeNames[ 'Other' ] )

      def degrade( self, dictRepr, revision ):
         if revision == 1:
            # HistoryEntry used to have a "Version" attribute containing the Aboot
            # version instead of a generic "Name" attribute.
            # This led to "Aboot-?.?.?" output for names representing something else
            # than an Aboot release string.
            # To keep compatibility, restore this behavior if necessary.
            version = BiosVersion()
            version.toModel( AbootVersion( self.name or "" ) )
            dictRepr[ 'version' ] = version.toDict()
            dictRepr.pop( 'name', None )
            dictRepr.pop( 'payloadType', None )
         return dictRepr

   sysName = Str( help="Name of the system" )
   history = List( valueType=HistoryEntry, help="List of installation history" )
   _detail = Bool( help='Render detailed installation history', optional=True )

   def degrade( self, dictRepr, revision ):
      if revision == 1:
         # See HistoryEntry degrade comment
         degradedHistory = []
         for i in range( 0, len( self.history ) ):
            degradedHistory.append(
                  self.history[ i ].degrade( dictRepr[ 'history' ][ i ], 1 ) )
         dictRepr[ 'history' ] = degradedHistory
      return dictRepr

   def insert( self, epoch, name, checksum, error ):
      historyEntry = self.HistoryEntry()
      historyEntry.epoch = epoch
      historyEntry.checksum = checksum
      historyEntry.name = name
      historyEntry.payloadType = historyEntry.assessType()
      historyEntry.error = error
      self.history.insert( 0, historyEntry )

   def render( self ):
      print( "%s BIOS version history\n" % ( self.sysName ) )

      tableHeaders = ( "Timestamp", "Name", "Type" )
      if self._detail:
         tableHeaders += ( "Checksum", )
      t = createTable( tableHeaders )
      f = Format( justify="left" )
      f.noPadLeftIs( True )
      f.padLimitIs( True )
      t.formatColumns( *( f for _ in tableHeaders ) )

      # Display first error
      try:
         h = self.history[ 0 ]
         historyErrorString = historyError( h.error )
         if historyErrorString:
            print( 'Last update failed on {}: {}.\n'.format( h.getTimestamp(),
                                                         historyErrorString ) )
            if self._detail:
               t.newRow( h.getTimestamp(), h.name, h.payloadType, h.checksum )
            else:
               t.newRow( h.getTimestamp(), h.name, h.payloadType )
      except IndexError:
         # No history
         pass

      for h in self.history:
         if h.error == HistoryCode.FLASH_SUCCESS:
            if self._detail:
               t.newRow( h.getTimestamp(), h.name, h.payloadType, h.checksum )
            else:
               t.newRow( h.getTimestamp(), h.name, h.payloadType )

      print( t.output() )

class DualSupBiosHistory( BiosHistory ):
   __revision__ = 2

   standbyHistory = Submodel( valueType=BiosHistory,
                              help="History on standby supervisor",
                              optional=True )

   def render( self ):
      super().render()

      if self.standbyHistory:
         self.standbyHistory.render()

   def degrade( self, dictRepr, revision ):
      standbyRepr = None
      if revision == 1:
         if self.standbyHistory:
            standbyRepr = self.standbyHistory.degrade( dictRepr[ 'standbyHistory' ],
                                                       revision )
         dictRepr = super().degrade( dictRepr, revision )
         if standbyRepr:
            dictRepr[ 'standbyHistory' ] = standbyRepr
      return dictRepr
