# Copyright (c) 2018 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import Cell
from CliModel import Model, Dict, Bool, Int, Str
import CliParser
import ConfigMount, LazyMount
import Tac
import TableOutput
import Tracing
import os.path
import subprocess
import time

blackboxCliConfig = None
blackboxHwDir = None

postFileIdBase = 100

t0 = Tracing.Handle( "Blackbox" ).trace0

def blackboxGuard( mode, token ):
   if 'config' in blackboxHwDir.entryState and blackboxHwDir[ 'config' ].supported:
      return None

   return CliParser.guardNotThisPlatform

def enableBlackbox( mode, args ):
   blackboxCliConfig.enabled = True

def disableBlackbox( mode, args ):
   blackboxCliConfig.enabled = False

def doEraseBlackbox():
   # Previous attempts to change this function were made to set 
   # blackboxCliConfig.enabled = False in order to get the BlackBox SM 
   # to react and erase the FRAM itself. This proves to be an issue on 
   # systems with multiple supervisors in both RPR and SSO modes. The
   # BlackBox SM does not take into account either of these cases and will
   # fail when trying to execute on the standby supervisor. 
   
   if blackboxHwDir and 'status' in blackboxHwDir.entryState:
      blackboxInventoried = blackboxHwDir[ 'status' ].inventoried
      if not blackboxInventoried:
         # Even if Blackbox is supported, until it is inventoried, initAloha
         # may not be able to determine the ham information.
         return False
      try:
         alohaUtilType = Tac.Type( "Blackbox::AlohaUtil" )
         alohaUtil = alohaUtilType( "BlackBox", blackboxHwDir[ 'config' ] )
         alohaUtil.initAloha()
         eraseCmd = [ "timeout", "-s9", "30", "ahaDumpBlackbox", "-e", "-m",
                      "-a", str( alohaUtil.addressBaseOffset ),
                      "--accelRegStride", str( alohaUtil.regStride ) ]
         if alohaUtil.hamType == "Pci":
            eraseCmd.append( "-s" )
            eraseCmd.append( str( alohaUtil.pciScd ) )
         elif alohaUtil.hamType == "MemMapped":
            eraseCmd.append( "-w" )
            eraseCmd.append( str( alohaUtil.hamMemOffset ) )
         if alohaUtil.hamType != "Sim":
            Tac.run( eraseCmd,
                     stdout=Tac.CAPTURE,
                     asRoot=True )
      # pylint: disable-next=try-except-raise
      except ( Tac.SystemCommandError, Tac.Timeout, SystemError ):
         raise
      return True
   else:
      return False

#-------------------------------------------------------------------------------
# show reload console status
#-------------------------------------------------------------------------------
class ConsoleLogFile( Model ):
   idVal = Int( help="ConsoleLog file id" )
   fileName = Str( help="ConsoleLog file name" )
   dumpTime = Str( help="ConsoleLog save time" )

class BlackboxStatusDetail( Model ):
   blackboxSupported = Bool( help="Console logging supported" )
   blackboxPowerOnRecordingSupported = Bool(
      help="Console logging supported during power-on sequence" )
   blackboxEnabled = Bool( help="Console logging enabled" )
   consoleLogFiles = Dict( keyType=int, valueType=ConsoleLogFile,
                           help="Available ConsoleLog files" )
   blackboxProductId = Int( help="FRAM Product ID" )
   blackboxManufacturer = Str( help="FRAM Manufacturer" )

   def render( self ):
      print( "Console logging:", end=' ' )
      if self.blackboxSupported:
         print( "Supported" )
      else:
         print( "Not supported" )
      print( "Power-on logging:", end=' ' )
      if self.blackboxPowerOnRecordingSupported:
         print( "Supported" )
      else:
         print( "Not supported" )
      print( "Recording:", end=' ' )
      if self.blackboxEnabled:
         print( "Enabled" )
      else:
         print( "Not enabled" )
      if self.blackboxSupported:
         # pylint: disable-next=consider-using-f-string
         print( "Manufacturer: {} (0x{:x})".format( self.blackboxManufacturer,
                                                    self.blackboxProductId ) )

      # Even if Blackbox console log files are present, the CLI
      # will hide them if Blackbox recording is disabled.
      if self.blackboxSupported and self.blackboxEnabled:
         self.renderConsoleLogFiles()

   def renderConsoleLogFiles( self ):
      print()
      print( 'Console log files:' )
      print()

      tableHeadings = ( "Id", "Name", "Save Time" )
      idFormat = TableOutput.Format( justify='right' )
      idFormat.padLimitIs( True )
      fnFormat = TableOutput.Format( justify='left' )
      fnFormat.padLimitIs( True )
      dtFormat = TableOutput.Format( justify='left' )
      dtFormat.padLimitIs( True )

      table = TableOutput.createTable( tableHeadings )
      table.formatColumns( idFormat, fnFormat, dtFormat )

      for idVal in sorted( self.consoleLogFiles ):
         bbFile = self.consoleLogFiles[ idVal ]
         table.newRow( idVal, bbFile.fileName, bbFile.dumpTime )
      print( table.output() )

def doShowReloadConsoleStatus( mode, args ):
   blackboxEnabled = False
   blackboxSupported = False
   blackboxPowerOnRecordingSupported = False
   consoleLogFiles = {}
   blackboxProductId = 0
   blackboxManufacturer = ""

   if blackboxHwDir and 'config' in blackboxHwDir.entryState:
      blackboxSupported = blackboxHwDir[ 'config' ].supported
      if blackboxSupported and 'status' in blackboxHwDir.entryState:
         bbStatus = blackboxHwDir[ 'status' ]
         blackboxEnabled = bbStatus.enabled
         blackboxPowerOnRecordingSupported = bbStatus.powerOnRecordingSupported
         blackboxProductId = bbStatus.framInfo.productId
         blackboxManufacturer = str( bbStatus.framInfo.mfrName )

         if blackboxEnabled:
            for idVal, bbFile in bbStatus.file.items():
               # Don't expose *post* files via CLI. When dumping the corresponding
               # base file, the *post* file will be automatically appended to the
               # output if it exists.
               if idVal >= postFileIdBase:
                  continue
               dumpTime = time.strftime( '%Y-%m-%d %H:%M:%S',
                  time.localtime( bbFile.dumpTime ) )
               consoleLogFile = ConsoleLogFile( idVal=idVal,
                  fileName=bbFile.fileName, dumpTime=dumpTime )
               consoleLogFiles[ idVal ] = consoleLogFile

   return BlackboxStatusDetail( 
      blackboxSupported=blackboxSupported,
      blackboxPowerOnRecordingSupported=blackboxPowerOnRecordingSupported,
      blackboxEnabled=blackboxEnabled,
      consoleLogFiles=consoleLogFiles,
      blackboxProductId=blackboxProductId,
      blackboxManufacturer=blackboxManufacturer )

#-------------------------------------------------------------------------------
# show reload console logs [ FILENUM ]
#-------------------------------------------------------------------------------
class ConsoleOutput( Model ):
   consoleLog = Str( help="Console log" )

   def render( self ):
      print( self.consoleLog )

def doShowPersistentLogs( mode, args ):
   assert 'status' in blackboxHwDir.entryState
   status = blackboxHwDir[ 'status' ]

   fileNum = args.get( 'FILENUM' )
   if fileNum:
      if fileNum not in status.file:
         raise CliParser.InformationalShowCmdError(
            "Specified file number not available" )
   else:
      if not status.file:
         raise CliParser.InformationalShowCmdError(
            "No persistent logs are available." )
      # By default, the lowest file number is chosen
      fileNum = min( status.file )

   fileName = status.file[ fileNum ].fileName
   filePath = os.path.join( status.flashDir, fileName )

   if not os.path.isfile( filePath ):
      raise CliParser.InformationalShowCmdError(
         # pylint: disable-next=consider-using-f-string
         'Could not read persistent log file %s' % filePath )
   # Tac.run normalizes newlines, so use subprocess instead.
   output = subprocess.check_output(
      [ '/usr/bin/ahaCatBlackbox', filePath ]
   ).decode( errors='backslashreplace' )

   if status.powerOnRecordingSupported:
      # append the post-power-on logs so that if the feature is supported, 
      # there is a full view of what has happened since the last reboot
      postFileNum = fileNum + postFileIdBase
      fileName = status.file[ postFileNum ].fileName
      filePath = os.path.join( status.flashDir, fileName )

      output += "\n"
      if os.path.isfile( filePath ):
         output += subprocess.check_output(
            [ '/usr/bin/ahaCatBlackbox', filePath ]
         ).decode( errors='backslashreplace' )
      else:
         # a raised exception at this point will hurt more than help. Instead
         # add a trace saying that there is no *post* log to report
         t0( "Power On Recording is supported, but console log ", 
             filePath, "not found" )

   return ConsoleOutput( consoleLog=output )

##-------------------------------------------------------------------------------
## Plugin
##-------------------------------------------------------------------------------
def Plugin( entityManager ):
   global blackboxCliConfig, blackboxHwDir
   blackboxCliConfig = ConfigMount.mount( entityManager,
      'hardware/blackbox/cliConfig', 'Hardware::Blackbox::CliConfig', 'w' )
   blackboxHwDir = LazyMount.mount( entityManager,
      # pylint: disable-next=consider-using-f-string
      'cell/%d/hardware/blackbox' % Cell.cellId(), 'Tac::Dir', 'ri' )

