# Copyright (c) 2016 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import Arnet.TcpUtils
import BasicCli
import BmpUtils
import CliCommand
from CliDynamicSymbol import LazyCallback
import CliMatcher
import CliParser
import ConfigMount
import Tac
import Tracing
import Toggles.BmpToggleLib
from BgpLib import (
      isValidPeerAddr,
      isValidV6PeerAddr,
)
from CliMode.Bmp import RoutingBmpStationMode
from CliPlugin.RoutingBgpCli import (
      bgpMatcherForConfig,
      BgpCmdBaseClass,
      RouterBgpVrfMode,
      RouterBgpDefaultVrfMode,
      RouterBgpBaseMode,
      RouterBgpSharedModelet,
      deleteRouterBgpVrfHook,
      PeerCliExpression,
)
import CliToken.RoutingBgp as bgpTokens

# Tracing
traceHandle = Tracing.Handle( 'Bmp' )
t0 = traceHandle.trace0
t1 = traceHandle.trace1
t2 = traceHandle.trace2

# Constants
U32_MAX_VALUE =  0xFFFFFFFF

# Tac Entities
bmpConfig = None

# Enum Type
ConnectionMode = Tac.Type( 'Routing::Bmp::ConnectionMode' )
TimestampMode = Tac.Type( 'Routing::Bmp::TimestampMode' )
IpAddrOrHostname = Tac.Type( 'Routing::Bmp::IpAddrOrHostname' )

matcherBgp = CliMatcher.KeywordMatcher(
   'bgp', helpdesc='BGP monitoring protocol BGP options' )
matcherRib = CliMatcher.KeywordMatcher(
   'rib', helpdesc='BGP monitoring protocol RIB selection' )
matcherBestPaths = CliMatcher.KeywordMatcher(
   'bestpaths', helpdesc='BGP monitoring protocol best paths selection' )

class RouterBmpStationMode( RoutingBmpStationMode, BasicCli.ConfigModeBase ):
   name = 'Bgp monitoring station configuration'

   def __init__( self, parent, session, bmpStationName ):
      self.bmpStationName = bmpStationName
      RoutingBmpStationMode.__init__( self, bmpStationName )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

#-------------------------------------------------------------------------------
# "monitoring station" config mode
#-------------------------------------------------------------------------------
class RouterBmpStationModelet( CliParser.Modelet ):
   def setBmpStationDescription( self, desc ):
      config = bmpConfig.bmpStation.get( self.mode.bmpStationName )
      config.description = desc

   def noBmpStationDescription( self ):
      config = bmpConfig.bmpStation.get( self.mode.bmpStationName )
      if config:
         config.description = config.descriptionDefault

   def setShutdown( self, msg=None ):
      config = bmpConfig.bmpStation.get( self.mode.bmpStationName )
      config.shutdownMsg = msg if msg else config.shutdownMsgDefault
      config.shutdown = True

   def noShutdown( self ):
      config = bmpConfig.bmpStation.get( self.mode.bmpStationName )
      config.shutdownMsg = config.shutdownMsgDefault
      config.shutdown = False

   def setBmpStationKey( self, authPasswd ):
      config = bmpConfig.bmpStation.get( self.mode.bmpStationName )
      config.authenticationKey = authPasswd

   def noBmpStationKey( self ):
      config = bmpConfig.bmpStation.get( self.mode.bmpStationName )
      if config:
         config.authenticationKey = config.authenticationKeyDefault

   def setStatistics( self, statistics ):
      config = bmpConfig.bmpStation.get( self.mode.bmpStationName )
      config.statistics = statistics

   def noStatistics( self ):
      config = bmpConfig.bmpStation.get( self.mode.bmpStationName )
      if config:
         config.statistics = config.statisticsDefault

   def sourceIntf( self, intf ):
      config = bmpConfig.bmpStation.get( self.mode.bmpStationName )
      config.sourceIntf = intf.name

   def noSourceIntf( self ):
      config = bmpConfig.bmpStation.get( self.mode.bmpStationName )
      if config:
         config.sourceIntf = config.sourceIntfDefault

   def setRemoteAddr( self, address, vrfName ):
      config = bmpConfig.bmpStation.get( self.mode.bmpStationName )
      if vrfName:
         config.vrfName = vrfName
      else:
         config.vrfName = config.vrfNameDefault

      if isValidV6PeerAddr( address ) or isValidPeerAddr( address ):
         ipGenAddr = Arnet.IpGenAddr( str( address ) )
         config.remoteHost = IpAddrOrHostname.fromIp( ipGenAddr )
      else:
         config.remoteHost = IpAddrOrHostname.fromHostname( address )

   def noRemoteAddr( self ):
      config = bmpConfig.bmpStation.get( self.mode.bmpStationName )
      if config:
         config.remoteHost = config.remoteHostDefault
         config.vrfName = config.vrfNameDefault

   def setActiveMode( self, port, reconnectTime ):
      config = bmpConfig.bmpStation.get( self.mode.bmpStationName )
      config.connectionMode = ConnectionMode.active
      config.remotePort = port
      if reconnectTime:
         config.reconnectTime = reconnectTime
      else:
         config.reconnectTime = config.reconnectTimeDefault

   def setPassiveMode( self ):
      config = bmpConfig.bmpStation.get( self.mode.bmpStationName )
      config.connectionMode = ConnectionMode.passive
      config.remotePort = config.remotePortDefault
      config.reconnectTime = config.reconnectTimeDefault

   def noMode( self ):
      config = bmpConfig.bmpStation.get( self.mode.bmpStationName )
      if config:
         config.connectionMode = ConnectionMode.unknown
         config.remotePort = config.remotePortDefault
         config.reconnectTime = config.reconnectTimeDefault

   def setTcpKeepalive( self, idleTime, probeInterval, probeCount ):
      config = bmpConfig.bmpStation.get( self.mode.bmpStationName )

      if config:
         keepAlive = Arnet.TcpUtils.TcpKeepaliveOptions(
                        idleTime, probeInterval, probeCount )
         config.keepaliveOption = keepAlive

   def noTcpKeepalive( self ):
      config = bmpConfig.bmpStation.get( self.mode.bmpStationName )

      if config:
         config.keepaliveOption = config.keepaliveOptionDefault

   def setBmpReceivedRoutes( self, exportPolicies ):
      config = bmpConfig.bmpStation.get( self.mode.bmpStationName )

      prePolicy = 'pre-policy' in exportPolicies
      postPolicy = 'post-policy' in exportPolicies

      config.ribInExportPolicyStationConfig = \
         BmpUtils.BmpRibInExportPolicyConfig( prePolicy, postPolicy )

   def defaultBmpReceivedRoutes( self ):
      config = bmpConfig.bmpStation.get( self.mode.bmpStationName )
      config.ribInExportPolicyStationConfig = \
         BmpUtils.BmpRibInExportPolicyConfigUnset()

   def noBmpReceivedRoutes( self ):
      config = bmpConfig.bmpStation.get( self.mode.bmpStationName )
      config.ribInExportPolicyStationConfig = \
         BmpUtils.BmpRibInExportPolicyConfig( False, False )

   def enableLocRibExport( self ):
      config = bmpConfig.bmpStation.get( self.mode.bmpStationName )
      config.locRibExportPolicyStationConfig = (
         BmpUtils.BmpLocRibExportPolicyConfig( True ) )

   def noLocRibExport( self ):
      config = bmpConfig.bmpStation.get( self.mode.bmpStationName )
      config.locRibExportPolicyStationConfig = (
            BmpUtils.BmpLocRibExportPolicyConfig( False ) )

   def defaultLocRibExport( self ):
      config = bmpConfig.bmpStation.get( self.mode.bmpStationName )
      config.locRibExportPolicyStationConfig = (
            BmpUtils.BmpLocRibExportPolicyConfigUnset() )

   def noBmpAdditionalPaths( self ):
      config = bmpConfig.bmpStation.get( self.mode.bmpStationName )
      config.addPathIdConfigured = False

   def setBmpAdditionalPaths( self ):
      config = bmpConfig.bmpStation.get( self.mode.bmpStationName )
      config.addPathIdConfigured = True

   def setBmpAdvertisedRoutes( self, exportPolicies ):
      config = bmpConfig.bmpStation.get( self.mode.bmpStationName )

      postPolicy = 'post-policy' in exportPolicies

      config.ribOutExportPolicyStationConfig = \
         BmpUtils.BmpRibOutExportPolicyConfig( postPolicy )

   def defaultBmpAdvertisedRoutes( self ):
      config = bmpConfig.bmpStation.get( self.mode.bmpStationName )
      config.ribOutExportPolicyStationConfig = \
         BmpUtils.BmpRibOutExportPolicyConfigUnset()

   def noBmpAdvertisedRoutes( self ):
      config = bmpConfig.bmpStation.get( self.mode.bmpStationName )
      config.ribOutExportPolicyStationConfig = \
         BmpUtils.BmpRibOutExportPolicyConfig( False )

RouterBmpStationMode.addModelet( RouterBmpStationModelet )

def gotoBmpStationMode( mode, bmpStationName ):
   bmpStationConfig = bmpConfig.bmpStation.get( bmpStationName )
   if not bmpStationConfig:
      bmpConfig.bmpStation.newMember( bmpStationName )
   childMode = mode.childMode( RouterBmpStationMode,
                               bmpStationName=bmpStationName )
   mode.session_.gotoChildMode( childMode )

def deleteBmpStationMode( mode, bmpStationName ):
   if not bmpStationName in bmpConfig.bmpStation:
      return
   del bmpConfig.bmpStation[ bmpStationName ]
   # pylint: disable-next=consider-using-f-string
   t0( 'Delete Bgp monitoring station %s' % bmpStationName )

#-------------------------------------------------------------------------------
# "bgp monitoring [ station <s1> <s2>+ ]" command in router bgp config mode
#-------------------------------------------------------------------------------
class RouterBgpMonitoring( CliCommand.CliCommandClass ):
   syntax = 'bgp monitoring'
   noOrDefaultSyntax = 'bgp monitoring'
   data = {
      'bgp': bgpMatcherForConfig,
      'monitoring': 'Enable Bgp monitoring for all/specified stations',
   }
   handler = "BmpCliHandler.handlerRouterBgpMonitoring"
   noOrDefaultHandler = "BmpCliHandler.noOrDefaultHandlerRouterBgpMonitoring"

RouterBgpSharedModelet.addCommandClass( RouterBgpMonitoring )

# -------------------------------------------------------------------------------
# "[no|default] bgp rib bestpaths monitoring [ disabled ]"
# command, in "router-bgp" mode.
# -------------------------------------------------------------------------------
class RouterBgpRibBestPathMonitoringBase( BgpCmdBaseClass ):
   syntax = 'bgp rib bestpaths monitoring [ disabled ]'
   noOrDefaultSyntax = syntax

   data = BgpCmdBaseClass._createSyntaxData( {
      'bgp': bgpMatcherForConfig,
      'rib': matcherRib,
      'bestpaths': matcherBestPaths,
      'monitoring': 'Enable BGP RIB monitoring for all stations',
   } )
   handler = "BmpCliHandler.handlerRouterBgpRibBestPathMonitoringBase"
   noOrDefaultHandler = \
      "BmpCliHandler.noOrDefaultHandlerRouterBgpRibBestPathMonitoringBase"

class RouterBgpRibBestPathMonitoringVrf( BgpCmdBaseClass ):
   syntax = 'bgp rib bestpaths monitoring [ disabled ]'
   noOrDefaultSyntax = syntax

   data = BgpCmdBaseClass._createSyntaxData( {
      'bgp': bgpMatcherForConfig,
      'rib': matcherRib,
      'bestpaths': matcherBestPaths,
      'monitoring': 'Enable BGP RIB monitoring for all stations',
   } )
   handler = "BmpCliHandler.handlerRouterBgpRibBestPathMonitoringVrf"
   noOrDefaultHandler = \
      "BmpCliHandler.noOrDefaultHandlerRouterBgpRibBestPathMonitoringVrf"

if Toggles.BmpToggleLib.toggleBmpLocRibEnabled():
   RouterBgpBaseMode.addCommandClass( RouterBgpRibBestPathMonitoringBase )
   RouterBgpDefaultVrfMode.addCommandClass( RouterBgpRibBestPathMonitoringVrf )
   RouterBgpVrfMode.addCommandClass( RouterBgpRibBestPathMonitoringVrf )

# -------------------------------------------------------------------------------
# "[ no | default ] bgp rib received routes monitoring disabled"
# command in "router-bgp-vrf" mode.
# -------------------------------------------------------------------------------
class RouterBgpVrfAdjRibinMonitorDisabled( CliCommand.CliCommandClass ):
   syntax = 'bgp rib received routes monitoring disabled'
   noOrDefaultSyntax = syntax
   data = {
      'bgp': bgpMatcherForConfig,
      'rib': matcherRib,
      'received': 'BGP monitoring protocol received route selection',
      'routes': 'BGP monitoring protocol route selection',
      'monitoring': 'Export BGP routes in VRF',
      'disabled': 'Disable exporting BGP routes',
   }
   handler = "BmpCliHandler.handlerRouterBgpVrfAdjRibinMonitorDisabled"
   noOrDefaultHandler = \
      "BmpCliHandler.noOrDefaultHandlerRouterBgpVrfAdjRibinMonitorDisabled"

if Toggles.BmpToggleLib.toggleBmpAdjRibinVrfFilteringEnabled():
   RouterBgpDefaultVrfMode.addCommandClass( RouterBgpVrfAdjRibinMonitorDisabled )
   RouterBgpVrfMode.addCommandClass( RouterBgpVrfAdjRibinMonitorDisabled )

# -------------------------------------------------------------------------------
# "[no|default] neighbor PEER monitoring"
# command, in "router-bgp" mode.
#-------------------------------------------------------------------------------
class RouterBgpNeighborBmp( CliCommand.CliCommandClass ):
   syntax = 'neighbor PEER monitoring'
   noOrDefaultSyntax = syntax
   data = {
      'neighbor': bgpTokens.neighbor,
      'PEER': PeerCliExpression,
      'monitoring': 'Enable BGP Monitoring Protocol for this peer',
   }
   handler = "BmpCliHandler.handlerRouterBgpNeighborBmp"
   noHandler = "BmpCliHandler.noHandlerRouterBgpNeighborBmp"
   defaultHandler = "BmpCliHandler.defaultHandlerRouterBgpNeighborBmp"

RouterBgpSharedModelet.addCommandClass( RouterBgpNeighborBmp )

def Plugin( entityManager ):
   global bmpConfig

   bmpConfig = ConfigMount.mount( entityManager, 'routing/bmp/config',
                                  'Routing::Bmp::BmpConfig', 'w' )
   deleteRouterBgpVrfHook.addExtension(
      LazyCallback( "BmpCliHandler.cleanupBmpConfig" ) )
