# Copyright (c) 2005-2009, 2010, 2011 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# pylint: disable=consider-using-f-string

"""
"boot" commands modify the Aboot boot loader settings stored in
flash:boot-config.  These commands are special because they:

* modify boot-config immediately
* do not affect Sysdb state
* are not reflected in running-config or startup-config
"""

from distutils import version
import functools
import os
import sys
import tempfile

import BasicCli
import BasicCliModes
import Cell
import CliCommand
import CliMatcher
import CliParser
import CliPlugin.BmcCli
import CliPlugin.TechSupportCli
from CliPlugin import SysMgrModels
import CliSession
import CliSessionDataStore
import LazyMount
import Logging
import FileUrl
import FirmwareRev
import SecretCli
import ShowCommand
import SimpleConfigFile
import Tac
import Url

# pkgdeps: library SecureBoot

SYS_BOOT_FAILED_UPDATE_BOOT_IMAGE = Logging.LogHandle(
              "SYS_BOOT_FAILED_UPDATE_BOOT_IMAGE",
              severity=Logging.logError,
              fmt="There was an issue with updating the boot image.",
              explanation="Updating the boot image failed, due to an issue "
              "specified in the 'boot system' CLI command",
              recommendedAction="Check the output of the 'boot system' CLI "
              "command to see what the error is." )

matcherBoot = CliMatcher.KeywordMatcher( 'boot',
                         helpdesc='System boot configuration' )

abootSbStatus = None

class BootImageException( Exception ):
   pass

def getStats( url ):
   stats = os.statvfs( url.fs.location_ )
   free = stats.f_frsize * stats.f_bavail
   return free

def bootImageUrl( mode ):
   context = Url.Context( *Url.urlArgsFromMode( mode ) )
   return Url.parseUrl( 'flash:/.boot-image.swi', context )

# The following are functions registered for config session data store -
# See CliSession/CliSessionDataStore.py. The purpose is to allow boot commands
# to act sort of like config commands that inside a config session they are
# committed only when the session is committed, while boot commands themselves
# don't show up in running-config (not really config commands).

# special value
NOFILE = 1

class BootConfigSessionStore( CliSessionDataStore.DataStore ):
   def __init__( self ):
      self.data_ = {}
      self.bootConfigFileName_ = FileUrl.localBootConfig( None, False ).realFilename_
      self.backupFiles_ = {}
      self.testError_ = {} # for testing
      # Get boot config
      CliSessionDataStore.DataStore.__init__( self )

   def setValue( self, mode, name, value ):
      sessionName = CliSession.currentSession( mode.entityManager )
      if sessionName:
         self.data_[ name ] = value
      else:
         # Directly set boot-config. Caller handles exceptions.
         config = FileUrl.bootConfig( mode, createIfMissing=True )
         if value is None:
            config.pop( name, None )
         else:
            config[ name ] = value

   def backup( self, mode, sessionName ):
      if not self.data_:
         return True
      bootFile = self.bootConfigFileName_
      if os.path.exists( bootFile ):
         # Generate a backup of the existing boot config
         fd = None
         backupFile = None
         try:
            self.checkTestError( 'backup' )

            fd, backupFile = tempfile.mkstemp( prefix='%s.' % bootFile )
            with open( bootFile, 'rb' ) as f:
               os.write( fd, f.read() )
            self.backupFiles_[ sessionName ] = backupFile
         except OSError as e:
            # cleanup
            if backupFile:
               try:
                  os.unlink( backupFile )
               except OSError:
                  pass
            mode.addError( "Failed to backup boot-config: %s" % e )
            return False
         finally:
            # close the fd regardless
            if fd is not None:
               os.close( fd )
      else:
         # no boot-config, remember this
         self.backupFiles_[ sessionName ] = NOFILE
      return True

   def restore( self, mode, sessionName ):
      backupFile = self.backupFiles_.get( sessionName )
      if backupFile:
         bootFile = self.bootConfigFileName_
         try:
            if backupFile is NOFILE:
               os.unlink( bootFile )
            else:
               os.rename( backupFile, bootFile )
         except OSError as e:
            mode.addWarning( "Failed to restore boot-config: %s" %
                             e.strerror )
      return True

   def commit( self, mode, sessionName ):
      if not self.data_:
         return True
      try:
         self.checkTestError( 'commit' )

         config = FileUrl.bootConfig( mode, createIfMissing=True )
         for n, v in self.data_.items():
            if v is None:
               config.pop( n, None )
            else:
               config[ n ] = v
         return True
      except ( OSError, SimpleConfigFile.ParseError ) as e:
         if 'SWI' in self.data_:
            Logging.log( SYS_BOOT_FAILED_UPDATE_BOOT_IMAGE )
         mode.addError( "Failed to commit boot-config: %s" % e )
         return False

   def cleanup( self, mode, sessionName ):
      self.data_ = {}
      backupFile = self.backupFiles_.pop( sessionName, None )
      if backupFile and backupFile is not NOFILE:
         try:
            os.unlink( backupFile )
         except OSError:
            pass
      return True

   def testErrorIs( self, name, error ):
      self.testError_[ name ] = error

   def checkTestError( self, name ):
      error = self.testError_.pop( name, None )
      if error:
         raise error

dataStore_ = BootConfigSessionStore()

#------------------------------------------------------------------------------------
# boot system <url>
#------------------------------------------------------------------------------------
def setSoftwareImageUrl( mode, args ):
   url = args[ 'URL' ]
   try:
      # Try opening the file, to ensure that it exists and isn't a directory.
      url.open()

      if not FileUrl.urlIsSwiFile( url ):
         raise BootImageException( "%s is not a valid EOS system image" %
                                   ( url.url ) )

      # Check if the image being set to boot is compatible (ReloadPolicy checks
      # pass, swEpoch >= hwEpoch and if 2Gb image, it supports the hardware)
      # with the hardware
      # pylint: disable-next=import-outside-toplevel
      import CliPlugin.ReloadImageEpochCheckCli as epochHook
      # pylint: disable-next=import-outside-toplevel
      import CliPlugin.ReloadPolicyCheckCli as rpHook

      # ReloadPolicy check includes SWI signature check and factors in whether
      # or not SecureBoot is enabled in that check
      rpComp, rpResult = rpHook.checkReloadPolicy( mode, url.realFilename_ )
      for rpWarning in rpResult.warnings:
         mode.addWarning( rpWarning )
      if not rpComp:
         rpError = '\n'.join( rpResult.errors )
         raise BootImageException( rpError )

      epochComp, epochErr = epochHook.checkImageEpochAllowed( mode,
                                                              url.realFilename_,
                                                              url.url )
      if not epochComp:
         raise BootImageException( epochErr )
      Tac.run( [ 'SyncFile', url.realFilename_ ], asRoot=True )
      dataStore_.setValue( mode, 'SWI', str( url ) )

   except ( OSError, SimpleConfigFile.ParseError ) as e:
      mode.addError( "Error setting software image to %s (%s)" % ( url.url, e ) )
      Logging.log( SYS_BOOT_FAILED_UPDATE_BOOT_IMAGE )
   except BootImageException as e:
      mode.addError( str( e ) )
      Logging.log( SYS_BOOT_FAILED_UPDATE_BOOT_IMAGE )

#--------------------------------------------------------------------------------
# boot system URL
#--------------------------------------------------------------------------------
class BootSystemCmd( CliCommand.CliCommandClass ):
   syntax = 'boot system URL'
   data = {
      'boot': matcherBoot,
      'system': 'Software image URL',
      'URL': Url.UrlMatcher( fsFunc=lambda fs: ( fs.fsType == 'flash' and
                                                 fs.scheme != 'file:' ),
                             helpdesc='Software image URL' ),
   }
   handler = setSoftwareImageUrl

BasicCliModes.GlobalConfigMode.addCommandClass( BootSystemCmd )

#------------------------------------------------------------------------------------
# [no] boot console speed <baud>
#------------------------------------------------------------------------------------
def setConsoleSpeed( mode, args ):
   speed = args[ 'BAUD' ]
   try:
      dataStore_.setValue( mode, 'CONSOLESPEED', speed )
   except ( OSError, SimpleConfigFile.ParseError ) as e:
      mode.addError( "Error setting console port speed to %s (%s)" % ( speed, e ) )

def clearConsoleSpeed( mode, args ):
   try:
      dataStore_.setValue( mode, 'CONSOLESPEED', None )
   except ( OSError, SimpleConfigFile.ParseError ) as e:
      mode.addError( "Error clearing console speed (%s)" % e )

#--------------------------------------------------------------------------------
# [ no | default ] boot console speed <BAUD>
#--------------------------------------------------------------------------------
class BootConsoleSpeedCmd( CliCommand.CliCommandClass ):
   syntax = 'boot console speed BAUD'
   noOrDefaultSyntax = 'boot console speed ...'
   data = {
      'boot': matcherBoot,
      'console': 'Console port settings',
      'speed': 'Console port speed',
      'BAUD': CliMatcher.PatternMatcher(
               pattern='(1200|2400|4800|9600|19200|38400|57600|115200)',
               helpdesc=
               'Console port speed (1200, 2400, 4800, 9600, 19200, 38400, 57600 '
               'and 115200)',
               helpname='baud' ),
   }
   handler = setConsoleSpeed
   noOrDefaultHandler = clearConsoleSpeed

BasicCliModes.GlobalConfigMode.addCommandClass( BootConsoleSpeedCmd )

#------------------------------------------------------------------------------------
# [no] boot secret { [0] <cleartext-passwd> | 5 <encrypted-passwd> }
#------------------------------------------------------------------------------------
@Tac.memoize
def bootSecretSha512Supported():
   abootRawVersion = FirmwareRev.abootFirmwareRevNumbers()
   abootParsedVersion = version.StrictVersion( "1.0.0" )
   if abootRawVersion:
      # There is a issue with version.StrictVersion being passed
      # an empty string.
      try:
         abootParsedVersion = version.StrictVersion( abootRawVersion )
      except ValueError:
         pass
   # Added SHA password support in Aboot 2.0.8
   # The mock variable is for testing purposes
   if abootParsedVersion >= version.StrictVersion( "2.0.8" ) or \
      "MOCK_ABOOT_2.0.8" in os.environ:
      return True
   return False

def bootSecretSha512SupportedGuard( mode, token ):
   return None if bootSecretSha512Supported() else CliParser.guardNotThisAbootVersion

def bootSecretScryptSupportedGuard( mode, token ):
   return CliParser.guardNotThisAbootVersion

def bootSecretYescryptSupportedGuard( mode, token ):
   return CliParser.guardNotThisAbootVersion

def tpmPasswordConfigured( mode ):
   if not abootSbStatus.supported or abootSbStatus.tpmPassword == 0:
      return False

   mode.addError( "This switch uses the TPM to store the Aboot password.\n"
                  "Please use 'securebootctl abootpassword -enable|-disable' "
                  "from the Aboot shell to change the password." )
   return True

class BootSecretCommand( CliCommand.CliCommandClass ):
   syntax = "boot secret SECRET"
   noOrDefaultSyntax = "boot secret ..."
   data = { 'boot' : matcherBoot,
            'secret' : 'Assign the Aboot password',
            'SECRET' : SecretCli.secretCliExpression(
               'boot-secret',
               shaSecretGuard=bootSecretSha512SupportedGuard,
               scryptSecretGuard=bootSecretScryptSupportedGuard,
               yescryptSecretGuard=bootSecretYescryptSupportedGuard ) }
   allowCache = False

   @staticmethod
   def handler( mode, args ):
      encryptedPasswd = args[ 'boot-secret' ]
      if tpmPasswordConfigured( mode ):
         return
      try:
         dataStore_.setValue( mode, 'PASSWORD', encryptedPasswd.hash() )
      except ( OSError, SimpleConfigFile.ParseError ) as e:
         mode.addError( "Error setting Aboot password (%s)" % e )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      if tpmPasswordConfigured( mode ):
         return
      try:
         dataStore_.setValue( mode, 'PASSWORD', None )
      except ( OSError, SimpleConfigFile.ParseError ) as e:
         mode.addError( "Error clearing Aboot password (%s)" % e )

BasicCli.GlobalConfigMode.addCommandClass( BootSecretCommand )

#------------------------------------------------------------------------------------
# boot test memory <iterations>
# [no|default] boot test memory
#------------------------------------------------------------------------------------
def setMemtest( mode, args ):
   iterations = args[ 'ITERATIONS' ]
   try:
      dataStore_.setValue( mode, 'memtest', iterations )
   except ( OSError, SimpleConfigFile.ParseError ) as e:
      mode.addError( "Error setting memory test iterations (%s)" % e )

def clearMemtest( mode, args ):
   try:
      dataStore_.setValue( mode, 'memtest', None )
   except ( OSError, SimpleConfigFile.ParseError ) as e:
      mode.addError( "Error clearing memory test iterations (%s)" % e )

#--------------------------------------------------------------------------------
# [ no | default ] boot test memory ITERATIONS
#--------------------------------------------------------------------------------
class BootTestMemoryCmd( CliCommand.CliCommandClass ):
   syntax = 'boot test memory ITERATIONS'
   noOrDefaultSyntax = 'boot test memory ...'
   data = {
      'boot': matcherBoot,
      'test': 'Boot test',
      'memory': 'Boot test memory',
      'ITERATIONS': CliMatcher.IntegerMatcher( 1, 17,
                    helpdesc='Iteration value between 1 and 17' ),
   }
   handler = setMemtest
   noOrDefaultHandler = clearMemtest

BasicCliModes.GlobalConfigMode.addCommandClass( BootTestMemoryCmd )

#-----------------------------------------------------------------------------------
# Utils for Platform Security status
# -----------------------------------------------------------------------------------
@functools.cache
def getFirstInstructionSupportInfo( label ):
   msrValue = 0
   labelToBit = {}
   cpuInfo = Tac.run( [ 'lscpu' ], stdout=Tac.CAPTURE, stderr=sys.stderr )

   if 'GenuineIntel' in cpuInfo:
      labelToBit = {
         'FirstInstructionSupport': 32,
         'Verified': 6,
         'Measured': 5,
      }

      try:
         msrValue = Tac.run( [ 'rdmsr', '0x13A' ],
                             stdout=Tac.CAPTURE, stderr=Tac.CAPTURE, asRoot=True )
      except Tac.SystemCommandError as e:
         # On some older systems, 0x13A is not accessible.
         # Error code 4: CPU cannot read this MSR
         assert 4 == e.error
         return False
      msrValue = int( msrValue, 16 )

   if msrValue == 0:
      return False
   return bool( msrValue & ( 1 << labelToBit.get( label ) ) )

# -----------------------------------------------------------------------------------
# show boot-config
#-----------------------------------------------------------------------------------
bootConfigKwMatcher = CliMatcher.KeywordMatcher(
   'boot-config',
   alternates=[ 'boot' ],
   helpdesc='Show boot configuration' )

def showBootConfig( mode, args ): # pylint: disable=inconsistent-return-statements
   try:
      config = FileUrl.bootConfig( mode )
      ret = SysMgrModels.BootConfig()
      ret.softwareImage = config.get( 'SWI', '(not set)' )

      if ( CliPlugin.BmcCli.bmcAgentIsRunning()
           and CliPlugin.BmcCli.bmcBootControllerIsAccessible() ):
         bmcModel = CliPlugin.BmcCli.ShowSystemBmcStatus.handler( mode, args )
         if not bmcModel:
            return
         ret.personalityOnReboot = bmcModel.personalityOnReboot

      conSpeed = config.get( 'CONSOLESPEED' )
      ret.consoleSpeed = int( conSpeed ) if conSpeed else None
      ret.abootPassword = config.get( 'PASSWORD', '(not set)' )

      ret.firstInstructionVerifiedbootSupported = \
         getFirstInstructionSupportInfo( 'FirstInstructionSupport' )
      ret.firstInstructionVerifiedbootEnabled = \
         getFirstInstructionSupportInfo( 'Verified' )
      ret.firstInstructionMeasuredbootSupported = \
         getFirstInstructionSupportInfo( 'FirstInstructionSupport' )
      ret.firstInstructionMeasuredbootEnabled = \
         getFirstInstructionSupportInfo( 'Measured' )

      ret.securebootSupported = abootSbStatus.supported
      ret.spiUpdateEnabled = False
      ret.securebootEnabled = False
      ret.spiFlashWriteProtected = False
      ret.tpmPassword = False
      ret.measuredbootEnabled = False
      ret.aristaCertEnabled = False
      ret.certsLoaded = False
      ret.aristaCert = ''
      ret.aristaCert2 = ''
      ret.aristaCert3 = ''
      ret.userCert = ''
      ret.upgradeCert = ''

      ret.fileChecksumAlg = abootSbStatus.measuredBootHashAlg or 'sha1'

      try:
         ret.fileChecksum = config.checksum( ret.fileChecksumAlg )
      except IOError:
         # We should have marked fileChecksum as optional, but now
         # we use "" to indicate it's not available.
         ret.fileChecksum = ""

      if abootSbStatus.supported:
         ret.securebootEnabled = abootSbStatus.securebootDisabled == 0
         if abootSbStatus.tpmPassword == 1:
            ret.tpmPassword = True
            ret.abootPassword = '(not set)'
            if abootSbStatus.password != '':
               ret.abootPassword = abootSbStatus.password

         # TODO: Also verify the protection using flashrom
         if abootSbStatus.hardwiredSpiFlashWP:
            ret.spiFlashWriteProtected = True
         else:
            ret.spiFlashWriteProtected = abootSbStatus.unlockSPI == 0
         ret.aristaCertEnabled = abootSbStatus.ignoreAristaCert == 0

         ret.certsLoaded = abootSbStatus.sbCertLoaded and \
                           abootSbStatus.sbCert2Loaded and \
                           abootSbStatus.sbCert3Loaded and \
                           abootSbStatus.userCertLoaded and \
                           abootSbStatus.upgradeCertLoaded
         ret.spiUpdateEnabled = abootSbStatus.enableSPIUpdate == 1
         ret.measuredbootEnabled = abootSbStatus.measuredboot == 1
         if ret.certsLoaded:
            def _readCa( path ):
               try:
                  with open( path ) as f:
                     ca = f.read()
               except OSError:
                  ca = ''
               return ca

            from VerifySwi import ( # pylint: disable=import-outside-toplevel
               ARISTA_ROOT_CA_FILE_NAME,
               ARISTA_ROOT_CA2_FILE_NAME,
               ARISTA_ROOT_CA3_FILE_NAME,
               USER_ROOT_CA_FILE_NAME,
            )
            ret.aristaCert = _readCa( ARISTA_ROOT_CA_FILE_NAME )
            ret.aristaCert2 = _readCa( ARISTA_ROOT_CA2_FILE_NAME )
            ret.aristaCert3 = _readCa( ARISTA_ROOT_CA3_FILE_NAME )
            ret.userCert = _readCa( USER_ROOT_CA_FILE_NAME )

            # pylint: disable-next=import-outside-toplevel
            from Auf import AUF_ARISTA_INT_CA_CERT
            ret.upgradeCert = _readCa( AUF_ARISTA_INT_CA_CERT )

      if ret.abootPassword != '(not set)' and \
         ( not SecretCli.getHashAlgorithm( ret.abootPassword ) ):
         mode.addError( "Password appears corrupted. Please reset the "
                        "password via the CLI" )
      ret.memTestIterations = int( config.get( 'memtest', 0 ) )
      return ret

   except ( OSError, SimpleConfigFile.ParseError ) as e:
      mode.addError( "Error reading boot configuration (%s)" % e )

class ShowBootConfig( ShowCommand.ShowCliCommandClass ):
   syntax = 'show boot-config'
   data = {
            'boot-config': bootConfigKwMatcher,
          }
   cliModel = SysMgrModels.BootConfig
   privileged = True
   handler = showBootConfig

BasicCli.addShowCommandClass( ShowBootConfig )

#-------------------------------------------------------------------------------
# Register "show tech-support" commands
#-------------------------------------------------------------------------------
CliPlugin.TechSupportCli.registerShowTechSupportCmd(
   '2021-06-22 09:04:32',
   cmds=[ 'show boot' ],
   summaryCmds=[ 'show boot' ] )

def Plugin( entityManager ):
   global abootSbStatus

   abootSbStatus = LazyMount.mount( entityManager,
                                    Cell.path( "aboot/sb/status" ),
                                    "Aboot::Secureboot::Status", "r" )
   CliSessionDataStore.registerDataStore( "boot-config", dataStore_ )
