# Copyright (c) 2017 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import Arnet
from CliPlugin import HttpService
from CliPlugin import VrfCli
from CliPlugin import AclCli
from CliPlugin import AclCliModel
from CliPlugin import CapiModel
from CliPlugin import ConfigMgmtMode
from CliPlugin import HttpServiceModel
from CliPlugin import TechSupportCli
import CliGlobal
import BasicCli
import CapiConstants
from CliMode.Capi import CapiMgmtMode
import CliCommand
import CliMatcher
import CliToken
import CliToken.Reset
import ConfigMount
from HttpServiceConstants import CapiCipherSuites
import LazyMount
import ShowCommand
import Tac

# -----------------------------------------------------------------------------------
# Mount path holders ( Define all mount path holders here )
# -----------------------------------------------------------------------------------
gv = CliGlobal.CliGlobal(
   dict(
      capiConfig=None,
      capiExecRequest=None,
      serverStatus=None,
      serviceStatusDir=None,
      ipStatus=None,
      ip6Status=None,
   )
)


DEFAULT_VRF = VrfCli.DEFAULT_VRF
CAPI_CONSTANTS = Tac.Type( "HttpService::Constants" )
DEFAULT_SESSION_TIMEOUT = CAPI_CONSTANTS.defaultSessionTimeout
apiShowHelpDesc = 'Show management APIs'
httpShowCommandsHelpDesc = 'Show details about the HTTP Command API'
httpsShowHelpDesc = 'Show HTTPS configuration details'
httpCommandsHelpdesc = 'Configure the HTTP Commands API'

#
# Management config mode for Capi.
#
class CapiConfigMode( CapiMgmtMode, BasicCli.ConfigModeBase ):
   name = "Capi configuration"

   def __init__( self, parent, session ):
      # the param is used in enterCmd(), so keep it literally
      CapiMgmtMode.__init__( self )
      BasicCli.ConfigModeBase.__init__( self, parent, session )
      self.config_ = gv.capiConfig

#-------------------------------------------------------------------------------
# The "[no|default] management api http-commands" mode command.
#-------------------------------------------------------------------------------
class EnterCapiMode( CliCommand.CliCommandClass ):
   syntax = """management api http-commands"""
   noOrDefaultSyntax = syntax

   data = { "management": ConfigMgmtMode.managementKwMatcher,
            "api": ConfigMgmtMode.apiKwMatcher,
            "http-commands": httpCommandsHelpdesc }

   handler = "CapiHandler.enterCapiModeHandler"
   noOrDefaultHandler = "CapiHandler.enterCapiModeNoOrDefaultHandler"

BasicCli.GlobalConfigMode.addCommandClass( EnterCapiMode )

# {no|default} vrf VRF in manamgent api http-commands mode
class EnterVrfMode( CliCommand.CliCommandClass ):
   syntax = '''vrf VRF'''
   noOrDefaultSyntax = '''vrf [ VRF ]'''
   data = { 'vrf': HttpService.vrfHelpdesc,
            'VRF': VrfCli.VrfNameExprFactory( inclDefaultVrf=True ) }
   handler = "CapiHandler.enterCapiVrfConfigMode"
   noOrDefaultHandler = "CapiHandler.capiNoVrf"

CapiConfigMode.addCommandClass( EnterVrfMode )

#-------------------------------------------------------------------------------
# The "[no|default] shutdown" command,
# in "management api http-commands" mode.
#-------------------------------------------------------------------------------
class ConfigureCapi( CliCommand.CliCommandClass ):
   syntax = """shutdown"""
   noOrDefaultSyntax = syntax

   data = { "shutdown": "Disable API access" }

   handler = "CapiHandler.configureCapiHandler"
   noHandler = "CapiHandler.configureCapiNoHandler"
   defaultHandler = handler # by default we are off

CapiConfigMode.addCommandClass( ConfigureCapi )

#-------------------------------------------------------------------------------
# The "[no|default] protocol ((http [localhost])|https) [port <number>]" command,
# in "management api http-commands" mode.
#-------------------------------------------------------------------------------
CapiConfigMode.addCommandClass( HttpService.HttpProtocolCmd )

#-------------------------------------------------------------------------------
# The "[no | default] protocol unix-socket" command,
# in "management api http-commands" mode.
#-------------------------------------------------------------------------------
CapiConfigMode.addCommandClass( HttpService.UnixProtocolCmd )

#-------------------------------------------------------------------------------
# The "[no | default] protocol https certificate" command,
# in "management api http-commands" mode.
#-------------------------------------------------------------------------------
class CapiHttpsCert( CliCommand.CliCommandClass ):
   syntax = """protocol https certificate"""
   noOrDefaultSyntax = syntax

   data = { "protocol": HttpService.protocolHelpdesc,
            "https": HttpService.httpsHelpdesc,
            "certificate": 'Set the HTTPS key and certificate to use' }

   handler = "CapiHandler.capiHttpsCertHandler"
   noOrDefaultHandler = "CapiHandler.capiHttpsCertNoOrDefaultHandler"

CapiConfigMode.addCommandClass( CapiHttpsCert )

#-------------------------------------------------------------------------------
# The "[no | default] session timeout <N> minutes" command,
# in "management api http-commands" mode.
# -------------------------------------------------------------------------------
sessionTimeoutValueMatcher = CliMatcher.IntegerMatcher( 1,
   DEFAULT_SESSION_TIMEOUT,
   helpdesc="User session timeout value in minutes" )

class SetSessionTimeout( CliCommand.CliCommandClass ):
   syntax = """session timeout TIMEOUT minutes"""
   noOrDefaultSyntax = """session timeout [TIMEOUT minutes]"""

   data = { "session": "Configure user session options",
            "timeout": "Set expiration timeout for user session",
            "TIMEOUT": sessionTimeoutValueMatcher,
             "minutes": "Session timeout unit" }

   handler = "CapiHandler.sessionTimeoutHandler"
   noOrDefaultHandler = "CapiHandler.sessionTimeoutNoOrDefaultHandler"

CapiConfigMode.addCommandClass( SetSessionTimeout )

# -------------------------------------------------------------------------------
# The "[no | default] protocol https ssl profile" command,
# in "management api http-commands" mode.
#-------------------------------------------------------------------------------
CapiConfigMode.addCommandClass( HttpService.HttpsProfile )

#-------------------------------------------------------------------------------
# The "[no|default] protocol https cipher" command,
# in "management api http-commands" mode.
#-------------------------------------------------------------------------------
# The union of acceptable cipher names (with their descriptions)
cipherData = { k: v[ 'helpdesc' ] for k, v in CapiCipherSuites.CIPHERS.items() }
class ProtocolHttpsCipher( CliCommand.CliCommandClass ):
   syntax = '''protocol https cipher <cipher>'''
   noOrDefaultSyntax = '''protocol https cipher ...'''
   data = { 'protocol': HttpService.protocolHelpdesc,
            'https': HttpService.httpsHelpdesc,
            'cipher': 'Exclusive list of cryptographic ciphers',
            '<cipher>': CliCommand.setCliExpression( cipherData ) }
   hidden = True

   handler = "CapiHandler.protocolHttpsCipherHandler"
   noOrDefaultHandler = "CapiHandler.protocolHttpsCipherNoOrDefaultHandler"

CapiConfigMode.addCommandClass( ProtocolHttpsCipher )

#-------------------------------------------------------------------------------
# The "[no|default] protocol https key-exchange" command,
# in "management api http-commands" mode.
#-------------------------------------------------------------------------------
# The union of acceptable key exchange algorithms (with their descriptions)
keyExchangeData = { k: v[ 'helpdesc' ]
                    for k, v in CapiCipherSuites.KEYEXCHANGE.items() }

class ProtocolHttpsKeyExchange( CliCommand.CliCommandClass ):
   syntax = '''protocol https key-exchange <keyExchange>'''
   noOrDefaultSyntax = '''protocol https key-exchange ...'''
   data = { 'protocol': HttpService.protocolHelpdesc,
            'https': HttpService.httpsHelpdesc,
            'key-exchange': 'Exclusive list of key-exchange algorithms',
            '<keyExchange>': CliCommand.setCliExpression( keyExchangeData ) }
   hidden = True

   handler = "CapiHandler.protocolHttpsKeyExchangeHandler"
   noOrDefaultHandler = "CapiHandler.protocolHttpsKeyExchangeNoOrDefaultHandler"

CapiConfigMode.addCommandClass( ProtocolHttpsKeyExchange )

#-------------------------------------------------------------------------------
# The "[no|default] protocol https mac" command,
# in "management api http-commands" mode.
#-------------------------------------------------------------------------------
macData = { k: v[ 'helpdesc' ] for k, v in CapiCipherSuites.MAC.items() }

class ProtocolHttpsMac( CliCommand.CliCommandClass ):
   syntax = '''protocol https mac <mac>'''
   noOrDefaultSyntax = '''protocol https mac ...'''
   data = { 'protocol': HttpService.protocolHelpdesc,
            'https': HttpService.httpsHelpdesc,
            'mac': 'Exclusive list of Mac algorithms',
            '<mac>': CliCommand.setCliExpression( macData ) }
   hidden = True

   handler = "CapiHandler.protocolHttpsMacHandler"
   noOrDefaultHandler = "CapiHandler.protocolHttpsMacNoOrDefaultHandler"

CapiConfigMode.addCommandClass( ProtocolHttpsMac )

#-------------------------------------------------------------------------------
# (config-mgmt-api-http-cmds)# [no] validate-output.
#
# This enables time-consuming output checks, which is ok in test environment.
# Output validation is useful for commands that print json directly (without
# going through a python model instance, see CliPrint). It will unmarshall the
# json back into a python model so that any error becomes visible (this only
# happens via http, "show blah | json" will not validate anything.
# "Art sanitize" will enable validation (it is off by default).
#-------------------------------------------------------------------------------
validateOutputMatcher = CliMatcher.KeywordMatcher( 'validate-output',
                           helpdesc='validate that json output conforms to model' )
class ValidateJsonOutput( CliCommand.CliCommandClass ):
   syntax = """validate-output"""
   noOrDefaultSyntax = syntax

   data = { "validate-output": CliCommand.Node( matcher=validateOutputMatcher,
                                                hidden=True ) }

   handler = "CapiHandler.validateJsonOutputHandler"
   noOrDefaultHandler = "CapiHandler.validateJsonOutputNoOrDefaultHandler"

CapiConfigMode.addCommandClass( ValidateJsonOutput )

#-------------------------------------------------------------------------------
# The "reset https diffie-hellman parameters" command in "config" mode
#-------------------------------------------------------------------------------
class ResetDhParams( CliCommand.CliCommandClass ):
   syntax = '''reset https diffie-hellman parameters'''
   data = { 'reset': CliToken.Reset.resetKwApi,
            'https': HttpService.httpsHelpdesc,
            'diffie-hellman': 'diffie-hellman parameters',
            'parameters': 'diffie-hellman parameters' }
   hidden = True

   handler = "CapiHandler.resetDhParamsHandler"

BasicCli.EnableMode.addCommandClass( ResetDhParams )

#-------------------------------------------------------------------------------
# The "clear management api http-commands" command,
# in "enable" mode.
#-------------------------------------------------------------------------------
class ClearCounters( CliCommand.CliCommandClass ):
   syntax = '''clear management api http-commands counters'''
   data = { 'clear': CliToken.Clear.clearKwNode,
            'management': ConfigMgmtMode.managementClearKwMatcher,
            'api': ConfigMgmtMode.apiKwMatcher,
            'http-commands': httpCommandsHelpdesc,
            'counters': 'Clear counters tracking API usage' }

   handler = "CapiHandler.clearCountersHandler"

BasicCli.EnableMode.addCommandClass( ClearCounters )

#-------------------------------------------------------------------------------
# The "[ no | default ] qos dscp <dscpValue>" command
#-------------------------------------------------------------------------------
CapiConfigMode.addCommandClass( HttpService.QosDscp )

#-------------------------------------------------------------------------------
# The "[ no | default ] log-level <severity>" command
#-------------------------------------------------------------------------------
CapiConfigMode.addCommandClass( HttpService.NginxSyslog )

#-------------------------------------------------------------------------------
# The "[ no | default ] default-services" command
#-------------------------------------------------------------------------------
CapiConfigMode.addCommandClass( HttpService.DefaultServices )

#-------------------------------------------------------------------------------
# The "[ no | default ] content frame ancestors <uri>" command
#-------------------------------------------------------------------------------
CapiConfigMode.addCommandClass( HttpService.FrameAncestors )

#-------------------------------------------------------------------------------
# The "cors allowed domian [ALL | URL]" command
#-------------------------------------------------------------------------------
CapiConfigMode.addCommandClass( HttpService.CorsOrigins )

#-------------------------------------------------------------------------------
# The "show management api http-commands" command
#-------------------------------------------------------------------------------
def checkUdsEnabledOnly( vrf, vrfStatus ):
   return ( vrf == DEFAULT_VRF and
            # default VRF is not configured specifically
            vrf not in gv.capiConfig.vrfConfig and
            # there is no other protocol enabled other than uds
            not vrfStatus.httpsEnabled and
            not vrfStatus.httpEnabled and
            not vrfStatus.httpLocalEnabled and
            vrfStatus.unixEnabled )

def showCapi( mode, args ):
   vrfList = []
   httpsEnabled = False
   httpEnabled = False
   httpLocalEnabled = False
   unixEnabled = False
   # Capi is enabled if it's enabled in any VRF
   for ( vrf, status ) in gv.serverStatus.vrfStatus.items():
      if 'http-commands' in status.vrfService:
         for service, error in status.vrfError.items():
            if error and service in [ 'nginx', 'vrf' ]:
               # Capi only cares about error message about nginx status
               # and vrf status, not other services
               mode.addWarning( error )
         if ( status.enabled and
              status.vrfService[ 'http-commands' ].enabled and
              # if Capi is running in default VRF with only uds
              # then hide it in show command
              not checkUdsEnabledOnly( vrf, status ) ):
            vrfList.append( vrf )

         httpsEnabled |= status.httpsEnabled
         httpEnabled |= status.httpEnabled
         httpLocalEnabled |= status.httpLocalEnabled
         unixEnabled |= status.unixEnabled

   ret = CapiModel.CommandApiStatus()
   ret.enabled = gv.capiConfig.service[ 'http-commands' ].enabled
   ret.sessionTimeout = gv.capiConfig.sessionTimeout
   ret.httpsServer = HttpServiceModel.HttpProtocolStatus(
      configured=gv.capiConfig.httpsConfig.enabled,
      running=httpsEnabled, port=gv.capiConfig.httpsConfig.port )
   ret.httpServer = HttpServiceModel.HttpProtocolStatus(
      configured=gv.capiConfig.httpConfig.enabled,
      running=httpEnabled, port=gv.capiConfig.httpConfig.port )
   ret.localHttpServer = HttpServiceModel.HttpProtocolStatus(
      configured=gv.capiConfig.localHttpConfig.enabled,
      running=httpLocalEnabled, port=gv.capiConfig.localHttpConfig.port )
   ret.unixSocketServer = HttpServiceModel.UnixProtocolStatus(
      configured=gv.capiConfig.unixConfig.enabled, running=unixEnabled )
   # show enabled VRFs only
   vrfList = sorted( vrfList )
   ret.vrfs = vrfList
   ret.vrf = vrfList[ 0 ] if vrfList else ""
   ret.corsOrigins = list( gv.capiConfig.corsAllowedOrigins )

   capiStatus = gv.serviceStatusDir.get( 'http-commands' )
   # if we have an outstanding clear request don't send back current values
   if ( capiStatus and
        capiStatus.lastStatsResetTime >= gv.capiExecRequest.lastStatsResetTime ):
      ret.hitCount = capiStatus.hitCount
      ret.requestCount = capiStatus.requestCount
      ret.commandCount = capiStatus.commandCount
      ret.bytesIn = capiStatus.bytesInCount
      ret.bytesOut = capiStatus.bytesOutCount
      if capiStatus.lastHitTime > 1:
         ret.lastHitTime = capiStatus.lastHitTime + Tac.utcNow() - Tac.now()
      else:
         ret.lastHitTime = None
      ret.executionTime = capiStatus.requestDuration

      for name, userStatus in capiStatus.user.items():
         if userStatus.lastHitTime > 1:
            lastHitTime = userStatus.lastHitTime + Tac.utcNow() - Tac.now()
         else:
            ret.lastHitTime = None
         ret.users[ name ] = CapiModel.UserStatistics(
            requestCount=userStatus.requestCount,
            bytesIn=userStatus.bytesInCount,
            bytesOut=userStatus.bytesOutCount,
            lastHitTime=lastHitTime )
   else:
      ret.hitCount = 0
      ret.requestCount = 0
      ret.commandCount = 0
      ret.bytesIn = 0
      ret.bytesOut = 0
      ret.lastHitTime = None
      ret.executionTime = 0.0

   ret.urls = getUrlList( vrfList, unixEnabled, httpLocalEnabled )
   HttpService.getCommonServerInfo( ret, gv.capiConfig )
   return ret

def getUrlList( vrfList, unixEnabled, httpLocalEnabled ):
   def formatUrl( intfName, protocol, address, port ):
      return f"{intfName:<12s}: {protocol:s}://{address:s}:{port:d}"

   urls = []
   internalIntfId = Tac.Value( 'Arnet::InternalIntfId' )

   for intfName in Arnet.sortIntf( gv.ipStatus.ipIntfStatus ):
      ipIntfStatus = gv.ipStatus.ipIntfStatus[ intfName ]
      # We don't want to list an interface with no ip address
      if ipIntfStatus.activeAddrWithMask.address == '0.0.0.0':
         continue

      # We don't want to display internal ports either
      if internalIntfId.isInternalIntfId( ipIntfStatus.intfId ):
         continue

      # don't display ip address if it's not in our VRF
      ipIntfStatusVrf = ipIntfStatus.vrf if ipIntfStatus.vrf else DEFAULT_VRF
      if ipIntfStatusVrf not in vrfList:
         continue

      if gv.capiConfig.httpsConfig.enabled:
         urls.append( formatUrl( intfName, 'https',
                                 ipIntfStatus.activeAddrWithMask.address,
                                 gv.capiConfig.httpsConfig.port ) )
      if gv.capiConfig.httpConfig.enabled:
         urls.append( formatUrl( intfName, 'http',
                                 ipIntfStatus.activeAddrWithMask.address,
                                 gv.capiConfig.httpConfig.port ) )

   for intfName in Arnet.sortIntf( gv.ip6Status.intf ):
      ip6IntfStatus = gv.ip6Status.intf[ intfName ]
      # We don't want to display internal ports either
      if internalIntfId.isInternalIntfId( ip6IntfStatus.intfId ):
         continue

      # don't display ip address if it's not in our VRF
      ip6IntfStatusVrf = ip6IntfStatus.vrf if ip6IntfStatus.vrf else DEFAULT_VRF
      if ip6IntfStatusVrf not in vrfList:
         continue

      for key in ip6IntfStatus.addr:
         addr = ip6IntfStatus.addr[ key ].address
         if addr == ip6IntfStatus.linkLocalAddrWithMask().address:
            continue

         if gv.capiConfig.httpsConfig.enabled:
            # pylint: disable-next=consider-using-f-string
            urls.append( formatUrl( intfName, 'https', "[%s]" % addr,
                                    gv.capiConfig.httpsConfig.port ) )
         if gv.capiConfig.httpConfig.enabled:
            # pylint: disable-next=consider-using-f-string
            urls.append( formatUrl( intfName, 'http', "[%s]" % addr,
                                    gv.capiConfig.httpConfig.port ) )

   if gv.capiConfig.unixConfig.enabled and unixEnabled:
      # pylint: disable-next=consider-using-f-string
      urls.append( "{:<12s}: unix:{:s}".format( "Unix Socket",
                                                CapiConstants.SOCKET_ENDPOINT ) )
   if gv.capiConfig.localHttpConfig.enabled and httpLocalEnabled:
      # pylint: disable-next=consider-using-f-string
      urls.append( "{}/{}".format( formatUrl( "Local", "http", "localhost",
                                          gv.capiConfig.localHttpConfig.port ),
                               CapiConstants.COMMAND_ENDPOINT ) )

   return urls

class ShowApiHttpCommands( ShowCommand.ShowCliCommandClass ):
   syntax = 'show management api http-commands'
   data = {
            'management': ConfigMgmtMode.managementShowKwMatcher,
            'api': apiShowHelpDesc,
            'http-commands': httpShowCommandsHelpDesc
          }
   cliModel = CapiModel.CommandApiStatus
   handler = showCapi

BasicCli.addShowCommandClass( ShowApiHttpCommands )
#-------------------------------------------------------------------------------
# The "show management api http-commands https certificate" command and
#-------------------------------------------------------------------------------
class ShowApiHttpCommandsCertificate( ShowCommand.ShowCliCommandClass ):
   syntax = 'show management api http-commands https certificate'
   data = {
            'management': ConfigMgmtMode.managementShowKwMatcher,
            'api': apiShowHelpDesc,
            'http-commands': httpShowCommandsHelpDesc,
            'https': httpsShowHelpDesc,
            'certificate': 'Show the HTTPS certificate'
          }
   cliModel = CapiModel.CommandApiCertificate
   handler = "CapiShowHandler.showCapiCertHandler"
   privileged = True

BasicCli.addShowCommandClass( ShowApiHttpCommandsCertificate )

#-------------------------------------------------------------------------------
# The "show management api http-commands https cipher" command
#-------------------------------------------------------------------------------
class ShowApiHttpCommandsHttpsCipher( ShowCommand.ShowCliCommandClass ):
   syntax = 'show management api http-commands https cipher'
   data = {
            'management': ConfigMgmtMode.managementShowKwMatcher,
            'api': apiShowHelpDesc,
            'http-commands': httpShowCommandsHelpDesc,
            'https': httpsShowHelpDesc,
            'cipher': 'Show the cipher suites in use'
          }
   cliModel = CapiModel.CommandApiCipherSuites
   handler = "CapiShowHandler.showCapiCiphersHandler"
   privileged = True

BasicCli.addShowCommandClass( ShowApiHttpCommandsHttpsCipher )

#-------------------------------------------------------------------------------
# The "show management api http-commands https diffie-hellman" command
#-------------------------------------------------------------------------------
class ShowApiHttpCommandsHttpsDiffieHellman( ShowCommand.ShowCliCommandClass ):
   syntax = 'show management api http-commands https diffie-hellman'
   data = {
            'management': ConfigMgmtMode.managementShowKwMatcher,
            'api': apiShowHelpDesc,
            'http-commands': httpShowCommandsHelpDesc,
            'https': httpsShowHelpDesc,
            'diffie-hellman': 'Show the diffie-hellman parameters in use'
          }
   cliModel = CapiModel.CommandApiDiffieHellman
   privileged = True
   handler = "CapiShowHandler.showCapiDhparamsHandler"

BasicCli.addShowCommandClass( ShowApiHttpCommandsHttpsDiffieHellman )

#-----------------------------------------------------------------------------
# "show management api http-commands ( ip | ipv6 ) access-list [<acl-name>]"
#-----------------------------------------------------------------------------
class ShowApiHttpCommandsAcl( ShowCommand.ShowCliCommandClass ):
   syntax = ('show management api http-commands '
             '('
             ' ( ip access-list [ <ipAclName> ] ) | '
             ' ( ipv6 access-list [ <ipv6AclName> ] ) '
             ')' )
   data = {
            'management': ConfigMgmtMode.managementShowKwMatcher,
            'api': apiShowHelpDesc,
            'http-commands': httpShowCommandsHelpDesc,
            'ip': AclCli.ipKwForShowServiceAcl,
            'ipv6': AclCli.ipv6KwForShowServiceAcl,
            'access-list': AclCli.accessListKwMatcherForServiceAcl,
            '<ipAclName>': AclCli.ipAclNameExpression,
            '<ipv6AclName>': AclCli.ip6AclNameExpression
          }
   cliModel = AclCliModel.AllAclList
   privileged = True
   handler = "CapiShowHandler.showCapiAclHandler"

BasicCli.addShowCommandClass( ShowApiHttpCommandsAcl )

#----------------------------------------------------------------
# "clear management api http-commands ip access-list counters"
# "clear management api http-commands ipv6 access-list counters"
#----------------------------------------------------------------
class ClearIpAclCounters( CliCommand.CliCommandClass ):
   syntax = 'clear management api http-commands ( ip | ipv6 ) access-list counters'
   data = { 'clear': CliToken.Clear.clearKwNode,
            'management': ConfigMgmtMode.managementClearKwMatcher,
            'api': ConfigMgmtMode.apiKwMatcher,
            'http-commands': httpCommandsHelpdesc,
            'ip': AclCli.ipKwForClearServiceAclMatcher,
            'ipv6': AclCli.ipv6KwMatcherForClearServiceAcl,
            'access-list': AclCli.accessListKwMatcherForServiceAcl,
            'counters': AclCli.countersKwMatcher }

   handler = "CapiHandler.clearIpAclCountersHandler"

BasicCli.EnableMode.addCommandClass( ClearIpAclCounters )

#-------------------------------------------------------------------------------
# Command-API commands in "show tech-support"
#-------------------------------------------------------------------------------
TechSupportCli.registerShowTechSupportCmd(
   '2014-03-05 15:08:19',
   cmds=[ 'show management api http-commands' ] )

# Plug-in definition:
def Plugin( entityManager ):
   gv.capiConfig = ConfigMount.mount( entityManager,
                                      "mgmt/capi/config",
                                      "HttpService::Config",
                                      "w" )
   gv.capiExecRequest = LazyMount.mount( entityManager,
                                         "mgmt/httpserver/execRequest",
                                         "HttpService::ExecRequest",
                                         "w" )
   gv.serverStatus = LazyMount.mount( entityManager,
                                      "mgmt/httpserver/status",
                                      "HttpService::Status",
                                      "r" )
   gv.serviceStatusDir = LazyMount.mount( entityManager,
                                          "mgmt/httpserver/service",
                                          "Tac::Dir",
                                          "ri" )
   gv.ipStatus = LazyMount.mount( entityManager,
                                  "ip/status",
                                  "Ip::Status",
                                  "r" )
   gv.ip6Status = LazyMount.mount( entityManager,
                                   "ip6/status",
                                   "Ip6::Status",
                                   "r" )
