#!/usr/bin/env python3
# Copyright (c) 2022 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import BasicCli
import CliCommand
import CliMatcher
import CliParser
from CliMode.Cbf import ( SrTeColorDscpModeBase,
                          TeCbfModeBase,
                          TeColorDscpModeBase,
                          TeColorTcModeBase,
                          TeColorVrfModeBase,
                          CbfFwdPlaneMplsSubModeBase )
from CliPlugin.VrfCli import getVrfNames
from CliPlugin.TeCli import RouterGlobalTeMode, TeDependentBase, TeModeDependents
from CliPlugin.SrTePolicyCli import SrTeMode, SrTeDependentBase, SrTeModeDependents
from TeCliLib import dscpAclNames
from MultiRangeRule import MultiRangeMatcher
import ConfigMount
import LazyMount
import Tac
from Toggles import IpLibToggleLib
from TypeFuture import TacLazyType

teConfig = None
routingHwStatus = None
srTeCbfColorDscpMap = None
teCbfColorDscpMap = None
teCbfColorTcMap = None
teCbfColorVrfMap = None
vrfSelPolicyHwCapability = None

TacColor = TacLazyType( 'TrafficEngineering::Color' )
TacDscp = TacLazyType( 'Arnet::DscpValue' )
TacTc = TacLazyType( 'Qos::TrafficClass' )

nonDefaultVrfMatcherExcludeAll = CliMatcher.DynamicNameMatcher( getVrfNames,
      helpdesc='VRF name',
      pattern=r'^(?!all$)(?!default$)[A-Za-z0-9_.:{}\[\]-]+' )

def getCbfValue( value, valueType ):
   kwargs = { valueType: value }
   return Tac.Value( "TrafficEngineering::CbfConfigModeValue", **kwargs )

def dscpValueFromCli( dscpVal ):
   dscpResult = dscpAclNames.get( dscpVal, False )
   return ( dscpResult[ 0 ], True ) if dscpResult else ( dscpVal, False )

def dscpAclNamesDef( mode ):
   return { k: v[ 1 ] for k, v in dscpAclNames.items() }

def dscpRange():
   return TacDscp.min, TacDscp.max

def tcRange():
   return TacTc.min, TacTc.max

# -------------------------------------------------------------------------------
# [ no | default ] color-dscp-mappings in te-sr mode
# -------------------------------------------------------------------------------
class SrTeColorDscpMappingsMode( SrTeColorDscpModeBase,
                                 BasicCli.ConfigModeBase ):
   name = "Color / DSCP mapping configuration"

   def __init__( self, parent, session ):
      SrTeColorDscpModeBase.__init__( self, None )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

def cbfToggleEnabledGuard( mode, token ):
   if routingHwStatus.srTeCbfSupported:
      return None
   return CliParser.guardNotThisPlatform

class SrTeColorDscpMappingsCmd( CliCommand.CliCommandClass ):
   syntax = 'color-dscp-mappings'
   noOrDefaultSyntax = syntax
   data = {
      'color-dscp-mappings': CliCommand.guardedKeyword(
         'color-dscp-mappings',
         SrTeColorDscpMappingsMode.name,
         guard=cbfToggleEnabledGuard ),
   }

   @staticmethod
   def handler( mode, args ):
      childMode = mode.childMode( SrTeColorDscpMappingsMode )
      mode.session_.gotoChildMode( childMode )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      srTeCbfColorDscpMap.colorToValue.clear()

SrTeMode.addCommandClass( SrTeColorDscpMappingsCmd )

# -------------------------------------------------------------------------------
# [ no | default ] color COLOR ( dscp DSCP | default )
# in te-sr color-dscp-mappings mode
# -------------------------------------------------------------------------------

class SrTeColorDscpMappingEntryCmd( CliCommand.CliCommandClass ):
   syntax = 'color COLOR ( dscp ( { DSCP | DSCPACL } | DSCPS ) ) | default'
   noOrDefaultSyntax = ( 'color COLOR [ ( dscp ( { DSCP | DSCPACL } | DSCPS ) ) '
                         ' | default ]' )
   data = {
      'color': "Configure the mapping mapping for a color",
      'COLOR': CliMatcher.IntegerMatcher( TacColor.min, TacColor.max,
                                          helpdesc='Color value' ),
      'default': 'Associate the specified color with any unmapped DSCP values',
      'dscp': "Map the specified color to specific DSCP values",
      'DSCP': CliMatcher.IntegerMatcher( *dscpRange(),
                                         helpdesc='DSCP value' ),
      'DSCPACL': CliMatcher.DynamicKeywordMatcher( dscpAclNamesDef ),
      'DSCPS': CliCommand.Node(
                  matcher=MultiRangeMatcher(
                               rangeFn=dscpRange,
                               noSingletons=False,
                               helpdesc='Differentiated Services Code Point (DSCP) '
                                        'numeric values or ranges',
                  priority=CliParser.PRIO_HIGH ) ),
   }

   @staticmethod
   def handler( mode, args ):
      if 'default' in args:
         color = args[ 'COLOR' ]
         if color in srTeCbfColorDscpMap.colorToValue:
            # Clean up any state for specific DSCPs
            if not srTeCbfColorDscpMap.colorToValue[ color ].isDefault:
               srTeCbfColorDscpMap.colorToValue[ color ].value.clear()
               srTeCbfColorDscpMap.colorToValue[ color ].cliSaveUseSymbolic_i.clear()
         colorToDscp = srTeCbfColorDscpMap.newColorToValue( color )
         colorToDscp.isDefault = True
         return
      dscpValues = args.get( 'DSCP', [] )
      dscpValues += args.get( 'DSCPACL', [] )
      dscpValues += list( args[ 'DSCPS' ].values() ) if 'DSCPS' in args else []
      colorToDscp = srTeCbfColorDscpMap.newColorToValue( args[ 'COLOR' ] )
      # Clean up any state for default DSCPs.
      if colorToDscp.isDefault:
         colorToDscp.isDefault = False
      for value in dscpValues:
         dscp, dscpUseName = dscpValueFromCli( value )
         dscp = Tac.Value( "TrafficEngineering::CbfConfigModeValue", dscp=dscp )
         colorToDscp.value[ dscp ] = True
         colorToDscp.cliSaveUseSymbolic_i[ dscp ] = dscpUseName

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      color = args[ 'COLOR' ]
      if color not in srTeCbfColorDscpMap.colorToValue:
         return
      if 'default' in args:
         # Removing default DSCPs setting.  If set delete the color, otherwise
         # do nothing.
         if srTeCbfColorDscpMap.colorToValue[ color ].isDefault:
            del srTeCbfColorDscpMap.colorToValue[ color ]
         return
      colorToDscp = srTeCbfColorDscpMap.colorToValue[ color ]
      dscpValues = args.get( 'DSCP', [] )
      dscpValues += args.get( 'DSCPACL', [] )
      dscpValues += list( args[ 'DSCPS' ].values() ) if 'DSCPS' in args else []
      if dscpValues:
         for value in dscpValues:
            dscp, _ = dscpValueFromCli( value )
            dscp = Tac.Value( "TrafficEngineering::CbfConfigModeValue",
                              dscp=dscp )
            del colorToDscp.value[ dscp ]
            del colorToDscp.cliSaveUseSymbolic_i[ dscp ]
         if not colorToDscp.value and \
               not srTeCbfColorDscpMap.colorToValue[ color ].isDefault:
            del srTeCbfColorDscpMap.colorToValue[ color ]
      else:
         del srTeCbfColorDscpMap.colorToValue[ color ]

SrTeColorDscpMappingsMode.addCommandClass( SrTeColorDscpMappingEntryCmd )

# -----------------------------------------------------------------------------
# Te Class Based Forwarding Mode
# -----------------------------------------------------------------------------
class TeCbfMode( TeCbfModeBase, BasicCli.ConfigModeBase ):
   name = "Class Based Forwarding configuration"

   def __init__( self, parent, session ):
      TeCbfModeBase.__init__( self, None )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

class TeCbfCmd( CliCommand.CliCommandClass ):
   syntax = 'class-based-forwarding'
   noOrDefaultSyntax = syntax
   data = {
         'class-based-forwarding': TeCbfMode.name,
   }

   @staticmethod
   def handler( mode, args ):
      childMode = mode.childMode( TeCbfMode )
      mode.session_.gotoChildMode( childMode )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      noOrDefaultCbfMode()

RouterGlobalTeMode.addCommandClass( TeCbfCmd )

# -----------------------------------------------------------------------------
# Helper Class. This is used as a base class for all dependent classes of
# 'config-te-cbf' mode. When the class-based-forwarding mode  is unconfigured
# the dependents can cleanup their configuration.
# -----------------------------------------------------------------------------
def noOrDefaultCbfMode():
   """
   Remove all configuration on 'no class-based-forwarding'
   or 'no router traffic-engineering'
   """
   srTeCbfColorDscpMap.colorToValue.clear()
   teCbfColorDscpMap.colorToValue.clear()
   teCbfColorTcMap.colorToValue.clear()
   teCbfColorVrfMap.colorToValue.clear()
   teConfig.cbfEcmpEndpoints = False
   teConfig.cbfMplsColorDefFallback = False
   teConfig.cbfMplsTunnelingLdp = False
   teConfig.cbfMplsTunnelingIsisSr = False

class TeCbfSubMode( TeDependentBase ):
   def setDefault( self ):
      noOrDefaultCbfMode()

class SrTeCbfSubMode( SrTeDependentBase ):
   def setDefault( self ):
      # Clear sr-te cbf config
      srTeCbfColorDscpMap.colorToValue.clear()

# -----------------------------------------------------------------------------
# Color Dscp Mapping in te-cbf mode
# ----------------------------------------------------------------------------
def rsvpCbfHwSupportedGuard( mode, token ):
   # RSVP CBF leverages all the platform related work done for SR-TE Policy CBF
   if routingHwStatus.srTeCbfSupported:
      return None
   return CliParser.guardNotThisPlatform

class TeColorDscpMappingsMode( TeColorDscpModeBase, BasicCli.ConfigModeBase ):
   name = "Color / DSCP mapping configuration"

   def __init__( self, parent, session ):
      TeColorDscpModeBase.__init__( self, None )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

class TeColorDscpMappingCmdBase( CliCommand.CliCommandClass ):
   syntax = 'color-dscp-mappings'
   noOrDefaultSyntax = syntax
   data = {
         'color-dscp-mappings': CliCommand.guardedKeyword(
            'color-dscp-mappings',
            TeColorDscpMappingsMode.name,
            guard=rsvpCbfHwSupportedGuard ),
   }

   @staticmethod
   def handler( mode, args ):
      if srTeCbfColorDscpMap.colorToValue:
         mode.addWarning( "Color/DSCP mappings in the traffic-engineering mode"
               " will not take effect until the mappings in the te-segment-routing"
               " mode are removed" )
      childMode = mode.childMode( TeColorDscpMappingsMode )
      mode.session_.gotoChildMode( childMode )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      teCbfColorDscpMap.colorToValue.clear()

class TeCbfColorDscpMappingsCmd( TeColorDscpMappingCmdBase ):
   data = TeColorDscpMappingCmdBase.data.copy()

TeCbfMode.addCommandClass( TeCbfColorDscpMappingsCmd )

# Backward compatibility. The CLI was originally in router-te mode, and was later
# moved to router-te-cbf mode
class TeColorDscpMappingsHiddenCmd( TeColorDscpMappingCmdBase ):
   data = TeColorDscpMappingCmdBase.data.copy()
   hidden = True

RouterGlobalTeMode.addCommandClass( TeColorDscpMappingsHiddenCmd )

# -------------------------------------------------------------------------------
# [ no | default ] color COLOR ( dscp DSCP | default )
# in te-cbf-color-dscp-mappings mode
# -------------------------------------------------------------------------------
class TeColorDscpMappingEntryCmd( CliCommand.CliCommandClass ):
   syntax = 'color COLOR ( dscp ( { DSCP | DSCPACL } | DSCPS ) ) | default'
   noOrDefaultSyntax = ( 'color COLOR [ ( dscp ( { DSCP | DSCPACL } | DSCPS ) ) '
                         ' | default ]' )
   data = {
      'color': "Configure the mapping for a color",
      'COLOR': CliMatcher.IntegerMatcher( TacColor.min, TacColor.max,
                                          helpdesc='Color value' ),
      'default': 'Associate the specified color with any unmapped DSCP values',
      'dscp': "Map the specified color to specific DSCP values",
      'DSCP': CliMatcher.IntegerMatcher( *dscpRange(),
                                         helpdesc='DSCP value' ),
      'DSCPACL': CliMatcher.DynamicKeywordMatcher( dscpAclNamesDef ),
      'DSCPS': CliCommand.Node(
                  matcher=MultiRangeMatcher(
                               rangeFn=dscpRange,
                               noSingletons=False,
                               helpdesc='Differentiated Services Code Point (DSCP) '
                                        'numeric values or ranges',
                  priority=CliParser.PRIO_HIGH ) ),
   }

   @staticmethod
   def handler( mode, args ):
      if 'default' in args:
         color = args[ 'COLOR' ]
         if color in teCbfColorDscpMap.colorToValue:
            # Clean up any state for specific DSCPs
            if not teCbfColorDscpMap.colorToValue[ color ].isDefault:
               teCbfColorDscpMap.colorToValue[ color ].value.clear()
               teCbfColorDscpMap.colorToValue[
                  color ].cliSaveUseSymbolic_i.clear()
         colorToDscp = teCbfColorDscpMap.newColorToValue( color )
         colorToDscp.isDefault = True
         return
      dscpValues = args.get( 'DSCP', [] )
      dscpValues += args.get( 'DSCPACL', [] )
      dscpValues += args[ 'DSCPS' ].values() if 'DSCPS' in args else []
      colorToDscp = teCbfColorDscpMap.newColorToValue( args[ 'COLOR' ] )
      # Clean up any state for default DSCPs.
      if colorToDscp.isDefault:
         colorToDscp.isDefault = False
      for value in dscpValues:
         dscp, dscpUseName = dscpValueFromCli( value )
         dscp = getCbfValue( dscp, "dscp" )
         colorToDscp.value[ dscp ] = True
         colorToDscp.cliSaveUseSymbolic_i[ dscp ] = dscpUseName

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      color = args[ 'COLOR' ]
      if color not in teCbfColorDscpMap.colorToValue:
         return
      if 'default' in args:
         # Removing default DSCPs setting.  If set delete the color, otherwise
         # do nothing.
         if teCbfColorDscpMap.colorToValue[ color ].isDefault:
            del teCbfColorDscpMap.colorToValue[ color ]
         return
      colorToDscp = teCbfColorDscpMap.colorToValue[ color ]
      dscpValues = args.get( 'DSCP', [] )
      dscpValues += args.get( 'DSCPACL', [] )
      dscpValues += args[ 'DSCPS' ].values() if 'DSCPS' in args else []
      if dscpValues:
         for value in dscpValues:
            dscp, _ = dscpValueFromCli( value )
            dscp = getCbfValue( dscp, "dscp" )
            del colorToDscp.value[ dscp ]
            del colorToDscp.cliSaveUseSymbolic_i[ dscp ]
         if not colorToDscp.value and \
               not teCbfColorDscpMap.colorToValue[ color ].isDefault:
            del teCbfColorDscpMap.colorToValue[ color ]
      else:
         del teCbfColorDscpMap.colorToValue[ color ]

TeColorDscpMappingsMode.addCommandClass( TeColorDscpMappingEntryCmd )

# -----------------------------------------------------------------------------
# Color TC Mapping in te-cbf mode
# ----------------------------------------------------------------------------
def rsvpTcCbfHwSupportedGuard( mode, token ):
   if routingHwStatus.tcBasedCbfSupported:
      return None
   return CliParser.guardNotThisPlatform

class TeColorTcMappingsMode( TeColorTcModeBase, BasicCli.ConfigModeBase ):
   name = "Color / TC mapping configuration"

   def __init__( self, parent, session ):
      TeColorTcModeBase.__init__( self, None )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

class TeColorTcMappingsBase( CliCommand.CliCommandClass ):
   syntax = 'color-tc-mappings'
   noOrDefaultSyntax = syntax
   data = {
         'color-tc-mappings': CliCommand.guardedKeyword(
            'color-tc-mappings',
            TeColorTcMappingsMode.name,
            guard=rsvpTcCbfHwSupportedGuard ),
   }

   @staticmethod
   def handler( mode, args ):
      if srTeCbfColorDscpMap.colorToValue or teCbfColorDscpMap.colorToValue:
         mode.addWarning( "Color/TC mappings in the traffic-engineering mode"
               " will not take effect until the Color/DSCP mappings in the"
               " te-segment-routing and traffic-engineering modes are removed" )
      childMode = mode.childMode( TeColorTcMappingsMode )
      mode.session_.gotoChildMode( childMode )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      teCbfColorTcMap.colorToValue.clear()

class TeCbfColorTcMappingsCmd( TeColorTcMappingsBase ):
   data = TeColorTcMappingsBase.data.copy()

TeCbfMode.addCommandClass( TeCbfColorTcMappingsCmd )

# Backward compatibility
class TeColorTcMappingsHiddenCmd( TeColorTcMappingsBase ):
   data = TeColorTcMappingsBase.data.copy()
   hidden = True

RouterGlobalTeMode.addCommandClass( TeColorTcMappingsHiddenCmd )

# -------------------------------------------------------------------------------
# [ no | default ] color COLOR ( traffic-class TC | default )
# in te-cbf-color-tc-mappings mode
# -------------------------------------------------------------------------------

class TeColorTcMappingEntryCmd( CliCommand.CliCommandClass ):
   syntax = '''color COLOR
               ( ( traffic-class ( { TRAFFIC_CLASS } | TRAFFIC_CLASSES ) )
               | default )'''
   noOrDefaultSyntax = '''color COLOR
                          [ ( traffic-class ( { TRAFFIC_CLASS } | TRAFFIC_CLASSES ) )
                          | default ]'''
   data = {
      'color': "Configure the mapping for a color",
      'COLOR': CliMatcher.IntegerMatcher( TacColor.min, TacColor.max,
                            helpdesc='Color value' ),
      'default': 'Associate the specified color with any unmapped traffic class'
                 ' values',
      'traffic-class': "Map the specified color to specific traffic class values",
      'TRAFFIC_CLASS': CliMatcher.IntegerMatcher( *tcRange(),
                                         helpdesc='traffic class value' ),
      'TRAFFIC_CLASSES': CliCommand.Node(
                               matcher=MultiRangeMatcher(
                               rangeFn=tcRange,
                               noSingletons=False,
                               helpdesc='Traffic Class (TC) '
                                        'numeric values or ranges',
                               priority=CliParser.PRIO_HIGH ) ),
   }

   @staticmethod
   def handler( mode, args ):
      if 'default' in args:
         color = args[ 'COLOR' ]
         if color in teCbfColorTcMap.colorToValue:
            # Clean up any state for specific traffic classes
            if not teCbfColorTcMap.colorToValue[ color ].isDefault:
               teCbfColorTcMap.colorToValue[ color ].value.clear()
         colorToTc = teCbfColorTcMap.newColorToValue( color )
         colorToTc.isDefault = True
         return
      tcValues = args.get( 'TRAFFIC_CLASS', [] )
      tcValues += \
            args[ 'TRAFFIC_CLASSES' ].values() if 'TRAFFIC_CLASSES' in args else []
      colorToTc = teCbfColorTcMap.newColorToValue( args[ 'COLOR' ] )
      # Clean up any state for default TCs.
      if colorToTc.isDefault:
         colorToTc.isDefault = False
      for tc in tcValues:
         colorToTc.value[ getCbfValue( tc, "tc" ) ] = True

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      color = args[ 'COLOR' ]
      if color not in teCbfColorTcMap.colorToValue:
         return
      if 'default' in args:
         # Removing default TCs setting.  If set delete the color, otherwise
         # do nothing.
         if teCbfColorTcMap.colorToValue[ color ].isDefault:
            del teCbfColorTcMap.colorToValue[ color ]
         return
      colorToTc = teCbfColorTcMap.colorToValue[ color ]
      tcValues = args.get( 'TRAFFIC_CLASS', [] )
      tcValues += \
            args[ 'TRAFFIC_CLASSES' ].values() if 'TRAFFIC_CLASSES' in args else []
      if tcValues:
         for tc in tcValues:
            del colorToTc.value[ getCbfValue( tc, "tc" ) ]
         if not colorToTc.value and \
            not teCbfColorTcMap.colorToValue[ color ].isDefault:
            del teCbfColorTcMap.colorToValue[ color ]
      else:
         del teCbfColorTcMap.colorToValue[ color ]

TeColorTcMappingsMode.addCommandClass( TeColorTcMappingEntryCmd )

# -----------------------------------------------------------------------------
# Color Vrf Mapping Mode
# ----------------------------------------------------------------------------
def vrfSelectionSupportedGuard( mode, token ):
   if vrfSelPolicyHwCapability.vrfSelectionSupported:
      return None
   return CliParser.guardNotThisPlatform

class TeColorVrfMappingsMode( TeColorVrfModeBase, BasicCli.ConfigModeBase ):
   name = "Color/VRF mapping configuration"

   def __init__( self, parent, session ):
      TeColorVrfModeBase.__init__( self, None )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

class TeCbfColorVrfMappingsCmd( CliCommand.CliCommandClass ):
   syntax = 'color-vrf-mappings'
   noOrDefaultSyntax = syntax
   data = {
         'color-vrf-mappings': CliCommand.guardedKeyword(
            'color-vrf-mappings',
            TeColorVrfMappingsMode.name,
            guard=vrfSelectionSupportedGuard ),
   }

   @staticmethod
   def handler( mode, args ):
      if srTeCbfColorDscpMap.colorToValue.items() or \
            teCbfColorDscpMap.colorToValue.items() or \
            teCbfColorTcMap.colorToValue.items():
         mode.addWarning( "Color/VRF mappings in the traffic-engineering mode"
               " will not take effect until the Color/DSCP mappings in the"
               " te-segment-routing, traffic-engineering modes and Color/TC"
               " mappings under traffic-engineering mode are removed" )
      childMode = mode.childMode( TeColorVrfMappingsMode )
      mode.session_.gotoChildMode( childMode )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      teCbfColorVrfMap.colorToValue.clear()

TeCbfMode.addCommandClass( TeCbfColorVrfMappingsCmd )

# -------------------------------------------------------------------------------
# [ no | default ] color COLOR ( vrf VRF | default )
# in te color-vrf-mappings mode
# -------------------------------------------------------------------------------

class TeColorVrfMappingEntryCmd( CliCommand.CliCommandClass ):
   syntax = '''color COLOR vrf ( default | { VRF_NAMES }  )'''
   noOrDefaultSyntax = '''color COLOR
                          [ ( vrf ( default | { VRF_NAMES } ) ) ]'''
   data = {
      'color': 'Configure the mapping for a color',
      'COLOR': CliMatcher.IntegerMatcher( TacColor.min, TacColor.max,
                            helpdesc='Color value' ),
      'vrf': 'Map the specified color to a specific VRF',
      'default': 'Map the specified color to default VRF',
      'VRF_NAMES': nonDefaultVrfMatcherExcludeAll,
   }

   @staticmethod
   def handler( mode, args ):
      if 'default' in args:
         color = args[ 'COLOR' ]
         if color in teCbfColorVrfMap.colorToValue:
            # Clean up any state for specific VRF
            if not teCbfColorVrfMap.colorToValue[ color ].isDefault:
               teCbfColorVrfMap.colorToValue[ color ].value.clear()
         colorToVrf = teCbfColorVrfMap.newColorToValue( color )
         colorToVrf.isDefault = True
         return
      vrfNames = args.get( 'VRF_NAMES', [] )
      colorToVrf = teCbfColorVrfMap.newColorToValue( args[ 'COLOR' ] )
      # Clean up any state for default VRFs.
      if colorToVrf.isDefault:
         colorToVrf.isDefault = False
      for vrfName in vrfNames:
         colorToVrf.value[ getCbfValue( vrfName, "vrf" ) ] = True

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      color = args[ 'COLOR' ]
      if color not in teCbfColorVrfMap.colorToValue:
         return
      if 'default' in args:
         # Removing default VRFs setting.  If set delete the color, otherwise
         # do nothing.
         if teCbfColorVrfMap.colorToValue[ color ].isDefault:
            del teCbfColorVrfMap.colorToValue[ color ]
         return
      colorToVrf = teCbfColorVrfMap.colorToValue[ color ]
      vrfNames = args.get( 'VRF_NAMES', [] )
      if vrfNames:
         for vrfName in vrfNames:
            del colorToVrf.value[ getCbfValue( vrfName, "vrf" ) ]
         if not colorToVrf.value and \
               not teCbfColorVrfMap.colorToValue[ color ].isDefault:
            del teCbfColorVrfMap.colorToValue[ color ]
      else:
         del teCbfColorVrfMap.colorToValue[ color ]

TeColorVrfMappingsMode.addCommandClass( TeColorVrfMappingEntryCmd )

# -----------------------------------------------------------------------------
# cbf ecmp endpoints multiple command in te-cbf mode
# ----------------------------------------------------------------------------
cbfKw = CliMatcher.KeywordMatcher( 'cbf',
                  helpdesc='Class Based Forwarding configuration' )
ecmpKw = CliMatcher.KeywordMatcher( 'ecmp',
                  helpdesc='Class Based Forwarding ECMP configuration' )
endpointsKw = CliMatcher.KeywordMatcher( 'endpoints',
                  helpdesc='Form ECMP to endpoints with CBF' )
multipleKw = CliMatcher.KeywordMatcher( 'multiple',
                  helpdesc='Form ECMP to multiple endpoints with CBF' )

def setCbfEcmpEndpoints( mode, args ):
   no = CliCommand.isNoOrDefaultCmd( args )
   teConfig.cbfEcmpEndpoints = not no

class CbfEcmpEndpointsBase( CliCommand.CliCommandClass ):
   syntax = 'cbf ecmp endpoints multiple'
   noOrDefaultSyntax = 'cbf ecmp endpoints ...'
   data = {
      'cbf': cbfKw,
      'ecmp': ecmpKw,
      'endpoints': endpointsKw,
      'multiple': multipleKw,
   }

   handler = setCbfEcmpEndpoints
   noOrDefaultHandler = handler

class TeCbfEcmpEndpointsCmd( CbfEcmpEndpointsBase ):
   data = CbfEcmpEndpointsBase.data.copy()

TeCbfMode.addCommandClass( TeCbfEcmpEndpointsCmd )

# Backward compatibility
class TeEcmpEndpointsHiddenCmd( CbfEcmpEndpointsBase ):
   data = CbfEcmpEndpointsBase.data.copy()
   hidden = True

RouterGlobalTeMode.addCommandClass( TeEcmpEndpointsHiddenCmd )

# --x--x--x--x--x--x--x--x--x--x--x-- MPLS CBF --x--x--x--x--x--x--x--x--x--x--x--x--

# --------------------------------------------------------------------------------
# forwarding-plane mpls submode in te-cbf mode
# --------------------------------------------------------------------------------
def mplsCbfHwSupportedGuard( mode, token ):
   if routingHwStatus.mplsCbfSupported:
      return None
   return CliParser.guardNotThisPlatform

class CbfFwdPlaneMplsSubMode( CbfFwdPlaneMplsSubModeBase, BasicCli.ConfigModeBase ):
   name = "Class Based Forwarding configuration"

   def __init__( self, parent, session ):
      CbfFwdPlaneMplsSubModeBase.__init__( self, None )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

class CbfFwdPlaneMplsSubModeCmd( CliCommand.CliCommandClass ):
   syntax = 'forwarding-plane mpls'
   noOrDefaultSyntax = syntax
   data = {
         'forwarding-plane': CliCommand.guardedKeyword(
                                                'forwarding-plane',
                                                CbfFwdPlaneMplsSubMode.name,
                                                guard=mplsCbfHwSupportedGuard ),
         'mpls': "Class Based Forwarding MPLS configuration",
   }

   @staticmethod
   def handler( mode, args ):
      childMode = mode.childMode( CbfFwdPlaneMplsSubMode )
      mode.session_.gotoChildMode( childMode )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      teConfig.cbfMplsColorDefFallback = False
      teConfig.cbfMplsTunnelingLdp = False
      teConfig.cbfMplsTunnelingIsisSr = False

if IpLibToggleLib.toggleMplsCbfLdpOverRsvpTeEnabled():
   TeCbfMode.addCommandClass( CbfFwdPlaneMplsSubModeCmd )

# -------------------------------------------------------------------------------
# [ no | default ] color default fallback highest
# in te-cbf-mpls mode
# -------------------------------------------------------------------------------
class CbfMplsColorDefaultFallbackCmd( CliCommand.CliCommandClass ):
   syntax = 'color default fallback highest'
   noOrDefaultSyntax = syntax
   data = {
         'color': "Associate the highest color of tunnels with any unmapped traffic"
                  " class",
         'default': "Associate the highest color of tunnels with any unmapped "
                    " class",
         'fallback': "Associate the highest color of tunnels with any unmapped "
                     "traffic class",
         'highest': "Associate the highest color of tunnels with any unmapped "
                    "traffic class",
   }

   @staticmethod
   def handler( mode, args ):
      teConfig.cbfMplsColorDefFallback = True

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      teConfig.cbfMplsColorDefFallback = False

if IpLibToggleLib.toggleMplsCbfLdpOverRsvpTeEnabled():
   CbfFwdPlaneMplsSubMode.addCommandClass( CbfMplsColorDefaultFallbackCmd )

# -------------------------------------------------------------------------------
# [ no | default ] tunneling ldp
# in te-cbf-mpls mode
# -------------------------------------------------------------------------------
class CbfMplsTunnelingLdpCmd( CliCommand.CliCommandClass ):
   syntax = 'tunneling ldp'
   noOrDefaultSyntax = syntax
   data = {
         'tunneling': "CBF configuration for tunneled packets",
         'ldp': "Enable CBF for labeled traffic related to LDPoverRSVP-TE",
   }

   @staticmethod
   def handler( mode, args ):
      teConfig.cbfMplsTunnelingLdp = True

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      teConfig.cbfMplsTunnelingLdp = False

if IpLibToggleLib.toggleMplsCbfLdpOverRsvpTeEnabled():
   CbfFwdPlaneMplsSubMode.addCommandClass( CbfMplsTunnelingLdpCmd )

# -------------------------------------------------------------------------------
# [ no | default ] tunneling segment-routing
# in te-cbf-mpls mode
# -------------------------------------------------------------------------------
class CbfMplsTunnelingIsisSrCmd( CliCommand.CliCommandClass ):
   syntax = 'tunneling segment-routing'
   noOrDefaultSyntax = syntax
   data = {
         'tunneling': "CBF configuration for tunneled packets",
         'segment-routing': "Enable CBF for labeled traffic through"
                            " ISIS-SR over RSVP-TE tunnels",
   }

   @staticmethod
   def handler( mode, args ):
      teConfig.cbfMplsTunnelingIsisSr = True

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      teConfig.cbfMplsTunnelingIsisSr = False

if IpLibToggleLib.toggleMplsCbfSrOverRsvpTeEnabled():
   CbfFwdPlaneMplsSubMode.addCommandClass( CbfMplsTunnelingIsisSrCmd )

def Plugin( entityManager ):
   global teConfig
   global routingHwStatus
   global srTeCbfColorDscpMap
   global teCbfColorDscpMap
   global teCbfColorTcMap
   global vrfSelPolicyHwCapability
   global teCbfColorVrfMap

   teConfig = ConfigMount.mount( entityManager, 'te/config',
                                 'TrafficEngineering::Config', 'w' )
   routingHwStatus = LazyMount.mount( entityManager,
                                      "routing/hardware/status",
                                      "Routing::Hardware::Status", "r" )
   srTeCbfColorDscpMap = LazyMount.mount( entityManager,
                                          'te/segmentrouting/cbf/color-dscp/config',
                                          'TrafficEngineering::CbfConfigMap', 'w' )
   teCbfColorDscpMap = ConfigMount.mount( entityManager, 'te/cbf/color-dscp/config',
                                          'TrafficEngineering::CbfConfigMap', 'w' )
   teCbfColorTcMap = ConfigMount.mount( entityManager, 'te/cbf/color-tc/config',
                                        'TrafficEngineering::CbfConfigMap', 'w' )
   vrfSelPolicyHwCapability = LazyMount.mount( entityManager,
                                               'vrfSelectionPolicy/hwcapabilities',
                                               'VrfSelectionPolicy::HwCapabilities',
                                               'r' )
   teCbfColorVrfMap = ConfigMount.mount( entityManager, 'te/cbf/color-vrf/config',
                                         'TrafficEngineering::CbfConfigMap', 'w' )

   SrTeModeDependents.registerDependentClass( SrTeCbfSubMode, priority=20 )
   TeModeDependents.registerDependentClass( TeCbfSubMode, priority=20 )
