# Copyright (c) 2022 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# pylint: disable=consider-using-f-string

import Ark
import Arnet
from CliModel import Model, Str, Int, List, Float, Dict, Bool
from IntfModels import Interface
import LazyMount
import TableOutput
import Tac

cdpStatus = None
cdpConfig = None

firstColumnFormat = TableOutput.Format( justify="left" )
firstColumnFormat.noPadLeftIs( True )
defaultColumnFormat = TableOutput.Format( justify="left" )

class CdpNeighbor( Model ):
   localPort = Interface( help="Name of the interface" )
   neighborDevice = Str( help="Name of the neighbor device", optional=True )
   neighborPort = Str( help="Name of the neighbor port", optional=True )
   ttl = Int( help="The Time To Live value (in seconds) advertised by the neighbor" )

   def addToTable( self, table ):
      table.newRow( self.localPort.stringValue,
                    self.neighborDevice or "None",
                    self.neighborPort or "None",
                    self.ttl )

   @staticmethod
   def create( nbr ):
      res = CdpNeighbor()
      res.localPort = nbr.intfId
      res.ttl = nbr.ttl

      deviceId = nbr.deviceId.decode( "utf-8", errors="backslashreplace" )
      if deviceId:
         res.neighborDevice = deviceId

      nbrPort = nbr.portId.decode( "utf-8", errors="backslashreplace" )
      if nbrPort:
         res.neighborPort = nbrPort

      return res

class CdpInterface( Model ):
   cdpNeighbors = List( valueType=CdpNeighbor, help="CDP neighbors" )

   def addToTable( self, table ):
      for neighbor in self.cdpNeighbors:
         neighbor.addToTable( table )

   @staticmethod
   def create( intfStatus ):
      res = CdpInterface()
      for neighbor in intfStatus.neighbor.values():
         res.cdpNeighbors.append( CdpNeighbor.create( neighbor ) )

      return res if res.cdpNeighbors else None

class CdpNeighbors( Model ):
   lastRxTime = Float( help="The last time CDP Agent received packet",
                       optional=True )
   rxFramesTotal = Int( help="The total number of packets that CDP Agent received",
                        default=0 )
   rxFramesValidTotal = Int( help="The total number of valid frames received by the"
                                  " CDP Agent", default=0 )
   rxFrameErrorsTotal = Int( help="The total numer of frames received by the CDP"
                                  " Agent that were discarded because of invalid"
                                  " frame", default=0 )
   rxProtocolErrorsTotal = Int( help="The total numer of frames received by the CDP"
                                     " Agent that were discarded because of errors"
                                     " in CDP protocol decoding", default=0 )
   rxTLVsDiscardedTotal = Int( help="The total numer of discarded TLVs received by"
                                    " the CDP Agent", default=0 )
   rxTLVsUnrecognizedTotal = Int( help="The total numer of unknown TLVs received by"
                                    " the CDP Agent", default=0 )
   nbrInserts = Int( help="The total numer of added neighbors", default=0 )
   nbrDeletes = Int( help="The total numer of removed neighbors", default=0 )

   rxEnabled = Bool( help="True if CDP packets are parsed" )

   cdpInterfaces = Dict( keyType=Interface,
                         valueType=CdpInterface,
                         help="CDP enabled interfaces" )

   def render( self ):
      if self.lastRxTime:
         timestamp = Ark.utcTimeRelativeToNowStr( self.lastRxTime )
      else:
         timestamp = "Never"
      rxStatus = "enabled" if self.rxEnabled else "not configured"
      print( 'Packet receive: %s' % rxStatus )
      print( 'Last received packet: %s' % timestamp )
      print( 'Received packets: %d' % self.rxFramesTotal )
      print( 'Received valid packets: %d' % self.rxFramesValidTotal )
      print( 'Discarded packets (frame parser error): %d' % self.rxFrameErrorsTotal )
      print( 'Discarded packets (CDP protocol error): %d' % \
             self.rxProtocolErrorsTotal )
      print( 'Discarded TLVs: %d' % self.rxTLVsDiscardedTotal )
      print( 'Unknown TLVs: %d' % self.rxTLVsUnrecognizedTotal )
      print( 'Neighbors added: %d' % self.nbrInserts )
      print( 'Neighbors removed: %d' % self.nbrDeletes )
      print()
      table = TableOutput.createTable( [ 'Local Interface',
                                         'Neighbor Device ID',
                                         'Neighbor Interface',
                                         'TTL (seconds)' ] )
      table.formatColumns( firstColumnFormat,
                           defaultColumnFormat,
                           defaultColumnFormat,
                           defaultColumnFormat )
      for interface in Arnet.sortIntf( self.cdpInterfaces ):
         cdpInterface = self.cdpInterfaces[ interface ]
         cdpInterface.addToTable( table )
      print( table.output() )

   @staticmethod
   def create( mode, args ):
      res = CdpNeighbors()
      if cdpStatus.lastRxTime > 0:
         res.lastRxTime = Ark.switchTimeToUtc( cdpStatus.lastRxTime )
      else:
         res.lastRxTime = None
      res.rxFramesTotal = cdpStatus.rxFramesTotal
      res.rxFramesValidTotal = cdpStatus.rxFramesValidTotal
      res.rxFrameErrorsTotal = cdpStatus.rxFrameErrorsTotal
      res.rxProtocolErrorsTotal = cdpStatus.rxProtocolErrorsTotal
      res.rxTLVsDiscardedTotal = cdpStatus.rxTLVsDiscardedTotal
      res.rxTLVsUnrecognizedTotal = cdpStatus.rxTLVsUnrecognizedTotal
      res.nbrInserts = cdpStatus.nbrInserts
      res.nbrDeletes = cdpStatus.nbrDeletes
      res.rxEnabled = cdpConfig.rxEnabled
      for intf, intfStatus in cdpStatus.intfStatus.items():
         cdpInterface = CdpInterface.create( intfStatus )
         if cdpInterface:
            res.cdpInterfaces[ intf ] = cdpInterface
      return res

def Plugin( entityManager ):
   global cdpStatus
   global cdpConfig
   cdpStatus = LazyMount.mount(
      entityManager, "l2discovery/cdp/status", "Cdp::CdpStatus", "r" )
   cdpConfig = LazyMount.mount(
      entityManager, "l2discovery/cdp/config", "Cdp::CdpConfig", "r" )
