# Copyright (c) 2017 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from __future__ import absolute_import, division, print_function
from collections import namedtuple
import datetime

import CliModel
from CliModel import Enum
from CliModel import Model
from CliModel import Str
from CliPlugin import SslModel
import TableOutput

rotationStateMapping = { 'importPending': 'Import Pending',
                         'importExpired': 'Import Expired',
                         'commitPending': 'Commit Pending',
                         'commitExpired': 'Commit Expired' }

statusCodes = [ "success", "invalidRotationId", "invalidCert",
                "certDoesNotExist","certDoesNotMatchKey", 
                "certNotYetValid","certExpired", "requestFailed",
                "invalidSslProfile", "invalidKey" ]
StatusCodeType = namedtuple( "StatusCode", statusCodes )
StatusCode = StatusCodeType( *statusCodes )

class RotationGenerateSignRequestStatus( Model ):
   statusCode = Enum( help="Status code",
                      values=( StatusCode.success, StatusCode.requestFailed ) )
   errorMessage = Str( help="Error message if not successful",
                       optional=True )
   rotationId = Str( help="Unique rotation ID", optional=True )
   csr = Str( help="PEM encoded CSR", optional=True )

   def render( self ):
      if self.statusCode == StatusCode.success:
         print( f"Rotation ID: {self.rotationId}" )
         print( "Certificate Signing Request:" )
         print( self.csr )
      else:
         print( f"Error: {self.errorMessage}" )

class RotationImportStatus( Model ):
   statusCode = Enum( help="Status code",
                      values=( StatusCode.success, 
                               StatusCode.invalidRotationId,
                               StatusCode.invalidCert,
                               StatusCode.certDoesNotMatchKey,
                               StatusCode.certExpired,
                               StatusCode.requestFailed ) )
   errorMessage = Str( help="Error message if not successful",
                       optional=True )
   rotationId = Str( help="Unique rotation ID" )
   
   def render( self ):
      if self.statusCode == StatusCode.success:
         print( "Success" )
      else:
         print( f"Error: {self.errorMessage}" )
   
class RotationCommitStatus( Model ):
   statusCode = Enum( help="Status code",
                      values=( StatusCode.success, 
                               StatusCode.invalidRotationId,
                               StatusCode.invalidSslProfile,
                               StatusCode.certDoesNotExist,
                               StatusCode.certNotYetValid,
                               StatusCode.certExpired,
                               StatusCode.requestFailed ) )
   errorMessage = Str( help="Error message if not successful",
                       optional=True )
   rotationId = Str( help="Unique rotation ID" )
      
   def render( self ):
      if self.statusCode == StatusCode.success:
         print( "Success" )
      else:
         print( f"Error: {self.errorMessage}" )

class RotationCommitSslProfileStatus( Model ):
   statusCode = Enum( help="Status code",
                      values=( StatusCode.success, 
                               StatusCode.invalidSslProfile,
                               StatusCode.invalidCert,
                               StatusCode.invalidKey,
                               StatusCode.certDoesNotMatchKey,
                               StatusCode.certNotYetValid,
                               StatusCode.certExpired,
                               StatusCode.requestFailed ) )
   errorMessage = Str( help="Error message if not successful",
                       optional=True )
      
   def render( self ):
      if self.statusCode == StatusCode.success:
         print( "Success" )
      else:
         print( f"Error: {self.errorMessage}" )

class RotationClearStatus( Model ):
   statusCode = Enum( help="Status code",
                      values=( StatusCode.success, 
                               StatusCode.invalidRotationId ) )
   errorMessage = Str( help="Error message if not successful",
                       optional=True )
   rotationId = Str( help="Unique rotation ID" )
      
   def render( self ):
      if self.statusCode == StatusCode.success:
         print( "Success" )
      else:
         print( f"Error: {self.errorMessage}" )

class CertificateRotation( CliModel.Model ):
   rotationId = CliModel.Str( help='Rotation ID' )
   profileName = CliModel.Str( help='SSL profile name for this rotation' )
   rotationState = CliModel.Enum( values=list( rotationStateMapping ),
                                  help='Rotation states. '
                                  'importPending: Pending import of '
                                  'certificate. '
                                  'importExpired: Certificate was not imported '
                                  'within importTimeout window. '
                                  'commitPending: Certificate has been '
                                  'imported, waiting on commit. '
                                  'commitExpired: Certificate expired before '
                                  ' rotation was comitted.' )
   importTimeout = CliModel.Int( help='Number of mins after rotation creation time '
                                      'that a certificate must be imported by' )
   creationTime = CliModel.Int( help='Time rotation was created' )
   certificateSigningRequest = CliModel.Submodel(
                                       valueType=SslModel.CertificateSigningRequest,
                                       help='The certificate signing request '
                                            'for the rotation' )
   certificate = CliModel.Submodel( valueType=SslModel.Certificate, optional=True,
                                    help='Certificate imported into this rotation' )

   def getExpiry( self ):
      ''' If a certificate has been imported then we display certificate's expiration
          date, otherwise we return the UTC of the import timeout'''
      if self.certificate:
         return self.certificate.notAfter
      else:
         return self.importTimeout * 60 + self.creationTime

   def render( self ):
      print( 'Rotation ID:', self.rotationId )
      dt = datetime.datetime.fromtimestamp( self.creationTime )
      print( '   Creation Time:', dt.strftime( '%Y-%m-%d %H:%M:%S' ) )
      print( '   SSL Profile Name:', self.profileName )
      print( '   State:', rotationStateMapping[ self.rotationState ] )
      print( f'   Import Timeout: {self.importTimeout} mins' )
      dt = datetime.datetime.fromtimestamp( self.getExpiry() )
      print( '   Expiry Time:', dt.strftime( '%Y-%m-%d %H:%M:%S' ) )
      print( '   ----------------Begin CSR----------------' )
      self.certificateSigningRequest.render()
      print( '   -----------------End CSR-----------------' )
      if self.certificate:
         print()
         print( '   ------------Begin Certificate------------' )
         self.certificate.render()
         print( '   -------------End Certificate-------------' )
      else:
         print()
         print( '   Cerificate has not been imported' )

class CertificateRotations( CliModel.Model ):
   rotations = CliModel.Dict( valueType=CertificateRotation,
                  help='A mapping between Rotation ID and rotation information' )
   _detail = CliModel.Bool( help='Should we print detailed information' )

   def renderSummary( self ):
      headings = ( 'Rotation ID', 'Profile Name', 'State', 'Expiry' )
      table = TableOutput.createTable( headings )
      fmt = TableOutput.Format( justify='left' )
      table.formatColumns( fmt, fmt, fmt, fmt )
      for rotationId, rotationInfo in sorted( self.rotations.items(),
                                              key=lambda x:x[ 1 ].profileName ):
         dt = datetime.datetime.fromtimestamp( rotationInfo.getExpiry() )
         expiryStr = dt.strftime( '%Y-%m-%d %H:%M:%S' )
         table.newRow( rotationId, rotationInfo.profileName,
                       rotationStateMapping[ rotationInfo.rotationState ],
                       expiryStr )
      print( table.output() )

   def renderDetail( self ):
      for _, rotationInfo in sorted( self.rotations.items(),
                                     key=lambda x:x[ 1 ].profileName ):
         rotationInfo.render()
         print()

   def render( self ):
      if self._detail:
         self.renderDetail()
      else:
         self.renderSummary()
