# Copyright (c) 2021 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from CliCommand import (
   CliExpression,
   guardedKeyword,
   Node,
)
from CliMatcher import (
   DynamicNameMatcher,
   IntegerMatcher,
   KeywordMatcher,
   PatternMatcher,
)
from CfmTypes import (
   tacMdNameFormat,
   tacMaNameFormat,
)
import CliParser
import ConfigMount
import LazyMount
from TypeFuture import TacLazyType

cfmConfig = None
cfmHwSupportStatus = None

#-----------------------------------------------------------
# Cli Guard
#-----------------------------------------------------------
def cfmSupported( mode, token ):
   if not cfmHwSupportStatus.hwSupported() and \
      not cfmHwSupportStatus.mepSwSupported():
      return CliParser.guardNotThisPlatform
   return None

def defaultMipSupported( mode, token ):
   if not cfmHwSupportStatus.defaultMipHwSupported:
      return CliParser.guardNotThisPlatform
   return None

def mepSupported( mode, token ):
   if ( not cfmHwSupportStatus.mepHwSupported() and
        not cfmHwSupportStatus.mepSwSupported() ):
      return CliParser.guardNotThisPlatform
   return None

def mepSwSupported( mode, token ):
   if not cfmHwSupportStatus.mepSwSupported():
      return CliParser.guardNotThisPlatform
   return None

def mepHwSupported( mode, token ):
   if not cfmHwSupportStatus.mepHwSupported():
      return CliParser.guardNotThisPlatform
   return None

def upMepSupported( mode, token ):
   if ( not cfmHwSupportStatus.upMepHwSupported and
        not cfmHwSupportStatus.mepSwSupported() ):
      return CliParser.guardNotThisPlatform
   return None

def downMepSupported( mode, token ):
   if ( not cfmHwSupportStatus.downMepHwSupported or
        cfmHwSupportStatus.mepSwSupported() ):
      return CliParser.guardNotThisPlatform
   return None

def ccmAlarmSupported( mode, token ):
   if ( not cfmHwSupportStatus.ccmAlarmHwSupported and
        not cfmHwSupportStatus.cfmSwSupported ):
      return CliParser.guardNotThisPlatform
   return None

def dmHwSupported( mode, token ):
   if not cfmHwSupportStatus.dmHwSupported:
      return CliParser.guardNotThisPlatform
   return None

def anyLmHwSupported( mode, token ):
   if cfmHwSupportStatus.lmHwSupported or cfmHwSupportStatus.slmHwSupported:
      return None
   return CliParser.guardNotThisPlatform

def lmHwSupported( mode, token ):
   if not cfmHwSupportStatus.lmHwSupported:
      return CliParser.guardNotThisPlatform
   return None

def slmHwSupported( mode, token ):
   if not cfmHwSupportStatus.slmHwSupported:
      return CliParser.guardNotThisPlatform
   return None

def pmRawReportHwSupported( mode, token ):
   if not cfmHwSupportStatus.pmRawReportHwSupported:
      return CliParser.guardNotThisPlatform
   return None

def pmPriorityHwSupported( mode, token ):
   if ( not cfmHwSupportStatus.pmPriorityHwSupported and
        not cfmHwSupportStatus.cfmSwSupported ):
      return CliParser.guardNotThisPlatform
   return None

def aisHwSupported( mode, token ):
   if not cfmHwSupportStatus.aisHwSupported:
      return CliParser.guardNotThisPlatform
   return None

def pmTxIntervalEnumNeeded( mode, token ):
   if not cfmHwSupportStatus.pmTxIntervalEnumNeeded:
      return CliParser.guardNotThisPlatform
   return None

def pmTxIntervalFullRangeSupported( mode, token ):
   if cfmHwSupportStatus.pmTxIntervalEnumNeeded:
      return CliParser.guardNotThisPlatform
   return None

MepDirectionEnum = TacLazyType( 'Cfm::MepDirection' )
mepDirectionUnknown = MepDirectionEnum.MepDirectionUnknown
mepDirectionUp = MepDirectionEnum.MepDirectionUp
mepDirectionDown = MepDirectionEnum.MepDirectionDown
mepDirectionShowStr = {
      mepDirectionUnknown : 'unconfigured',
      mepDirectionUp : 'up',
      mepDirectionDown : 'down',
}

PmOperModeEnum = TacLazyType( 'Cfm::PmOperMode' )
pmOperModeProactive = PmOperModeEnum.OperModeProactive
pmOperModeOnDemand = PmOperModeEnum.OperModeOnDemand
pmOperModeKwStr = {
   pmOperModeProactive : 'proactive',
   pmOperModeOnDemand : 'on-demand',
}

AisDefectConditionEnum = TacLazyType( 'Cfm::AisDefectCondition' )
continuityCheckDefect = AisDefectConditionEnum.continuityCheck
aisReceivedDefect = AisDefectConditionEnum.alarmIndicationReceived
aisDefectKwStr = {
   continuityCheckDefect : 'Continuity check',
   aisReceivedDefect : 'AIS received',
}

def getPmOperModeFromArgs( args ):
   pmOperMode = None
   if pmOperModeKwStr[ pmOperModeProactive ] in args:
      pmOperMode = pmOperModeProactive
   elif pmOperModeKwStr[ pmOperModeOnDemand ] in args:
      pmOperMode = pmOperModeOnDemand
   else:
      assert 0
   return pmOperMode

def getDomainNames( mode ):
   return list( cfmConfig.mdConfig )

cfmDomainMatcher = DynamicNameMatcher( getDomainNames,
                                       'Configure a maintenance domain' )

cfmShowKw = guardedKeyword( 'cfm',
                            helpdesc='Connectivity fault management information',
                            guard=cfmSupported )
endPointShowKw = guardedKeyword( 'end-point',
                                 helpdesc='Maintenance end point information',
                                 guard=mepSupported )
continuityCheckShowKw = guardedKeyword( 'continuity-check',
                                 helpdesc='Continuity check information',
                                 guard=mepSupported )
endPointDetailShowKw = guardedKeyword( 'detail',
                                 helpdesc='Continuity check detailed information',
                                 guard=ccmAlarmSupported )
remoteDefectShowKw = guardedKeyword( 'remote-defect',
                                 helpdesc='End point remote defect information',
                                 guard=mepSupported )
measurementShowKw = guardedKeyword( 'measurement',
                                    helpdesc='Peformance monitoring information',
                                    guard=mepHwSupported )
delayMeasurementKw = guardedKeyword( 'delay',
                                     helpdesc='Delay measurement information',
                                     guard=dmHwSupported )
lossMeasurementKw = guardedKeyword( 'loss',
                                    helpdesc='Loss measurement information',
                                    guard=lmHwSupported )
anyLossMeasurementKw = guardedKeyword( 'loss',
                                    helpdesc='Loss measurement information',
                                    guard=anyLmHwSupported )
syntheticLossMeasurementKw = guardedKeyword( 'synthetic',
                                 helpdesc='Synthetic loss measurement information',
                                 guard=slmHwSupported )
detailShowKw = KeywordMatcher( 'detail',
                               helpdesc='Detailed measurement information' )
lmDetailShowKw = guardedKeyword( 'detail',
                                 helpdesc='Detailed measurement information',
                                 guard=pmRawReportHwSupported )
proactiveShowKw = KeywordMatcher( 'proactive',
                             helpdesc='Proactive peformance monitoring information' )
onDemandShowKw = KeywordMatcher( 'on-demand',
                             helpdesc='On-demand peformance monitoring information' )
alarmIndicationShowKw = guardedKeyword(
   'alarm', helpdesc='Alarm indication information', guard=aisHwSupported )
domainKw = KeywordMatcher( 'domain', helpdesc='Maintenance domain' )
domainNameValue = Node( cfmDomainMatcher, maxMatches=1 )
endPointKw = KeywordMatcher( 'end-point', helpdesc='Maintenance end point' )
_endPointMatcher = IntegerMatcher( 0, 8191, helpdesc='Maintenance end point ID' )
endPointValue = Node( _endPointMatcher, maxMatches=1 )
associationKw = KeywordMatcher( 'association', helpdesc='Maintenance association' )
_associationMatcher = PatternMatcher( r'.{1,45}', helpname='STRING',
                                   helpdesc='Maintenance association name' )
associationValue = Node( _associationMatcher, maxMatches=1 )
domainLevelKw = KeywordMatcher( 'domain-level', helpdesc='Maintenance domain level' )
_domainLevelMatcher = IntegerMatcher( 0, 7, helpdesc='Maintenance domain level' )
domainLevelValue = Node( _domainLevelMatcher, maxMatches=1 )
cfmCcCountersKw = guardedKeyword( 'counters',
                                  helpdesc='CFM CCM counters',
                                  guard=mepSwSupported )
cfmCcCountersDetailShowKw = KeywordMatcher( 'detail',
                                            helpdesc='Detailed CCM counters '
                                                     'information' )
cfmEndPointDetailShowKw = KeywordMatcher( 'detail',
   helpdesc='Detailed maintenance end point information' )
cfmSummaryShowKw = KeywordMatcher( 'summary',
                                   helpdesc='CFM summary' )

class EndPointFilter( CliExpression ):
   # using "END_POINT_KW" as a placeholder workaround to make CliParser happy
   # with repeated ( first in command and then in the filter ) "end-point" tokens
   expression = '''[ ( domain DOMAIN_NAME [ association MA_NAME
                           [ END_POINT_KW MEP_ID ] ] ) ]'''
   data = {
      'domain' : domainKw,
      'DOMAIN_NAME' : domainNameValue,
      'association' : associationKw,
      'MA_NAME' : associationValue,
      'END_POINT_KW' : endPointKw,
      'MEP_ID' : endPointValue,
   }

def Plugin( entityManager ):
   global cfmConfig
   global cfmHwSupportStatus

   cfmConfig = ConfigMount.mount( entityManager, "cfm/config",
                                  "CfmAgent::Config", "w" )
   cfmHwSupportStatus = LazyMount.mount( entityManager, "cfm/hwSupportStatus",
                                         "Cfm::HwSupportStatus", "r" )

mdFormatLegend = {
   tacMdNameFormat.mdNameFormatReserved : '',
   tacMdNameFormat.mdNameFormatNoName : 'N',
   tacMdNameFormat.mdNameFormatDnsName : 'D',
   tacMdNameFormat.mdNameFormatMacAddrPlusShortInt : 'M',
   tacMdNameFormat.mdNameFormatCharString : 'C',
   }

maFormatLegend = {
   tacMaNameFormat.maNameFormatReserved : '',
   tacMaNameFormat.maNameFormatPrimaryVid : 'P',
   tacMaNameFormat.maNameFormatCharString : 'C',
   tacMaNameFormat.maNameFormatShortInt : 'I',
   tacMaNameFormat.maNameFormatRfc2685VpnId : 'V',
   }
