#!/usr/bin/env python3
# Copyright (c) 2021 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import BasicCli
from collections import defaultdict
from CfmTypes import (
   tacCfmConst,
   tacSlmSessionKey,
   tacStatEntryIdx,
   tacRemoteMepDmState,
   tacRemoteMepLmState,
   tacRemoteMepSlmState,
   tacSmashLocalMepKey,
   tacSmashRemoteMepKey,
   tacSmashRemoteMepSlmSessionKey,
   tacMepIntfType,
   tacLocalMepProgState,
   tacRemoteMepCcmState,
   tacLocalMepCcmRxState,
   tacLocalMepCcmTxState,
   tacLocalMepAisState,
)
from CliPlugin.CfmCliLib import (
   aisDefectKwStr,
   alarmIndicationShowKw,
   anyLossMeasurementKw,
   cfmShowKw,
   continuityCheckShowKw,
   delayMeasurementKw,
   detailShowKw,
   EndPointFilter,
   endPointShowKw,
   endPointDetailShowKw,
   getPmOperModeFromArgs,
   lmDetailShowKw,
   lossMeasurementKw,
   measurementShowKw,
   mepDirectionShowStr,
   pmOperModeKwStr,
   proactiveShowKw,
   remoteDefectShowKw,
   syntheticLossMeasurementKw,
   cfmCcCountersKw,
   cfmCcCountersDetailShowKw,
   cfmEndPointDetailShowKw,
   cfmSummaryShowKw,
)
from CliPlugin.CfmCliModel import (
   EndPointModel,
   EndPointCcModel,
   EndPointDmModel,
   EndPointLmModel,
   EndPointRdiModel,
   EndPointSlmModel,
   LocalMepModel,
   LocAndTimeChangeModel,
   RdiAndTimeChangeModel,
   MaModel,
   MdModel,
   RemoteMepCcModel,
   CcmDefectModel,
   LocalMepCcModel,
   MaCcModel,
   MdCcModel,
   RemoteMepRdiModel,
   LocalMepRdiModel,
   MaRdiModel,
   MdRdiModel,
   RemoteMepModel,
   MaDmModel,
   MdDmModel,
   LocalMepDmModel,
   RemoteMepDmModel,
   DmStatsModel,
   MaLmModel,
   MdLmModel,
   LocalMepLmModel,
   RemoteMepLmModel,
   LmStatsModel,
   MaSlmModel,
   MdSlmModel,
   LocalMepSlmModel,
   RemoteMepSlmModel,
   SlmStatsModel,
   AisDefectTimeChangeModel,
   RemoteAisRxEntryModel,
   LocalMepAisModel,
   MaAisModel,
   MdAisModel,
   EndPointAisModel,
   LocalMepVlanModel,
   CcCountersDetailModel,
   CcCountersModel,
   IntfCcCountersModel,
   IntfVlanCcCountersModel,
   MdCcCountersModel,
   MaCcCountersModel,
   LocalMepCcCountersModel,
   LocalMepDetailModel,
   RemoteMepDetailModel,
   CfmSummaryModel,
   CcSummaryLocActionModel,
   CfmSummaryProfileModel,
   CcSummaryDefectAlarmModel,
   DmMiStatsModel,
   SlmMiStatsModel,
)
import ShowCommand
import ConfigMount
import LazyMount
import SmashLazyMount
import Tac
from Toggles.CfmToggleLib import (
   toggleDownMepL2EthIntfEnabled,
   toggleCfmSwUpMepCcmEnabled,
   toggleCfmProgStateEnabled,
   toggleCfmPmMiEnabled,
)
import Tracing
from TypeFuture import TacLazyType
import SharkLazyMount

CcmDefectKey = TacLazyType( 'Cfm::CcmDefectKey' )
DefectCodeEnum = Tac.Type( 'Cfm::DefectCode' )
ServiceIntfType = TacLazyType( 'Ebra::ServiceIntfType' )
CcmDefectAlarm = TacLazyType( 'Cfm::CcmDefectAlarm' )

traceHandle = Tracing.Handle( 'CfmCliShow' )
t0 = traceHandle.trace0

#-----------------------------------------------------------
# The Cfm globals
#-----------------------------------------------------------
cfmConfig = None
cfmStatus = None
aleCfmConfig = None
aleCfmStatus = None
cfmHwSupportStatus = None
trapConfig = None
mepStats = None
aisRxStatus = None
cfmRxCounters = None
cfmTxCounters = None
mepMiStatsHistory = None

class ShowCfmBase:
   def __init__( self, args ):
      self.args = args

   def getMdModel( self, mdConfig ):
      raise NotImplementedError

   def getMaModel( self, maConfig ):
      raise NotImplementedError

   def addOneMep( self, localMepConfig, localMepStatus, maModel ):
      raise NotImplementedError

   def addMaModel( self, mdModel, maName, maModel ):
      raise NotImplementedError

   def addMdModel( self, model, domainName, mdModel ):
      raise NotImplementedError

   def addMeps( self, maConfig, maStatus, maModel ):
      localMepIdFilter = self.args.get( 'MEP_ID' )

      if localMepIdFilter:
         localMepConfig = maConfig.localMepConfig.get( localMepIdFilter )
         if not localMepConfig:
            t0( 'mepId ', localMepIdFilter, ' does not exist' )
            return False
         localMepStatus = maStatus.localMepStatus.get( localMepIdFilter )
         self.addOneMep( localMepConfig, localMepStatus, maModel )
      else:
         for mepId in sorted( maConfig.localMepConfig ):
            localMepConfig = maConfig.localMepConfig.get( mepId )
            localMepStatus = maStatus.localMepStatus.get( mepId )
            self.addOneMep( localMepConfig, localMepStatus, maModel )
      return True

   def addMas( self, mdConfig, mdStatus, mdModel ):
      associations = set( mdConfig.maConfig ) & set( mdStatus.maStatus )
      maNameFilter = self.args.get( 'MA_NAME' )
      t0( 'using maNameFilter ', maNameFilter )
      if maNameFilter and maNameFilter not in mdConfig.maConfig:
         t0( 'maName ', maNameFilter, ' does not exist' )
         return False

      for maName in associations:
         if maNameFilter and maNameFilter != maName:
            continue
         maConfig = mdConfig.maConfig.get( maName )
         maStatus = mdStatus.maStatus.get( maName )
         if not maConfig or not maStatus:
            continue

         maModel = self.getMaModel( maConfig )
         if self.addMeps( maConfig, maStatus, maModel ) or maNameFilter:
            # ignore adding an empty MA, if mepIdFilter matches none
            self.addMaModel( mdModel, maName, maModel )
      return True

   def addDomains( self, mode, model ):
      domainNameFilter = self.args.get( 'DOMAIN_NAME' )

      t0( 'using domainNameFilter ', domainNameFilter )
      if domainNameFilter is not None and \
         domainNameFilter not in aleCfmConfig.mdConfig:
         t0( 'domainName ', domainNameFilter, 'does not exist in aleCfmConfig' )
         return
      domains = set( aleCfmConfig.mdConfig ) & set( cfmStatus.mdStatus )
      t0( 'using domains ', domains )
      for domainName in domains:
         if domainNameFilter is not None and domainNameFilter != domainName:
            continue
         mdConfig = aleCfmConfig.mdConfig.get( domainName )
         mdStatus = cfmStatus.mdStatus.get( domainName )
         if not mdConfig or not mdStatus:
            continue
         t0( 'found domain: ', domainName )
         mdModel = self.getMdModel( mdConfig )
         if self.addMas( mdConfig, mdStatus, mdModel ) or \
            domainNameFilter is not None:
            # ignore adding an empty MD, if maNameFilter matches none
            self.addMdModel( model, domainName, mdModel )

#--------------------------------------------------------------------------------
# show cfm end-point
# [ domain < Domain name > [ association < MA name > [ end-point < MEP ID > ] ] ]
#--------------------------------------------------------------------------------
class ShowEndPoint( ShowCfmBase ):
   def getRemoteMepState( self, remoteMepStatus ):
      status = "inactive"
      if toggleCfmProgStateEnabled():
         # Remote MEP is active if it is programmed for any CFM protocol
         slmProgrammed = False
         for slmSession in remoteMepStatus.mepSlmProgrammingState:
            if remoteMepStatus.mepSlmProgrammingState.get( slmSession ).mepState == \
               tacRemoteMepSlmState.remoteMepSlmProgrammingSuccessful:
               slmProgrammed = True
               break
         if remoteMepStatus.mepCcmProgrammingState.mepState == \
            tacRemoteMepCcmState.remoteMepCcmProgrammed or \
            remoteMepStatus.mepLmProgrammingState.mepState == \
            tacRemoteMepLmState.remoteMepLmProgrammingSuccessful or \
            remoteMepStatus.mepDmProgrammingState.mepState == \
            tacRemoteMepDmState.remoteMepDmProgrammingSuccessful or \
            slmProgrammed:
            status = "active"
      else:
         if remoteMepStatus.mepProgrammingState.mepState == \
            "remoteMepProgrammmed":
            status = "active"
         if remoteMepStatus.rdiState.rdi:
            status = "rdi"
         if remoteMepStatus.locState.locDetected:
            status = "unreachable"
      return status

   def addRemoteMeps( self, localMepConfig, localMepStatus, localMepModel ):
      for remoteMepId in localMepConfig.remoteMepConfig:
         remoteMepModel = RemoteMepModel()
         status = "inactive"
         if localMepStatus:
            remoteMepStatus = localMepStatus.remoteMepStatus.get( remoteMepId )
            if remoteMepStatus:
               status = self.getRemoteMepState( remoteMepStatus )
         remoteMepModel.status = status

         if toggleCfmProgStateEnabled() and 'detail' in self.args:
            rmepCcStatusModel = RemoteMepDetailModel()
            rmepLmStatusModel = RemoteMepDetailModel()
            rmepSlmStatusModel = RemoteMepDetailModel()
            rmepDmStatusModel = RemoteMepDetailModel()
            ccmEnabled = localMepConfig.maConfig.ccmEnable
            lmEnabled = localMepConfig.remoteMepConfig.get( remoteMepId ).lmEnable
            slmEnabled = localMepConfig.remoteMepConfig.get( remoteMepId ).slmEnable
            dmEnabled = localMepConfig.remoteMepConfig.get( remoteMepId ).dmEnable

            remoteMepStatus = None
            if localMepStatus:
               remoteMepStatus = localMepStatus.remoteMepStatus.get( remoteMepId )
            if remoteMepStatus and \
               remoteMepStatus.mepCcmProgrammingState.lastStateChangeTime:
               # Update the model with current state if programming
               # status exists and is updated.
               rmepCcStatusModel.status = 'inactive'
               ccmProgState = remoteMepStatus.mepCcmProgrammingState.mepState
               if ccmProgState == tacRemoteMepCcmState.remoteMepCcmProgrammed:
                  rmepCcStatusModel.status = 'active'
               elif ccmProgState == tacRemoteMepCcmState.remoteMepCcmUnprogrammed:
                  if ccmEnabled:
                     rmepCcStatusModel.reason = 'waiting'
               elif ccmProgState == \
                    tacRemoteMepCcmState.remoteMepCcmProgrammingFailed:
                  rmepCcStatusModel.reason = 'programmingFailed'
               else:
                  t0( 'Unexpected remoteMep CCM programming state', ccmProgState )
            elif ccmEnabled:
               # If CCM is enabled but remoteMep CCM status is not updated,
               # update the model appropriately
               rmepCcStatusModel.status = 'inactive'
               rmepCcStatusModel.reason = 'waiting'
            else:
               t0( 'CCM not enabled/programmed' )

            # LM status
            if remoteMepStatus and \
               remoteMepStatus.mepLmProgrammingState.lastStateChangeTime:
               # Update the model with current state if programming
               # status exists and is updated.
               rmepLmStatusModel.status = 'inactive'
               lmProgState = remoteMepStatus.mepLmProgrammingState.mepState
               if lmProgState == \
                  tacRemoteMepLmState.remoteMepLmProgrammingSuccessful:
                  rmepLmStatusModel.status = 'active'
               elif lmProgState == tacRemoteMepLmState.remoteMepLmDisabled:
                  if lmEnabled:
                     rmepLmStatusModel.reason = 'waiting'
               elif lmProgState == \
                    tacRemoteMepLmState.remoteMepLmProgrammingInProgress:
                  rmepLmStatusModel.reason = 'waiting'
               elif lmProgState == tacRemoteMepLmState.remoteMepLmProgrammingFailed:
                  rmepLmStatusModel.reason = 'programmingFailed'
               else:
                  t0( 'Unexpected remoteMep LM programming state', lmProgState )
            elif lmEnabled:
               # If LM is enabled but remoteMep LM status is not updated,
               # update the model appropriately
               rmepLmStatusModel.status = 'inactive'
               rmepLmStatusModel.reason = 'waiting'
            else:
               t0( 'LM not enabled/programmed' )

            # SLM status
            # For SLM there can be multiple sessions configured and active
            # at the same time. Based on the approved CLI syntax, we will show:
            # - "active" if any SLM session is programmed,
            # - "inactive" if no SLM session is programmed.
            # Based on the scope of current CLI syntax,
            # we cannot show reason for each SLM session for now.
            slmStatusUpdated = False
            if remoteMepStatus:
               for slmSession in remoteMepStatus.mepSlmProgrammingState.values():
                  if slmSession.lastStateChangeTime:
                     slmStatusUpdated = True
                     break
            if slmStatusUpdated:
               rmepSlmStatusModel.status = 'inactive'
               totalSessionCount = 0
               activeSessionCount = 0
               for slmSession in remoteMepStatus.mepSlmProgrammingState.values():
                  totalSessionCount += 1
                  if slmSession.mepState == \
                     tacRemoteMepSlmState.remoteMepSlmProgrammingSuccessful:
                     activeSessionCount += 1
               if activeSessionCount:
                  rmepSlmStatusModel.status = 'active'
               rmepSlmStatusModel.totalSessions = totalSessionCount
               rmepSlmStatusModel.activeSessions = activeSessionCount
            elif slmEnabled:
               # If SLM is enabled but remoteMep SLM status is not updated,
               # update the model appropriately
               rmepSlmStatusModel.status = 'inactive'
               rmepSlmStatusModel.reason = 'waiting'
            else:
               t0( 'SLM not enabled/programmed' )

            # DM status
            if remoteMepStatus and \
               remoteMepStatus.mepDmProgrammingState.lastStateChangeTime:
               # Update the model with current state if programming
               # status exists and is updated.
               rmepDmStatusModel.status = 'inactive'
               dmProgState = remoteMepStatus.mepDmProgrammingState.mepState
               if dmProgState == \
                  tacRemoteMepDmState.remoteMepDmProgrammingSuccessful:
                  rmepDmStatusModel.status = 'active'
               elif dmProgState == tacRemoteMepDmState.remoteMepDmDisabled:
                  if dmEnabled:
                     rmepDmStatusModel.reason = 'waiting'
               elif dmProgState == \
                    tacRemoteMepDmState.remoteMepDmProgrammingInProgress:
                  rmepDmStatusModel.reason = 'waiting'
               elif dmProgState == tacRemoteMepDmState.remoteMepDmProgrammingFailed:
                  rmepDmStatusModel.reason = 'programmingFailed'
               else:
                  t0( 'Unexpected remoteMep DM programming state', dmProgState )
            elif dmEnabled:
               # If DM is enabled but remoteMep DM status is not updated,
               # update the model appropriately
               rmepDmStatusModel.status = 'inactive'
               rmepDmStatusModel.reason = 'waiting'
            else:
               t0( 'DM not enabled/programmed' )
            remoteMepModel.ccStatus = rmepCcStatusModel
            remoteMepModel.lmStatus = rmepLmStatusModel
            remoteMepModel.slmStatus = rmepSlmStatusModel
            remoteMepModel.dmStatus = rmepDmStatusModel
         localMepModel.remoteMeps[ remoteMepId ] = remoteMepModel

   def addOneMep( self, localMepConfig, localMepStatus, maModel ):
      mepModel = LocalMepModel()
      mepModel.intf = localMepConfig.intfId
      status = "inactive"
      if toggleCfmProgStateEnabled():
         if localMepConfig.ready and localMepStatus and \
            localMepStatus.mepCfmProgrammingState.mepState == \
            tacLocalMepProgState.localMepProgrammed:
            status = "active"
      else:
         if localMepConfig.ready and localMepStatus and \
            localMepStatus.mepProgrammingState.mepState == "localMepCcmRxTxActive":
            status = "active"
      mepModel.status = status
      # Add VLAN information for L2 front panel ports if they are supported
      l2EthIntfSupported = cfmHwSupportStatus.isMepIntfAndServiceTypeSupported(
         tacMepIntfType.ethIntf, ServiceIntfType.l2 )
      if l2EthIntfSupported and toggleDownMepL2EthIntfEnabled():
         # If l2 eth intf is supported, then irrespective of interface type,
         # we will populate the LocalMepVlanModel so that we can display
         # "none" where needed for "Active VLAN" or "Primary VLAN" or
         # "Inactive VLAN".
         # If l2 eth intf is not supported, then localMepVlanModel is not
         # needed and will remain "None", so that we don't display the
         # VLAN information line at all.
         mepIntfAndServiceType = localMepConfig.mepIntfAndServiceType
         localMepVlanModel = LocalMepVlanModel()
         if mepIntfAndServiceType.mepIntfType == tacMepIntfType.ethIntf and \
            mepIntfAndServiceType.serviceIntfType == ServiceIntfType.l2:
            if localMepConfig.primaryVlanId:
               # For now we only allow one VLAN.
               # If localMepAleConfig.primaryVlanId exists, that is the active
               # VLAN as well as the primary VLAN.
               localMepVlanModel.activeVlans.append( localMepConfig.primaryVlanId )
               localMepVlanModel.primaryVlan = localMepConfig.primaryVlanId
            else:
               # Since localMepAleConfig.primaryVlanId does not exist,
               # see if there is some VLAN configured under MA. If yes,
               # that is the inactive VLAN.
               maName = localMepConfig.maConfig.maNameId
               mdName = localMepConfig.maConfig.mdConfig.name
               cliMdConfig = cfmConfig.mdConfig.get( mdName )
               if cliMdConfig:
                  cliMaConfig = cliMdConfig.maConfig.get( maName )
                  if cliMaConfig and cliMaConfig.primaryVlanId:
                     localMepVlanModel.inactiveVlans.append(
                        cliMaConfig.primaryVlanId )
         mepModel.vlanStatus = localMepVlanModel

      if toggleCfmProgStateEnabled() and 'detail' in self.args:
         ccmEnabled = localMepConfig.maConfig.ccmEnable
         localMepCcStatusModel = LocalMepDetailModel()
         # Detailed CCM programming state
         if localMepConfig.ready and localMepStatus and \
            localMepStatus.mepCcmProgrammingState.lastStateChangeTime:
            # If MEP is ready and programming status is updated,
            # update the model with the current status.
            localMepCcStatusModel.rxProgrammed = False
            localMepCcStatusModel.txProgrammed = False
            mepRxState = localMepStatus.mepCcmProgrammingState.mepRxState
            if mepRxState == tacLocalMepCcmRxState.localMepCcmRxProgrammed:
               localMepCcStatusModel.rxProgrammed = True
            elif mepRxState == tacLocalMepCcmRxState.localMepCcmRxInactive:
               if ccmEnabled:
                  localMepCcStatusModel.rxReason = 'waiting'
            elif mepRxState == tacLocalMepCcmRxState.localMepCcmRxProgrammingFailed:
               localMepCcStatusModel.rxReason = 'programmingFailed'
            else:
               t0( 'Unexpected mepRxState', mepRxState )
            mepTxState = localMepStatus.mepCcmProgrammingState.mepTxState
            if mepTxState == tacLocalMepCcmTxState.localMepCcmTxProgrammed:
               localMepCcStatusModel.txProgrammed = True
            elif mepTxState == tacLocalMepCcmTxState.localMepCcmTxInactive:
               if ccmEnabled:
                  localMepCcStatusModel.txReason = 'waiting'
            elif mepTxState == tacLocalMepCcmTxState.localMepCcmTxProgrammingFailed:
               localMepCcStatusModel.txReason = 'programmingFailed'
            else:
               t0( 'Unexpected mepTxState', mepTxState )
         elif ccmEnabled:
            # If ccm is enabled but either the MEP is not ready or
            # MEP is not programmed, update the model appropriately.
            localMepCcStatusModel.rxProgrammed = False
            localMepCcStatusModel.txProgrammed = False
            localMepCcStatusModel.rxReason = 'waiting'
            localMepCcStatusModel.txReason = 'waiting'
         else:
            t0( 'CCM not enabled/programmed' )
         mepModel.ccStatus = localMepCcStatusModel

         # Detailed AIS programming state
         aisEnabled = localMepConfig.maConfig.aisEnable
         localMepAisStatusModel = LocalMepDetailModel()
         if localMepConfig.ready and localMepStatus and \
            localMepStatus.mepAisProgrammingState.lastStateChangeTime:
            # If MEP is ready and programming status is updated,
            # update the model with the current status.
            localMepAisStatusModel.rxProgrammed = False
            localMepAisStatusModel.txProgrammed = False
            mepAisState = localMepStatus.mepAisProgrammingState.mepState
            if mepAisState == tacLocalMepAisState.localMepAisRxActive:
               localMepAisStatusModel.rxProgrammed = True
            elif mepAisState == tacLocalMepAisState.localMepAisTxActive:
               localMepAisStatusModel.txProgrammed = True
            elif mepAisState == tacLocalMepAisState.localMepAisRxTxActive:
               localMepAisStatusModel.rxProgrammed = True
               localMepAisStatusModel.txProgrammed = True
            elif mepAisState == tacLocalMepAisState.localMepAisProgrammingFailed:
               localMepAisStatusModel.rxReason = 'programmingFailed'
               localMepAisStatusModel.txReason = 'programmingFailed'
            elif mepAisState == tacLocalMepAisState.localMepAisInactive:
               if aisEnabled:
                  localMepAisStatusModel.rxReason = 'waiting'
                  localMepAisStatusModel.txReason = 'waiting'
            else:
               t0( 'Unexpected mepAisState', mepAisState )
         elif aisEnabled:
            # If AIS is enabled but either the MEP is not ready or
            # MEP is not programmed, update the model appropriately.
            localMepAisStatusModel.rxProgrammed = False
            localMepAisStatusModel.txProgrammed = False
            localMepAisStatusModel.rxReason = 'waiting'
            localMepAisStatusModel.txReason = 'waiting'
         else:
            t0( 'AIS not enabled/programmed' )
         mepModel.aisStatus = localMepAisStatusModel

      self.addRemoteMeps( localMepConfig, localMepStatus, mepModel )
      maModel.localMeps[ localMepConfig.localMepId ] = mepModel

   def addMaModel( self, mdModel, maName, maModel ):
      mdModel.associations[ maName ] = maModel

   def addMdModel( self, model, domainName, mdModel ):
      model.domains[ domainName ] = mdModel

   def getMaModel( self, maConfig ):
      maModel = MaModel()
      maModel.direction = mepDirectionShowStr[ maConfig.direction ]
      maModel.ccmTxInterval = maConfig.ccmConfig.ccmTxInterval
      return maModel

   def getMdModel( self, mdConfig ):
      mdModel = MdModel()
      mdModel.level = mdConfig.mdLevel
      return mdModel

def showCfmEndPoint( mode, args ):
   s = ShowEndPoint( args )
   model = EndPointModel()
   s.addDomains( mode, model )
   return model

class ShowCfmEndPoint( ShowCommand.ShowCliCommandClass ):
   syntax = 'show cfm end-point [ FILTER ]'
   data = {
      'cfm' : cfmShowKw,
      'end-point' : endPointShowKw,
      'FILTER' : EndPointFilter
   }
   if toggleCfmProgStateEnabled():
      syntax += '[ detail ]'
      data[ 'detail' ] = cfmEndPointDetailShowKw

   cliModel = EndPointModel
   handler = showCfmEndPoint

BasicCli.addShowCommandClass( ShowCfmEndPoint )

#--------------------------------------------------------------------------------
# show cfm continuity-check end-point
# [ domain < Domain name > [ association < MA name > [ end-point < MEP ID > ] ] ]
# [ detail ]
#--------------------------------------------------------------------------------
class ShowContinuityCheck( ShowCfmBase ):
   def addRemoteMeps( self, localMepConfig, localMepStatus, localMepModel ):
      for remoteMepId in localMepConfig.remoteMepConfig:
         remoteMepStatus = localMepStatus.remoteMepStatus.get( remoteMepId )
         if not remoteMepStatus:
            t0( 'remoteMep ', remoteMepId, ' is missing from remoteMepStatus' )
            continue
         remoteMepCcModel = RemoteMepCcModel()
         locState = LocAndTimeChangeModel()
         locState.loc = remoteMepStatus.locState.locDetected
         if 'detail' in self.args and remoteMepStatus.locState.locDetected:
            ccmDefectKey = CcmDefectKey( localMepConfig.maConfig.mdConfig.mdLevel,
                                         localMepConfig.maConfig.mdConfig.mdName,
                                         localMepConfig.maConfig.maName,
                                         localMepConfig.localMepId,
                                         remoteMepId,
                                         DefectCodeEnum.defectCodeCcmTxMismatch )
            ccmDefect = localMepStatus.ccmDefect.get( ccmDefectKey )
            if ccmDefect:
               remoteMepCcModel.mismatchCcmTxInterval = ccmDefect.ccmTxInterval
         locState.lastDetectedTime = remoteMepStatus.locState.lastDetectedTime
         locState.lastClearedTime = remoteMepStatus.locState.lastClearedTime
         remoteMepCcModel.locState = locState
         localMepModel.remoteMeps[ remoteMepId ] = remoteMepCcModel
         if toggleCfmSwUpMepCcmEnabled() and cfmHwSupportStatus.mepSwSupported():
            remoteMepKey = \
               tacSmashRemoteMepKey( localMepConfig.maConfig.mdConfig.name,
                                     localMepConfig.maConfig.mdConfig.mdLevel,
                                     int( localMepConfig.maConfig.maNameId ),
                                     localMepConfig.localMepId,
                                     remoteMepId )
            mepCcmRxCtr = cfmRxCounters.remoteMepCcmRxCounter.get( remoteMepKey )
            ccmPduReceived = 0
            if mepCcmRxCtr:
               ccmPduReceived = mepCcmRxCtr.rxCount
            remoteMepCcModel.ccmPduReceived = ccmPduReceived

   def addCcmDefects( self, localMepStatus, localMepModel ):
      localMepModel.ccmDefects = []
      if self.args.get( 'detail' ):
         for defectKey, ccmDefect in localMepStatus.ccmDefect.items():
            # currently handling only "unknown rmep" defect codes
            defectCode = Tac.enumValue( DefectCodeEnum, defectKey.defectCode )
            if defectCode < Tac.enumValue( DefectCodeEnum,
                                       DefectCodeEnum.defectCodeMaidMismatch ) or \
               defectCode > Tac.enumValue( DefectCodeEnum,
                                       DefectCodeEnum.defectCodeDuplicateMep ):
               continue
            ccmDefectModel = CcmDefectModel()
            ccmDefectModel.level = defectKey.mdLevel
            ccmDefectModel.mdName = defectKey.mdName.name
            ccmDefectModel.mdNameFormat = defectKey.mdName.format
            ccmDefectModel.maName = defectKey.maName.name
            ccmDefectModel.maNameFormat = defectKey.maName.format
            ccmDefectModel.remoteMepId = defectKey.rmepId
            ccmDefectModel.defectId = defectKey.defectCode
            ccmDefectModel.cfmDefectPkts = ccmDefect.count
            ccmDefectModel.firstSeenTime = ccmDefect.firstSeenTime
            ccmDefectModel.lastSeenTime = ccmDefect.lastSeenTime
            localMepModel.ccmDefects.append( ccmDefectModel )

   def addOneMep( self, localMepConfig, localMepStatus, maModel ):
      if toggleCfmProgStateEnabled():
         if not localMepConfig or not localMepConfig.ready or not localMepStatus or \
            ( localMepStatus.mepCcmProgrammingState.mepRxState !=
              tacLocalMepCcmRxState.localMepCcmRxProgrammed and
              localMepStatus.mepCcmProgrammingState.mepTxState !=
              tacLocalMepCcmTxState.localMepCcmTxProgrammed ):
            # localMep is "inactive" for CCM
            return
      else:
         if not localMepConfig or not localMepConfig.ready or not localMepStatus or \
            localMepStatus.mepProgrammingState.mepState == 'localMepInactive':
            # localMep is "inactive"
            return
      mepModel = LocalMepCcModel()
      mepModel.intf = localMepConfig.intfId
      if toggleCfmSwUpMepCcmEnabled() and cfmHwSupportStatus.mepSwSupported():
         localMepKey = tacSmashLocalMepKey( localMepConfig.maConfig.mdConfig.name,
                                            localMepConfig.maConfig.mdConfig.mdLevel,
                                            int( localMepConfig.maConfig.maNameId ),
                                            localMepConfig.localMepId )
         mepCcmTxCtr = cfmTxCounters.localMepCcmTxCounter.get( localMepKey )
         ccmPduSent = 0
         if mepCcmTxCtr:
            ccmPduSent = mepCcmTxCtr.txCount
         mepModel.ccmPduSent = ccmPduSent
      self.addRemoteMeps( localMepConfig, localMepStatus, mepModel )
      self.addCcmDefects( localMepStatus, mepModel )
      rdiTxState = RdiAndTimeChangeModel()
      rdiTxState.rdi = localMepStatus.rdiTxCondition.rdi
      rdiTxState.lastDetectedTime = localMepStatus.rdiTxCondition.lastDetectedTime
      rdiTxState.lastClearedTime = localMepStatus.rdiTxCondition.lastClearedTime
      mepModel.rdiTxCondition = rdiTxState
      maModel.localMeps[ localMepStatus.mepId ] = mepModel

   def addMaModel( self, mdModel, maName, maModel ):
      mdModel.associations[ maName ] = maModel

   def addMdModel( self, model, domainName, mdModel ):
      model.domains[ domainName ] = mdModel

   def getMaModel( self, maConfig ):
      maModel = MaCcModel()
      maModel.direction = mepDirectionShowStr[ maConfig.direction ]
      maModel.ccmTxInterval = maConfig.ccmConfig.ccmTxInterval
      return maModel

   def getMdModel( self, mdConfig ):
      mdModel = MdCcModel()
      mdModel.level = mdConfig.mdLevel
      return mdModel

def showCfmContinuityCheck( mode, args ):
   s = ShowContinuityCheck( args )
   model = EndPointCcModel()
   s.addDomains( mode, model )
   return model

class ShowCfmContinuityCheck( ShowCommand.ShowCliCommandClass ):
   syntax = 'show cfm continuity-check end-point [ FILTER ] [ detail ] '
   data = {
      'cfm' : cfmShowKw,
      'continuity-check' : continuityCheckShowKw,
      'end-point' : endPointShowKw,
      'FILTER' : EndPointFilter,
      'detail' : endPointDetailShowKw,
   }

   cliModel = EndPointCcModel
   handler = showCfmContinuityCheck

BasicCli.addShowCommandClass( ShowCfmContinuityCheck )

#--------------------------------------------------------------------------------
# show cfm continuity-check end-point remote-defect
# [ domain < Domain name > [ association < MA name > [ end-point < MEP ID > ] ] ]
#--------------------------------------------------------------------------------
class ShowRemoteDefect( ShowCfmBase ):
   def addRemoteMeps( self, localMepConfig, localMepStatus, localMepModel ):
      for remoteMepId in localMepConfig.remoteMepConfig:
         remoteMepStatus = localMepStatus.remoteMepStatus.get( remoteMepId )
         if not remoteMepStatus:
            t0( 'remoteMep ', remoteMepId, ' is missing from remoteMepStatus' )
            continue
         remoteMepRdiModel = RemoteMepRdiModel()
         rdiState = RdiAndTimeChangeModel()
         rdiState.rdi = remoteMepStatus.rdiState.rdi
         rdiState.lastDetectedTime = remoteMepStatus.rdiState.lastDetectedTime
         rdiState.lastClearedTime = remoteMepStatus.rdiState.lastClearedTime
         remoteMepRdiModel.rdiState = rdiState
         localMepModel.remoteMeps[ remoteMepId ] = remoteMepRdiModel

   def addOneMep( self, localMepConfig, localMepStatus, maModel ):
      if not localMepConfig or not localMepConfig.ready or not localMepStatus:
         # localMep is "inactive"
         return
      mepModel = LocalMepRdiModel()
      mepModel.intf = localMepConfig.intfId
      rdiCondition = RdiAndTimeChangeModel()
      rdiCondition.rdi = localMepStatus.rdiRxCondition.rdi
      rdiCondition.lastDetectedTime = \
            localMepStatus.rdiRxCondition.lastDetectedTime
      rdiCondition.lastClearedTime = localMepStatus.rdiRxCondition.lastClearedTime
      mepModel.rdiCondition = rdiCondition
      rdiTxState = RdiAndTimeChangeModel()
      rdiTxState.rdi = localMepStatus.rdiTxCondition.rdi
      rdiTxState.lastDetectedTime = localMepStatus.rdiTxCondition.lastDetectedTime
      rdiTxState.lastClearedTime = localMepStatus.rdiTxCondition.lastClearedTime
      mepModel.rdiTxCondition = rdiTxState
      self.addRemoteMeps( localMepConfig, localMepStatus, mepModel )
      maModel.localMeps[ localMepStatus.mepId ] = mepModel

   def addMaModel( self, mdModel, maName, maModel ):
      mdModel.associations[ maName ] = maModel

   def addMdModel( self, model, domainName, mdModel ):
      model.domains[ domainName ] = mdModel

   def getMaModel( self, maConfig ):
      maModel = MaRdiModel()
      maModel.direction = mepDirectionShowStr[ maConfig.direction ]
      return maModel

   def getMdModel( self, mdConfig ):
      mdModel = MdRdiModel()
      mdModel.level = mdConfig.mdLevel
      return mdModel

def showCfmRemoteDefect( mode, args ):
   s = ShowRemoteDefect( args )
   model = EndPointRdiModel()
   s.addDomains( mode, model )
   return model

class ShowCfmRemoteDefect( ShowCommand.ShowCliCommandClass ):
   syntax = 'show cfm continuity-check end-point remote-defect [ FILTER ]'
   data = {
      'cfm' : cfmShowKw,
      'continuity-check' : continuityCheckShowKw,
      'end-point' : endPointShowKw,
      'remote-defect' : remoteDefectShowKw,
      'FILTER' : EndPointFilter
   }

   cliModel = EndPointRdiModel
   handler = showCfmRemoteDefect

BasicCli.addShowCommandClass( ShowCfmRemoteDefect )

def microSeconds( delay ):
   oneSecInMicroSeconds = 1000000
   return delay * oneSecInMicroSeconds

def smashRMepKey( localMepConfig, remoteMepId ):
   return tacSmashRemoteMepKey( localMepConfig.maConfig.mdConfig.name,
                                localMepConfig.maConfig.mdConfig.mdLevel,
                                int( localMepConfig.maConfig.maNameId ),
                                localMepConfig.localMepId,
                                remoteMepId )

# --------------------------------------------------------------------------------
# show cfm measurement delay proactive
# [ domain < Domain name > [ association < MA name > [ end-point < MEP ID > ] ] ]
# --------------------------------------------------------------------------------
class ShowDelayMeasurement( ShowCfmBase ):
   def addRemoteMepDetail( self, dmStats, remoteMepModel ):
      if not self.args.get( 'detail' ):
         return
      index = tacStatEntryIdx.toIdx( dmStats.dmSamplesCount )
      if index:
         # if the next update index is not the first entry
         index -= 1
      else:
         # else it is the last entry in the circular buffer
         index = tacStatEntryIdx.max
      count = min( dmStats.dmSamplesCount, tacStatEntryIdx.max + 1 )
      remoteMepModel.delayHistory = []
      for _ in range( count ):
         delay = dmStats.delay.get( index )
         if not delay:
            # Sysdb changed during show
            break
         dmStatsModel = DmStatsModel()
         dmStatsModel.delay = microSeconds( delay.delay )
         dmStatsModel.updateTime = delay.lastUpdateTime
         remoteMepModel.delayHistory.append( dmStatsModel )
         # walk backwards from the most recent update
         if index - 1 >= 0:
            index -= 1
         else:
            # wrap around for the next old entry
            index = count - 1

   def addRemoteMeps( self, localMepConfig, localMepStatus, localMepModel ):
      # pylint: disable=too-many-nested-blocks
      for remoteMepId, remoteMepConfig in localMepConfig.remoteMepConfig.items():
         # This is the initial state which can get overwrite to something else
         # in the subsequent code
         if remoteMepConfig.dmEnable:
            reason = "inProgress"
         else:
            reason = "notConfigured"

         remoteMepModel = RemoteMepDmModel()
         remoteMepModel.delayHistory = None
         miEnabled = localMepConfig.maConfig.dmConfig.miConfig != \
                     Tac.Value( 'Cfm::MeasurementIntervalConfig' )
         if toggleCfmPmMiEnabled():
            remoteMepModel.historicMiStats = None
            remoteMepModel.miEnabled = miEnabled
         remoteMepStatus = localMepStatus.remoteMepStatus.get( remoteMepId )
         smashRemoteMepKey = smashRMepKey( localMepConfig, remoteMepId )
         if remoteMepStatus:
            if ( remoteMepStatus.mepDmProgrammingState.mepState ==
                 tacRemoteMepDmState.remoteMepDmProgrammingSuccessful ):
               reason = None
               dmStats = mepStats.dmStats.get( smashRemoteMepKey )
               if dmStats:
                  remoteMepModel.numSamples = dmStats.dmSamplesCount
                  remoteMepModel.startTime = \
                     remoteMepStatus.mepDmProgrammingState.lastStateChangeTime

                  if dmStats.dmSamplesCount:
                     remoteMepModel.avgDelay = \
                       microSeconds( dmStats.totalDelay ) / dmStats.dmSamplesCount
                     remoteMepModel.avgJitter = \
                      microSeconds( dmStats.totalJitter ) / dmStats.dmSamplesCount
                     remoteMepModel.bestDelay = microSeconds( dmStats.minDelay )
                     remoteMepModel.bestDelayTime = dmStats.minDelayLastUpdateTime
                     remoteMepModel.worstDelay = microSeconds( dmStats.maxDelay )
                     remoteMepModel.worstDelayTime = \
                        dmStats.maxDelayLastUpdateTime
                     self.addRemoteMepDetail( dmStats, remoteMepModel )
               else:
                  remoteMepModel.numSamples = 0
            elif ( remoteMepStatus.mepDmProgrammingState.mepState ==
                   tacRemoteMepDmState.remoteMepDmProgrammingFailed ):
               reason = "failed"
            elif ( remoteMepStatus.mepDmProgrammingState.mepState ==
                   tacRemoteMepDmState.remoteMepDmProgrammingInProgress ):
               reason = "inProgress"
            elif ( remoteMepStatus.mepDmProgrammingState.mepState ==
                   tacRemoteMepDmState.remoteMepDmDisabled ):
               # If DM is configured via CLI then it should be treated as inProgress
               if remoteMepConfig.dmEnable:
                  reason = "inProgress"
                  # check if performance function is suspended due to LOC
                  if remoteMepConfig.suspendPm and \
                     remoteMepStatus.locState.locDetected:
                     reason = "suspendedForLoc"
               else:
                  reason = "notConfigured"
            else:
               assert False, ( "Unhandled DM status %s" %
                               remoteMepStatus.mepDmProgrammingState.mepState )
         if toggleCfmPmMiEnabled():
            # From the CLI perspective, the current/historic MI stats present in
            # Smash/Shark will be displayed. Such stats will be managed at the agent
            # level depending on whether the remote MEP is configured, DM/MI is
            # enabled/configured.
            dmMiStatsCurr = mepStats.dmMiStatsCurrent.get( smashRemoteMepKey )
            if dmMiStatsCurr and dmMiStatsCurr.dmMiStats.startTime:
               dmMiStatsModel = DmMiStatsModel()
               dmMiStats = dmMiStatsCurr.dmMiStats
               dmMiStatsModel.startTime = dmMiStats.startTime
               if miEnabled:
                  # There can be an agent level race condition, where MI is
                  # unconfigured, but the current MI stats is not yet cleaned up.
                  # In such case, the currnet MI stats will be displayed, but the
                  # measurement interval will be considered n/a.
                  dmMiStatsModel.measurementInterval = \
                     int( localMepConfig.maConfig.dmConfig.miConfig.interval / 60 )
               dmMiStatsModel.numSamples = dmMiStats.numSamples
               dmMiStatsModel.minDelay = \
                  microSeconds( dmMiStats.delayTwoWayMin )
               dmMiStatsModel.maxDelay = \
                  microSeconds( dmMiStats.delayTwoWayMax )
               dmMiStatsModel.avgDelay = \
                  microSeconds( dmMiStats.delayTwoWayAvg )
               dmMiStatsModel.minVariation = \
                  microSeconds( dmMiStats.jitterTwoWayMin )
               dmMiStatsModel.maxVariation = \
                  microSeconds( dmMiStats.jitterTwoWayMax )
               dmMiStatsModel.avgVariation = \
                  microSeconds( dmMiStats.jitterTwoWayAvg )
               remoteMepModel.currentMiStats = dmMiStatsModel
            if self.args.get( 'detail' ):
               remoteMepModel.historicMiStats = []
               dmSessionStats = \
                  mepMiStatsHistory.dmSessionStats.get( smashRemoteMepKey )
               if dmSessionStats:
                  for dmMiStatsHistory in \
                        dmSessionStats.dmMiStatsHistory.values():
                     dmMiStats = dmMiStatsHistory.dmMiStats
                     dmMiStatsModel = DmMiStatsModel()
                     dmMiStatsModel.startTime = dmMiStats.startTime
                     dmMiStatsModel.endTime = dmMiStats.endTime
                     dmMiStatsModel.suspect = dmMiStats.suspect
                     dmMiStatsModel.numSamples = dmMiStats.numSamples
                     dmMiStatsModel.minDelay = \
                        microSeconds( dmMiStats.delayTwoWayMin )
                     dmMiStatsModel.maxDelay = \
                        microSeconds( dmMiStats.delayTwoWayMax )
                     dmMiStatsModel.avgDelay = \
                        microSeconds( dmMiStats.delayTwoWayAvg )
                     dmMiStatsModel.minVariation = \
                        microSeconds( dmMiStats.jitterTwoWayMin )
                     dmMiStatsModel.maxVariation = \
                        microSeconds( dmMiStats.jitterTwoWayMax )
                     dmMiStatsModel.avgVariation = \
                        microSeconds( dmMiStats.jitterTwoWayAvg )
                     remoteMepModel.historicMiStats.append( dmMiStatsModel )
         remoteMepModel.reason = reason
         remoteMepModel.enabled = not reason
         localMepModel.remoteMeps[ remoteMepId ] = remoteMepModel

   def addOneMep( self, localMepConfig, localMepStatus, maModel ):
      if not localMepConfig or not localMepConfig.ready or not localMepStatus:
         # localMep is "inactive"
         return
      mepModel = LocalMepDmModel()
      mepModel.intf = localMepConfig.intfId
      self.addRemoteMeps( localMepConfig, localMepStatus, mepModel )
      maModel.localMeps[ localMepStatus.mepId ] = mepModel

   def addMaModel( self, mdModel, maName, maModel ):
      mdModel.associations[ maName ] = maModel

   def addMdModel( self, model, domainName, mdModel ):
      model.domains[ domainName ] = mdModel

   def getMaModel( self, maConfig ):
      pmOperMode = getPmOperModeFromArgs( self.args )
      maModel = MaDmModel()
      maModel.direction = mepDirectionShowStr[ maConfig.direction ]
      maModel.operMode = pmOperModeKwStr[ pmOperMode ]
      maModel.txInterval = maConfig.dmConfig.txInterval
      return maModel

   def getMdModel( self, mdConfig ):
      mdModel = MdDmModel()
      mdModel.level = mdConfig.mdLevel
      return mdModel

def showCfmDm( mode, args ):
   s = ShowDelayMeasurement( args )
   model = EndPointDmModel()
   s.addDomains( mode, model )
   return model

class ShowCfmDm( ShowCommand.ShowCliCommandClass ):
   syntax = 'show cfm measurement delay proactive [ FILTER ] [ detail ]'
   data = {
      'cfm' : cfmShowKw,
      'measurement' : measurementShowKw,
      'delay' : delayMeasurementKw,
      'proactive' : proactiveShowKw,
      'FILTER' : EndPointFilter,
      'detail' : detailShowKw,
   }

   cliModel = EndPointDmModel
   handler = showCfmDm

BasicCli.addShowCommandClass( ShowCfmDm )

# --------------------------------------------------------------------------------
# show cfm measurement loss proactive
# [ domain < Domain name > [ association < MA name > [ end-point < MEP ID > ] ] ]
# --------------------------------------------------------------------------------
class ShowLossMeasurement( ShowCfmBase ):
   def addRemoteMepDetail( self, lmStats, remoteMepModel ):
      if not self.args.get( 'detail' ):
         return
      count = tacStatEntryIdx.max + 1
      index = tacStatEntryIdx.toIdx( lmStats.samplesCount )
      if index:
         # if the next update index is not the first entry
         index -= 1
      else:
         # else it is the last entry in the circular buffer
         index = tacStatEntryIdx.max
      if lmStats.samplesCount < tacStatEntryIdx.max + 1:
         count = lmStats.samplesCount
      remoteMepModel.lmHistory = []
      for _ in range( count ):
         lmEntry = lmStats.lmEntry.get( index )
         if not lmEntry:
            # Entries changed during show
            break
         lmStatsModel = LmStatsModel()
         lmStatsModel.localTx = lmEntry.localTx
         lmStatsModel.localRx = lmEntry.localRx
         lmStatsModel.remoteRx = lmEntry.remoteRx
         lmStatsModel.remoteTx = lmEntry.remoteTx
         lmStatsModel.nearEndLoss = \
                        lmEntry.remoteTx - lmEntry.localRx
         lmStatsModel.farEndLoss = \
                        lmEntry.localTx - lmEntry.remoteRx
         lmStatsModel.updateTime = lmEntry.lastUpdateTime
         remoteMepModel.lmHistory.append( lmStatsModel )
         # walk backwards from the most recent update
         if index - 1 >= 0:
            index -= 1
         else:
            # wrap around for the next old entry
            index = count - 1

   def addRemoteMeps( self, localMepConfig, localMepStatus, localMepModel ):
      for remoteMepId, remoteMepConfig in localMepConfig.remoteMepConfig.items():
         # This is the initial state which can get overwrite to something else
         # in the subsequent code
         if remoteMepConfig.lmEnable:
            reason = "inProgress"
         else:
            reason = "notConfigured"

         remoteMepModel = RemoteMepLmModel()
         remoteMepModel.lmHistory = None
         remoteMepStatus = localMepStatus.remoteMepStatus.get( remoteMepId )
         if remoteMepStatus:
            smashRemoteMepKey = smashRMepKey( localMepConfig, remoteMepId )
            if ( remoteMepStatus.mepLmProgrammingState.mepState ==
                 tacRemoteMepLmState.remoteMepLmProgrammingSuccessful ):
               reason = None
               remoteMepModel.startTime = \
                     remoteMepStatus.mepLmProgrammingState.lastStateChangeTime
               remoteMepModel.numSamples = 0
               lmStats = mepStats.lmStats.get( smashRemoteMepKey )
               if lmStats:
                  remoteMepModel.numSamples = lmStats.samplesCount
                  if lmStats.samplesCount:
                     remoteMepModel.avgFarEndFrameLoss = \
                           int( lmStats.farEndFrameLoss / lmStats.samplesCount )
                     remoteMepModel.avgNearEndFrameLoss = \
                           int( lmStats.nearEndFrameLoss / lmStats.samplesCount )
                     self.addRemoteMepDetail( lmStats, remoteMepModel )
            elif ( remoteMepStatus.mepLmProgrammingState.mepState ==
                   tacRemoteMepLmState.remoteMepLmProgrammingFailed ):
               reason = "failed"
            elif ( remoteMepStatus.mepLmProgrammingState.mepState ==
                   tacRemoteMepLmState.remoteMepLmProgrammingInProgress ):
               reason = "inProgress"
            elif ( remoteMepStatus.mepLmProgrammingState.mepState ==
                   tacRemoteMepLmState.remoteMepLmDisabled ):
               if remoteMepConfig.lmEnable:
                  reason = "inProgress"
                  # check if performance function is suspended due to LOC
                  if remoteMepConfig.suspendPm and \
                     remoteMepStatus.locState.locDetected:
                     reason = "suspendedForLoc"
               else:
                  reason = "notConfigured"
            else:
               assert False, ( "Unhandled LM status %s" %
                               remoteMepStatus.mepLmProgrammingState.mepState )
         remoteMepModel.reason = reason
         remoteMepModel.enabled = not reason
         localMepModel.remoteMeps[ remoteMepId ] = remoteMepModel

   def addOneMep( self, localMepConfig, localMepStatus, maModel ):
      if not localMepConfig or not localMepConfig.ready or not localMepStatus:
         # localMep is "inactive"
         return
      mepModel = LocalMepLmModel()
      mepModel.intf = localMepConfig.intfId
      self.addRemoteMeps( localMepConfig, localMepStatus, mepModel )
      maModel.localMeps[ localMepStatus.mepId ] = mepModel

   def addMaModel( self, mdModel, maName, maModel ):
      mdModel.associations[ maName ] = maModel

   def addMdModel( self, model, domainName, mdModel ):
      model.domains[ domainName ] = mdModel

   def getMaModel( self, maConfig ):
      pmOperMode = getPmOperModeFromArgs( self.args )
      maModel = MaLmModel()
      maModel.direction = mepDirectionShowStr[ maConfig.direction ]
      maModel.operMode = pmOperModeKwStr[ pmOperMode ]
      maModel.txInterval = maConfig.lmConfig.txInterval
      return maModel

   def getMdModel( self, mdConfig ):
      mdModel = MdLmModel()
      mdModel.level = mdConfig.mdLevel
      return mdModel

def showCfmLm( mode, args ):
   s = ShowLossMeasurement( args )
   model = EndPointLmModel()
   s.addDomains( mode, model )
   return model

class ShowCfmLm( ShowCommand.ShowCliCommandClass ):
   syntax = 'show cfm measurement loss proactive [ FILTER ] [ detail ]'
   data = {
      'cfm' : cfmShowKw,
      'measurement' : measurementShowKw,
      'loss' : lossMeasurementKw,
      'proactive' : proactiveShowKw,
      'FILTER' : EndPointFilter,
      'detail' : lmDetailShowKw,
   }

   cliModel = EndPointLmModel
   handler = showCfmLm

BasicCli.addShowCommandClass( ShowCfmLm )

def smashSlmRMepKey( localMepConfig, remoteMepId, cosValue ):
   smashRemoteMepKey = smashRMepKey( localMepConfig, remoteMepId )
   slmSessionKey = tacSlmSessionKey( tacCfmConst.slmTestId, cosValue )
   return tacSmashRemoteMepSlmSessionKey( smashRemoteMepKey, slmSessionKey )

# --------------------------------------------------------------------------------
# show cfm measurement loss synthetic proactive
# [ domain < Domain name > [ association < MA name > [ end-point < MEP ID > ] ] ]
# --------------------------------------------------------------------------------
class ShowSyntheticLossMeasurement( ShowCfmBase ):
   def addRemoteMepDetail( self, slmStats, remoteMepModel ):
      if not self.args.get( 'detail' ) or not \
         cfmHwSupportStatus.pmRawReportHwSupported:
         return
      count = tacStatEntryIdx.max + 1
      index = tacStatEntryIdx.toIdx( slmStats.samplesCount )
      if index:
         # if the next update index is not the first entry
         index -= 1
      else:
         # else it is the last entry in the circular buffer
         index = tacStatEntryIdx.max
      if slmStats.samplesCount < tacStatEntryIdx.max + 1:
         count = slmStats.samplesCount
      remoteMepModel.slmHistory = []
      for _ in range( count ):
         slmEntry = slmStats.slmEntry.get( index )
         if not slmEntry:
            # Entries changed during show
            break
         slmStatsModel = SlmStatsModel()
         slmStatsModel.localTx = slmEntry.localTx
         slmStatsModel.localRx = slmEntry.localRx
         slmStatsModel.remoteRx = slmEntry.remoteRx
         slmStatsModel.updateTime = slmEntry.lastUpdateTime
         remoteMepModel.slmHistory.append( slmStatsModel )
         # walk backwards from the most recent update
         if index - 1 >= 0:
            index -= 1
         else:
            # wrap around for the next old entry
            index = count - 1

   def addRemoteMepsForCos( self, cosValue, localMepConfig, localMepStatus,
                            localMepModel ):
      maName = localMepConfig.maConfig.maNameId
      domainName = localMepConfig.maConfig.mdConfig.name
      maAleStatus = None
      if domainName in aleCfmStatus.mdStatus:
         maAleStatus = \
               aleCfmStatus.mdStatus.get( domainName ).maAleStatus.get( maName )
      for remoteMepId in sorted( localMepConfig.remoteMepConfig ):
         remoteMepModel = RemoteMepSlmModel()
         remoteMepModel.measurementPeriod = tacCfmConst.defaultSlmTxFrames * \
                                       localMepConfig.maConfig.slmConfig.txInterval
         if maAleStatus:
            remoteMepModel.measurementPeriod = \
                                          maAleStatus.slmMeasurementPeriod * 1000
         remoteMepModel.slmHistory = None
         miEnabled = localMepConfig.maConfig.slmConfig.miConfig != \
                     Tac.Value( 'Cfm::MeasurementIntervalConfig' )
         if toggleCfmPmMiEnabled():
            remoteMepModel.historicMiStats = None
            remoteMepModel.miEnabled = miEnabled
         reason = "notConfigured"
         remoteMepConfig = localMepConfig.remoteMepConfig.get( remoteMepId )
         slmEnabled = remoteMepConfig.slmEnable
         remoteMepStatus = localMepStatus.remoteMepStatus.get( remoteMepId )
         remoteMepSlmSessionKey = smashSlmRMepKey( localMepConfig, remoteMepId,
                                                   cosValue )
         if not remoteMepStatus:
            if slmEnabled:
               reason = "inProgress"
         elif slmEnabled:
            slmSessionKey = tacSlmSessionKey( tacCfmConst.slmTestId, cosValue )
            if slmSessionKey in remoteMepStatus.mepSlmProgrammingState:
               mepSlmProgState = \
                     remoteMepStatus.mepSlmProgrammingState.get( slmSessionKey )
               if ( mepSlmProgState.mepState ==
                      tacRemoteMepSlmState.remoteMepSlmProgrammingFailed ):
                  reason = "failed"
               elif ( mepSlmProgState.mepState ==
                      tacRemoteMepSlmState.remoteMepSlmProgrammingInProgress ):
                  reason = "inProgress"
               elif ( mepSlmProgState.mepState ==
                      tacRemoteMepSlmState.remoteMepSlmDisabled ):
                  reason = "inProgress"
                  # check if performance function is suspended due to LOC
                  if remoteMepConfig.suspendPm and \
                     remoteMepStatus.locState.locDetected:
                     reason = "suspendedForLoc"
               elif mepSlmProgState.mepState == \
                     tacRemoteMepSlmState.remoteMepSlmProgrammingSuccessful:
                  reason = None
                  remoteMepModel.startTime = mepSlmProgState.lastStateChangeTime
                  remoteMepModel.numMeasurements = 0
                  slmStats = mepStats.slmStats.get( remoteMepSlmSessionKey )
                  if slmStats and slmStats.samplesCount:
                     remoteMepModel.numMeasurements = slmStats.samplesCount
                     remoteMepModel.avgFarEndFrameLoss = \
                                 slmStats.farEndFrameLoss / slmStats.samplesCount
                     remoteMepModel.avgNearEndFrameLoss = \
                                 slmStats.nearEndFrameLoss / slmStats.samplesCount
                     self.addRemoteMepDetail( slmStats, remoteMepModel )
            else:
               # SLM is enabled but remoteMepStatus.mepSlmProgrammingState is not
               # populated. It means that the programming is in progress
               reason = "inProgress"
         if toggleCfmPmMiEnabled():
            # From the CLI perspective, the current/historic MI stats present in
            # Smash/Shark will be displayed. Such stats will be managed at the agent
            # level depending on whether the remote MEP is configured, SLM/MI is
            # enabled/configured.
            slmMiStatsCurr = mepStats.slmMiStatsCurrent.get( remoteMepSlmSessionKey )
            if slmMiStatsCurr and slmMiStatsCurr.slmMiStats.startTime:
               slmMiStatsModel = SlmMiStatsModel()
               slmMiStats = slmMiStatsCurr.slmMiStats
               slmMiStatsModel.startTime = slmMiStats.startTime
               if miEnabled:
                  # There can be an agent level race condition, where MI is
                  # unconfigured, but the current MI stats is not yet cleaned up.
                  # In such case, the currnet MI stats will be displayed, but the
                  # measurement interval will be considered n/a.
                  slmMiStatsModel.measurementInterval = \
                     int( localMepConfig.maConfig.slmConfig.miConfig.interval / 60 )
               slmMiStatsModel.numMeasurements = slmMiStats.numSamples
               # Ratio needs to be converted to percentage.
               slmMiStatsModel.minFarEndFrameLossPercentage = \
                  slmMiStats.forwardMinFlr * 100
               slmMiStatsModel.maxFarEndFrameLossPercentage = \
                  slmMiStats.forwardMaxFlr * 100
               slmMiStatsModel.avgFarEndFrameLossPercentage = \
                  slmMiStats.forwardAvgFlr * 100
               slmMiStatsModel.minNearEndFrameLossPercentage = \
                  slmMiStats.backwardMinFlr * 100
               slmMiStatsModel.maxNearEndFrameLossPercentage = \
                  slmMiStats.backwardMaxFlr * 100
               slmMiStatsModel.avgNearEndFrameLossPercentage = \
                  slmMiStats.backwardAvgFlr * 100
               remoteMepModel.currentMiStats = slmMiStatsModel
            if self.args.get( 'detail' ):
               remoteMepModel.historicMiStats = []
               slmSessionStats = \
                  mepMiStatsHistory.slmSessionStats.get( remoteMepSlmSessionKey )
               if slmSessionStats:
                  for slmMiStatsHistory in \
                        slmSessionStats.slmMiStatsHistory.values():
                     slmMiStats = slmMiStatsHistory.slmMiStats
                     slmMiStatsModel = SlmMiStatsModel()
                     slmMiStatsModel.startTime = slmMiStats.startTime
                     slmMiStatsModel.endTime = slmMiStats.endTime
                     slmMiStatsModel.suspect = slmMiStats.suspect
                     slmMiStatsModel.numMeasurements = slmMiStats.numSamples
                     slmMiStatsModel.minFarEndFrameLossPercentage = \
                        slmMiStats.forwardMinFlr * 100
                     slmMiStatsModel.maxFarEndFrameLossPercentage = \
                        slmMiStats.forwardMaxFlr * 100
                     slmMiStatsModel.avgFarEndFrameLossPercentage = \
                        slmMiStats.forwardAvgFlr * 100
                     slmMiStatsModel.minNearEndFrameLossPercentage = \
                        slmMiStats.backwardMinFlr * 100
                     slmMiStatsModel.maxNearEndFrameLossPercentage = \
                        slmMiStats.backwardMaxFlr * 100
                     slmMiStatsModel.avgNearEndFrameLossPercentage = \
                        slmMiStats.backwardAvgFlr * 100
                     remoteMepModel.historicMiStats.append( slmMiStatsModel )
         remoteMepModel.reason = reason
         remoteMepModel.enabled = not reason
         localMepModel.remoteMeps[ remoteMepId ] = remoteMepModel

   def addOneMep( self, localMepConfig, localMepStatus, maModel ):
      if not localMepConfig or not localMepConfig.ready or not localMepStatus:
         # localMep is "inactive"
         return
      slmConfig = localMepConfig.maConfig.slmConfig
      for cosValue in sorted( slmConfig.priority ):
         mepModel = LocalMepSlmModel()
         mepModel.mepId = localMepStatus.mepId
         mepModel.intf = localMepConfig.intfId
         mepModel.cos = cosValue
         self.addRemoteMepsForCos( cosValue, localMepConfig, localMepStatus,
                                   mepModel )
         maModel.localMeps.append( mepModel )

   def addMaModel( self, mdModel, maName, maModel ):
      mdModel.associations[ maName ] = maModel

   def addMdModel( self, model, domainName, mdModel ):
      model.domains[ domainName ] = mdModel

   def getMaModel( self, maConfig ):
      pmOperMode = getPmOperModeFromArgs( self.args )
      maModel = MaSlmModel()
      maModel.direction = mepDirectionShowStr[ maConfig.direction ]
      maModel.operMode = pmOperModeKwStr[ pmOperMode ]
      maModel.txInterval = maConfig.slmConfig.txInterval
      return maModel

   def getMdModel( self, mdConfig ):
      mdModel = MdSlmModel()
      mdModel.level = mdConfig.mdLevel
      return mdModel

def showCfmSlm( mode, args ):
   s = ShowSyntheticLossMeasurement( args )
   model = EndPointSlmModel()
   s.addDomains( mode, model )
   return model

class ShowCfmSlm( ShowCommand.ShowCliCommandClass ):
   syntax = 'show cfm measurement loss synthetic proactive [ FILTER ] [ detail ]'
   data = {
      'cfm' : cfmShowKw,
      'measurement' : measurementShowKw,
      'loss' : anyLossMeasurementKw,
      'synthetic' : syntheticLossMeasurementKw,
      'proactive' : proactiveShowKw,
      'FILTER' : EndPointFilter,
   }
   if toggleCfmPmMiEnabled():
      # Historic SLM MI stats must be displayed if present in Shark, regardless of
      # whether non-MI SLM stats are available to display, so relaxing the guard for
      # detail keyword.
      data[ 'detail' ] = detailShowKw
   else:
      data[ 'detail' ] = lmDetailShowKw

   cliModel = EndPointSlmModel
   handler = showCfmSlm

BasicCli.addShowCommandClass( ShowCfmSlm )

# --------------------------------------------------------------------------------
# show cfm alarm indication
# [ domain < Domain name > [ association < MA name > [ end-point < MEP ID > ] ] ]
# --------------------------------------------------------------------------------
class ShowCfmAis( ShowCfmBase ):
   def __init__( self, args ):
      super().__init__( args )
      # Map of localMepKey --> list of (lastReceived, remoteMac, txInterval, count).
      # Tuple is structured so that when we sort the list, we get back
      # entries latest received AIS PDU.
      self.aisRxStatsPerMep = defaultdict( list )
      self.buildAisRxStats()

   def getLocalMepKey( self, localMepConfig ):
      return Tac.const( Tac.Value( 'Cfm::LocalMepKey',
                                   localMepConfig.maConfig.mdConfig.name,
                                   localMepConfig.maConfig.mdConfig.mdLevel,
                                   localMepConfig.maConfig.maNameId,
                                   localMepConfig.localMepId ) )

   def buildAisRxStats( self ):
      rxStats = list( aisRxStatus.aisRxActiveStats.items() )
      rxStats.extend( aisRxStatus.aisRxStaleStats.items() )
      for ( aisRxMepKey, stats ) in rxStats:
         localMepKey = aisRxMepKey.localMepKey.toSysdbLocalMepKey()
         self.aisRxStatsPerMep[ localMepKey ].append( ( stats.lastReceived,
                                                        aisRxMepKey.remoteMacAddr,
                                                        stats.txIntervalInSeconds,
                                                        stats.count ) )

   def getMdModel( self, mdConfig ):
      mdModel = MdAisModel()
      mdModel.level = mdConfig.mdLevel
      return mdModel

   def getMaModel( self, maConfig ):
      maModel = MaAisModel()
      maModel.direction = mepDirectionShowStr[ maConfig.direction ]
      return maModel

   def addMaModel( self, mdModel, maName, maModel ):
      mdModel.associations[ maName ] = maModel

   def addMdModel( self, model, domainName, mdModel ):
      model.domains[ domainName ] = mdModel

   def addOneMep( self, localMepConfig, localMepStatus, maModel ):
      if not localMepConfig or not localMepStatus:
         return
      # Filter out MEPs which don't have AIS enabled in CFM profile.
      maName = localMepConfig.maConfig.maNameId
      mdName = localMepConfig.maConfig.mdConfig.name
      cliMdConfig = cfmConfig.mdConfig.get( mdName )
      if not cliMdConfig:
         return
      cliMaConfig = cliMdConfig.maConfig.get( maName )
      if not cliMaConfig:
         return
      profileName = cliMaConfig.cfmProfileName
      profileConfig = cfmConfig.cfmProfile.get( profileName )
      if not profileConfig or not profileConfig.aisEnable:
         return
      mepModel = LocalMepAisModel()
      mepModel.intf = localMepConfig.intfId
      aisConfig = localMepConfig.maConfig.aisConfig
      if aisConfig.clientDomainState == 'validConfig':
         mepModel.clientDomainLevel = aisConfig.clientDomainLevel
         mepModel.aisTxInterval = aisConfig.getTxIntervalInSeconds()
      elif aisConfig.clientDomainState == 'invalidConfig':
         mepModel.clientDomainLevelError = 'invalid'
      elif aisConfig.clientDomainState == 'notConfigured':
         mepModel.clientDomainLevelError = 'notConfigured'
      else:
         assert False, "Unknown or unexpected clientDomainState"
      aisState = AisDefectTimeChangeModel()
      aisState.detected = localMepStatus.aisDetectedState.aisDetected
      aisState.lastDetectedTime = localMepStatus.aisDetectedState.lastDetectedTime
      aisState.lastClearedTime = localMepStatus.aisDetectedState.lastClearedTime
      mepModel.aisCondition = aisState
      # Populate AIS defect condition table.
      defaultDefectState = Tac.Value( 'Cfm::AisDefect' )
      for ( defectCondition, defectConditionStr ) in aisDefectKwStr.items():
         defectModel = AisDefectTimeChangeModel()
         defectState = localMepStatus.aisDefect.get( defectCondition,
                                                     defaultDefectState )
         defectModel.detected = defectState.defectDetected
         defectModel.lastDetectedTime = defectState.lastDetectedTime
         defectModel.lastClearedTime = defectState.lastClearedTime
         mepModel.aisDefects[ defectConditionStr ] = defectModel
      # Populate AIS received information table.
      localMepKey = self.getLocalMepKey( localMepConfig )
      statsList = []
      if localMepKey in self.aisRxStatsPerMep:
         statsList = self.aisRxStatsPerMep[ localMepKey ]
         statsList.sort( reverse=True )
      for ( lastReceived, remoteMac, txInterval, count ) in statsList:
         aisRxEntryModel = RemoteAisRxEntryModel()
         aisRxEntryModel.lastReceivedTime = lastReceived
         aisRxEntryModel.remoteMac = remoteMac
         aisRxEntryModel.txInterval = txInterval
         aisRxEntryModel.count = count
         mepModel.remoteAisRxEntries.append( aisRxEntryModel )
      maModel.localMeps[ localMepStatus.mepId ] = mepModel

def showCfmAlarmIndication( mode, args ):
   showAis = ShowCfmAis( args )
   model = EndPointAisModel()
   showAis.addDomains( mode, model )
   return model

class ShowCfmAlarmIndication( ShowCommand.ShowCliCommandClass ):
   syntax = 'show cfm alarm indication [ FILTER ]'
   data = {
      'cfm' : cfmShowKw,
      'alarm' : alarmIndicationShowKw,
      'indication' : "Alarm indication information",
      'FILTER' : EndPointFilter
   }

   cliModel = EndPointAisModel
   handler = showCfmAlarmIndication

BasicCli.addShowCommandClass( ShowCfmAlarmIndication )

# --------------------------------------------------------------------------------
# show cfm continuity-check counters
# [ domain < Domain name > [ association < MA name > [ end-point < MEP ID > ] ] ]
# [ detail ]
# --------------------------------------------------------------------------------

class ShowCfmCcCounters( ShowCfmBase ):
   # The current implementation of ShowCfmBase assumes that an MD/MA model gets
   # created once per MD/MA. However for this show command, multiple MD/MA models
   # could exist for a single MD/MA if its MEPs belong to different intfVlans. This
   # way, the MD/MA models will be created upon determining a MEP's intfVlan, so the
   # following methods will be ignored.
   def addMaModel( self, _, __, ____ ):
      pass

   def getMaModel( self, _ ):
      pass

   def addMdModel( self, _, __, ___ ):
      pass

   def getMdModel( self, _ ):
      pass

   def addOneMep( self, _, __, ___ ):
      pass

   def applyLMepTxCtr( self, model, counter ):
      model.txCount = counter.txCount

   def applyLMepTxFailCtr( self, model, counter ):
      model.txFailureCount = counter.txFailureCount

   def applyLMepRxDropCtr( self, model, counter ):
      model.ccmPacketParseError = counter.ccmPacketParseError
      model.invalidMdName = counter.invalidMdName
      model.invalidMaName = counter.invalidMaName
      model.directionMismatch = counter.directionMismatch
      model.ccmIntervalMismatch = counter.ccmIntervalMismatch
      model.invalidRemoteMepId = counter.invalidRemoteMepId
      model.duplicateRemoteMep = counter.duplicateRemoteMep
      model.unconfiguredRemoteMep = counter.unconfiguredRemoteMep
      model.rxDropCount = counter.ccmPacketParseError + counter.invalidMdName + \
                          counter.invalidMaName + counter.directionMismatch + \
                          counter.ccmIntervalMismatch + \
                          counter.invalidRemoteMepId + counter.duplicateRemoteMep + \
                          counter.unconfiguredRemoteMep

   def applyLMepRxCtr( self, model, counter ):
      model.rxCount = counter.rxCount

   def applyIntfVlanRxCtr( self, model, counter ):
      model.rxCount = counter.rxCount

   def applyIntfVlanRxDropCtr( self, model, counter ):
      model.noMepConfigured = counter.noMepConfigured
      model.ethHeaderParseError = counter.ethHeaderParseError
      model.cfmHeaderParseError = counter.cfmHeaderParseError
      model.mdLevelMismatch = counter.mdLevelMismatch
      model.firstTlvOffsetInvalid = counter.firstTlvOffsetInvalid
      model.rxDropCount = counter.noMepConfigured + counter.ethHeaderParseError \
                          + counter.cfmHeaderParseError + counter.mdLevelMismatch \
                          + counter.firstTlvOffsetInvalid

   def initializeIntfVlanCounterModel( self, model ):
      model.txCount = 0
      model.txFailureCount = 0
      model.rxCount = 0
      model.rxDropCount = 0
      model.noMepConfigured = 0
      model.ethHeaderParseError = 0
      model.cfmHeaderParseError = 0
      model.mdLevelMismatch = 0
      model.firstTlvOffsetInvalid = 0
      return model

   def initializeLocalMepCounterModel( self, model ):
      model.intfId = ""
      model.vlanId = 0
      model.txCount = 0
      model.txFailureCount = 0
      model.rxCount = 0
      model.rxDropCount = 0
      model.ccmPacketParseError = 0
      model.invalidMdName = 0
      model.invalidMaName = 0
      model.directionMismatch = 0
      model.ccmIntervalMismatch = 0
      model.invalidRemoteMepId = 0
      model.duplicateRemoteMep = 0
      model.unconfiguredRemoteMep = 0
      return model

   def addCounters( self, model ):
      if self.args.get( 'detail' ):
         def applyIntfVlanCounter( counters, func ):
            for intfVlan in counters:
               intfId = intfVlan.intfId
               vlanId = intfVlan.vlanId
               if intfId not in model.interfaces:
                  model.interfaces[ intfId ] = IntfCcCountersModel()
               intfModel = model.interfaces[ intfId ]
               if vlanId not in intfModel.vlans:
                  intfModel.vlans[ vlanId ] = \
                     self.initializeIntfVlanCounterModel( IntfVlanCcCountersModel() )
               # populate the intfVlan counter stats
               func( intfModel.vlans[ vlanId ], counters.get( intfVlan ) )
         # Build the intfVlan models based on the intfVlans present in the two
         # counters.
         applyIntfVlanCounter( cfmRxCounters.cfmIntfVlanRxCounter,
                               self.applyIntfVlanRxCtr )
         applyIntfVlanCounter( cfmRxCounters.cfmIntfVlanRxDropCounter,
                               self.applyIntfVlanRxDropCtr )
      domainFilter = self.args.get( 'DOMAIN_NAME' )
      maNameFilter = self.args.get( 'MA_NAME' )
      localMepFilter = self.args.get( 'MEP_ID' )
      def applyLMepCounter( counters, func ):
         for localMepKey in counters:
            domainName = localMepKey.domainName()
            mdLevel = localMepKey.mdLevel
            maName = str( localMepKey.maName )
            localMepId = localMepKey.localMepId
            # Local MEP counters with no Sysdb mapping(the local MEP is not
            # configured in aleConfig) will be mapped to a "dummy" intfVlan.
            intfId = "none"
            vlanId = 0
            ccCtrsModel = model
            mdConfig = aleCfmConfig.mdConfig.get( domainName )

            if mdConfig:
               maConfig = mdConfig.maConfig.get( maName )
               if maConfig:
                  lMepConfig = maConfig.localMepConfig.get( localMepId )
                  if lMepConfig:
                     intfId = lMepConfig.intfId
                     vlanId = lMepConfig.primaryVlanId
            counter = counters.get( localMepKey )
            filterMatch = False
            if domainFilter is None or domainName == domainFilter:
               if maNameFilter is None or maName == maNameFilter:
                  if localMepFilter is None or localMepId == localMepFilter:
                     filterMatch = True
            if self.args.get( 'detail' ):
               if intfId == "none":
                  if ccCtrsModel.noIntfVlanMapping is None and filterMatch:
                     # The model will not be created if the mep with no Sysdb mapping
                     # is filtered out, as it does not need to keep track of
                     # aggregate counters.
                     ccCtrsModel.noIntfVlanMapping = CcCountersModel()
                  ccCtrsModel = ccCtrsModel.noIntfVlanMapping
               else:
                  if intfId not in model.interfaces:
                     ccCtrsModel.interfaces[ intfId ] = IntfCcCountersModel()
                  intfModel = ccCtrsModel.interfaces[ intfId ]
                  if vlanId not in intfModel.vlans:
                     intfModel.vlans[ vlanId ] = \
                        self.initializeIntfVlanCounterModel(
                           IntfVlanCcCountersModel() )
                  ccCtrsModel = intfModel.vlans[ vlanId ]
                  # Aggregate the intfVlan tx(failure)count.
                  if isinstance( counter,
                                 Tac.Type( "Cfm::Smash::LocalMepCcmTxCounter" ) ):
                     ccCtrsModel.txCount += counter.txCount
                  if isinstance( counter,
                        Tac.Type( "Cfm::Smash::LocalMepCcmTxFailureCounter" ) ):
                     ccCtrsModel.txFailureCount += counter.txFailureCount
            if not filterMatch:
               continue
            if domainName not in ccCtrsModel.domains:
               ccCtrsModel.domains[ domainName ] = MdCcCountersModel()
               ccCtrsModel.domains[ domainName ].level = mdLevel
            mdModel = ccCtrsModel.domains[ domainName ]
            if maName not in mdModel.associations:
               mdModel.associations[ maName ] = MaCcCountersModel()
            maModel = mdModel.associations[ maName ]
            if localMepId not in maModel.localMeps:
               maModel.localMeps[ localMepId ] = \
                  self.initializeLocalMepCounterModel( LocalMepCcCountersModel() )
               maModel.localMeps[ localMepId ].intfId = intfId
               maModel.localMeps[ localMepId ].vlanId = vlanId
            lMepModel = maModel.localMeps[ localMepId ]
            # populate the lMepModel here
            func( lMepModel, counter )
      # Build the MEP models based on the MEPs present in the four counters, and
      # assign them to the appropriate intfVlan model in the case of detail option.
      applyLMepCounter( cfmTxCounters.localMepCcmTxCounter, self.applyLMepTxCtr )
      applyLMepCounter( cfmTxCounters.localMepCcmTxFailureCounter,
                        self.applyLMepTxFailCtr )
      applyLMepCounter( cfmRxCounters.localMepCcmRxCounter, self.applyLMepRxCtr )
      applyLMepCounter( cfmRxCounters.localMepCcmRxDropCounter,
                        self.applyLMepRxDropCtr )
      if self.args.get( 'detail' ):
         if domainFilter is not None or maNameFilter is not None or \
            localMepFilter is not None:
            # Purge intf/intfVlan models whose MEP models were all filtered out.
            intfVlansToDelete = set()
            for intfId in model.interfaces:
               intfModel = model.interfaces[ intfId ]
               for vlanId in intfModel.vlans:
                  intfVlanModel = intfModel.vlans[ vlanId ]
                  if not intfVlanModel.domains:
                     intfVlansToDelete.add( ( intfId, vlanId ) )
            intfsToDelete = set()
            for intfId, vlanId in intfVlansToDelete:
               intfModel = model.interfaces[ intfId ]
               del intfModel.vlans[ vlanId ]
               if not intfModel.vlans:
                  intfsToDelete.add( intfId )
            for intfId in intfsToDelete:
               del model.interfaces[ intfId ]

def showCfmContinuityCheckCounters( mode, args ):
   s = ShowCfmCcCounters( args )
   model = CcCountersDetailModel() if 'detail' in args else CcCountersModel()
   s.addCounters( model )
   return model

# Two separate command classes were created in order to use different
# counter models based on whether the 'detail' keyword is present
# in the command.
class ShowCfmContinuityCheckCounters( ShowCommand.ShowCliCommandClass ):
   syntax = 'show cfm continuity-check counters [ FILTER ]'
   data = {
      'cfm' : cfmShowKw,
      'continuity-check' : continuityCheckShowKw,
      'counters' : cfmCcCountersKw,
      'FILTER' : EndPointFilter,
   }

   cliModel = CcCountersModel
   handler = showCfmContinuityCheckCounters

if toggleCfmSwUpMepCcmEnabled():
   BasicCli.addShowCommandClass( ShowCfmContinuityCheckCounters )

class ShowCfmContinuityCheckDetailCounters( ShowCommand.ShowCliCommandClass ):
   syntax = ShowCfmContinuityCheckCounters.syntax + ' detail'
   data = ShowCfmContinuityCheckCounters.data.copy()
   data[ 'detail' ] = cfmCcCountersDetailShowKw

   cliModel = CcCountersDetailModel
   handler = showCfmContinuityCheckCounters

if toggleCfmSwUpMepCcmEnabled():
   BasicCli.addShowCommandClass( ShowCfmContinuityCheckDetailCounters )

# --------------------------------------------------------------------------------
# show cfm summary
# --------------------------------------------------------------------------------

class ShowSummary():
   def addSummary( self, model ):
      model.mdCount = 0
      model.maCount = 0
      model.mepCount = 0
      if cfmHwSupportStatus.mepHwSupported():
         model.ccHwOffloadActive = True
      else:
         model.ccHwOffloadActive = False
         model.ccHwOffloadInactiveReason = 'unsupported'
         if not cfmHwSupportStatus.mepSwSupported():
            # 'cfm' keyword for the show command will be guarded if neither support
            # is present, so this is a hypothetical situation.
            model.ccInactiveReason = 'unsupported'
      model.ccActive = False
      # Maps from profile to MAs that are associated with the profile.
      profileToMa = {}
      domains = set( aleCfmConfig.mdConfig ) & set( cfmStatus.mdStatus )
      model.mdCount = len( domains )
      for domainName in domains:
         mdAleConfig = aleCfmConfig.mdConfig.get( domainName )
         mdStatus = cfmStatus.mdStatus.get( domainName )
         associations = set( mdAleConfig.maConfig ) & set( mdStatus.maStatus )
         model.maCount += len( associations )
         for maName in associations:
            maAleConfig = mdAleConfig.maConfig.get( maName )
            maStatus = mdStatus.maStatus.get( maName )
            meps = set( maAleConfig.localMepConfig ) & set( maStatus.localMepStatus )
            model.mepCount += len( meps )
            mdConfig = cfmConfig.mdConfig.get( domainName )
            if mdConfig is not None:
               t0( 'MdConfig: ', domainName, ' is present' )
               maConfig = mdConfig.maConfig.get( maName )
               if maConfig is not None:
                  t0( 'MaConfig: ', maName, ' is present' )
                  profile = maConfig.cfmProfileName
                  if profile not in profileToMa:
                     t0( 'Profile: ', profile, ' has an associated MA' )
                     profileToMa[ profile ] = []
                  profileToMa[ profile ].append( ( domainName, maName ) )

      routingDisabledModel = CcSummaryLocActionModel()
      routingDisabledModel.action = "interfaceRoutingDisable"
      if cfmConfig.locActionRoutingDisabled:
         if not cfmHwSupportStatus.mepHwSupported():
            # Configured, but not operational.
            routingDisabledModel.reason = 'unsupported'
            model.inactiveLocActions.append( routingDisabledModel )
         else:
            # Configured, and operational.
            model.locActions.append( routingDisabledModel )
      else:
         # Neither configured or operational.
         model.inactiveLocActions.append( routingDisabledModel )

      logging = False
      # pylint: disable=too-many-nested-blocks
      for profileName in cfmConfig.cfmProfile:
         profileModel = CfmSummaryProfileModel()
         profileModel.maCount = 0
         profileConfig = cfmConfig.cfmProfile.get( profileName )
         profileCcmConfig = profileConfig.ccmConfig
         operCcmConfig = None
         if profileName in profileToMa and len( profileToMa[ profileName ] ) > 0:
            profileModel.maCount = len( profileToMa[ profileName ] )
            for domainName, maName in profileToMa[ profileName ]:
               # In the case where aleConfig has changed between here and the point
               # where profileToMa was constructed, walk through the MD/MA pairs
               # until we find a valid corresponding maAleConfig.
               if domainName in aleCfmConfig.mdConfig:
                  mdAleConfig = aleCfmConfig.mdConfig.get( domainName )
                  if maName in mdAleConfig.maConfig:
                     maAleConfig = mdAleConfig.maConfig.get( maName )
                     t0( 'An maAleConfig associated with profile: ', profileName,
                         ' was found' )
                     operCcmConfig = maAleConfig.ccmConfig
                     profileModel.ccmTxInterval = operCcmConfig.ccmTxInterval
                     if model.ccInactiveReason != 'unsupported' and \
                        maAleConfig.ccmEnable:
                        # There is at least one MA for which ccmEnable is
                        # (operationally) true.
                        model.ccActive = True
                     break
            if operCcmConfig is not None and CcmDefectAlarm.ccmDefectAlarmRemoteCcm \
               in operCcmConfig.ccmDefectAlarm:
               # There is at least one profile for which LOC defect alarm is active.
               logging = True
            for defectAlarm in CcmDefectAlarm.attributes:
               defectModel = CcSummaryDefectAlarmModel()
               defectModel.defectAlarm = defectAlarm
               if operCcmConfig is None or (
                  defectAlarm not in profileCcmConfig.ccmDefectAlarm and
                  defectAlarm not in operCcmConfig.ccmDefectAlarm ):
                  t0( 'DefectAlarm: ', defectAlarm, ' is neither configured or '
                      'operational' )
                  profileModel.inactiveDefectAlarms.append( defectModel )
               elif defectAlarm in operCcmConfig.ccmDefectAlarm:
                  t0( 'DefectAlarm: ', defectAlarm, ' is configured and '
                       'operational' )
                  profileModel.activeDefectAlarms.append( defectModel )
               else:
                  t0( 'DefectAlarm: ', defectAlarm, ' is configured, but not '
                      'operational' )
                  defectModel.reason = 'unsupported'
                  profileModel.inactiveDefectAlarms.append( defectModel )
         else:
            t0( 'Profile: ', profileName, ' has no associated MA' )
         model.profiles[ profileName ] = profileModel
      loggingModel = CcSummaryLocActionModel()
      loggingModel.action = 'logging'
      if logging:
         model.locActions.append( loggingModel )
      else:
         model.inactiveLocActions.append( loggingModel )
      return model

def showCfmSummary( mode, args ):
   s = ShowSummary()
   model = CfmSummaryModel()
   s.addSummary( model )
   return model

class ShowCfmSummary( ShowCommand.ShowCliCommandClass ):
   syntax = 'show cfm summary'
   data = {
      'cfm' : cfmShowKw,
      'summary' : cfmSummaryShowKw,
   }
   cliModel = CfmSummaryModel
   handler = showCfmSummary

if toggleCfmSwUpMepCcmEnabled():
   BasicCli.addShowCommandClass( ShowCfmSummary )

#-----------------------------------------------------------
# Have the Cli Agent mount all needed state from sysdb
#-----------------------------------------------------------
def Plugin( entityManager ):
   global cfmConfig
   global cfmStatus
   global aleCfmConfig
   global aleCfmStatus
   global cfmHwSupportStatus
   global trapConfig
   global mepStats
   global aisRxStatus
   global cfmRxCounters
   global cfmTxCounters
   global mepMiStatsHistory

   cfmConfig = ConfigMount.mount( entityManager, "cfm/config",
                                  "CfmAgent::Config", "w" )
   trapConfig = ConfigMount.mount( entityManager, "hardware/trap/config/trapConfig",
                                   "Arnet::TrapConfig", 'w' )
   cfmStatus = LazyMount.mount( entityManager, "cfm/status",
                                "CfmAgent::Status", "r" )
   cfmHwSupportStatus = LazyMount.mount( entityManager, "cfm/hwSupportStatus",
                                         "Cfm::HwSupportStatus", "r" )
   aleCfmConfig = LazyMount.mount( entityManager, "cfm/aleConfig",
                                   "Cfm::AleConfig", "r" )
   aleCfmStatus = LazyMount.mount( entityManager, "cfm/aleStatus",
                                   "Cfm::AleStatus", "r" )
   mepStats = SmashLazyMount.mount( entityManager, "cfm/mepSmashTable",
                                    "Cfm::Smash::MepStatus",
                                    SmashLazyMount.mountInfo( 'reader' ) )
   aisRxStatusMountPath = Tac.Type( 'Cfm::Smash::AisRxStatus' ).mountPath
   aisRxStatus = SmashLazyMount.mount( entityManager, aisRxStatusMountPath,
                                       "Cfm::Smash::AisRxStatus",
                                       SmashLazyMount.mountInfo( 'reader' ) )
   cfmRxCountersMountPath = Tac.Type( 'Cfm::Smash::CfmRxCounters' ).mountPath
   cfmRxCounters = SmashLazyMount.mount( entityManager, cfmRxCountersMountPath,
                                         "Cfm::Smash::CfmRxCounters",
                                         SmashLazyMount.mountInfo( 'reader' ) )
   cfmTxCountersMountPath = Tac.Type( 'Cfm::Smash::CfmTxCounters' ).mountPath
   cfmTxCounters = SmashLazyMount.mount( entityManager, cfmTxCountersMountPath,
                                         "Cfm::Smash::CfmTxCounters",
                                         SmashLazyMount.mountInfo( 'reader' ) )
   mepMiStatsHistoryMountPath = Tac.Type( 'CfmShark::MepMiStatsHistory' ).mountPath
   mepMiStatsHistory = SharkLazyMount.mount( entityManager,
                                             mepMiStatsHistoryMountPath,
                                             "CfmShark::MepMiStatsHistory",
                                             SharkLazyMount.mountInfo( 'shadow' ),
                                             True )
