# Copyright (c) 2016 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import BasicCli
import BasicCliModes
import CliCommand
import CliMatcher
from CliPlugin import ConfigSessionCli
from CliPlugin import ConfigSessionModel
from CliPlugin import ConfigSessionShowCli
from CliPlugin import TechSupportCli
from CliSession import sessionLock
import CliToken.Configure
from CliToken.Service import configMatcherForAfterService, serviceMatcherForConfig
import ConfigMount
import FileCliUtil
import GitCliLib
import GitLib
import ShowCommand
import Tracing
import Url

t0 = Tracing.trace0

cliConfig = None
# Filename shouldn't have / in the name
# under the hood the checkpoint name is a git tag. We are restricting git tags
# from having : in the name
CHECKPOINT_NAME_REGEX = r'[^/:]+'

matcherCheckpoint = CliMatcher.KeywordMatcher( 'checkpoint',
      helpdesc='Creating a new checkpoint' )

#--------------------------------------------------------------------------------
# [ no | default ] service configuration checkpoint max saved MAX_CHECKPOINTS
#--------------------------------------------------------------------------------
# set max checkpoint file number in configure mode
def setMaxCheckpointNum( mode, args ):
   maxCheckpoints = args.get( 'MAX_CHECKPOINTS', cliConfig.defaultMaxCheckpoints )
   cliConfig.maxCheckpoints = maxCheckpoints

   GitLib.maybeSquashGitHistory( mode.entityManager.sysname(),
         cliConfig.maxCheckpoints )

class GlobalMaxCheckpointNumCmd( CliCommand.CliCommandClass ):
   syntax = 'service configuration checkpoint max saved MAX_CHECKPOINTS'
   noOrDefaultSyntax = 'service configuration checkpoint max saved ...'
   data = {
      'service': serviceMatcherForConfig,
      'configuration': configMatcherForAfterService,
      'checkpoint': 'Creating a new checkpoint',
      'max': 'Set a max limit',
      'saved': 'Maximum number of saved checkpoints',
      'MAX_CHECKPOINTS': CliMatcher.IntegerMatcher( 0, 1000,
         helpdesc='Maximum number of saved checkpoints' ),
   }
   handler = setMaxCheckpointNum
   noOrDefaultHandler = handler

BasicCliModes.GlobalConfigMode.addCommandClass( GlobalMaxCheckpointNumCmd )

#--------------------------------------------------------------------------------
# configure checkpoint restore CHECKPOINT_NAME
#--------------------------------------------------------------------------------
def getCheckpointNames( mode ):

   checkpointNames = []
   for commit in GitLib.getGitCommits( mode.entityManager.sysname() ):
      checkpointNames.append( commit[ 'commitHash' ] )
      checkpointNames += commit[ 'tags' ]

   return checkpointNames

def restoreCheckpoint( mode, args ):
   # get the url from checkpoint name
   checkpointName = args[ 'CHECKPOINT_NAME' ]
   commits = GitLib.getGitCommits( mode.entityManager.sysname(),
         commitHash=checkpointName )
   if not commits:
      mode.addError( f'Checkpoint {checkpointName} not found' )
      return
   url = Url.parseUrl( f'checkpoint:{checkpointName}',
                       Url.Context( *Url.urlArgsFromMode( mode ) ) )
   FileCliUtil.checkUrl( url )
   # replace the running-config with the checkpoint file
   ConfigSessionCli.configureReplace( mode, url )

class ConfigureCheckpointRestoreCmd( CliCommand.CliCommandClass ):
   syntax = 'configure checkpoint restore CHECKPOINT_NAME'
   data = {
      'configure': CliToken.Configure.configureParseNode,
      'checkpoint': matcherCheckpoint,
      'restore': 'Restore running-config from a previous checkpoint file',
      'CHECKPOINT_NAME': CliMatcher.DynamicNameMatcher(
         getCheckpointNames, 'Restore checkpoint name',
         pattern='\\S+' ),
   }
   handler = restoreCheckpoint

BasicCliModes.EnableMode.addCommandClass( ConfigureCheckpointRestoreCmd )

#--------------------------------------------------------------------------------
# configure checkpoint save [ CHECKPOINT_NAME ]
#--------------------------------------------------------------------------------
def configureCheckpoint( mode, args ):
   if cliConfig.maxCheckpoints == 0:
      return

   # if a session is being modified don't allow a manual checkpoint to be created
   with sessionLock:
      GitCliLib.saveRunningConfigToGit( mode.entityManager,
            subject="User triggered a manual checkpoint to be created",
            trailers={ "Commit-type": "manual" },
            allowEmptyCommit=True,
            author=GitCliLib.getCurrentUser( mode ),
            maxHistorySize=cliConfig.maxCheckpoints,
            tag=args.get( 'CHECKPOINT_NAME' ) )

class ConfigureCheckpointSaveCmd( CliCommand.CliCommandClass ):
   syntax = 'configure checkpoint save [ CHECKPOINT_NAME ]'
   data = {
      'configure': CliToken.Configure.configureParseNode,
      'checkpoint': matcherCheckpoint,
      'save': 'Save running-config to a checkpoint file',
      'CHECKPOINT_NAME': CliMatcher.PatternMatcher( pattern=CHECKPOINT_NAME_REGEX,
         helpdesc='Checkpoint name', helpname='WORD' ),
   }
   handler = configureCheckpoint

BasicCliModes.EnableMode.addCommandClass( ConfigureCheckpointSaveCmd )

#-----------------------------------------------------------------------------------
# show configuration checkpoints
# show the checkpoint file name, creating date and creating user
#-----------------------------------------------------------------------------------
def showConfigureCheckpoint( mode, args ):
   checkpoints = {}
   for commit in GitLib.getGitCommits( mode.entityManager.sysname() ):
      # converting nanoseconds to fractional seconds
      timestamp = float( commit[ 'trailers' ][ 'Timestamp-ns' ] ) / 1000000000
      # only 0 or 1 tags should be in this list
      assert len( commit[ 'tags' ] ) in ( 0, 1 )
      checkpoints[ commit[ 'commitHash' ] ] = ConfigSessionModel.Checkpoint(
         name=commit[ 'tags' ][ 0 ] if commit[ 'tags' ] else None,
         fileDate=timestamp,
         fileUser=commit[ 'author' ] )

   return ConfigSessionModel.Checkpoints( checkpoints=checkpoints,
         maxCheckpoints=cliConfig.maxCheckpoints )

class ShowConfigCheckpoints( ShowCommand.ShowCliCommandClass ):
   syntax = 'show configuration checkpoints'
   data = {
            'configuration': ConfigSessionShowCli.configKwMatcher,
            'checkpoints': 'List all saved checkpoints',
          }
   cliModel = ConfigSessionModel.Checkpoints
   handler = showConfigureCheckpoint

BasicCli.addShowCommandClass( ShowConfigCheckpoints )

def Plugin( entityManager ):
   global cliConfig
   cliConfig = ConfigMount.mount( entityManager,
                                  "cli/session/input/config",
                                  "Cli::Session::CliConfig", "w" )

   TechSupportCli.registerShowTechSupportCmd(
   '2020-05-12 01:22:30',
   summaryCmds=[ 'show configuration checkpoints' ] )
