#!/usr/bin/env python3
# Copyright (c) 2019 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# pylint: disable=consider-using-f-string

import Tac
import BasicCli
import ConfigMount
import CliCommand
import CliMatcher
import CliParser
from CliMode.Classification import ( AppTrafficRecConfigMode,
                                     AppProfileConfigMode,
                                     AppConfigModeIpv4,
                                     AppConfigModeL4,
                                     CategoryConfigMode,
                                     FieldSetL4PortConfigMode,
                                     FieldSetIpPrefixConfigMode )

from CliPlugin.ClassificationCliLib import ( AppRecognitionContext,
                                   AppProfileContext,
                                   AppContext,
                                   applicationModeGuards,
                                   CategoryContext,
                                   CommitAbortModelet,
                                   FieldSetL4PortBaseConfigCmd,
                                   FieldSetL4PortConfigCmds,
                                   FieldSetL4PortExceptConfigCmds,
                                   FieldSetIpPrefixBaseConfigCmd,
                                   FieldSetIpPrefixConfigCmds,
                                   PrefixFieldSetCmdBase,
                                   generateFieldSetExpression,
                                   ProtocolFieldSetBaseCmd,
                                   generateTcpFlagExpression,
                                   ipv4ProtoExpr, invalidPortConflictMsg,
                                   invalidProtocolConflictMsg,
                                   protectedFieldSetNamesRegex,
                                   AppTransport, DEFAULT_SERVICE )
from CliPlugin import ClassificationModel
from TypeFuture import TacLazyType
from TeCliLib import dscpAclNames
from Toggles.ClassificationToggleLib import toggleAppDscpMatchEnabled
import ShowCommand
import LazyMount
import os
import MultiRangeRule

appRecognitionConfig = None
fieldSetConfig = None
classificationStatus = None
entityMib = None

TacDscp = TacLazyType( 'Arnet::DscpValue' )

def isDpiSupportedOnPlatform( mode ):
   return ( ( entityMib.root.modelName == 'vEOS' or
              entityMib.root.vendorType == 'Caravan' ) if entityMib.root else
            ( os.getenv( 'SIMULATION_CLASSIFICATION' ) ) )

def guardClassDpi( mode, token ):
   if isDpiSupportedOnPlatform( mode ):
      return None
   return CliParser.guardNotThisPlatform

applicationKw = CliMatcher.KeywordMatcher( 'application',
                                           helpdesc='Show application' )
trafficKw = CliMatcher.KeywordMatcher( 'traffic', helpdesc='Application Traffic' )
recognitionKw = CliMatcher.KeywordMatcher( 'recognition',
                                           helpdesc='Traffic recognition' )

appNameKw = CliCommand.guardedKeyword(
      'name',
      helpdesc='Match only the specific application name',
      maxMatches=1,
      guard=guardClassDpi )

classifierKw = CliCommand.guardedKeyword(
      'classifier',
      helpdesc='Match only the specific application classifier',
      maxMatches=1,
      guard=guardClassDpi )

appServiceKw = CliCommand.guardedKeyword(
      'service',
      helpdesc='Match only the specific application service',
      maxMatches=1,
      guard=guardClassDpi )

appCategoryKw = CliCommand.guardedKeyword(
      'category',
      helpdesc='Match only the specific application category',
      maxMatches=1,
      guard=guardClassDpi )

def appServiceNames( mode, context ):
   appName = context.sharedResult[ 'APP' ]

   serviceStr = {
      'audio-video': 'audio-video',
      'chat': 'chat',
      'default': 'default',
      'file-transfer': 'file transfer',
      'networking-protocols': 'networking protocols',
      'peer-to-peer': 'peer to peer',
      'software-update': 'software update'
   }
   helpStr = 'Match the {} service for this application'

   # Get the services supported for an application
   services = set()
   if appName in appRecognitionConfig.app:
      services.update(
         appRecognitionConfig.app[ appName ].defaultServiceCategory )
   else:
      # Ipv4 app not yet created. Only default service is supported for ipv4 apps
      services.add( DEFAULT_SERVICE )
   return { s: helpStr.format( serviceStr.get( s, s ) )
            for s in services }

appServiceNamesMatcher = CliMatcher.DynamicKeywordMatcher( appServiceNames,
      passContext=True, alwaysMatchInStartupConfig=True )

classifierMatcher = CliCommand.Node(
   matcher=CliMatcher.EnumMatcher( {
      'dpi': 'Predefined DPI application',
      'custom-dpi': 'User defined DPI application',
      'ipv4': 'IPv4 application', } ),
   maxMatches=1 )

def serviceNames( mode ):
   return appRecognitionConfig.service

serviceNamesMatcher = CliCommand.Node(
   matcher=CliMatcher.DynamicNameMatcher(
      serviceNames,
      helpdesc="Service name",
      priority=CliParser.PRIO_LOW ),
   maxMatches=1 )

def categoryNames( mode ):
   return appRecognitionConfig.category

categoryNameMatcher = CliCommand.Node(
   matcher=CliMatcher.DynamicNameMatcher(
      categoryNames,
      helpdesc="Category name",
      priority=CliParser.PRIO_LOW ),
   maxMatches=1 )

def categoryServiceNames( mode ):
   services = {}
   for service in appRecognitionConfig.service:
      services[ service ] = \
         "Select the %s service for this category" % service
   return services

categoryServiceNamesMatcher = CliMatcher.DynamicKeywordMatcher(
      categoryServiceNames, alwaysMatchInStartupConfig=True )

#---------------------------------------------------
# The "application traffic recognition" mode command
#---------------------------------------------------
class AppTrafficRecConfigModeCmd( CliCommand.CliCommandClass ):
   syntax = '''application traffic recognition'''
   noOrDefaultSyntax = syntax
   data = {
      'application': 'Configure application',
      'traffic': trafficKw,
      'recognition': recognitionKw,
   }

   @staticmethod
   def handler( mode, args ):
      context = AppRecognitionContext( appRecognitionConfig,
                                       fieldSetConfig )
      context.copyEditAppRecognitionConfig()
      childMode = mode.childMode( AppTrafficRecConfigMode, context=context )
      mode.session_.gotoChildMode( childMode )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      fieldSetConfig.fieldSetL4Port.clear()
      fieldSetConfig.fieldSetIpPrefix.clear()
      for app, appConfig in appRecognitionConfig.app.items():
         if not appConfig.defaultApp and not appConfig.readonly:
            del appRecognitionConfig.app[ app ]
      for cat, catConfig in appRecognitionConfig.category.items():
         if catConfig.defaultCategory:
            catConfig.appService.clear()
         else:
            del appRecognitionConfig.category[ cat ]
      appRecognitionConfig.appProfile.clear()

# --------------------------------------------------------------------------
# The "field-set l4-port PORT_SET_NAME" command
# --------------------------------------------------------------------------
def getL4PortFieldSetNames( mode ):
   if fieldSetConfig is None:
      return []
   return fieldSetConfig.fieldSetL4Port

l4PortFieldSetNameMatcher = CliMatcher.DynamicNameMatcher(
   getL4PortFieldSetNames,
   "Layer-4 port field-set name",
   pattern=protectedFieldSetNamesRegex( 'port' ),
   priority=CliParser.PRIO_LOW )

class FieldSetL4PortConfigCmd( FieldSetL4PortBaseConfigCmd ):
   syntax = 'field-set l4-port FIELD_SET_NAME'
   noOrDefaultSyntax = syntax
   data = {
      'FIELD_SET_NAME': l4PortFieldSetNameMatcher,
   }
   data.update( FieldSetL4PortBaseConfigCmd._baseData )

   @classmethod
   def _getContextKwargs( cls, fieldSetL4PortName, mode=None ):
      parentContext = mode.getContext()
      config = parentContext.fieldSetEditConfig
      return {
         'fieldSetL4PortName': fieldSetL4PortName,
         'fieldSetConfig': config,
         'childMode': FieldSetL4PortConfigMode,
         'featureName': 'app'
      }

# --------------------------------------------------------------------------
# The "field-set ipv4 prefix FIELD_SET_NAME" command
# --------------------------------------------------------------------------
def getIpPrefixFieldSetNames( mode ):
   if fieldSetConfig is None:
      return []
   # pylint: disable-next=unnecessary-comprehension
   return [ key for key in fieldSetConfig.fieldSetIpPrefix ]

ipPrefixFieldSetNameMatcher = CliMatcher.DynamicNameMatcher(
   getIpPrefixFieldSetNames,
   "IP prefix field-set name",
   pattern=protectedFieldSetNamesRegex( 'prefix' ),
   priority=CliParser.PRIO_LOW )

class FieldSetIpPrefixConfigCmd( FieldSetIpPrefixBaseConfigCmd ):
   # XXX ipv6 is not supported yet
   syntax = 'field-set ipv4 prefix FIELD_SET_NAME'
   noOrDefaultSyntax = syntax
   data = {
      'FIELD_SET_NAME': ipPrefixFieldSetNameMatcher,
   }
   data.update( FieldSetIpPrefixBaseConfigCmd._baseData )

   @classmethod
   def _getContextKwargs( cls, fieldSetIpPrefixName, setType, mode=None ):
      assert setType == 'ipv4' # only ipv4 prefix field-set is supported
      parentContext = mode.getContext()
      config = parentContext.fieldSetEditConfig
      return {
         'fieldSetIpPrefixName': fieldSetIpPrefixName,
         'fieldSetConfig': config,
         'setType': setType,
         'childMode': FieldSetIpPrefixConfigMode,
         'featureName': 'app',
      }

# --------------------------------------------------------------------------
# The "application-profile <name>" command
# --------------------------------------------------------------------------
def appProfileNames( mode ):
   if appRecognitionConfig is None:
      return []
   return appRecognitionConfig.appProfile

appProfileNameMatcher = CliCommand.Node(
   matcher=CliMatcher.DynamicNameMatcher(
      appProfileNames,
      "Application profile name",
      priority=CliParser.PRIO_LOW ),
   maxMatches=1 )

# Returns only custom apps
def appNames( mode, af ):
   if appRecognitionConfig is None:
      return []

   apps = []
   for name, app in appRecognitionConfig.app.items():
      if app.defaultApp or app.readonly:
         continue
      if af is None or app.af == af:
         apps.append( name )
   return apps

# Return custom and default/built-in apps
def allAppNames( mode ):
   return appRecognitionConfig.app if appRecognitionConfig else []

# Matches only custom apps
def makeAppNameMatcher( af ):
   return CliMatcher.DynamicNameMatcher( lambda mode: appNames( mode, af ),
                                         "Application name",
                                         priority=CliParser.PRIO_LOW )

allAppNamesHelpStr = "Specify a predefined or custom application name"

# Matches default and custom apps
allAppNamesMatcher = CliMatcher.DynamicNameMatcher( allAppNames,
                                                    allAppNamesHelpStr,
                                                    priority=CliParser.PRIO_LOW )

appNameNode = CliCommand.Node( matcher=allAppNamesMatcher, maxMatches=1,
                               storeSharedResult=True )

class AppProfileConfigCmd( CliCommand.CliCommandClass ):
   syntax = 'application-profile PROFILE'
   noOrDefaultSyntax = syntax
   data = {
      'application-profile': 'Configure application profile',
      'PROFILE': appProfileNameMatcher,
   }

   @staticmethod
   def handler( mode, args ):
      name = args[ 'PROFILE' ]
      parentContext = mode.getContext()
      context = AppProfileContext( name, parentContext )

      if context.hasAppProfile( name ):
         context.copyEditAppProfile()
      else:
         context.newEditAppProfile()

      childMode = mode.childMode( context.childMode, context=context )
      mode.session_.gotoChildMode( childMode )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      name = args[ 'PROFILE' ]
      parentContext = mode.getContext()
      context = AppProfileContext( name, parentContext )
      if context.hasAppProfile( name ):
         context.delAppProfile( name )

class AppProfileConfigCmds( CliCommand.CliCommandClass ):
   syntax = 'application APP [ service SERVICE ]'
   noOrDefaultSyntax = syntax
   data = {
      'application': 'Add application to application profile',
      'APP': appNameNode,
      'service': appServiceKw,
      'SERVICE': appServiceNamesMatcher,
   }

   @staticmethod
   def handler( mode, args ):
      appName = args.get( 'APP' )
      context = mode.getContext()
      serviceName = args.get( 'SERVICE' )
      context.updateAppService( appName, serviceName )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      appName = args.get( 'APP' )
      context = mode.getContext()
      serviceName = args.get( 'SERVICE' )
      context.updateAppService( appName, serviceName, add=False )

def getTransport( mode, context ):
   appName = context.sharedResult[ 'APP' ]
   helpStr = \
        'Match the application only if it is used to transport another application'

   if AppTransport.appTransport( appName ):
      return { "transport": helpStr }
   else:
      return {}

transportKwMatcher = CliMatcher.DynamicKeywordMatcher( getTransport,
        passContext=True )
transportNode = CliCommand.Node( matcher=transportKwMatcher, guard=guardClassDpi )

class AppProfileAppTransportConfigCmds( CliCommand.CliCommandClass ):
   syntax = 'application APP TRANSPORT'
   noOrDefaultSyntax = syntax
   data = {
      'application': 'Add application to application profile',
      'APP': appNameNode,
      'TRANSPORT': transportNode,
   }

   @staticmethod
   def handler( mode, args ):
      appName = args[ 'APP' ]
      context = mode.getContext()
      context.addAppTransport( appName )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      appName = args[ 'APP' ]
      context = mode.getContext()
      context.delAppTransport( appName )

class AppProfileCategoryConfigCmds( CliCommand.CliCommandClass ):
   syntax = 'category CATEGORY [ service SERVICE ]'
   noOrDefaultSyntax = syntax
   data = {
      'category': CliCommand.guardedKeyword( 'category',
                                 helpdesc='Add category to application profile',
                                 guard=guardClassDpi ),
      'CATEGORY': categoryNameMatcher,
      'service': 'Select a specific service in the category',
      'SERVICE': categoryServiceNamesMatcher,
   }

   @staticmethod
   def handler( mode, args ):
      categoryName = args[ 'CATEGORY' ]
      context = mode.getContext()
      serviceName = args.get( 'SERVICE' )
      context.updateCategoryService( categoryName, serviceName )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      categoryName = args[ 'CATEGORY' ]
      context = mode.getContext()
      serviceName = args.get( 'SERVICE' )
      context.updateCategoryService( categoryName, serviceName, add=False )

def applicationGuard( mode, token ):
   if any( guardCallback( token )
           for guardCallback in applicationModeGuards.extensions() ):
      return None
   return CliParser.guardNotThisPlatform

# --------------------------------------------------------------------------
# The "application (ipv4|l4) <name>" commands, XXX ipv6 not supported yet
# --------------------------------------------------------------------------
_afToToken = { 'ipv4': 'ipv4',
               'bothIpv4AndIpv6': 'l4' }

def warnAfMismatch( mode, cmd ):
   mode.addWarning(
      "Trying to edit application with incorrect type. "
      "Did you mean '%s'?" % cmd )

def makeAppConfigCmd( applicationType, af ):
   appToken = applicationType.lower()

   class AppConfigCmd( CliCommand.CliCommandClass ):
      syntax = 'application %s APP' % appToken
      noOrDefaultSyntax = syntax
      data = {
         'application': 'Configure application',
         appToken: CliCommand.guardedKeyword(
               appToken,
               helpdesc='Configure %s application' % applicationType,
               guard=applicationGuard ),
         'APP': makeAppNameMatcher( af ),
      }

      @staticmethod
      def handler( mode, args ):
         name = args[ 'APP' ]
         parentContext = mode.getContext()
         context = AppContext( name, parentContext, af )

         app = context.getApp( name )
         if app:
            if app.readonly:
               mode.addWarning( "Cannot edit readonly application " + name )
               return
            if app.defaultApp and isDpiSupportedOnPlatform( mode ):
               return

            if app.defaultApp:
               # Override the default app with user configured app
               context.newEditApp()
            else:
               if app.af != af:
                  cmd = f'application {_afToToken[ app.af ]} {name}'
                  warnAfMismatch( mode, cmd )
                  context.af = app.af
               context.copyEditApp()
         else:
            context.newEditApp()

         childMode = mode.childMode( context.childMode, context=context )
         mode.session_.gotoChildMode( childMode )

      @staticmethod
      def noOrDefaultHandler( mode, args ):
         name = args[ 'APP' ]
         parentContext = mode.getContext()
         context = AppContext( name, parentContext, af )
         app = context.getApp( name )
         if app:
            if app.readonly:
               mode.addWarning( "Cannot delete readonly application " + name )
               return
            if app.defaultApp:
               return
            if app.af != af:
               cmd = f'no application {_afToToken[ app.af ]} {name}'
               warnAfMismatch( mode, cmd )
            context.delApp( name )

   return AppConfigCmd

AppConfigCmdIpv4 = makeAppConfigCmd( 'IPv4', 'ipv4' )
AppConfigCmdL4 = makeAppConfigCmd( 'L4', 'bothIpv4AndIpv6' )

prefixFieldSetExpr = generateFieldSetExpression( ipPrefixFieldSetNameMatcher,
                                                 'FIELD_SET',
                                                 allowMultiple=False )

class AppConfigFieldSetCmd( PrefixFieldSetCmdBase ):
   data = {
      'FIELD_SET': prefixFieldSetExpr,
   }
   data.update( PrefixFieldSetCmdBase._baseData )

#------------------------------------------------------------------------------------
# "protocol PROTOCOL | TCP_UDP [ (source|destination) port field-set NAME ]"
#  where PROTOCOL is:
#  ahp      Authentication Header Protocol
#  icmp     Internet Control Message Protocol
#  igmp     Internet Group Management Protocol (IGMP)
#  ospf     OSPF routing protocol
#  pim      Protocol Independent Multicast (PIM)
#  rsvp     Resource Reservation Protocol (RSVP)
#  tcp      Transmission Control Protocol
#  udp      User Datagram Protocol
#  vrrp     Virtual Router Redundancy Protocol
#  <1-255>  IP protocol number
#------------------------------------------------------------------------------------
portKwMatcher = CliMatcher.KeywordMatcher( 'port', helpdesc='Port' )
fieldSetKwMatcher = CliMatcher.KeywordMatcher( 'field-set',
                                               helpdesc='field-set' )
tcpUdpL4PortFieldSetNameMatcher = CliMatcher.DynamicNameMatcher(
   getL4PortFieldSetNames,
   "Layer-4 port field-set name for UDP and TCP",
   priority=CliParser.PRIO_LOW )

class ProtocolListIPv4ConfigCmd( CliCommand.CliCommandClass ):
   syntax = ( 'protocol PROTOCOL' )
   noOrDefaultSyntax = ( 'protocol [ PROTOCOL ]' )

   data = {
      'protocol': 'Protocol',
      'PROTOCOL': ipv4ProtoExpr,
   }

   @staticmethod
   def handler( mode, args ):
      protoList = args.get( 'PROTOCOL' )
      context = mode.getContext()
      if ProtocolListIPv4ConfigCmd._hasConflict( mode, protoList ):
         return

      rangeSet = protoList
      context.updateRangeAttr( 'proto', rangeSet, "Classification::ProtocolRange",
                               add=True )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      rangeSet = args.get( 'PROTOCOL' )
      context = mode.getContext()
      if not rangeSet:
         # no protoccol
         rangeSet = set()
      context.updateRangeAttr( 'proto', rangeSet, "Classification::ProtocolRange",
                               add=False )

   @staticmethod
   def _hasConflict( mode, protoList ):
      rangeSet = protoList
      context = mode.getContext()
      ( currentProtoSet, hasPort ) = context.portAndProtoConfigured()
      updatedSet = rangeSet | currentProtoSet
      tcpUdpSet = { 6, 17 } # tcp and udp
      nonTcpUdpSet = updatedSet - tcpUdpSet

      if nonTcpUdpSet and hasPort:
         mode.addError( invalidProtocolConflictMsg )
         return True
      return False

l4PortSrcFieldSetExpr = generateFieldSetExpression( tcpUdpL4PortFieldSetNameMatcher,
                                                   'SRC_FIELD_SET_NAME',
                                                    allowMultiple=False )
l4PortDstFieldSetExpr = generateFieldSetExpression( tcpUdpL4PortFieldSetNameMatcher,
                                                   'DST_FIELD_SET_NAME',
                                                    allowMultiple=False )

class ProtocolFieldSetConfigCmd( ProtocolFieldSetBaseCmd ):
   _sportFieldSetAttr = "srcPortFieldSet"
   _dportFieldSetAttr = "dstPortFieldSet"
   data = {
      'FLAGS_EXPR': generateTcpFlagExpression( tcpFlagsSupported=False ),
      'SRC_FIELD_SET_NAME': l4PortSrcFieldSetExpr,
      'DST_FIELD_SET_NAME': l4PortDstFieldSetExpr
   }
   data.update( ProtocolFieldSetBaseCmd._baseData )

   @classmethod
   def _updatePort( cls, mode, args, attrName, argListName, add=True ):
      pass

   @classmethod
   def _maybeHandleErrors( cls, mode, args, proto, source=False, destination=False,
                           fieldSet=False ):
      hasError = False
      context = mode.getContext()
      ( currentProtoSet, hasPort ) = context.portAndProtoConfigured()
      tcpUdpSet = { 6, 17 } # tcp and udp
      nonTcpUdpSet = currentProtoSet - tcpUdpSet
      if nonTcpUdpSet and hasPort:
         if proto and nonTcpUdpSet:
            # we already have non-tcp/Udp proto configured
            mode.addError( invalidPortConflictMsg % 'source|destination port' )
         else:
            mode.addError( invalidProtocolConflictMsg )
         hasError = True
      if not hasError:
         return
      cls._updateProtoAndPort( mode, args, proto, source, destination,
                               fieldSet=fieldSet, add=False )


# --------------------------------------------------------------------------
# The "category <name>" command
# --------------------------------------------------------------------------
class CategoryConfigCmd( CliCommand.CliCommandClass ):
   syntax = 'category CATEGORY'
   noOrDefaultSyntax = syntax
   data = {
      'category': CliCommand.guardedKeyword( 'category',
                                             helpdesc='Configure category',
                                             guard=guardClassDpi ),
      'CATEGORY' : categoryNameMatcher,
   }

   @staticmethod
   def handler( mode, args ):
      name = args[ 'CATEGORY' ]
      parentContext = mode.getContext()
      context = CategoryContext( name, parentContext )

      if context.hasCategory( name ):
         context.copyEditCategory()
      else:
         context.newEditCategory()

      childMode = mode.childMode( context.childMode, context=context )
      mode.session_.gotoChildMode( childMode )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      name = args[ 'CATEGORY' ]
      parentContext = mode.getContext()
      context = CategoryContext( name, parentContext )
      category = context.getCategory( name )
      if category:
         if category.defaultCategory:
            context.clearCategoryApps( name )
         else:
            context.delCategory( name )

class CategoryConfigCmds( CliCommand.CliCommandClass ):
   syntax = 'application APP [ service SERVICE ]'
   noOrDefaultSyntax = syntax
   data = {
      'application' : 'Add application to a category',
      'APP' : appNameNode,
      'service' : 'Match only the specific application service',
      'SERVICE' : appServiceNamesMatcher,
   }

   @staticmethod
   def handler( mode, args ):
      appName = args[ 'APP' ]
      serviceName = args.get( 'SERVICE' )
      mode.getContext().addCategoryApps( appName, serviceName )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      appName = args[ 'APP' ]
      serviceName = args.get( 'SERVICE', None )
      mode.getContext().delCategoryApps( appName, serviceName )

# ------------------------------------------------------------
# The "dscp <values>" command in "application ipv4 <name>"
# ------------------------------------------------------------
def dscpValueFromCli( dscpVal ):
   dscpResult = dscpAclNames.get( dscpVal, None )
   return ( dscpResult[ 0 ], True ) if dscpResult else ( dscpVal, False )

def dscpClassificationNamesDef( mode ):
   return { k: v[ 1 ] for k, v in dscpAclNames.items() }

def dscpRange():
   return TacDscp.min, TacDscp.max

def decodeArgs( mode, args ):
   names = args.get( 'DSCP_NAME', [] )
   values = [ value for rng in args[ 'DSCP_RANGE' ]
              for value in rng.values() ] if 'DSCP_RANGE' in args else []
   names.extend( values )
   return names

class AppConfigDscpCmd( CliCommand.CliCommandClass ):
   syntax = 'dscp { DSCP_NAME | DSCP_RANGE }'
   noOrDefaultSyntax = 'dscp [ { DSCP_NAME | DSCP_RANGE } ]'
   data = {
      'dscp': 'Specify DSCP values',
      'DSCP_NAME': CliMatcher.DynamicKeywordMatcher( dscpClassificationNamesDef ),
      'DSCP_RANGE': CliCommand.Node(
                  matcher=MultiRangeRule.MultiRangeMatcher(
                               rangeFn=dscpRange,
                               noSingletons=False,
                               helpdesc='Differentiated Services Code Point (DSCP) '
                                        'numeric values or ranges',
                  priority=CliParser.PRIO_HIGH ) ),
   }

   @staticmethod
   def handler( mode, args ):
      context = mode.getContext()
      dscpValues = decodeArgs( mode, args )
      rangeSet = set()
      for value in dscpValues:
         dscp, dscpUseName = dscpValueFromCli( value )
         rangeSet.add( dscp )
         context.updateDscpSymbolicValues( dscp, dscpUseName )
      context.updateRangeAttr( 'dscp', rangeSet, "Classification::DscpRangeConfig",
            add=True )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      context = mode.getContext()
      dscpValues = decodeArgs( mode, args )
      rangeSet = set()
      for value in dscpValues:
         dscp, _ = dscpValueFromCli( value )
         rangeSet.add( dscp )
         context.clearDscpSymbolicValues( dscp )
      context.updateRangeAttr( 'dscp', rangeSet, "Classification::DscpRangeConfig",
            add=False )

class ShowAppTraficRecFilterExp( CliCommand.CliExpression ):
   expression = '''[ { ( name APP )
                     | ( classifier CLASSIFIER_TYPE )
                     | ( service SERVICE_NAME )
                     | ( category CATEGORY_NAME ) } ]'''
   data = {
      'name': appNameKw,
      'APP': appNameNode,
      'classifier': classifierKw,
      'CLASSIFIER_TYPE': classifierMatcher,
      'service': appServiceKw,
      'SERVICE_NAME': serviceNamesMatcher,
      'category': appCategoryKw,
      'CATEGORY_NAME': categoryNameMatcher,
   }

class ShowAppTrafficRecCmd( ShowCommand.ShowCliCommandClass ):
   syntax = ( '''show application traffic recognition [ FILTER ] ''' )
   data = {
      'application': applicationKw,
      'traffic': trafficKw,
      'recognition': recognitionKw,
      'FILTER': ShowAppTraficRecFilterExp
   }

   cliModel = ClassificationModel.ShowApplications

   @staticmethod
   def handler( mode, args ):
      applicationStatus = classificationStatus.applicationStatus
      applications = {}
      appName = args.get( 'APP', None )
      categoryName = args.get( 'CATEGORY_NAME', None )
      serviceName = args.get( 'SERVICE_NAME', None )

      def getServices( application, serviceCategory=None ):
         services = {}
         serviceCatories = application.serviceCategory

         for name, category in serviceCatories.items():
            if ( serviceCategory and category != serviceCategory ) or \
                    ( serviceName and name != serviceName ):
               continue

            services[ name ] = ClassificationModel.ServiceModel()
            services[ name ].category = category

         return services

      def populateApplication( applicationName, application ):
         classifier = None
         if application.dpiApp and application.customApp:
            classifier = 'custom-dpi'
         elif application.dpiApp:
            classifier = 'dpi'
         else:
            classifier = 'ipv4'

         inClassifier = args.get( 'CLASSIFIER_TYPE', None )
         if inClassifier and classifier != inClassifier:
            return

         services = getServices( application, categoryName )
         if services:
            applications[ applicationName ] = ClassificationModel.ApplicationModel()
            applications[ applicationName ].classifier = \
               ClassificationModel.classifierNameToModel[ classifier ]
            applications[ applicationName ].services = services

      if appName:
         if appName in applicationStatus:
            populateApplication( appName, applicationStatus[ appName ] )
      else:
         for applicationName, application in applicationStatus.items():
            populateApplication( applicationName, application )

      applicationModel = ClassificationModel.ShowApplications()
      applicationModel.applications = applications
      return applicationModel

BasicCli.GlobalConfigMode.addCommandClass( AppTrafficRecConfigModeCmd )
AppTrafficRecConfigMode.addModelet( CommitAbortModelet )

AppTrafficRecConfigMode.addCommandClass( FieldSetL4PortConfigCmd )
FieldSetL4PortConfigMode.addModelet( CommitAbortModelet )
FieldSetL4PortConfigMode.addCommandClass( FieldSetL4PortConfigCmds )
FieldSetL4PortConfigMode.addCommandClass( FieldSetL4PortExceptConfigCmds )

AppTrafficRecConfigMode.addCommandClass( FieldSetIpPrefixConfigCmd )
FieldSetIpPrefixConfigMode.addModelet( CommitAbortModelet )
FieldSetIpPrefixConfigMode.addCommandClass( FieldSetIpPrefixConfigCmds )

AppTrafficRecConfigMode.addCommandClass( AppProfileConfigCmd )
AppProfileConfigMode.addModelet( CommitAbortModelet )
AppProfileConfigMode.addCommandClass( AppProfileConfigCmds )
AppProfileConfigMode.addCommandClass( AppProfileCategoryConfigCmds )
AppProfileConfigMode.addCommandClass( AppProfileAppTransportConfigCmds )

AppTrafficRecConfigMode.addCommandClass( AppConfigCmdIpv4 )
AppConfigModeIpv4.addModelet( CommitAbortModelet )
AppConfigModeIpv4.addCommandClass( AppConfigFieldSetCmd )
AppConfigModeIpv4.addCommandClass( ProtocolListIPv4ConfigCmd )
AppConfigModeIpv4.addCommandClass( ProtocolFieldSetConfigCmd )
if toggleAppDscpMatchEnabled():
   AppConfigModeIpv4.addCommandClass( AppConfigDscpCmd )

AppTrafficRecConfigMode.addCommandClass( AppConfigCmdL4 )
AppConfigModeL4.addModelet( CommitAbortModelet )
AppConfigModeL4.addCommandClass( ProtocolListIPv4ConfigCmd )
AppConfigModeL4.addCommandClass( ProtocolFieldSetConfigCmd )

AppTrafficRecConfigMode.addCommandClass( CategoryConfigCmd )
CategoryConfigMode.addModelet( CommitAbortModelet )
CategoryConfigMode.addCommandClass( CategoryConfigCmds )
BasicCli.addShowCommandClass( ShowAppTrafficRecCmd )

def Plugin( entityManager ):
   global appRecognitionConfig, fieldSetConfig
   global classificationStatus, entityMib

   appRecognitionConfig = ConfigMount.mount( entityManager,
                                             'classification/app-recognition/config',
                                             'Classification::AppRecognitionConfig',
                                             'w' )
   fieldSetConfig = ConfigMount.mount( entityManager,
                                       'classification/app-recognition/fieldset',
                                       'Classification::FieldSetConfig', 'w' )

   classificationStatus = LazyMount.mount( entityManager,
                                           'classification/app-recognition/status',
                                           'Classification::Status',
                                           'r' )
   entityMib = LazyMount.mount( entityManager, 'hardware/entmib',
                                 'EntityMib::Status', 'r' )
