#!/usr/bin/env python3
# Copyright (c) 2023 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import BasicCli
import CliCommand
import CliGlobal
import CliMatcher
from CliPlugin.ClbCli import clbShowMatcher
from CliPlugin.EthIntfCli import EthPhyAutoIntfType
from CliPlugin.IpGenAddrMatcher import IpGenAddrMatcher
from CliPlugin.LagIntfCli import LagAutoIntfType
from CliPlugin.MacAddr import MacAddrMatcher
from CliPlugin.SwitchIntfCli import SwitchAutoIntfType
from CliPlugin.TechSupportCli import registerShowTechSupportCmd
from CliPlugin.VrfCli import vrfMatcher
from CliToken.LoadBalance import loadBalanceMatcherForShow
import Intf
import LazyMount
import ShowCommand

queuePairMatcher = CliMatcher.IntegerMatcher(
   0, 0xffffff, helpdesc='Display flows by RoCE queue pair' )

flowMatcher = CliMatcher.KeywordMatcher(
   'flows', helpdesc='Display flow information' )

groupMatcher = CliMatcher.KeywordMatcher(
   'group', helpdesc='Display flows group information' )

gv = CliGlobal.CliGlobal( requirements=None )

srcAddrKwMatcher = CliMatcher.KeywordMatcher(
   'ip', helpdesc='Display flows by IP source address' )
srcAddrMatcher = IpGenAddrMatcher( 'Display flows by IP source address' )
dstAddrKwMatcher = CliMatcher.KeywordMatcher(
   'ip', helpdesc='Display flows by IP destination address' )
dstAddrMatcher = IpGenAddrMatcher( 'Display flows by IP destination address' )

intfTypes = ( EthPhyAutoIntfType, LagAutoIntfType, SwitchAutoIntfType )
srcIntfKwMatcher = CliMatcher.KeywordMatcher(
   'interface', helpdesc='Display flows by source interface range' )
srcIntfRangeMatcher = Intf.IntfRange.IntfRangeMatcher(
   explicitIntfTypes=intfTypes )
dstIntfKwMatcher = CliMatcher.KeywordMatcher(
   'interface', helpdesc='Display flows by destination interface range' )
dstIntfRangeMatcher = Intf.IntfRange.IntfRangeMatcher(
   explicitIntfTypes=intfTypes )

vlanIdMatcher = CliMatcher.IntegerMatcher(
   1, 4095, helpdesc='Identifier for a Virtual LAN' )

class ShowClbFlowCmd( ShowCommand.ShowCliCommandClass ):
   syntax = '''show load-balance cluster flows [ counters ] [ VRF_KW VRF ]
               [ source [ INTF_KW_SRC SRC_INTF ] [ ADDR_KW_SRC SRC_IP ] ]
               [ destination [ ADDR_KW_DST DST_IP ] [ mac DST_MAC ]
                             [ INTF_KW_DST DST_INTF ] ]
               [ queue-pair QP ]
               [ vlan VLAN_ID ]
               [ detail ]'''
   data = {
      'load-balance': loadBalanceMatcherForShow,
      'cluster': clbShowMatcher,
      'flows': flowMatcher,
      'counters': 'Display counters instead of other flow information',
      'VRF_KW': CliCommand.Node(
         matcher=CliMatcher.KeywordMatcher( 'vrf', 'Display flows in a VRF' ) ),
      'VRF': vrfMatcher,
      'vlan': 'Display flows by VLAN ID',
      'VLAN_ID': vlanIdMatcher,
      'source': 'Display flows by source information',
      'INTF_KW_SRC': srcIntfKwMatcher,
      'SRC_INTF': srcIntfRangeMatcher,
      'ADDR_KW_SRC': srcAddrKwMatcher,
      'SRC_IP': srcAddrMatcher,
      'destination': 'Display flows by destination information',
      'ADDR_KW_DST': dstAddrKwMatcher,
      'DST_IP': dstAddrMatcher,
      'INTF_KW_DST': dstIntfKwMatcher,
      'DST_INTF': dstIntfRangeMatcher,
      'queue-pair': 'Display flows by RoCE queue pair',
      'QP': queuePairMatcher,
      'mac': 'Display flows by MAC address',
      'DST_MAC': MacAddrMatcher(),
      'detail': 'Include more detailed information in output',
   }

   cliModel = "ClbFlowModel.ShowFlowModel"

   handler = "ClbShowFlowHandlers.showClbFlow"

BasicCli.addShowCommandClass( ShowClbFlowCmd )

class ShowClbLearnedFlowCmd( ShowCommand.ShowCliCommandClass ):
   syntax = '''show load-balance cluster learned flows
               [ source [ INTF_KW_SRC SRC_INTF ] [ ADDR_KW_SRC SRC_IP ] ]
               [ destination [ ADDR_KW_DST DST_IP ] [ mac DST_MAC ] ]
               [ queue-pair QP ]
               [ vlan VLAN_ID ]
               [ detail ]'''
   data = {
      'load-balance': loadBalanceMatcherForShow,
      'cluster': clbShowMatcher,
      'learned': 'Display learned flow information',
      'flows': 'Display flow information',
      'vlan': 'Display flows by VLAN ID',
      'VLAN_ID': vlanIdMatcher,
      'source': 'Display flows by source information',
      'INTF_KW_SRC': srcIntfKwMatcher,
      'SRC_INTF': srcIntfRangeMatcher,
      'ADDR_KW_SRC': srcAddrKwMatcher,
      'SRC_IP': srcAddrMatcher,
      'destination': 'Display flows by destination information',
      'ADDR_KW_DST': dstAddrKwMatcher,
      'INTF_KW_DST': dstIntfKwMatcher,
      'DST_IP': dstAddrMatcher,
      'queue-pair': 'Display flows by RoCE queue pair',
      'QP': queuePairMatcher,
      'mac': 'Display flows by MAC address',
      'DST_MAC': MacAddrMatcher(),
      'detail': 'Include more detailed information in output',
   }

   cliModel = "ClbFlowModel.ShowLearnedFlowModel"

   handler = "ClbShowFlowHandlers.showClbLearnedFlow"

BasicCli.addShowCommandClass( ShowClbLearnedFlowCmd )

# show load-balance cluster stats
class ShowClbStatsCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show load-balance cluster statistics'

   data = {
      'load-balance': loadBalanceMatcherForShow,
      'cluster': clbShowMatcher,
      'statistics': 'Display statistics for feature',
   }

   cliModel = "ClbFlowModel.ShowStatsModel"

   handler = "ClbShowFlowHandlers.showClbStats"

BasicCli.addShowCommandClass( ShowClbStatsCmd )

# show load-balance cluster destination
class ShowClbDestCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show load-balance cluster destination [ detail ]'

   data = {
      'load-balance': loadBalanceMatcherForShow,
      'cluster': clbShowMatcher,
      'destination': 'Show information about CLB destinations',
      'detail': 'Include detailed information about CLB destinations',
   }

   cliModel = "ClbFlowModel.ShowClbDestModel"

   handler = "ClbShowFlowHandlers.showClbDest"

BasicCli.addShowCommandClass( ShowClbDestCmd )

class ShowClbComGroupCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show load-balance cluster flows group communicator [ summary ]'

   data = {
      'load-balance': loadBalanceMatcherForShow,
      'cluster': clbShowMatcher,
      'flows': flowMatcher,
      'group': groupMatcher,
      'communicator': 'Show communicator group info',
      'summary': 'Show communicator group summary information'
   }

   cliModel = "ClbFlowModel.ShowClbComGroupModel"

   handler = "ClbShowFlowHandlers.showClbComGroup"

BasicCli.addShowCommandClass( ShowClbComGroupCmd )

class ShowClbRouteGroupCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show load-balance cluster flows group route [ summary ]'

   data = {
      'load-balance': loadBalanceMatcherForShow,
      'cluster': clbShowMatcher,
      'flows': flowMatcher,
      'group': groupMatcher,
      'route': 'Show route group info',
      'summary': 'Show route group summary information'
   }

   cliModel = "ClbFlowModel.ShowClbRouteGroupModel"

   handler = "ClbShowFlowHandlers.showClbRouteGroup"

BasicCli.addShowCommandClass( ShowClbRouteGroupCmd )

class ShowClbWarningsCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show load-balance cluster warnings'
   data = {
      'load-balance': loadBalanceMatcherForShow,
      'cluster': clbShowMatcher,
      'warnings': 'Display configuration warnings.'
   }

   cliModel = "ClbFlowModel.ShowClbWarningsModel"
   handler = "ClbShowFlowHandlers.showClbWarnings"

BasicCli.addShowCommandClass( ShowClbWarningsCmd )

class ShowClbStatusCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show load-balance cluster status'
   data = {
      'load-balance': loadBalanceMatcherForShow,
      'cluster': clbShowMatcher,
      'status': 'Display high level feature status.'
   }

   cliModel = "ClbFlowModel.ShowClbStatusModel"
   handler = "ClbShowFlowHandlers.showClbStatus"

BasicCli.addShowCommandClass( ShowClbStatusCmd )

def clbShowTechGuard():
   return gv.requirements.loadBalanceMethod != 'methodNone'

def Plugin( entityManager ):
   gv.requirements = LazyMount.mount( entityManager, 'clb/hardware/requirements',
                                      'Clb::Hardware::Requirements', 'r' )

   registerShowTechSupportCmd( '2024-04-01 11:30:00',
                               [ 'show load-balance cluster warnings',
                                 'show load-balance cluster statistics',
                                 'show load-balance cluster status',
                                 'show load-balance cluster flows',
                                 'show load-balance cluster learned flows',
                                 "show load-balance cluster destination detail" ],
                               cmdsGuard=clbShowTechGuard )
