# Copyright (c) 2006-2010 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import BasicCli
import BasicCliModes
import CliCommand
import CliMatcher
import CliToken.Clear
import CliToken.Cli
import ConfigMount
import ShowCommand

import Tac

cliConfig = None

#------------------------------------------------------------------------------------
# show cli pid
#------------------------------------------------------------------------------------
class ShowCliPid( ShowCommand.ShowCliCommandClass ):
   syntax = 'show cli pid'
   data = {
      'cli': 'Show CLI commands',
      'pid': 'PID of the current CLI'
   }
   cliModel = "CliCliModel.CliPid"
   handler = "CliCli.showCliPid"

BasicCli.addShowCommandClass( ShowCliPid )

#------------------------------------------------------------------------------------
# [ no | default ] prompt PROMPT
#------------------------------------------------------------------------------------
class PromptCmd( CliCommand.CliCommandClass ):
   syntax = 'prompt PROMPT'
   noOrDefaultSyntax = 'prompt ...'
   data = {
      'prompt': 'Configure the CLI prompt',
      'PROMPT': CliMatcher.PatternMatcher( '.+', helpname='WORD',
         helpdesc='Prompt format string' )
   }

   handler = "CliCli.promptCmd"
   noOrDefaultHandler = handler

BasicCli.GlobalConfigMode.addCommandClass( PromptCmd )

#-----------------------------------------------------------------------------------
# show history [ current ]
#-----------------------------------------------------------------------------------
class ShowHistoryCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show history [ current ]'
   data = {
      'history': 'CLI shell history of entered commands',
      'current': CliCommand.Node(
         matcher=CliMatcher.KeywordMatcher( 'current',
            helpdesc='Display history for the current mode' ),
         hidden=True ),
   }
   cliModel = "CliCliModel.ShowHistory"
   handler = "CliCli.showHistoryCmd"

BasicCli.addShowCommandClass( ShowHistoryCmd )

#------------------------------------------------------------------------------------
# clear history
#------------------------------------------------------------------------------------
class ClearHistory( CliCommand.CliCommandClass ):
   syntax = 'clear history'
   data = {
      'clear': CliToken.Clear.clearKwNode,
      'history': 'Clear the session command history',
   }

   handler = "CliCli.clearHistory"

BasicCli.ExecMode.addCommandClass( ClearHistory )

#-----------------------------------------------------------------------------------
# show privilege
#-----------------------------------------------------------------------------------
class ShowPrivilege( ShowCommand.ShowCliCommandClass ):
   syntax = 'show privilege'
   data = {
      'privilege': 'Display the current privilege level',
   }
   cliModel = "CliCliModel.PrivilegeLevel"
   handler = "CliCli.showPrivilege"

BasicCli.addShowCommandClass( ShowPrivilege )

#------------------------------------------------------------------------------------
# [ no | default ] activity-lock-monitor
# In ConfigAgent there is an ActivityLock monitor. If this command is enabled
# it will log a syslog if the ActivityLock is held for too long in ConfigAgent
#------------------------------------------------------------------------------------
class ActivityLockMonitor( CliCommand.CliCommandClass ):
   syntax = 'activity-lock-monitor'
   noOrDefaultSyntax = syntax
   data = {
      'activity-lock-monitor': 'Enable the activity lock monitor syslog message'
   }
   hidden = True

   handler = "CliCli.activityLockMonitor"
   noOrDefaultHandler = "CliCli.activityLockMonitorNot"

BasicCli.GlobalConfigMode.addCommandClass( ActivityLockMonitor )

cliDebugNodeHidden = CliCommand.Node( CliMatcher.KeywordMatcher( 'debug',
                                          helpdesc='Cli debug commands' ),
                                      hidden=True )

#------------------------------------------------------------------------------------
# cli debug parser result
#------------------------------------------------------------------------------------
class CliDebugParserResultCmd( CliCommand.CliCommandClass ):
   syntax = 'cli debug parser result'
   noOrDefaultSyntax = syntax
   data = {
      'cli': CliToken.Cli.cliForExecMatcher,
      'debug': cliDebugNodeHidden,
      'parser': 'Parser Debugs',
      'result': 'Parser Results displayed',
   }

   handler = "CliCli.cliDebugParserResultCmd"
   noOrDefaultHandler = "CliCli.cliDebugParserResultCmdNot"

BasicCli.EnableMode.addCommandClass( CliDebugParserResultCmd )

#------------------------------------------------------------------------------------
# cli debug parser guard
#------------------------------------------------------------------------------------
class CliDebugParserGuardCmd( CliCommand.CliCommandClass ):
   syntax = 'cli debug parser guard'
   noSyntax = syntax
   data = {
      'cli': CliToken.Cli.cliForExecMatcher,
      'debug': cliDebugNodeHidden,
      'parser': 'Parser Debugs',
      'guard': 'Disable parser guards',
   }

   handler = "CliCli.cliDebugParserGuardCmd"
   noHandler = "CliCli.cliDebugParserGuardCmdNot"

BasicCli.EnableMode.addCommandClass( CliDebugParserGuardCmd )

#------------------------------------------------------------------------------------
# cli debug memory handler { heapcheck | mallinfo | rss | gc | log }
#------------------------------------------------------------------------------------
class CliDebugMemoryHandlerCmd( CliCommand.CliCommandClass ):
   noOrDefaultSyntax = 'cli debug memory handler ...'
   syntax = 'cli debug memory handler [ OPTIONS ]'
   data = {
      'cli': CliToken.Cli.cliForExecMatcher,
      'debug': cliDebugNodeHidden,
      'memory': 'Analyze memory usage',
      'handler': 'Analyze memory usage during cli command handler execution',
      'OPTIONS': CliCommand.setCliExpression( {
         'heapcheck': ( 'Run libtcmalloc\'s heapcheck, which needs to be '
            '\'preloaded\' (LD_PRELOAD=/usr/lib/libtcmalloc.so HEAPCHECK=local), '
            'which will invalidate the mallinfo results '
            '(returns free=allocated=0)' ),
         'mallinfo': 'Display stats on free/available bytes using mallinfo',
         'rss': 'Display number of rss pages (4k) used by ConfigAgent',
         'py-objects': ( 'Display the count of python objects known to the '
                         'garbage collector' ),
         'gc': ( 'Invoke garbage collector before protocolling mem usage (will be '
                 'slow)' ),
         'log': 'Don\'t display the memory stats in the cli prompt (instead log it)',
      } )
   }

   handler = "CliCli.cliDebugMemoryHandlerCmd"

   noOrDefaultHandler = handler

BasicCli.EnableMode.addCommandClass( CliDebugMemoryHandlerCmd )

#-----------------------------------------------------------------------------------
# run [ ignore-errors ] COMMAND
#-----------------------------------------------------------------------------------
class runCommands( CliCommand.CliCommandClass ):
   syntax = 'run [ ignore-errors ] COMMAND'
   data = {
            'run': 'Run multiple commands in one line',
            'ignore-errors': 'Do not stop on errors',
            'COMMAND': CliMatcher.StringMatcher(
               helpname='COMMAND',
               helpdesc="Commands separated by ';'" ),
          }
   handler = "CliCli.runCommands"

BasicCli.ExecMode.addCommandClass( runCommands )

#-------------------------------------------------------------------------------
# The "cli update-handler <mode> <cmd> <module> <handler>" command
# This command allows to change the code of a cli command at runtime (no restart).
# We assume the <module> with the fixed <handler> is in dir CliHandlerPatches.
#-------------------------------------------------------------------------------
modeMatcher = CliMatcher.PatternMatcher( '[a-zA-Z.0-9_]+',
   helpname='MODE NAME',
   helpdesc='The name of the mode where the command to fix is, '
            'as in "EnableMode"' )
cmdMatcher = CliMatcher.QuotedStringMatcher( pattern='[-a-zA-Z.0-9_]+',
   helpdesc='An instance of the cli command to update, as in "mtu 1234"' )
fileMatcher = CliMatcher.PatternMatcher( '[a-zA-Z.0-9_]+',
   helpname='FILE NAME',
   helpdesc='The filename in CliHandlerPatches dir '
            'where the new handler code is' )
functionMatcher = CliMatcher.PatternMatcher( '[a-zA-Z.0-9_]+',
   helpname='FUNCTION NAME',
   helpdesc='Name of the handler function (in just specified file);'
            'use class.function if the handler is inside a class' )

class UpdateHandlerFunction( CliCommand.CliCommandClass ):
   syntax = 'cli update-handler MODE CMD MODULE HANDLER'
   noOrDefaultSyntax = syntax

   data = { "cli": CliToken.Cli.cliForExecMatcher,
            "update-handler": "Runtime update the handler code of a cli command",
            "MODE": modeMatcher,
            "CMD": cmdMatcher,
            "MODULE": fileMatcher,
            "HANDLER": functionMatcher,
          }

   handler = "CliCli.updateHandlerFunction"
   noOrDefaultHandler = handler

BasicCli.EnableMode.addCommandClass( UpdateHandlerFunction )

#------------------------------------------------------------------------------------
# Plugin Func
#------------------------------------------------------------------------------------
def Plugin( entityManager ):
   global cliConfig

   cliConfig = ConfigMount.mount( entityManager, 'cli/config', 'Cli::Config', 'w' )
   BasicCliModes.cliConfig_ = cliConfig
