#!/usr/bin/env python3
# Copyright (c) 2023 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import io
import os
import sys

import BasicCli
import CliCommand
import CliFragment
import CliMatcher
import CliModel
from CliPlugin import CliFragmentModel
from CliSave import saveSessionConfig
import CliSession as CS
import CliToken
import CommonGuards
import FileCliUtil
from FileUrlDefs import MANAGED_CONFIG_DIR
import LazyMount
import ShowRunOutputModel
import Tac
import Tracing

from UrlPlugin.StartupConfigUrl import StartupConfigUrl

t0 = Tracing.trace0

fragmentConfig = None
fragmentStatus = None

fragmentKwForConfig = CliMatcher.KeywordMatcher( 'fragment',
      helpdesc='Enter fragment session; committed only on config fragment commit' )
fragmentNameMatcher = CliMatcher.PatternMatcher(
      r'[-_a-zA-Z0-9]+',
      helpname='WORD',
      helpdesc='Name for fragment' )

fragmentNamesMatcher = CliMatcher.DynamicNameMatcher(
   lambda _mode: CliFragment.cliFragmentNames( None ),
   helpdesc='Name of fragment(s) to commit',
   pattern=r'[-_a-zA-Z0-9]+',
   helpname='FRAGMENT' )

def waitForFragmentConfigs( mode ):
   # Ignore this wait while loading fragment configs (not relevant during
   # loading of startup-config itself), but the --startup-config flag is
   # shared by both loads)
   if ( mode.session_.startupConfig() or
        # If we're not in the middle of loading fragment configs, and the
        # startup-config hasn't been attempted to load, then we shouldn't
        # wait for fragment configs to be loaded, either (the fragment config
        # files only describe what's in the running-config, so if we skipped
        # startup-config then the fragment config files are not relevant,
        # either.
        ( CliFragment.sysdbStatus and
          CliFragment.sysdbStatus.startupConfigStatus == "notLoaded" ) ):
      return True

   def isReady():
      state = None
      if CliFragment.fragmentStatus:
         state = CliFragment.fragmentStatus.fragmentConfigStatus
      return state in ( "completed", "incomplete" )
   Tac.waitFor( isReady, description="fragment config files to be loaded" )

# This is a temporary placeholder to allow FragmentCopyHookTest to run
def prepareFragmentSession( mode, args ):
   fragmentName = args.get( '<fragmentName>' )
   copy = 'update' in args # Either 'update' or 'replace'
   waitForFragmentConfigs( mode )
   t0( "Opening fragment session for fragment", fragmentName )
   cfg = fragmentConfig.cliFragmentDir.config.get( fragmentName )
   response = CliFragment.prepareFragmentSession( fragmentName, copy, mode=mode )
   if response:
      mode.addError( response )
      # if fragment did not exist before this cmd; restore previous state
      if not cfg:
         # ignore result of deleteFragments
         CliFragment.deleteFragments( [ fragmentName ], mode=mode )
      return

def loadFragmentSession( mode, args ):
   fragmentName = args.get( '<fragmentName>' )
   surl = args.get( 'SRC_URL' )
   waitForFragmentConfigs( mode )
   response = CliFragment.loadFragmentSession( mode, fragmentName, surl )
   if response:
      mode.addError( response )
      return

def verifyPendingSessions( operation, mode, fragmentNames, warn=False ):
   cfgDir = fragmentConfig.cliFragmentDir.config
   for fname in fragmentNames:
      cfg = cfgDir.get( fname )
      if not ( cfg and cfg.pendingSession ):
         msg = f"Fragment {fname} "
         if cfg:
            msg += f"has no pending session to {operation}."
         else:
            msg += "does not exist."
         if warn:
            mode.addWarning( msg )
         else:
            mode.addError( msg )
            return msg
   return ""

def commitFragments( mode, args ):
   fragmentNames = args.get( 'FRAGMENT' )
   waitForFragmentConfigs( mode )
   # Used to be either 'merge' or 'replace', now always true.
   replace = 'replace' in args
   error = verifyPendingSessions( "commit", mode, fragmentNames )
   if error:
      return error
   error = CliFragment.commitFragments( fragmentNames, mode, replace=replace )
   if error:
      mode.addError( error )
   return error
   # At the moment, CliFragment.commitFragments copies running-config to
   # startup-config, triggering the copy-hook at the bottom of this file.
   # If we change that behavior, then we should call FileCliUtil.copyFile()
   # here, as follows, so all the hooks get run, also, and we go through
   # the normal path::
   # runConfig = FileUrl.localRunningConfig( *Url.urlArgsFromMode( mode ) )
   # startConfig = FileUrl.localStartupConfig( *Url.urlArgsFromMode( mode ) )
   # return FileCliUtil.copyFile( None, mode, runConfig, startConfig,
   #                              commandSource="configure replace" )

def abortFragments( mode, args ):
   fragmentNames = args.get( 'FRAGMENT' )
   t0( "Aborting fragments in the process of being committed", fragmentNames )
   verifyPendingSessions( "abort", mode, fragmentNames, warn=True )
   response = CliFragment.abortFragments( fragmentNames, mode )
   # if a fragment has neither a pending session nor a committed session, then
   # it was aborted on its initial setup. Delete it. Regardless, it would
   # require a new `prepare` to do anything more with it, which will (re)create
   # it if needed.
   for frag in fragmentNames:
      cfg = fragmentConfig.cliFragmentDir.config.get( frag )
      stat = fragmentStatus.cliFragmentDir.status.get( frag )
      # if it exists, but is completely empty (never committed, and
      # has no pending)
      if cfg and ( not stat ) or stat.committedVersion == 0:
         # Then aborting it should delete it
         CliFragment.deleteFragments( [ frag ], mode=mode )
   return response

def deleteFragment( mode, args ):
   fragmentNames = args.get( 'FRAGMENT' )
   t0( "Deleting fragment", fragmentNames )
   response = CliFragment.deleteFragments( fragmentNames, mode=mode )
   if response:
      mode.addError( response )
      return

def showFragmentConfigs( mode, args ):
   fragments = {}
   cfgDir = fragmentConfig.cliFragmentDir.config
   statDir = fragmentStatus.cliFragmentDir.status
   hasDescription = False
   for fname in CliFragment.cliFragmentNames( None ):
      cfg = cfgDir.get( fname )
      if not cfg:
         mode.addError(
            f"Internal error: fragment {fname} listed, but does not exist" )
         return None
      stat = statDir.get( fname )
      model = CliFragmentModel.Fragment()
      if cfg.pendingVersion:
         model.pendingSession = cfg.pendingSessionName
      if stat and stat.committedVersion:
         model.committedSession = stat.committedSessionName
      if cfg.description:
         hasDescription = True
         model.description = cfg.description
      if cfg.commitGroup:
         model.commitGroupId = cfg.commitGroup
      fragments[ fname ] = model
   return CliFragmentModel.Fragments(
      fragmentConfigStatus=fragmentStatus.fragmentConfigStatus,
      fragments=fragments,
      _hasDescription=hasDescription )

def showFragmentConfigCombinedContents( mode, args ):
   em = mode.entityManager
   sessionName = ""
   pending = 'pending' in args
   # Any fragment not in fragmentNames will include its committed
   # session, if it has one.
   fragmentNames = []
   if pending:
      for fname in CliFragment.cliFragmentNames( em ):
         fcfg = fragmentConfig.cliFragmentDir.config.get( fname )
         if fcfg.pendingVersion > 0:
            fragmentNames.append( fname )
   try:
      sessionName = CS.uniqueSessionName( entityManager=em,
                                          prefix="combinedFragments" )
      response = CS.enterSession( sessionName,
                                  entityManager=em,
                                  transient=True,
                                  noCommit=True )
      if response:
         mode.addError( response )
         return CliModel.noValidationModel( ShowRunOutputModel.Mode )
      response = CS.rollbackSession( em, sessionName=sessionName )
      if response:
         mode.addError( response )
         return CliModel.noValidationModel( ShowRunOutputModel.Mode )
      CliFragment.copyFragmentContentsIntoSession( fragmentNames,
                                          sessionName=sessionName,
                                          mode=mode )
      saveSessionConfig( em,
                         sys.stdout,
                         sessionName,
                         cleanConfig=False,
                         showJson=not mode.session_.shouldPrint() )
      # the config should be able to be JSON and we return the model type.
      # the result itself would have been streamed out
      return CliModel.noValidationModel( ShowRunOutputModel.Mode )
   finally:
      if sessionName and sessionName == CS.currentSession( em ):
         CS.exitSession( em )

def showFragmentConfigContents( mode, args ):
   fragModel = CliModel.noValidationModel( CliFragmentModel.ContentsOfFragments )
   fragments = []
   fragName = args.get( 'FRAGMENT' )
   if fragName:
      fragments = [ fragName ]
   else:
      fragments = CliFragment.cliFragmentNames( None )
   pending = 'pending' in args
   cfgDir = fragmentConfig.cliFragmentDir.config
   statDir = fragmentStatus.cliFragmentDir.status
   contents = {}
   for fname in fragments:
      cfg = cfgDir.get( fname )
      if not cfg:
         internal = ""
         if not fragName:
            internal = "Internal error: "
         mode.addError(
            f"{internal}fragment {fname} listed, but does not exist" )
         return fragModel
      stat = statDir.get( fname )
      if pending and cfg.pendingVersion:
         sessionName = cfg.pendingSessionName
      else:
         if not ( stat and stat.committedVersion ):
            if fragName:
               mode.addError(
                  f"fragment {fname} has no associated session" )
               return fragModel
            else:
               continue
         sessionName = stat.committedSessionName
      contents[ fname ] = sessionName
   if not mode.session_.shouldPrint():
      vers = "pending" if pending else "committed"
      print( f'{{ "versionName": "{vers}", "contents": {{', end='' )
   for frag, config in contents.items():
      cfg = cfgDir.get( frag )
      isPending = ( pending and
                    bool( cfg.pendingVersion ) and
                    config == cfg.pendingSessionName )
      firstOne = True
      if mode.session_.shouldPrint():
         vers = "pending" if isPending else "committed"
         print( "================" )
         print( f"{vers} session-config for {frag}:" )
         print( "================" )
      else:
         if not firstOne:
            print( "," )
         vers = "true" if isPending else "false"
         print( f'"{frag}": {{ "pending": {vers}, "config": ', end='' )
         firstOne = False
      # config.render() if not firstOne:
      saveSessionConfig( mode.entityManager,
                         sys.stdout,
                         config,
                         cleanConfig=False,
                         showJson=not mode.session_.shouldPrint() )
      if mode.session_.shouldPrint():
         print( "" )
      else:
         print( "}", end='' )

   if not mode.session_.shouldPrint():
      print( '} }' )
   return fragModel

###################
# Cli syntax:
###################

class PrepareFragmentSession( CliCommand.CliCommandClass ):
   syntax = 'configure fragment prepare <fragmentName> ( update | replace )'
   data = { 'configure': CliToken.Configure.configureParseNode,
            'fragment': CliCommand.Node( matcher=fragmentKwForConfig,
                                         # Fragment commands are not yet meant
                                         # to be enabled in general. Only
                                         # used for testing, so far.
                                         hidden=True ),
            'prepare': 'prepare pending session for fragment',
            # For now, only allow fragment cmds on active supervisor
            '<fragmentName>': CliCommand.Node(
               matcher=fragmentNameMatcher,
               guard=CommonGuards.ssoStandbyGuard ),
            'update': 'copy current committed version into session',
            'replace': 'pending session starts as clean-config'
          }
   handler = prepareFragmentSession

BasicCli.EnableMode.addCommandClass( PrepareFragmentSession )
BasicCli.GlobalConfigMode.addCommandClass( PrepareFragmentSession )

class LoadFragmentSession( CliCommand.CliCommandClass ):
   syntax = 'configure fragment copy SRC_URL <fragmentName>'
   data = {
            'configure': CliToken.Configure.configureParseNode,
            'fragment': CliCommand.Node( matcher=fragmentKwForConfig,
                                         # Fragment commands are not yet meant
                                         # to be enabled in general. Only
                                         # used for testing, so far.
                                         hidden=True ),
            'copy': 'apply config from URL into pending fragment session',
            'SRC_URL': FileCliUtil.copySourceUrlExpr(),
            # For now, only allow fragment cmds on active supervisor
            '<fragmentName>': CliCommand.Node(
               matcher=fragmentNameMatcher,
               guard=CommonGuards.ssoStandbyGuard ),
          }
   handler = loadFragmentSession

BasicCli.EnableMode.addCommandClass( LoadFragmentSession )
BasicCli.GlobalConfigMode.addCommandClass( LoadFragmentSession )

class CommitFragments( CliCommand.CliCommandClass ):
   syntax = 'configure fragment commit replace {FRAGMENT}'
   data = {
      'configure': CliToken.Configure.configureParseNode,
      'fragment': CliCommand.Node( matcher=fragmentKwForConfig, hidden=True ),
      'commit': 'Commit specified fragments',
      # require 'replace' even though there's no other choice any more.
      'replace': 'replace the running-config',
      'FRAGMENT': CliCommand.Node(
         matcher=fragmentNamesMatcher,
         guard=CommonGuards.ssoStandbyGuard ),
   }
   handler = commitFragments

BasicCli.EnableMode.addCommandClass( CommitFragments )

class AbortFragments( CliCommand.CliCommandClass ):
   syntax = 'configure fragment abort {FRAGMENT}'
   data = {
      'configure': CliToken.Configure.configureParseNode,
      'fragment': CliCommand.Node( matcher=fragmentKwForConfig, hidden=True ),
      'abort': 'Abort specified pending fragments',
      'FRAGMENT': CliCommand.Node(
         matcher=fragmentNamesMatcher,
         guard=CommonGuards.ssoStandbyGuard ),
   }
   handler = abortFragments

BasicCli.EnableMode.addCommandClass( AbortFragments )

class DeleteFragment( CliCommand.CliCommandClass ):
   syntax = 'configure fragment delete {FRAGMENT}'
   data = {
      'configure': CliToken.Configure.configureParseNode,
      'fragment': CliCommand.Node( matcher=fragmentKwForConfig, hidden=True ),
      'delete': 'Delete specified pending fragments',
      # For now, only allow fragment cmds on active supervisor
      'FRAGMENT': CliCommand.Node(
         matcher=fragmentNamesMatcher,
         guard=CommonGuards.ssoStandbyGuard ),
   }
   handler = deleteFragment

BasicCli.EnableMode.addCommandClass( DeleteFragment )

def fragmentConfigCopyHook( mode, _surl, _durl, commandSource, targetDir ):
   for fragName, fragStat in fragmentStatus.cliFragmentDir.status.items():
      if not fragStat.committedVersion:
         continue
      sessionName = fragStat.committedSessionName

      # We will write the session config to the local file, under targetDir,
      # not to the managed-config directory on flash:
      fragmentFilePath = os.path.join( targetDir, f"{fragName}.fragment" )

      # Copied the strategy from ConfigSession/UrlPlugin/SessionUrl.py
      # Store the config in a temporary bytes seq.
      dstData = io.BytesIO()
      wrapperFunc = io.TextIOWrapper( dstData, write_through=True )
      saveSessionConfig( mode.entityManager,
                         wrapperFunc,
                         sessionName,
                         cleanConfig=False,
                         saveAll=False,
                         saveAllDetail=False,
                         showNoSeqNum=False,
                         showSanitized=False,
                         showJson=False )
      configContents = dstData.getvalue()
      dstData.close()
      error = ""
      fragFile = f"flash:/{MANAGED_CONFIG_DIR}/{fragName}.fragment"
      try:
         with open( fragmentFilePath, 'wb' ) as f:
            f.write( configContents )
      except OSError as e:
         error = e.strerror or f"Unknown OS error writing {fragFile}"
      if not error:
         if not os.path.exists( fragmentFilePath ):
            error = f"Did not successfully save {fragFile}"
         elif os.path.getsize( fragmentFilePath ) == 0:
            error = f"Wrote empty file to {fragFile}"
      if error:
         mode.addWarning( error )
   # all current fragment files should now be written, or warnings added
   # to mode.

def Plugin( entityManager ):
   global fragmentConfig, fragmentStatus
   fragmentConfig = LazyMount.mount( entityManager,
                                     "cli/fragment/config",
                                     "Cli::Session::FragmentConfig",
                                     "r" )
   fragmentStatus = LazyMount.mount( entityManager,
                                     "cli/fragment/status",
                                     "Cli::Session::FragmentStatus",
                                     "r" )
   # register copy hook in startup-config to save individual fragment files
   StartupConfigUrl.registerCopyHook( fragmentConfigCopyHook )
