# Copyright (c) 2019 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import cProfile
import io

import BasicCli
import CliCommand
import CliToken.Cli
import ShowCommand

#------------------------------------------------------------------------------------
# The 'show cli debug [ timing | profiler ]' command.
#------------------------------------------------------------------------------------
def printProfilerInfo( mode, args ):
   import pstats # pylint: disable=import-outside-toplevel
   sortby = 'cumulative'
   print( '\n\n\nParser Profiler\n\n\n' )
   s = io.StringIO()
   profiler = mode.session.sessionData( 'BasicCliCommands.parserProfiler' )
   try:
      ps = pstats.Stats( profiler, stream=s ).sort_stats( sortby )
      if 'callers' in args:
         ps.print_callers()
      elif 'callees' in args:
         ps.print_callees()
      else:
         ps.print_stats()
      print( s.getvalue() )
   except Exception: # pylint: disable-msg=W0703
      print( 'No profiling data available' )

   print( '\n\n\nValue Function Profiler\n\n\n' )
   s = io.StringIO()
   profiler = mode.session.sessionData( 'BasicCliCommands.valueFuncProfiler' )
   try:
      ps = pstats.Stats( profiler, stream=s ).sort_stats( sortby )
      if 'callers' in args:
         ps.print_callers()
      elif 'callees' in args:
         ps.print_callees()
      else:
         ps.print_stats()
      print( s.getvalue() )
   except Exception: # pylint: disable-msg=W0703
      print( 'No profiling data available' )

def printTimingInfo( mode, args ):
   print( 'Timing information:' )
   timingInfo = mode.session.sessionData( 'BasicCliCommands.timingInfo' )
   if timingInfo is None:
      print( 'No timing information available' )
      return

   for cmd, info in sorted( timingInfo.items(),
                            key=lambda x: x[ 1 ][ 'totalTime' ] ):
      # pylint: disable-next=consider-using-f-string
      print( '{}: {}'.format( cmd, info[ 'totalTime' ] ) )
      if 'detail' in args:
         for tim in info[ 'details' ]:
            print( '\t%s' % tim ) # pylint: disable=consider-using-f-string

class ShowCliDebugCmd( ShowCommand.ShowCliCommandClass ):
   syntax = ( 'show cli debug ( timing [ detail ] ) | '
               '( profiler [ stats | callers | callees ] )' )
   data = { 'cli': CliToken.Cli.cliForShowMatcher,
            'debug': 'Cli debug commands',
            'timing': 'Per command timings',
            'detail': 'Per command invokation timings',
            'profiler': 'Profiler information',
            'stats': 'Profiler data by total time',
            'callers': 'Profiler data by callers',
            'callees': 'Profiler data by callees' }
   hidden = True

   @staticmethod
   def handler( mode, args ):
      if 'profiler' in args:
         printProfilerInfo( mode, args )

      if 'timing' in args:
         printTimingInfo( mode, args )

#------------------------------------------------------------------------------------
# The '[no/default] cli debug profiler' command.
#------------------------------------------------------------------------------------
class ConfigCliDebugProfilerCmd( CliCommand.CliCommandClass ):
   syntax = 'cli debug profiler'
   noOrDefaultSyntax = 'cli debug profiler'
   data = { 'cli': CliToken.Cli.cliForExecMatcher,
            'debug': 'Cli debug commands',
            'profiler': 'Profiler information' }
   hidden = True

   @staticmethod
   def handler( mode, args ):
      mode.session.sessionDataIs( 'BasicCliCommands.profilerEnabled', True )
      mode.session.sessionDataIs( 'BasicCliCommands.parserProfiler',
                                  cProfile.Profile() )
      mode.session.sessionDataIs( 'BasicCliCommands.valueFuncProfiler',
                                  cProfile.Profile() )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      mode.session.sessionDataIs( 'BasicCliCommands.profilerEnabled', False )

#------------------------------------------------------------------------------------
# The '[no/default] cli debug timing' command.
#------------------------------------------------------------------------------------
class ConfigCliDebugTimingCmd( CliCommand.CliCommandClass ):
   syntax = 'cli debug timing'
   noOrDefaultSyntax = 'cli debug timing'
   data = { 'cli': CliToken.Cli.cliForExecMatcher,
            'debug': 'Cli debug commands',
            'timing': 'Per command timings' }
   hidden = True

   @staticmethod
   def handler( mode, args ):
      mode.session.sessionDataIs( 'BasicCliCommands.timingInfo', {} )
      mode.session.sessionDataIs( 'BasicCliCommands.timingEnabled', True )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      mode.session.sessionDataIs( 'BasicCliCommands.timingEnabled', False )

BasicCli.addShowCommandClass( ShowCliDebugCmd )
BasicCli.EnableMode.addCommandClass( ConfigCliDebugProfilerCmd )
BasicCli.EnableMode.addCommandClass( ConfigCliDebugTimingCmd )
