# Copyright (c) 2011, 2013 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import BasicCli
import CliCommand
import CliMatcher
import CliParserCommon
import CliSchedulerLib
import ConfigMount
import DateTimeRule
import ShowCommand
import Url

config = None

gconfigMode = BasicCli.GlobalConfigMode

scheduleKwMatcher = CliMatcher.KeywordMatcher(
   'schedule',
   helpdesc='Configure a CLI command to be run periodically' )

def allSchduledJobs( mode ):
   return { name: f'CLI command {value.cliCommand}'
            for name, value in config.scheduledCli.items() }

class ScheduleCommand( CliCommand.CliCommandClass ):
   syntax = ( "schedule NAME "
      f"( [ {CliSchedulerLib.scheduleNowStr} ] interval INT_MIN ) | "
      "( at TIME [ DATE ] ( interval INT_MIN ) | "
      f"{CliSchedulerLib.scheduleOnceStr} ) "
      "[ timeout TIMEOUT_MIN ] max-log-files MAXLOGS "
      "[ max-total-size MAX_TOTAL_SIZE ] "
      "[ logging verbose ] [ loglocation LOGPATH ] "
      "[ compression COMPRESSION_ALGO ] "
      "command COMMAND" )

   noOrDefaultSyntax = "schedule NAME ..."
   data = {
      'schedule' : scheduleKwMatcher,
      # config being a keyword should not be allowed as job name
      'NAME' : CliMatcher.PatternMatcher( r'^(?!config)[\w_-]+',
                                          helpname='WORD',
                                          helpdesc='Scheduled job name' ),
      CliSchedulerLib.scheduleNowStr :
      'Set the start time of the schedule to now',
      'interval' : 'Set interval for CLI command execution',
      'INT_MIN' : CliMatcher.IntegerMatcher(
         CliSchedulerLib.intervalMin,
         CliSchedulerLib.intervalMax,
         helpdesc='Interval in minutes for CLI command execution' ),
      'at' : 'Set the start time of the schedule',
      'DATE' : DateTimeRule.dateExpression( 'DATE', helpdesc="Start date" ),
      'TIME' : DateTimeRule.ValidTimeMatcher( helpdesc="Start time" ),
      CliSchedulerLib.scheduleOnceStr :
      'Schedule the command for single execution',
      'timeout' : 'Set timeout for CLI command execution',
      'TIMEOUT_MIN' : CliMatcher.IntegerMatcher(
         CliSchedulerLib.timeoutMin,
         CliSchedulerLib.timeoutMax,
         helpdesc='Timeout in minutes for CLI command execution' ),
      'max-log-files' : 'Set maximum number of logfiles to be stored',
      'MAXLOGS' : CliMatcher.IntegerMatcher(
         CliSchedulerLib.maxLogFilesMin,
         CliSchedulerLib.maxLogFilesMax,
         helpdesc='Number of logfiles to be stored' ),
      'max-total-size' : 'Set total size of log files',
      'MAX_TOTAL_SIZE' : CliMatcher.PatternMatcher(
                                    pattern='[0-9]+[b|k|m|g|B|K|M|G]?',
                                    helpdesc='Maximum total size of log files'
         'in bytes, k for kilobytes, m for megabytes, g for gigabytes ',
                                    helpname='WORD' ),
      'logging' : 'Logging level for scheduler',
      'verbose' : 'Detailed logging',
      'loglocation' : 'Choose which path scheduler log files can be saved to',
      'LOGPATH' : Url.UrlMatcher(
         lambda fs: fs.supportsWrite() and
         fs.fsType == 'flash' and not 'file:' in fs.scheme,
         'Destination for scheduled command output log files',
         notAllowed=[ 'all-filesystems', 'file' ] ),
      'compression' : 'Set compression algorithm for log files',
      'COMPRESSION_ALGO' : CliMatcher.EnumMatcher( 
         CliSchedulerLib.compressAlgoEnum ),
      'command' : 'CLI command',
      'COMMAND' : CliMatcher.StringMatcher( helpname='WORD',
                                            helpdesc='Argument' )
      }
   handler = "CliSchedulerCliHandler.handlerScheduleCommand"
   noHandler = "CliSchedulerCliHandler.noHandlerScheduleCommand"
   defaultHandler = "CliSchedulerCliHandler.defaultHandlerScheduleCommand"

gconfigMode.addCommandClass( ScheduleCommand )

configKwMatcher = CliMatcher.KeywordMatcher(
   'config',
   helpdesc='Set CLI scheduler configuration parameters' )

class ScheduleMaxJobsCommand( CliCommand.CliCommandClass ):
   syntax = "schedule config max-concurrent-jobs JOBS"
   noOrDefaultSyntax = "schedule config max-concurrent-jobs ..."
   data = {
      'schedule' : scheduleKwMatcher,
      'config' : configKwMatcher,
      'max-concurrent-jobs' : 'Set limit for maximum number of concurrent jobs',
      'JOBS' : CliMatcher.IntegerMatcher(
         CliSchedulerLib.jobsInProgressMin,
         CliSchedulerLib.jobsInProgressMax,
         helpdesc='Maximum number of concurrent jobs' )
      }
   handler = "CliSchedulerCliHandler.handlerScheduleMaxJobsCommand"
   noOrDefaultHandler = \
      "CliSchedulerCliHandler.noOrDefaultHandlerScheduleMaxJobsCommand"

gconfigMode.addCommandClass( ScheduleMaxJobsCommand )

class ScheduleHostnameCommand( CliCommand.CliCommandClass ):
   syntax = "schedule config prepend-hostname-logfile"
   noSyntax = syntax
   defaultSyntax = syntax
   data = {
      'schedule' : scheduleKwMatcher,
      'config' : configKwMatcher,
      'prepend-hostname-logfile' : 'Prepend hostname to logfile name',
      }
   handler = "CliSchedulerCliHandler.handlerScheduleHostnameCommand"
   noHandler = "CliSchedulerCliHandler.noHandlerScheduleHostnameCommand"
   defaultHandler = handler

gconfigMode.addCommandClass( ScheduleHostnameCommand )

class ShowScheduleCommand( ShowCommand.ShowCliCommandClass ):
   syntax = "show schedule NAME | summary"
   data = {
      "schedule" : "Show CLI Scheduler information",
      "NAME" : CliMatcher.DynamicNameMatcher( allSchduledJobs,
                                              "Scheduled job name",
                                              priority=CliParserCommon.PRIO_LOW ),
      "summary" : "Summarized output"
      }
   handler = "CliSchedulerCliHandler.showSchedule"
   cliModel = "CliSchedulerCliModels.ShowScheduleJobModels"
   privileged = True

BasicCli.addShowCommandClass( ShowScheduleCommand )

# --------------------------------------------------------------------------
def Plugin( entityManager ):
   global config
   config = ConfigMount.mount( entityManager, "sys/clischeduler/config",
                               "System::CliScheduler::Config", "w" )
