# Copyright (c) 2007-2011 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# pylint: disable=consider-using-f-string

""" Implements cli commands for displaying and configuring the system clock.
In enable mode:
    - show clock [ timezone UTC ]
    - clock set <hh:mm:ss> <mm/dd/yyyy> [ timezone UTC ]
    - clock set <hh:mm:ss> Month Day Year [ timezone UTC ]
    - clock set <hh:mm:ss> Day Month Year [ timezone UTC ]

In config mode:
    - clock timezone <timezone>
    - no clock timezone
    - no clock source
"""
import datetime
import os
import time

import BasicCli
import Cell
import CliCommand
import CliMatcher
import CliParser
import CliParserCommon
import CliPlugin.TechSupportCli
import ConfigMount
import DateTimeRule
import LazyMount
import ShowCommand
import Tac

from ArnetModel import IpGenericAddress
from CliModel import Bool, Float, Int, Model, Str, Submodel

clockConfig = None
clockStatus = None
timezoneConfig = None
timezoneStatus = None

configClockMatcher = CliMatcher.KeywordMatcher(
   'clock',
   helpdesc='Configure the system clock' )

sourceMatcher = CliMatcher.KeywordMatcher(
   'source',
   helpdesc='Set the system clock source' )

minYear = 2000
maxYear = 2037

def doSetClock( mode, time, date, utc ): # pylint: disable=redefined-outer-name
   """Note that we currently only support years from minYear to maxYear 
      since Linux currently only supports years from 1969 to maxYear."""
   ( hour, minute, sec ) = time
   ( month, day, year ) = date

   try:
      datetime.date( year, month, day )
   except ValueError:
      mode.addError( "Invalid date entered." )
      return

   if year < minYear or year > maxYear:
      mode.addError( "Invalid date entered. Please enter an year within the range " +
                     "<%d-%d>" % ( minYear, maxYear ) )
      return
   cmd = [ 'date' ]
   if utc:
      cmd.append( '-u' )
   cmd.append( '%02d%02d%02d%02d%s.%02d' %( month, day, hour,
                                            minute, year, sec ))
   try:
      Tac.run( cmd, stdout=Tac.CAPTURE, asRoot=True )
   except Tac.SystemCommandError:
      mode.addError( "Invalid date entered." )
      return

   # Push the new time to hardware
   try:
      Tac.run( [ 'touch', '-m', '/mnt/flash/persist/clock' ],
               stdout=Tac.CAPTURE,
               asRoot=True )
      Tac.run( [ 'SyncFile', '/mnt/flash/persist/clock' ],
               stdout=Tac.CAPTURE,
               asRoot=True )
   except Tac.SystemCommandError:
      print( "Warning: failed to write persistent clock file." )

   try:
      Tac.run( [ '/sbin/hwclock', '--systohc', '--utc' ],
               stdout=Tac.CAPTURE,
               asRoot=True )
      clockStatus.clockSetAtTime = Tac.now()
      doShowClock( mode ).render()
   except Tac.SystemCommandError:
      print( "Warning: failed to update hardware clock." )

class ClockSetCommand( CliCommand.CliCommandClass ):
   syntax = ( 'clock set TIME '
              '( ( DATE ) '
              '| ( MONTH DAY YEAR ) '
              '| ( DAY MONTH YEAR ) ) [ timezone UTC ]' )
   data = {
      'clock' : configClockMatcher,
      'set' : 'Set the system date and time',
      'TIME' : DateTimeRule.ValidTimeMatcher( helpdesc="Current time" ),
      'DATE' : DateTimeRule.dateExpression( 'DATE', helpdesc="Today's date" ),
      'MONTH' : DateTimeRule.monthMatcher,
      'DAY' : DateTimeRule.dayMatcher,
      'YEAR' : CliMatcher.IntegerMatcher( minYear, maxYear, helpdesc="Year" ),
      'timezone' : 'Specify timezone',
      'UTC' : 'Provided time is in UTC instead of local timezone',
      }
   @staticmethod
   def handler( mode, args ):
      date = args.get( 'DATE' )
      if date is None:
         date = ( args[ 'MONTH' ], args[ 'DAY' ], args[ 'YEAR' ] )
      utc = args.get( 'UTC', False )
      doSetClock( mode, args[ 'TIME' ], date, utc )

BasicCli.EnableMode.addCommandClass( ClockSetCommand )

clockSourceHooks = []
def registerClockSourceHook( hookFunction ):
   """
   hookFunction for any particular clock source takes mode as argument and
   outputs the source if it's enabled or an empty string otherwise.
   See getClockSource() to figure out how the output string is used.
   In general only one registered hookFunction should return us a non-empty
   string since the system can't have two clock synchronization sources.
   """
   clockSourceHooks.append( hookFunction )

def doSetClockSource( mode, source ):
   if source is None:
      source = clockConfig.defaultSource
   clockConfig.source = source

# The no/default version of the command is defined here, but the forms for
# specific sources may require guards based on certain config (e.g., PTP).
# To avoid dependency problems, those commands are defined in source-specific
# packages (Ntp, Ptp, etc.).
#-------------------------------------------------------------------------------
# no|default clock source
#-------------------------------------------------------------------------------
class NoClockSourceCommand( CliCommand.CliCommandClass ):
   noOrDefaultSyntax = 'clock source ...'
   data = {
      'clock' : configClockMatcher,
      'source' : sourceMatcher,
   }
   noOrDefaultHandler = lambda mode, args: doSetClockSource( mode, None )

BasicCli.GlobalConfigMode.addCommandClass( NoClockSourceCommand )

def getClockSource( mode ):
   sources = []
   for hook in clockSourceHooks:
      sourceInfo = hook( mode )
      if sourceInfo:
         sources.append( sourceInfo )
   if not sources:
      sources.append ( "local" )
   elif len( sources ) > 1:
      # This should never happen in practice, 
      # but I am still tolerating the error in the cli.
      mode.addWarning( "Multiple clock sources are "
                       "simultaneously active." )

   s = sources[ 0 ]
   cs = ClockSourceModel()
   if s == "local":
      cs.local = True
   elif s.find ( "NTP server (" ) != -1:
      # expected source format: NTP server (fd7a:629f:52a4:1616:250:56ff:fea8:c)
      source = s [ s.find ( "(" ) + 1 : s.find ( ")" ) ]
      cs.local = False
      cs.ntpServer = source
   elif s.find( "PTP grandmaster (" ) != -1:
      # expected source format: PTP grandmaster (444ca8.fffe.801be4)
      source = s [ s.find ( "(" ) + 1 : s.find ( ")" ) ]
      cs.local = False
      cs.ptpGrandmaster = source
   else:
      # This should never happen in practice,
      mode.addWarning( "Unrecognised clock source %s " % s )
      cs.local = False

   return cs

class ClockSourceModel( Model ):
   local = Bool( help='Clock source is local' )
   ntpServer = IpGenericAddress( help='NTP server address', optional=True )
   ptpGrandmaster = Str( help='PTP grandmaster clock identity', optional=True )

   def render( self ):
      if self.local:
         print( 'Clock source: local' )
      elif self.ntpServer:
         print( 'Clock source: NTP server (%s)' % self.ntpServer )
      elif self.ptpGrandmaster:
         print( 'Clock source: PTP grandmaster (%s)' % self.ptpGrandmaster )
      else:
         print( 'Clock source: unrecognised' )

class LocalTimeModel( Model ):
   year = Int( help='Year (for example, 2019)' )
   month = Int( help='Month (1 - 12)' )
   dayOfMonth = Int( help='Day of month (1 - 31)' )
   hour = Int( help='Hour (0 - 23)' )
   min = Int( help='Minute (0 - 59)' )
   sec = Int( help='Second ( (0 - 61, 60 for leap secs, 61 for double leap secs)' )
   dayOfWeek = Int( help='Day of week (0 - 6, Monday is 0)' )
   dayOfYear = Int( help='Day of year (1 - 366)' )
   daylightSavingsAdjust = Int( help='Daylight savings time (-1 - 1, -1 unknown)' )

class ShowClockModel( Model ):
   # non-public: operating timezone - normally expected to be identical
   # to timezone arg but can hold a different value from timezone if
   # SuperServer is not running such that config values are not getting applied
   # (generally expected to differ in test environs only)
   _tzStatus = Str( help="private" )
   # non-public: current seconds relative to the machine's understanding of its
   # timezone.  May or may not be the same as utcTime depending on actual timezone in
   # use except where 'timezone utc' variant of 'show clock' is used.
   _now = Float( help="private" )
   utcTime = Float( help="The current timestamp (UTC)" )
   timezone = Str( help="Local timezone in use" )
   localTime = Submodel( valueType=LocalTimeModel, help='Local time components' )
   clockSource = Submodel( valueType=ClockSourceModel, help="Clock source" )

   def render( self ):

      print( time.ctime( self._now ) )

      suffix = ""
      if self.timezone != self._tzStatus:
         suffix = " (using %s)" % ( self._tzStatus )
      print( "Timezone: %s%s" % ( self.timezone, suffix ) )

      self.clockSource.render()

def doShowClock( mode, args=None ):
   # Make sure we're using the correct timezone. tzset documentation says
   # that "changing the TZ environment variable without calling tzset *may* change
   # the local timezone used by methods such as localtime, but this behaviour
   # should not be relied on" so we call it explicitly.
   time.tzset()
   utcTime = time.time()

   if args and 'UTC' in args:
      now = time.gmtime( utcTime )
      tzConfig = "UTC"
      tzStatus = "UTC"
   else:
      now = time.localtime( utcTime )
      tzConfig = timezoneConfig.zone if timezoneConfig else "not set"
      tzStatus = timezoneStatus.zone if timezoneStatus else "UTC"

   source = getClockSource( mode )

   lt = LocalTimeModel( )
   lt.year = now.tm_year
   lt.month = now.tm_mon
   lt.dayOfMonth = now.tm_mday
   lt.hour = now.tm_hour
   lt.min = now.tm_min
   lt.sec = now.tm_sec
   lt.dayOfWeek = now.tm_wday
   lt.dayOfYear = now.tm_yday
   lt.daylightSavingsAdjust = now.tm_isdst

   return ShowClockModel( _tzStatus=tzStatus,
                          _now=time.mktime( now ),
                          utcTime=utcTime,
                          timezone=tzConfig,
                          localTime=lt,
                          clockSource=source )

#-----------------------------------------------------------------------------------
# show clock
#-----------------------------------------------------------------------------------
class ShowClock( ShowCommand.ShowCliCommandClass ):
   syntax = 'show clock [ timezone UTC ]'
   data = {
            'clock': 'System time',
            'timezone': 'Specify timezone',
            'UTC': 'Show current time in UTC instead of local timezone',
          }
   cliModel = ShowClockModel
   handler = doShowClock

BasicCli.addShowCommandClass( ShowClock )

# ------------------------------------------
# clock config commands :
#    clock timezone <zonefile>

def findfiles( root ):
   """Find all files in the subdir starting at 'root' and builds a
   data structure organized in the following way:
   
   A directory is a 2-tuple
   first element: path to that directory from root
   second element: a dict that maps lower case names of the files and
   dirs in that directory to file and directory types as described
   here.

   A file is the same except that its second element is always None

   Here is an example if the files are US/Pacific and UTC:

   ('',
    { "us" : ( "US",
               { "pacific" : ("US/Pacific", None) } ),
           
      "utc" : ("UTC", None )})
   """
   def walk( d, rel ):
      z = {}
      for p in os.listdir( os.path.join( root, d ) ):
         f = os.path.join( d, p )
         rp = os.path.join( rel, p )
         if os.path.isdir( f ):
            z[ p.lower() ] = walk( f, rp )
         else:
            z[ p.lower() ] = (rp, None)
      return (rel, z)

   return walk( root, '' )

def traverseFiles( node, path ):
   if path == '':
      return node
   seperated = path.lower().split( os.sep )
   for f in seperated:
      if node[ 1 ] and f in node[ 1 ]:
         node = node[ 1 ][ f ]
      else:
         return None
   return node

def isdir( node ):
   return node and node[ 1 ] is not None

def isfile( node ):
   return node and node[ 1 ] is None

def getpath( node ):
   return node[0]

def getchildren( node ):
   return node[1]

class FileMatcher( CliMatcher.Matcher ):
   """It matches the relative path of any normal
   (non-dir) file in rootdir.  It memoizes the result of find and
   attempts to avoid a directory walk unless completions is called."""

   def __init__( self, rootdir, helpsuffix, **kargs ):
      self.rootdir_ = rootdir
      self.helpsuffix_ = helpsuffix
      super().__init__( helpdesc='', **kargs )

   @Tac.memoize
   def _files( self ):
      return findfiles( self.rootdir_ )

   def match( self, mode, context, token ):
      # First check if we've got a path to an existant file.  This
      # makes system initialization a little faster because it is
      # O(1) rather than O(# of timezones supported).
      if os.path.isfile( os.path.join( self.rootdir_, token ) ):
         return CliParserCommon.MatchResult( token, token )
      node = traverseFiles( self._files(), token )
      if isfile( node ):
         path = getpath( node )
         return CliParserCommon.MatchResult( path, path )
      else:
         return CliParserCommon.noMatch

   def completions( self, mode, context, token ):
      dirfile = os.path.split( token )
      d = traverseFiles( self._files(), dirfile[0] )
      if d is None: 
         return []
      def getHelpMessage( node ):
         timezoneStr = "%s %s" % ( getpath( node ), self.helpsuffix_ )
         if isdir( node ):
            timezoneStr += "s"
         return timezoneStr
      completionList = []
      children = getchildren( d )
      if not children:
         # We were actually passed a file
         return completionList
      for name, node in children.items():
         if name.startswith( dirfile[ 1 ].lower() ):
            completion = CliParser.Completion( 
                                  getpath( node ) + ("/" if isdir( node ) else ''),
                                  getHelpMessage( node ),
                                  partial = not isfile( node ) )
            completionList.append( completion )
      return completionList

class TimeZoneMatcher( FileMatcher ):
   """Matches available timezones for use in 'clock timezone ...' 
   config command."""
   def __init__( self, **kargs ):
      super().__init__( '/usr/share/zoneinfo/posix',
                        'timezone', **kargs )

class ConfigTimeZone( CliCommand.CliCommandClass ):
   syntax = 'clock timezone TIMEZONE'
   noOrDefaultSyntax = 'clock timezone ...'
   data = {
            'clock': configClockMatcher,
            'timezone': 'Configure timezone',
            'TIMEZONE': TimeZoneMatcher()
          }

   @staticmethod
   def handler( mode, args ):
      timezoneConfig.zone = args[ 'TIMEZONE' ]

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      timezoneConfig.zone = timezoneConfig.defaultZone

BasicCli.GlobalConfigMode.addCommandClass( ConfigTimeZone )

#------------------------------------------------------
# Register show clock command into "show tech-support".
#------------------------------------------------------

# Timestamps are made up to maintain historical order within show tech-support
CliPlugin.TechSupportCli.registerShowTechSupportCmd( 
   '2010-01-01 00:02:30',
   cmds=[ 'show clock' ],
   summaryCmds=[ 'show clock' ] )

def Plugin( entityManager ):
   global clockConfig, clockStatus, timezoneConfig, timezoneStatus
   clockConfig = ConfigMount.mount( entityManager, "sys/time/clock/config",
                                    "Time::Clock::Config", "w" )
   clockStatus = LazyMount.mount( entityManager,
                                    Cell.path( "mgmt/security/clockStatus" ),
                                    "Mgmt::Security::ClockStatus", "w" )
   timezoneConfig = ConfigMount.mount( entityManager, "sys/time/zone/config",
                                       "Time::Zone::Config", "w" )
   timezoneStatus = LazyMount.mount( entityManager, 
                                     Cell.path( "sys/time/zone/status" ),
                                     "Time::Zone::Status", "r" )
