# Copyright (c) 2022 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from collections import namedtuple
import BasicCli
from CliPlugin import ConfigTagCommon
from CliPlugin.FileCli import runningConfigAfterShowKw
from CliMatcher import EnumMatcher
import ConfigMount
import LazyMount
import ShowCommand
import CliGlobal
import CliCommand
import Tac

gv = CliGlobal.CliGlobal( dict( configTagConfig=None, configTagInput=None,
                                configTagInputCli=None, configTagIdState=None,
                                configTagStatus=None ) )
tagState = Tac.Type( "ConfigTag::ConfigTagState" )

# Information required for command tag validation
CommandTagValidationInfo = namedtuple( 'CommandTagValidationInfo',
                                     [ 'commandTag', 'commandTagId', 'operState' ] )

@Tac.memoize
def configTagAllocator():
   # Forcefully mount lazyMounted entities before use by this allocator.
   gv.configTagInput.force()
   gv.configTagIdState.force()
   return Tac.newInstance( "ConfigTag::ConfigTagAllocator" )

@Tac.memoize
def configTagInputAllocator():
   return Tac.newInstance( "ConfigTag::ConfigTagInputAllocator" )

configTagExpr = ConfigTagCommon.ConfigTagExpr
configTagState = ConfigTagCommon.ConfigTagState()

def manageConfigTagConfig( mode, tagStr, operState ):
   ConfigTagCommon.checkTagInRemovedOrDisassociatedState( mode, tagStr )

   # Check if post this command-tag action, config is still valid.
   # Abort without making changes if validation fails.
   if ( tagStr in gv.configTagConfig.configTagEntry ) and \
         ( tagStr in gv.configTagInputCli.configTagEntry ):
      tagId = gv.configTagConfig.configTagEntry[ tagStr ].tagId
      configTagState.processAndValidateConfig( mode,
            CommandTagValidationInfo( tagStr, tagId, operState ) )

   # Handle all possible tag states.
   if operState == tagState.disassociated:
      # Disassociate all the config associated with the command-tag.
      configTagState.disassociateConfigFromTag( mode, tagStr )
      # Since cleanup is complete at this point, remove tag from input.
      configTagInputAllocator().deleteConfigTagInputEntry( gv.configTagInputCli,
         tagStr )
      # Create/Modify tag with the given operState.
      res = configTagAllocator().newConfigTagEntry( gv.configTagInput,
            gv.configTagConfig, gv.configTagIdState, tagStr, operState )
   elif operState == tagState.removed:
      # Remove all the config associated with the command-tag.
      configTagState.removeTaggedConfig( mode, tagStr )
      # Since cleanup is complete at this point, remove tag from input.
      configTagInputAllocator().deleteConfigTagInputEntry( gv.configTagInputCli,
         tagStr )
      # Delete config entry for tag.
      res = configTagAllocator().deleteConfigTagEntry( gv.configTagInput,
            gv.configTagConfig, gv.configTagIdState, tagStr )

   elif operState in [ tagState.disabled, tagState.enabled ]:
      # Create/Modify tag with the given operState.
      res = configTagAllocator().newConfigTagEntry( gv.configTagInput,
            gv.configTagConfig, gv.configTagIdState, tagStr, operState )
   else:
      # Any other state is not supported.
      assert False, "ConfigTagState is not handled"

   if not res:
      mode.addErrorAndStop( configTagAllocator().errorMsg )

# -----------------------------------------------------------------------------------
# [ no ] command-tag TAG [ disabled ]
# command-tag TAG disassociated
# -----------------------------------------------------------------------------------
class ConfigTagCmd( CliCommand.CliCommandClass ):
   syntax = '''CONFIG_TAG_EXPR [ ACTION ]'''
   noOrDefaultSyntax = '''CONFIG_TAG_EXPR ...'''
   data = {
      'CONFIG_TAG_EXPR': configTagExpr,
      'ACTION': EnumMatcher( {
         'disabled': 'Disable command-tag',
         'disassociated': 'Disassociate all config associated with this command-tag',
      } )
   }

   @staticmethod
   def handler( mode, args ):
      state = args.get( 'ACTION', 'enabled' )
      operState = getattr( tagState, state )
      tagStr = args[ 'CONFIG_TAG' ]
      manageConfigTagConfig( mode, tagStr, operState )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      tagStr = args[ 'CONFIG_TAG' ]
      manageConfigTagConfig( mode, tagStr, operState=tagState.removed )

BasicCli.GlobalConfigMode.addCommandClass( ConfigTagCmd )

# -----------------------------------------------------------------------------------
# show command-tag [ TAG ]
# -----------------------------------------------------------------------------------
class ShowCommandTag( ShowCommand.ShowCliCommandClass ):
   syntax = "show command-tag [ COMMAND_TAG ]"
   data = {
            'command-tag': ConfigTagCommon.configTagKwNode,
            'COMMAND_TAG': ConfigTagCommon.configTagNameNode
   }
   privileged = True
   cliModel = "ConfigTagCliModels.CommandTags"
   handler = "ConfigTagCliHandlers.showCommandTagHandler"

BasicCli.addShowCommandClass( ShowCommandTag )

# -----------------------------------------------------------------------------------
# show running-config command-tag [ TAG ]
# -----------------------------------------------------------------------------------
class ShowRunningConfigCommandTag( ShowCommand.ShowCliCommandClass ):
   syntax = """show running-config command-tag [ COMMAND_TAG ]"""
   data = {
      'running-config': runningConfigAfterShowKw,
      'command-tag': ConfigTagCommon.configTagKwNode,
      'COMMAND_TAG': ConfigTagCommon.configTagNameNode
   }
   privileged = True
   handler = "ConfigTagCliHandlers.showRunningConfigCommandTagHandler"

BasicCli.addShowCommandClass( ShowRunningConfigCommandTag )

def Plugin( entityManager ):
   gv.configTagConfig = ConfigMount.mount(
         entityManager, "configTag/config", "ConfigTag::ConfigTagConfig", "w" )
   gv.configTagIdState = LazyMount.mount(
         entityManager, "configTag/configTagIdState", "ConfigTag::ConfigTagIdState",
         "w" )
   gv.configTagInput = LazyMount.mount(
         entityManager, "configTag/input", "Tac::Dir", "ri" )
   gv.configTagInputCli = ConfigMount.mount(
         entityManager, "configTag/input/cli", "ConfigTag::ConfigTagInput", "w" )
