# Copyright (c) 2022 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import CliMatcher
import CliCommand
import CliGlobal
import CliParser
import LazyMount
import Tac
import ConfigMount

gv = CliGlobal.CliGlobal( dict( configTagConfig=None, configTagPlatformStatus=None,
                                configTagInput=None, configTagInputCli=None ) )

configTagNameMatcher = CliMatcher.DynamicNameMatcher(
                          lambda mode: gv.configTagConfig.configTagEntry,
                          helpname='WORD',
                          helpdesc='Command tag name' )
tagState = Tac.Type( "ConfigTag::ConfigTagState" )

def configTagSupportedByPlatform( mode, configTagStr=None ):
   if mode.session_.startupConfig():
      return bool( configTagStr )
   else:
      return gv.configTagPlatformStatus.configTagSupported

def configTagSupportedGuard( mode, token ):
   if gv.configTagPlatformStatus.configTagSupported:
      return None
   return CliParser.guardNotThisPlatform

configTagKwMatcher = CliMatcher.KeywordMatcher( 'command-tag',
      helpdesc='Configure command tag' )

configTagKwNode = CliCommand.Node( matcher=configTagKwMatcher,
                                          guard=configTagSupportedGuard )

configTagNameNode = CliCommand.Node( matcher=configTagNameMatcher )

class ConfigTagExpr( CliCommand.CliExpression ):
   expression = 'command-tag CONFIG_TAG'
   data = {
            'command-tag': configTagKwNode,
            'CONFIG_TAG': configTagNameNode
          }

def commandTagErrStr( configTag ):
   return "Command tag %s undefined. " % ( configTag ) + \
          "Please configure the command tag with \"command-tag %s\"." % ( configTag )

def commandTagConfigCheck( mode, configTag ):
   # Check whether configTag has been created before allowing it to be referenced.
   # If tag is present, check if tagId has been set.
   # Throw error otherwise.
   if configTag:
      tagInConfig = gv.configTagConfig.configTagEntry.get( configTag )
      if not tagInConfig or ( tagInConfig and not tagInConfig.tagId ):
         mode.addErrorAndStop( commandTagErrStr( configTag ) )


def checkTagInUse( mode, tagStr ):
   # Although configTagInputCli ( 'configTag/input/cli' ) is an entry of
   # configTagInput ( 'configTag/input' ), we check for tag presence seperately
   # for configTagInputCli because it is configMounted while configTagInput is not.
   # When this check is run inside a configSession, configTagInputCli points to the
   # local entity created for the session since it is configMounted while the entry
   # for cli in configTagInput still points to the entity in the global sysdb.
   # Since any changes made to this path in the session will only be updated in
   # configTagInputCli, we should check for tag usage in it seperately.
   # TODO:Have a seperate path for cli outside of the configTag/input dir and only
   # have entries for eosSdk agents in the configTag/input dir.
   if gv.configTagInputCli.configTagEntry.get( tagStr, False ):
      return True
   inputs = gv.configTagInput
   for name in inputs:
      if name != 'cli':
         curInput = gv.configTagInput.get( name, None )
         if curInput:
            if curInput.configTagEntry.get( tagStr, False ):
               return True
   return False

def checkTagInRemovedOrDisassociatedState( mode, tagStr, featureCheck=False ):
   # Check the tag config for 'removed' or 'disassociated' state.
   tag = gv.configTagConfig.configTagEntry.get( tagStr, None )
   inUse = checkTagInUse( mode, tagStr )
   # If tag is not present in config but present in input, it is in removed state.
   inRemovedState = inUse and ( not tag )

   if ( tag and tag.operState == tagState.disassociated and ( featureCheck or inUse \
         ) ) or inRemovedState:
      # This means that the tag is in either 'removed' or 'disassociated' state
      # and has not been removed by all users. We should not allow any
      # modification to this tag until all the related tagged config is either
      # disassociated or removed ( indicated by absence from input ).
      operState = tagState.removed if inRemovedState else tagState.disassociated
      mode.addErrorAndStop( "Command tag %s is being %s. "
      "Please wait for the operation to complete." % ( tagStr, operState ) )

# This class is instantiated by the ConfigTagCli plugin to cleanup
# all tagged config from the system when the command handlers for
# `no command-tag TAG` or `command-tag disassociated` are invoked.
# This class is also used to validate config before modifying
# a tag's config in sysdb.
class ConfigTagState:
   @classmethod
   def registerConfigTagSupportingClass( cls, dependent ):
      cls.configTagSupportingClass_ += [ dependent() ]

   configTagSupportingClass_ = []

   def removeTaggedConfig( self, mode, tag ):
      for dependent in ConfigTagState.configTagSupportingClass_:
         dependent.removeTaggedConfig( mode, tag )

   def disassociateConfigFromTag( self, mode, tag ):
      for dependent in ConfigTagState.configTagSupportingClass_:
         dependent.disassociateConfigFromTag( mode, tag )

   def processAndValidateConfig( self, mode, commandTagInfo ):
      for dependent in ConfigTagState.configTagSupportingClass_:
         dependent.processAndValidateConfig( mode, commandTagInfo )

# This is used as a base class for removing command-specific
# tagged configuration in the system. For example, the
# StaticRouteConfigTagState in Vxlan inherits this class
# and defines its own removeTaggedConfig function, which is
# responsible for removing config related specifically to
# tagged static vxlan routes and disassociateConfigFromTag which is
# responsible for only untagging static vxlan routes config
# associated with tag.
class ConfigTagDependentBase:
   def removeTaggedConfig( self, mode, tag ):
      raise NotImplementedError

   def disassociateConfigFromTag( self, mode, tag ):
      raise NotImplementedError

   def processAndValidateConfig( self, mode, commandTagInfo ):
      raise NotImplementedError

def Plugin( entityManager ):
   gv.configTagConfig = ConfigMount.mount(
         entityManager, "configTag/config", "ConfigTag::ConfigTagConfig", "w" )
   gv.configTagPlatformStatus = LazyMount.mount( entityManager,
         "configTag/platformStatus", "ConfigTag::ConfigTagPlatformStatus", "r" )
   gv.configTagInput = LazyMount.mount(
         entityManager, "configTag/input", "Tac::Dir", "ri" )
   gv.configTagInputCli = ConfigMount.mount(
         entityManager, "configTag/input/cli", "ConfigTag::ConfigTagInput", "w" )
