# Copyright (c) 2010, 2011 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import BasicCli
import CliCommand
import ConfigMount
import CliMatcher
from CliPlugin import ConfigMgmtMode
import Tac

# ------------------------------
# Console config commands
#
# From config mode
#    management console    - enter console config mode
#    idle-timeout x - set the idle session timeout to x minutes
#

consoleConfig = None

class ConsoleConfigMode( ConfigMgmtMode.ConfigMgmtMode ):
   name = "Console configuration"

   def __init__( self, parent, session ):
      ConfigMgmtMode.ConfigMgmtMode.__init__( self, parent, session, "console" )
      self.config_ = consoleConfig

   def setIdleTimeout( self, tokenIdleTime ):
      to = Tac.Value( "Mgmt::SessionTimeout::IdleSessionTimeout" )
      to.timeout = tokenIdleTime * 60
      self.config_.idleTimeout = to

   def noIdleTimeout( self ):
      self.setIdleTimeout( self.config_.idleTimeout.defaultTimeout // 60 )

   def setPromptFQDN( self ):
      self.config_.longHostname = True

   def noPromptFQDN( self ):
      self.config_.longHostname = False

   def setConsoleUserMatchList( self, matchListString ):
      self.config_.consoleUserMatchList = matchListString

   def noConsoleUserMatchList( self ):
      self.config_.consoleUserMatchList = ""

class MgmtConsole( CliCommand.CliCommandClass ):
   syntax = "management console"
   noOrDefaultSyntax = syntax
   data = {
      "management" : ConfigMgmtMode.managementKwMatcher,
      "console" : 'Configure console'
      }
   @staticmethod
   def handler( mode, args ):
      childMode = mode.childMode( ConsoleConfigMode )
      mode.session_.gotoChildMode( childMode )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      childMode = mode.childMode( ConsoleConfigMode )
      childMode.noIdleTimeout()
      childMode.noPromptFQDN()
      childMode.noConsoleUserMatchList()
      childMode.removeComment()

BasicCli.GlobalConfigMode.addCommandClass( MgmtConsole )

class IdleTimeout( CliCommand.CliCommandClass ):
   syntax = "idle-timeout TIMEOUT"
   noOrDefaultSyntax = "idle-timeout ..."
   data = {
      "idle-timeout" : ConfigMgmtMode.idleTimeoutKwMatcher,
      "TIMEOUT" : ConfigMgmtMode.idleTimeoutMatcher
      }
   @staticmethod
   def handler( mode, args ):
      mode.setIdleTimeout( args[ 'TIMEOUT' ] )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      mode.noIdleTimeout()

ConsoleConfigMode.addCommandClass( IdleTimeout )

class LoginUser( CliCommand.CliCommandClass ):
   syntax = "login user match-list <matchListName>"
   noOrDefaultSyntax = "login user match-list ..."
   hidden = True
   data = {
      "login": 'Configure console login',
      "user": 'Configure login user',
      "match-list": 'Configure match-list format for allowed users',
      "<matchListName>": CliMatcher.PatternMatcher(
            pattern=".+",
            helpname='WORD',
            helpdesc='Match-list name containing format for allowed users'
        )
      }

   @staticmethod
   def handler( mode, args ):
      matchListString = args[ "<matchListName>" ]
      mode.setConsoleUserMatchList( matchListString )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      mode.noConsoleUserMatchList()

ConsoleConfigMode.addCommandClass( LoginUser )

class LoginPrompt( CliCommand.CliCommandClass ):
   syntax = "login prompt hostname fqdn"
   noOrDefaultSyntax = syntax
   data = {
      "login" : 'Configure console login',
      "prompt" : 'Configure login prompt',
      "hostname" : 'Configure hostname format in the prompt',
      "fqdn" : 'Display fully qualified domain name in prompt'
      }
   @staticmethod
   def handler( mode, args ):
      mode.setPromptFQDN()

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      mode.noPromptFQDN()

ConsoleConfigMode.addCommandClass( LoginPrompt )

def Plugin( entityManager ):
   global consoleConfig
   consoleConfig = ConfigMount.mount( entityManager, "mgmt/console/config",
                                      "Mgmt::Console::Config", "w" )

