#!/usr/bin/env python3
# Copyright (c) 2023 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import BasicCli
import CliCommand
import CliMatcher
from CliMode.RegistryMode import RegistryMode
from CliPlugin.ContainerMgrCliLib import pathRe
import ConfigMount
import DesCrypt
import Tac

registryConfigDir = None

registryNameMatcher = CliMatcher.DynamicNameMatcher(
   lambda mode: registryConfigDir.registry,
   pattern=pathRe,
   helpname='WORD',
   helpdesc='name of the registry' )

def setConfigParam( mode, paramName, paramVal ):
   registryConfig = mode.getConfig()
   params = Tac.nonConst( registryConfig.params )
   setattr( params, paramName, paramVal )
   registryConfig.params = params

class RegistryConfigMode( RegistryMode, BasicCli.ConfigModeBase ):
   name = "Container Manager Registry Configuration"

   def __init__( self, parent, session, registryName ):
      self.registryName_ = registryName
      self.session_ = session
      RegistryMode.__init__( self, registryName )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

   def getConfig( self ):
      return registryConfigDir.registry[ self.registryName_ ]

def addRegistryConfig( mode, args ):
   registryName = args[ 'REGISTRY_NAME' ]
   childMode = mode.childMode( RegistryConfigMode,
                               registryName=registryName )
   mode.session_.gotoChildMode( childMode )
   registryConfigDir.newRegistry( registryName )

def deleteRegistryConfig( mode, args ):
   registryName = args[ 'REGISTRY_NAME' ]
   if registryName in registryConfigDir.registry:
      del registryConfigDir.registry[ registryName ]

def deleteAllRegistryConfigs():
   registryConfigDir.registry.clear()

# --------------------------------------------------------------------------------
# [ no | default ] insecure
# --------------------------------------------------------------------------------
class InsecureCmd( CliCommand.CliCommandClass ):
   syntax = 'insecure'
   noOrDefaultSyntax = syntax
   data = {
      'insecure': 'Configure the registry to be insecure',
   }

   @staticmethod
   def handler( mode, args ):
      setConfigParam( mode, 'insecure', True )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      setConfigParam( mode, 'insecure', False )

RegistryConfigMode.addCommandClass( InsecureCmd )

# --------------------------------------------------------------------------------
# [ no | default ] username USERNAME
# --------------------------------------------------------------------------------
class UsernameUsernameCmd( CliCommand.CliCommandClass ):
   syntax = 'username USERNAME'
   noOrDefaultSyntax = 'username ...'
   data = {
      'username': 'Configure username for the registry',
      'USERNAME': CliMatcher.PatternMatcher( pattern=pathRe,
         helpdesc='username for the registry', helpname='WORD' ),
   }

   @staticmethod
   def handler( mode, args ):
      setConfigParam( mode, 'userName', args[ 'USERNAME' ] )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      setConfigParam( mode, 'userName', "" )

RegistryConfigMode.addCommandClass( UsernameUsernameCmd )

# --------------------------------------------------------------------------------
# [ no | default ] password ( [ 0 ] UNENCRYPTED_PASSWD | ( 7 ENCRYPTED_PASSWD ) )
# --------------------------------------------------------------------------------
class PasswordCmd( CliCommand.CliCommandClass ):
   syntax = 'password ( [ 0 ] UNENCRYPTED_PASSWD | ( 7 ENCRYPTED_PASSWD ) )'
   noOrDefaultSyntax = 'password ...'
   data = {
      'password': 'Configure password for the registry',
      '0': 'Encryption type - unencrypted',
      'UNENCRYPTED_PASSWD': CliCommand.Node(
         matcher=CliMatcher.PatternMatcher( pattern=r'[^\s]{1,80}',
            helpdesc='password (up to 80 characterss)', helpname='LINE' ),
         sensitive=True ),
      '7': 'Encryption type - encrypted',
      'ENCRYPTED_PASSWD': CliCommand.Node(
         matcher=CliMatcher.PatternMatcher( pattern=r'[^\s]+',
            helpdesc='encrypted password', helpname='LINE' ),
         sensitive=True ),
   }

   @staticmethod
   def handler( mode, args ):
      if 'UNENCRYPTED_PASSWD' in args:
         regPasswd = args[ 'UNENCRYPTED_PASSWD' ]
      else:
         try:
            regPasswd = DesCrypt.decrypt(
               mode.registryName_.encode(),
               args[ 'ENCRYPTED_PASSWD' ].encode() ).decode()
         except: # pylint: disable-msg=bare-except
            mode.addError( "Invalid encrypted password" )
            return
      setConfigParam( mode, 'password', regPasswd )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      setConfigParam( mode, 'password', "" )

RegistryConfigMode.addCommandClass( PasswordCmd )

# --------------------------------------------------------------------------------
# server SERVER_NAME
# --------------------------------------------------------------------------------
class ServerServernameCmd( CliCommand.CliCommandClass ):
   syntax = 'server SERVER_NAME'
   noOrDefaultSyntax = 'server ...'
   data = {
      'server': 'Server URL of the registry',
      'SERVER_NAME': CliMatcher.PatternMatcher( pattern=pathRe, helpdesc='server',
         helpname='WORD' ),
   }

   @staticmethod
   def handler( mode, args ):
      setConfigParam( mode, 'serverName', args[ 'SERVER_NAME' ] )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      setConfigParam( mode, 'serverName', "" )

RegistryConfigMode.addCommandClass( ServerServernameCmd )

def Plugin( entityManager ):
   global registryConfigDir
   registryConfigDir = ConfigMount.mount( entityManager,
                                          'containerMgr/registry/config',
                                          'ContainerMgr::RegistryConfigDir', 'w' )
