# Copyright (c) 2018 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import itertools

from CliModel import Bool, Enum, Dict, Model, Str
import TableOutput

def renderTable( headings, rowGenerator ):
   table = TableOutput.createTable( headings )
   fmt = TableOutput.Format( justify='left' )
   fmt.noPadLeftIs( True )
   table.formatColumns( *itertools.repeat( fmt, len( headings ) ) )

   for row in rowGenerator:
      table.newRow( *row )
   return table.output()

class ClusterConfigModel( Model ):
   __public__ = False

   name = Str( help="Name of the Kubernetes cluster" )
   url = Str( help="URL for the Kubernetes API" )
   authType = Str( help="Service account authentication method" )
   authSecret = Str( help="Service account authentication secret" )
   apiVersion = Str( help="Kubernetes API version to use" )


class ClustersModel( Model ):
   __public__ = False

   enabled = Bool( help='The configured state of the ContainerTracer service' )
   clusters = Dict(
      valueType=ClusterConfigModel,
      help="The configurations for kubernetes clusters",
   )

   def render( self ):
      if not self.clusters:
         return

      headings = ( 'Name', 'URL', 'Auth Type', 'API Version' )
      g = (
         ( c.name, c.url, c.authType, c.apiVersion )
         for _, c in sorted( self.clusters.items() )
      )
      print( renderTable( headings, g ) )


class PodModel( Model ):
   __public__ = False
   name = Str( help="Name of the pod" )
   phase = Enum(
      help="Current status of the pod",
      # Phase values statically defined in
      # https://github.com/kubernetes/api/blob/master/core/v1/types.go#L2285
      values=( 'Pending', 'Running', 'Succeeded', 'Failed', 'Unknown' )
   )

class NodeModel( Model ):
   __public__ = False

   name = Str( help="Name of the Kubernetes cluster node" )
   switch = Str( help="Host name for the neighbor switch" )
   switchInterface = Str( help="Name for the neighbor switch interface" )
   hostInterface = Str( help="Name of the host interface" )
   pods = Dict(
      valueType=PodModel,
      help="Pods scheduled to the node",
      optional=True
    )


class NodesModel( Model ):
   __public__ = False

   nodes = Dict(
      valueType=NodeModel,
      help="The nodes that form the cluster",
   )
   podDetails = Bool( help="Indicates presence of pod details", default=False )

   def render( self ):
      if not self.nodes:
         return None
      elif self.podDetails:
         return self.renderPodsView()
      else:
         return self.renderNodesView()

   def renderPodsView( self ):
      if not any( len( n.pods ) for n in self.nodes.values() ):
         return

      headings = (
         'Node Name',
         'Switch',
         'Interface',
         'Host Interface',
         'Pod Name',
         'Pod Status'
      )

      def rows():
         for _, n in sorted( self.nodes.items() ):
            first4 = ( n.name, n.switch, n.switchInterface, n.hostInterface )

            for _, p in sorted( n.pods.items() ):
               yield first4 + ( p.name, p.phase )
               first4 = ( '', '', '', '' )
      print( renderTable( headings, rows() ) )

   def renderNodesView( self ):
      headings = ( 'Node', 'Switch', 'Interface', 'Host Interface' )
      g = (
         ( n.name, n.switch, n.switchInterface, n.hostInterface )
         for _, n in sorted( self.nodes.items() )
      )
      print( renderTable( headings, g ) )
