# Copyright (c) 2019 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import AgentCommandRequest
import AgentDirectory
import BasicCliModes
import CliCommand
import CliMatcher
# pylint: disable-next=consider-using-from-import
import CliPlugin.ControllerCli as ControllerCli
# pylint: disable-next=consider-using-from-import
import CliPlugin.ControllerdbLib as ControllerdbLib
from CliToken.Clear import clearKwNode
import Tac

#--------------------------------------------------------------------------------
# clear cvx connections [ client ( SYSTEMID | HOSTNAME ) ]
#--------------------------------------------------------------------------------
def isControllerOobRunning( mode ):
   return AgentDirectory.agentIsRunning( mode.entityManager.sysname(),
                                         'ControllerOob' )

def runAgentCommand( mode, *args, **kwargs ):
   if isControllerOobRunning( mode ):
      AgentCommandRequest.runSocketCommand( mode.entityManager, *args, **kwargs )

def clearCvxConnection( mode, name, systemId ):
   name = name or systemId.stringValue
   inactivePeers = ControllerCli.inactive()
   if systemId in ControllerdbLib.established().peer:
      mode.addError( 'Active connection cannot be cleared' )
   elif systemId in inactivePeers:
      command = '%s' % systemId.stringValue # pylint: disable=consider-using-f-string
      runAgentCommand( mode, 'controllerOob', 'clearConnection',
            command )
   else:
      # pylint: disable-next=consider-using-f-string
      mode.addError( "Unknown switch: %s" % name )

def clearCvxConnectionSystemId( mode, systemId ):
   systemId = Tac.Value( 'Controller::SystemId', systemId )
   name = None
   peer = ControllerCli.inactive().get( systemId )
   if peer:
      name = peer.connectionConfig.hostname

   clearCvxConnection( mode, name, systemId )

def clearCvxConnections( mode ):
   runAgentCommand( mode, 'controllerOob', 'clearConnection', '' )

def clearCvxConnectionHostname( mode, hostname ):
   for peer in ControllerdbLib.established().peer.values():
      if peer.connectionConfig.hostname == hostname:
         mode.addError( 'Active connection cannot be cleared' )
         return

   for peer in ControllerCli.inactive().values():
      if peer.connectionConfig.hostname == hostname:
         clearCvxConnection( mode, None, peer.systemId )
         return

   # pylint: disable-next=consider-using-f-string
   mode.addError( "Unknown hostname: '%s'" % hostname )

class ClearCvxConnectionsCmd( CliCommand.CliCommandClass ):
   syntax = 'clear cvx connections [ client ( SYSTEMID | HOSTNAME ) ]'
   data = {
      'clear': clearKwNode,
      'cvx': 'Clear controller client connection',
      'connections':
         CliCommand.Node(
            CliMatcher.KeywordMatcher( 'connections',
               helpdesc='Clear connection states of inactive switches' ),
            guard=ControllerdbLib.controllerGuard ),
      'client': ControllerCli.clientMatcher,
      'SYSTEMID': ControllerCli.systemIdMatcher,
      'HOSTNAME': ControllerCli.hostnameMatcher,
   }

   @staticmethod
   def handler( mode, args ):
      systemId = args.get( 'SYSTEMID' )
      hostname = args.get( 'HOSTNAME' )
      if systemId:
         clearCvxConnectionSystemId( mode, systemId )
      elif hostname:
         clearCvxConnectionHostname( mode, hostname )
      else:
         clearCvxConnections( mode )

BasicCliModes.EnableMode.addCommandClass( ClearCvxConnectionsCmd )
