# Copyright (c) 2013 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# pylint: disable=consider-using-f-string

import Ark
from CliPlugin import FileReplicationModels
import TableOutput
import Tac
import re
from CliModel import List
from CliModel import Bool
from CliModel import Str
from CliModel import Model
from CliModel import Float
from CliModel import Dict
from CliModel import Submodel
from CliModel import Enum
from ArnetModel import MacAddress
from ArnetModel import Ip4Address
from ControllerModel import ServiceStatus
from ControllerModel import HeartbeatStatus
from ControllerModel import MountState
from natsort import natsorted
import Toggles.ControllerdbToggleLib as cvxToggle

partialToggle = cvxToggle.toggleCvxPartialServiceCompatibilityEnabled()

SslDisableReason = Tac.Type( "Controller::SslProfileStatus::DisableReason" )
def _createCvxTable( headings, **kwargs ):
   table = TableOutput.createTable( headings, **kwargs )
   f = TableOutput.Format( justify='left' )
   f.noPadLeftIs( True )
   f.padLimitIs( True )
   table.formatColumns( *( [ f ] * len( headings ) ) )
   return table

class SslProfileStatus( Model ):
   sslProfileName = Str( help="SSL profile name" )
   enabled = Bool( help="Whether SSL profile is enabled" )
   disableReason = Enum( help="Reason for disabled SSL profile" ,
                         values=SslDisableReason.attributes, optional=True )

   def disableReasonStr( self ):
      return { 'profileNotExist': 'SSL profile does not exist',
               'profileInvalid': 'Invalid SSL profile',
               'certNotConfigured': 'Certificate not configured in SSL profile',
               'trustedCertNotConfigured': 
               'Trusted certificates not configured in SSL profile',
               'dhparamsNotReady': 
               'Diffie-Hellman parameters not yet ready' } [ self.disableReason ]
   
   def render( self ):
      print( '{}SSL profile: {}'.format( ' ' * 2, self.sslProfileName ) )
      print( '{}Status: {}'.format( ' ' * 4,
            'Enabled' if self.enabled else 'Disabled' ) )
      if not self.enabled: 
         print( '{}Disable Reason: {}'.format( ' ' * 4, self.disableReasonStr() ) )

def overallServiceCompatibilityDesc( overallServiceCompatibility ):
   unknownCompatibilityStatus = 'unknown'
   fullyCompatibleStatus = 'ok'
   notFullyCompatibleStatuses = [ 'partial', 'mismatch' ]

   if partialToggle:
      if overallServiceCompatibility == unknownCompatibilityStatus:
         return 'unknown'
      elif overallServiceCompatibility == fullyCompatibleStatus:
         return 'ok'
      elif overallServiceCompatibility in notFullyCompatibleStatuses:
         return 'not ok'
   else:
      if overallServiceCompatibility == unknownCompatibilityStatus:
         return 'Version unknown'
      elif overallServiceCompatibility == fullyCompatibleStatus:
         return 'Version ok'
      elif overallServiceCompatibility in notFullyCompatibleStatuses:
         return 'Version mismatch'

   return 'Unexpected compatibility status'

OverallServiceCompatibility = \
   Tac.Type( "ControllerCluster::ClusterPeerStatus::VersionStatus" )

# BUG 586422 workaround
overallServiceCompatibilityMask = {
   OverallServiceCompatibility.versionOk: 'ok',
   OverallServiceCompatibility.versionPartial: 'partial',
   OverallServiceCompatibility.versionMismatch: 'mismatch',
   OverallServiceCompatibility.versionUnknown: 'unknown'
}

class PeerStatus( Model ):
   peerName = Str( help="Peer name" )
   peerIp = Ip4Address( help="Peer IP address" )
   peerSystemId = Str( help="Peer Id", optional=True )
   registrationState = Str( help="Peer registration state" )
   registrationErrorStatus = Str( help="Peer registration error description", 
         optional=True )
   if partialToggle:
      serviceVersionStatus = Enum( help="Overall service compatibility for peer",
                                   values=
                                   list( overallServiceCompatibilityMask.values() ) )
      isLeader = Bool( help="Peer is leader" )
   else:
      serviceVersionStatus = Str( help="Peer service version compatibility",
                                  optional=True )

   def render( self ):
      if partialToggle:
         indent = ' ' * 4
         leaderTag = " (leader)" if self.isLeader else ""
         print( f'{indent}Peer Status for {self.peerName}{leaderTag}' )
         indent = indent + ' ' * 2
         if str( self.peerIp ) != self.peerName:
            print( f"{indent}IP address: {self.peerIp}" )
         if self.peerSystemId:
            print( f"{indent}Id: {self.peerSystemId}" )
         print( "{}Registration state: {}".format( indent,
               self.registrationState ) )
         if self.registrationErrorStatus:
            print ( "%sRegistration error description: %s" % 
                    ( indent, self.registrationErrorStatus ) )
         if self.serviceVersionStatus:
            osc = overallServiceCompatibilityDesc( self.serviceVersionStatus )
            print( f"{indent}Service compatibility: {osc}" )
      else:
         indent = ' ' * 4
         print( f'{indent}Peer Status for {self.peerName}' )
         indent = indent + ' ' * 2
         if str( self.peerIp ) != self.peerName:
            print( f"{indent}IP address: {self.peerIp}" )
         if self.peerSystemId:
            print( f"{indent}Id: {self.peerSystemId}" )
         print( "{}Registration state: {}".format( indent,
               self.registrationState ) )
         if self.registrationErrorStatus:
            print ( "%sRegistration error description: %s" % 
                    ( indent, self.registrationErrorStatus ) )
         if self.serviceVersionStatus:
            print ( "%sService compatibility: %s" % 
                    ( indent, self.serviceVersionStatus ) )

class PeerStatusDebug( PeerStatus ):
   sshKeyCopyStatus = Str( help="Ssh key copy status" )

   def render( self ):
      if partialToggle:
         indent = ' ' * 2
         leaderTag = " (leader)" if self.isLeader else ""
         print( f'{indent}Peer Status for {self.peerName}{leaderTag}' )
         indent = indent + ' ' * 2
         if self.peerSystemId:
            print( f"{indent}Id: {self.peerSystemId}" )
         print( f"{indent}Registration state: {self.registrationState}" )
         print( f"{indent}Ssh key copy status: {self.sshKeyCopyStatus}" )
         if self.registrationErrorStatus:
            print ( "%sRegistration error description: %s" % 
                  ( indent, self.registrationErrorStatus ) )
         if self.serviceVersionStatus:
            osc = overallServiceCompatibilityDesc( self.serviceVersionStatus )
            print( f"{indent}Service compatibility: {osc}" )
      else:
         indent = ' ' * 2
         print( f'{indent}Peer Status for {self.peerName}' )
         indent = indent + ' ' * 2
         if self.peerSystemId:
            print( f"{indent}Id: {self.peerSystemId}" )
         print( f"{indent}Registration state: {self.registrationState}" )
         print( f"{indent}Ssh key copy status: {self.sshKeyCopyStatus}" )
         if self.registrationErrorStatus:
            print ( "%sRegistration error description: %s" % 
                    ( indent, self.registrationErrorStatus ) )
         if self.serviceVersionStatus:
            print ( "%sService compatibility: %s" % 
                    ( indent, self.serviceVersionStatus ) )

class ClusterStatus( Model ):
   clusterName = Str( help="Name" )
   role = Enum( values=( "Disconnected", "Master", "Standby" ),
                help='Controller Role' )
   currLeader = Str( help="Leader", optional=True )
   peerTimeout = Float( help="Heartbeat timeout in seconds for intra cluster "
                        "communication" )
   peerStatus = Dict( valueType=PeerStatus, help="Cluster's peer status" ) 
   lastLeaderChangeTs = Float( help="Cluster's last leadership change timestamp" )

   def render( self ):
      indent = ' ' * 2
      print( '%sCluster Status' % indent )
      indent = indent + ' ' * 2
      print( f"{indent}Name: {self.clusterName}" )
      print( f"{indent}Role: {self.role}" )
      if self.currLeader and not partialToggle:
         print( f"{indent}Leader: {str( self.currLeader )}" )
      print( f"{indent}Peer timeout: {self.peerTimeout}" )
      print( "{}Last leader switchover timestamp: {}".format(
         indent, Ark.timestampToStr( self.lastLeaderChangeTs, now=Tac.utcNow() ) ) )

      if self.peerStatus:
         for peerName in self.peerStatus:
            self.peerStatus[ peerName ].render()

class ClusterStatusDebug( ClusterStatus ):
   currTerm = Str( help="Cluster's current term" )
   currState = Str( help="Election State" )
   lastLeaderChangeTerm = Str( help="Cluster's last leadership change term" )

   def render( self ):
      indent = ' ' * 2
      print( 'Cluster Status' )
      indent = indent + ' ' * 2
      print( f"{indent}Name: {self.clusterName}" )
      print( f"{indent}Role: {self.role}" )
      if self.currLeader and not partialToggle:
         print( f"{indent}Leader: {str( self.currLeader )}" )
      print( f"{indent}Election State: {self.currState}" )
      print( f"{indent}Current Term: {self.currTerm}" )
      print( f"{indent}Peer timeout: {self.peerTimeout}" )
      print( "{}Last leader switchover timestamp: {}".format( indent,
            self.lastLeaderChangeTs ) )
      print( "{}Last leader switchover term: {}".format( indent,
            self.lastLeaderChangeTerm ) )
      if self.peerStatus:
         for peerName in self.peerStatus:
            self.peerStatus[ peerName ].render()

class CvxStatus( Model ):
   __revision__ = 2
   controllerUUID = Str( 'Controller UUID' )
   enabled = Bool( 'Status of the CVX server' )
   clusterMode = Bool( 'CVX server in cluster mode' )
   heartbeatInterval = Float( 'Heartbeat interval in seconds' )
   heartbeatTimeout = Float( 'Heartbeat timeout in seconds' )
   clusterStatus = Submodel( help="Cluster status", 
         valueType=ClusterStatus, optional=True )
   sslProfileStatus = Submodel( help="SSL profile status", 
                                valueType=SslProfileStatus, optional=True )
   connectionPreserved = Bool( 'Client connection state preserving is enabled' )

   def render( self ):
      print( 'CVX Server' )
      # Uncomment when we have controller redundancy.
      print( '  Status: %s' % ( 'Enabled' if self.enabled else 'Disabled' ) )
      if self.enabled:
         print( '  UUID: %s' % self.controllerUUID )
      print( '  Mode: %s' % ( 'Cluster' if self.clusterMode else 'Standalone' ) )
      print( '  Heartbeat interval: %s' % self.heartbeatInterval )
      print( '  Heartbeat timeout: %s' % self.heartbeatTimeout )
      print( '  Client connection state preserving: %s' % (
            'Enabled' if self.connectionPreserved else 'Disabled' ) )
      if self.clusterStatus:
         self.clusterStatus.render()

      if self.sslProfileStatus:
         self.sslProfileStatus.render()

   def degrade( self, dictRepr, revision ):
      if revision == 1:
         dictRepr[ 'role' ] = 'master'
         del dictRepr[ 'clusterMode' ]
      return dictRepr

class GroupMountStates( Model ):
   group = Str( help='Group' )
   state = Str( help='Publish state of group' )
   pathStates = List( valueType=MountState,
                      help='Publish states of paths within a group' )
   systemWhitelist = List( valueType=str,
                           help='System whitelist' )
   versionWhitelist = List( valueType=int,
                           help='Version whitelist' )

   def display( self, indent=2 ):
      print( '{}Group \"{}\": ({}) :'.format( ( indent * ' ' ), self.group, 
                                                                self.state ) )
      indent2 = indent + 2

      systemWhitelist = 'wildcard' if not self.systemWhitelist else \
         ', '.join( sorted( self.systemWhitelist ) )
      print( '{}System Whitelist: {}'.format( ( indent * ' ' ), systemWhitelist ) )
      
      versionWhitelist = 'wildcard' if not self.versionWhitelist else \
         ', '.join( str( entry ) for entry in sorted( self.versionWhitelist ) )
      print( '{}Version Whitelist: {}'.format( ( indent * ' ' ), versionWhitelist ) )

      headings = ( 'MountPath', 'MountType', 'MountState' )
      table = _createCvxTable( headings, indent=indent2)
      for state in self.pathStates:
         mountPath = state.path
         mountType = state.type
         mountState = state.state
         table.newRow( mountPath, mountType, mountState )
      print( table.output() )

class ServiceMountStates( Model ):
   service = Str( help='Service.' )
   mountStates = List( valueType=GroupMountStates,
                       help='States of groups of mounts' )

   def display( self, indent=2 ):
      print( '{}{}:'.format( ( indent * ' ' ) ,
                                self.service ) )
      indent2 = indent + 2
      for group in self.mountStates:
         group.display( indent2 )

class CvxMounts( Model ):
   switchId = MacAddress( help='Switch Id' )
   hostname = Str( help='Switch hostname' )
   mounts = List( valueType=ServiceMountStates,
                  help='CVX service mount states' )

   def render( self ):
      name = self.hostname or str( self.switchId )
      print( 'Switch:', name )
      for service in self.mounts:
         service.display( indent=2 )
 
class CvxAllMounts( Model ):
   connections = List( valueType=CvxMounts, help='Cvx mounts for all connections' )

   def render( self ):
      for connection in self.connections:
         connection.render()

# pylint: disable-next=inconsistent-return-statements
def degradeCvxConnection( dictRepr, revision ):
   if revision == 1:
      # The 'offset' field of HeartbeatStatus is removed in rev2.
      # Add this field back for rev1.
      hbStatus = dictRepr.get( 'heartbeatStatus' )
      if hbStatus:
         hbStatus[ 'offset' ] = 0
      dictRepr[ 'status' ] = True
      if dictRepr.get( 'oobState' ):
         del dictRepr[ 'oobState' ]
      return dictRepr

class CvxConnection( Model ):
   __revision__ = 2
   switchId = MacAddress( help='Switch Id' )
   hostname = Str( help='Switch hostname' )
   oobState = Enum( values=( 'connecting',
                             'negotiating',
                             'established',
                             'disconnecting' ),
                    help='Out-of-band state' )
   oobConnectionActive = Bool( help='Connection state' )
   heartbeatStatus = Submodel( valueType=HeartbeatStatus,
                               help='Heartbeat status of the switch' )
   connectionTime = Float( help='Connection establishment UTC timestamp',
                           optional=True )
   oobConnectionSecured = Bool( help="Whether out-of-band connection is SSL secured",
                                optional=True )
   ibConnectionSecured = Bool( help="Whether in-band connection is SSL secured",
                               optional=True )

   def connectionSecuredStr( self, secured ):
      retStr = "SSL secured" if secured else "Not secured"
      return retStr

   def connectionStatusStr( self, connectionStatus ):
      return 'active' if connectionStatus else 'inactive'

   def render( self ):
      print( 'Switch %s' % self.switchId )
      print( '  Hostname: %s' % self.hostname )
      print( '  State: %s' % self.oobState )
      print( '  Connection Status: %s' % self.connectionStatusStr(
                                        self.oobConnectionActive ) )
      if self.connectionTime:
         print( '  Connection timestamp: %s' % Ark.timestampToStr(
            self.connectionTime, now=Tac.utcNow() ) )
      if self.heartbeatStatus:
         self.heartbeatStatus.display( indent=' ' * 2 )
      if self.oobConnectionSecured is not None:
         print( '  Out-of-band connection: %s' % self.connectionSecuredStr( 
                                                self.oobConnectionSecured ) )
      if self.ibConnectionSecured is not None:
         print( '  In-band connection: %s' % self.connectionSecuredStr( 
                                            self.ibConnectionSecured ) )
   
   def degrade( self, dictRepr, revision ):
      return degradeCvxConnection( dictRepr, revision )

class CvxAllConnections( Model ):
   __revision__ = 2
   connections = List( valueType=CvxConnection, help='Cvx connections' )
   brief = Bool( help='Show brief status only' )

   def render( self ):
      if self.brief:
         table = TableOutput.createTable( [ ("CVX Connection Table", 'hc',
                                             ('Switch', 'Hostname', 'Status') ) ] )
         fmt = TableOutput.Format( justify='left' )
         fmt.noPadLeftIs( True )
         fmt.padLimitIs( True )
         table.formatColumns( fmt, fmt, fmt )
         for c in self.connections:
            statusStr = c.connectionStatusStr( c.oobConnectionActive )
            table.newRow( c.switchId, c.hostname, statusStr )
         print( table.output() )
      else:
         for connection in self.connections:
            connection.render()

   def degrade( self, dictRepr, revision ):
      degradedList = []
      for connection in dictRepr.get( "connections" ):
         degradedConn = degradeCvxConnection( connection, revision )
         degradedList.append(degradedConn)
      dictRepr[ 'connections' ] = degradedList
      return dictRepr

class CvxService( Model ):
   __revision__ = 2
   name = Str( help='Service name' )
   status = Bool( help='Service status' )
   versions = List( valueType=int, help='Supported versions' )
   switches = Dict( valueType=ServiceStatus,
                    help='Status of all the switches running the service' )
   
   def render( self ):
      print( self.name )
      print( '  Status: %s' % ( 'Enabled' if self.status else 'Disabled' ) )
      print( '  Supported versions: %s' %
             ', '.join( sorted( map( str, self.versions ) ) ) )
      print()
      headings = ( 'Switch', 'Status', 'Negotiated Version' )
      table = _createCvxTable( headings, indent=2 )
      for switch in self.switches:
         serviceStatus = self.switches[ switch ]
         table.newRow( switch, serviceStatus.statusString(),
                       serviceStatus.versionString() )
      print( table.output() )

   def degrade( self, dictRepr, revision ):
      if revision == 1:
         for switch in list( dictRepr[ 'switches' ] ):
            # Remove all switches that have hostnames
            if not re.match( r"([a-f0-9]{2}[:-]){5}", switch ):
               del dictRepr[ 'switches' ][ switch ]
      return dictRepr

class CvxAllServices( Model ):
   __revision__ = 2
   services = List( valueType=CvxService,
                    help='Status of all services running on the controller' )

   def render( self ):
      for service in sorted( self.services, key=lambda s: s.name ):
         service.render()

   def degrade( self, dictRepr, revision ):
      if revision == 1:
         degradedServices = []
         for serviceRepr in dictRepr[ 'services' ]:
            for service in self.services:
               if service.name == serviceRepr[ 'name' ]:
                  degradedServices.append( service.degrade( serviceRepr, revision ) )
         dictRepr[ 'services' ] = degradedServices
      return dictRepr

class ServiceFileReplicationStatus( Model ):
   services = Dict( keyType=str, valueType=FileReplicationModels.Status,
                    help=( 'A mapping from service name to sets of '
                           'replicated files' ) )
   
   def render( self ):
      for serviceName in sorted( self.services.keys() ):
         print( 'Service %s' % serviceName )
         # pylint: disable-msg=W0212
         self.services[ serviceName ]._render( indentLevel=1 )

IndividualServiceCompatibility = \
   Tac.Type( "ControllerCluster::ServiceCompatibility::Compatibility" )
class CvxPeerServiceCompatibility( Model ):
   compatibility = Enum( help="Service compatibility status",
                         values=IndividualServiceCompatibility.attributes )
   details = Str( help="Service compatibility details", optional=True )

   def cliCompatibility( self ):
      return { 'unknownLocal' : 'unavailable',
         'unknownPeer' : 'unavailable',
         'disabledLocal': 'config mismatch',
         'disabledPeer' : 'config mismatch',
         'disabledBoth' : 'disabled',
         'versionMismatch' : 'version mismatch',
         'compatible': 'ok' } [ self.compatibility ]

   def display( self, serviceName ):
      description = ""
      if self.compatibility == 'disabledBoth':
         description = self.details
      else:
         description = self.cliCompatibility()
         details = " (" + self.details + ")" if self.details else ""
         description += details

      print( f"{serviceName}: {description}" )

class CvxPeerServiceStatus( Model ):
   isLeader = Bool( help="Peer is leader" )
   overallCompatibility = Enum( help="Overall service compatibility for peer",
                           values=list( overallServiceCompatibilityMask.values() ) )
   services = Dict( help="A mapping of service name to its "
                                      "compatibility information",
                                 valueType=CvxPeerServiceCompatibility )

   def display( self, peerName ):
      print ( "Service status for {}{}".format( peerName, 
                                            " (leader)" if self.isLeader else "" ) )
      osc = overallServiceCompatibilityDesc( self.overallCompatibility )
      print( "Compatibility: %s" % ( osc ) )
      for serviceName in natsorted( self.services ):
         self.services[ serviceName ].display( serviceName )

class CvxClusterServiceStatus( Model ):
   clusterPeers = Dict( valueType=CvxPeerServiceStatus,
      help="A mapping of peer name to its service status" )

   def render( self ):
      if self.clusterPeers:
         for peerName in natsorted( self.clusterPeers ):
            self.clusterPeers[ peerName ].display( peerName )
            print( "" )
