# Copyright (c) 2020 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

#-------------------------------------------------------------------------------
# This module implements the Cpu interface type.  In particular, it provides
# the CpuIntf class.
#-------------------------------------------------------------------------------
'''Commands supported on a Cpu interface'''

import Cell
import CliParser
from CliPlugin import EthIntfCli
from CliPlugin import IntfCli
from CliPlugin.IntfStatusMatcher import IntfStatusMatcher
import LazyMount
import CliMatcher
from IntfRangePlugin.CpuIntf import CpuAutoIntfType
import Intf.IntfRange as IntfRange # pylint: disable=consider-using-from-import
import Tac
from os import getenv

cpuPhyIntfStatusDir = None
cpuPhyIntfStatusPath = 'interface/status/eth/phy/all'

def valueFunc_( mode, intf ):
   return CpuIntf( intf, mode )

def allCpuIntfStatus( mode ):
   intfs = [ intf for intf in cpuPhyIntfStatusDir.intfStatus
             if intf.startswith( 'Cpu' ) ]
   return intfs

def cpuIntfGuard( mode, token ):
   if getenv( "SIMULATION_MAKO_PRODUCT" ):
      return None
   if CpuIntf.getAllPhysical( mode ):
      return None
   return CliParser.guardNotThisPlatform

defaultMatcher = CliMatcher.IntegerMatcher( 1, 4, helpdesc='CPU interface number' )

#-------------------------------------------------------------------------------
# A subclass of the base IntfCli.Intf class for Cpu interfaces.
#-------------------------------------------------------------------------------
class CpuIntf( EthIntfCli.EthPhyIntf ):

   #----------------------------------------------------------------------------
   # Creates a new CpuIntf instance of the specified name.
   #----------------------------------------------------------------------------
   def __init__( self, name, mode ):
      EthIntfCli.EthPhyIntf.__init__( self, name, mode )

   matcher = IntfStatusMatcher( 'Cpu', allCpuIntfStatus, defaultMatcher,
                                helpdesc='CPU interface', value=valueFunc_,
                                guard=cpuIntfGuard )

   def sliceName( self, fixedSystemCardSlotId=False ):
      return str( Cell.cellId() )

   #----------------------------------------------------------------------------
   # Returns the EthIntfXcvrStatus object for this interface.
   #----------------------------------------------------------------------------
   def intfXcvrStatus( self ):
      dummyEixs = Tac.newInstance( 'Interface::EthIntfXcvrStatus', '' )
      if self.status():
         dummyEixs.xcvrPresence = 'xcvrPresent'
      return dummyEixs

   #----------------------------------------------------------------------------
   # Outputs information about this interface in an interface type-specific
   # manner.
   #----------------------------------------------------------------------------
   def countersSupported( self ):
      return True

   def routingSupported( self ):
      return True

   def switchPortEligible( self ):
      return False

   def vrfSupported( self ):
      return True

   def xcvrTypeStr( self ):
      return "N/A"

   def flowcontrolRxPause( self ):
      return 0

   def flowcontrolTxPause( self ):
      return 0

   def hardware( self ):
      return "cpu"

   def linkModeUnsupportedErrorMsg( self ):
      # pylint: disable-next=consider-using-f-string
      return ( 'Speed and duplex settings are not available on interface %s.'
                % self.name )

   def internal( self ):
      return True

   #----------------------------------------------------------------------------
   # Returns an unsorted list of CpuIntf objects representing all the existing
   # Interface::CpuIntfStatus objects.
   #----------------------------------------------------------------------------
   @staticmethod
   def getAllPhysical( mode ):
      intfs = []
      for name in cpuPhyIntfStatusDir.intfStatus:
         if not name.startswith( 'Cpu' ):
            continue
         intf = CpuIntf( name, mode )
         if intf.lookupPhysical():
            intfs.append( intf )
      return intfs

#-------------------------------------------------------------------------------
# Register the CpuIntf class as a type of physical interface.
#-------------------------------------------------------------------------------
IntfCli.Intf.addPhysicalIntfType( CpuIntf, CpuAutoIntfType )
CpuAutoIntfType.registerEthPhyIntfClass( CpuIntf )

IntfRange.registerIntfTypeGuard( CpuAutoIntfType, cpuIntfGuard )

#-------------------------------------------------------------------------------
# Have the Cli Agent mount all needed state from sysdb
#-------------------------------------------------------------------------------
def Plugin( entityManager ):
   global cpuPhyIntfStatusDir

   cpuPhyIntfStatusDir = LazyMount.mount( entityManager, cpuPhyIntfStatusPath,
                                          "Interface::AllEthPhyIntfStatusDir", "r" )
