#!/usr/bin/env python3
# Copyright (c) 2017 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from CliModel import Model, Submodel, Dict
from CliPlugin import QueueCountersModel

class IngressCpuPortCounters( Model ):
   # 'ports' identifies ingress port where packets received. Valid keys include
   # 'Portxx', 'Ethernetxx', and 'All'.
   ports = Dict( keyType=str,
                 valueType=QueueCountersModel.Counters,
                 help="Mapping between ingress port and corresponding counter data" )

class IngressCpuClassCounters( Model ):
   # 'cpuClasses' represent the CPU queue, e.g., CpuQueueCfmSnoop.
   cpuClasses = Dict( keyType=str,
                      valueType=IngressCpuPortCounters,
                      help="Mapping between CPU classes and ingress port" )

class IngressCpuCounters( Model ):
   # 'sources' (e.g., FAP or core) refers to where packets are ingressing from
   # the CPU.
   sources = Dict( keyType=str, valueType=IngressCpuClassCounters,
                   help="Mapping between source and CPU class" )

class EgressCpuQueueCounters( Model ):
   # 'queues' represents the CPU queue, e.g., Sflow. This level of hierarchy
   # currently is only used by Strata. Sand currently specifies '0' for each of its
   # queues.
   queues = Dict( keyType=str,
                  valueType=QueueCountersModel.Counters,
                  help="Mapping between CPU queue and counter data" )

class EgressCpuDestinationTypeCounters( Model ):
   ucastQueues = Submodel( valueType=EgressCpuQueueCounters,
                           help="Unicast CPU traffic class counters",
                           optional=True )
   mcastQueues = Submodel( valueType=EgressCpuQueueCounters,
                           help="Multicast CPU traffic class counters" )

class EgressCpuPortCounters( Model ):
   # 'cpuPorts' represents the CPU port, e.g., CpuTm0. This level of hierarchy
   # currently is only used by Sand. Strata specifies 'CPU' for each of its queues.
   cpuPorts = Dict( keyType=str,
                    valueType=EgressCpuDestinationTypeCounters,
                    help="Mapping between CPU ports and queues" )

class EgressCpuCounters( Model ):
   sources = Dict( keyType=str, valueType=EgressCpuPortCounters,
                   help="Mapping between sources and CPU queue counters" )

class CpuQueueCounters( Model ):
   ingressVoqs = Submodel( valueType=IngressCpuCounters,
                           help="CPU VOQ counters on the ingress source",
                           optional=True )

   egressQueues = Submodel( valueType=EgressCpuCounters,
                            help="CPU queue counters on the egress chip" )

