#!/usr/bin/env python3
# Copyright (c) 2018 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import CliMatcher
import CliCommand
import ConfigMount
from CliPlugin.TeCli import (
      RouterGlobalTeModelet,
      TeModeDependents,
      TeDependentBase,
)
from TypeFuture import TacLazyType

CspfIntervalMsec = TacLazyType( 'Cspf::CspfIntervalMsec' )
CspfThrottleTimer = TacLazyType( 'Cspf::CspfThrottleTimer' )

config = None

#---------------------------------------------------------------------------------
# [no|default] cspf delay [initial <initial-interval> back-off <back-off-interval>]
# max <max-interval> command in config-te mode
#---------------------------------------------------------------------------------
intervalMatcher = CliMatcher.IntegerMatcher( CspfIntervalMsec.min,
      CspfIntervalMsec.max, helpdesc='Time in milliseconds' )

class CspfDelay( TeDependentBase ):
   '''
   This gets registerd in Te so it can clear the config.
   We've given it default `args` so it can be used as
   the no/def handler for `CspfDelayCmd`.
   '''
   def setDefault( self, args=None ):
      config.cspfTimer = config.cspfThrottleTimerDefault

TeModeDependents.registerDependentClass( CspfDelay )

class CspfDelayCmd( CliCommand.CliCommandClass ):
   syntax = ( 'cspf delay [ initial INITIAL_INTERVAL back-off BACK_OFF_INTERVAL ] '
              'max MAX_INTERVAL' )
   noOrDefaultSyntax = 'cspf delay ...'
   data = {
            'cspf' : 'Configure CSPF',
            'delay' : 'Configure CSPF throttle timers',
            'initial' : 'Specify initial wait interval for CSPF',
            'INITIAL_INTERVAL' : intervalMatcher,
            'back-off' : 'Specify back-off interval for CSPF',
            'BACK_OFF_INTERVAL' : intervalMatcher,
            'max' : 'Specify maximum wait interval for CSPF',
            'MAX_INTERVAL' : intervalMatcher
          }

   @staticmethod
   def handler( mode, args ):
      initialInterval = args.get( 'INITIAL_INTERVAL', config.cspfInitialIntDefault )
      backOffInterval = args.get( 'BACK_OFF_INTERVAL', config.cspfBackOffIntDefault )
      maxInterval = args[ 'MAX_INTERVAL' ]

      if initialInterval > maxInterval:
         mode.addError(
               'CSPF initial wait interval cannot exceed max wait interval' )
         return
      if backOffInterval > maxInterval:
         mode.addError( 'CSPF back-off interval cannot exceed max wait interval' )
         return

      config.cspfTimer = CspfThrottleTimer( initialInterval, backOffInterval,
                                            maxInterval )

   noOrDefaultHandler = CspfDelay.setDefault

RouterGlobalTeModelet.addCommandClass( CspfDelayCmd )

def Plugin( entityManager ):
   global config

   config = ConfigMount.mount( entityManager, "te/cspf/cliConfig",
                               "Cspf::CspfCliConfig", 'w' )
