#!/usr/bin/env python3
# Copyright (c) 2021 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# pylint: disable=consider-using-f-string

from CliModel import Model, Enum, List, Int, Str, Submodel, Bool, Dict
from IntfModels import Interface
from CliPlugin.IsisCliModels import bw_best_value_units
from CliPlugin.TeCli import adminGroupAndNameListToStr, adminGroupDictToDecimalList
from Toggles import TeToggleLib
from Arnet import sortIntf

class TeDelay( Model ):
   style = Enum( values=( 'static', 'twamp', 'dynamic' ),
                 help="Method for determining the delay" )
   dynamicDelayType = Enum( values=( 'fallback', 'measured' ),
                            help="Dynamic min delay value category",
                      optional=True )
   value = Int( help="Interface delay" )
   units = Enum( values=( 'milliseconds', 'microseconds' ), help="Delay units" )

   def render( self ):
      if self.value:
         if self.style == 'static':
            print( f"Minimum delay ({self.style}): {self.value} {self.units}" )
         else:
            print( f"Minimum delay ({self.style}): {self.value} {self.units} "
                   f"({self.dynamicDelayType})" )

class IntfSrlg( Model ):
   gid = Int( help="Shared Risk Link Group identifier" )
   name = Str( help='Shared Risk Link Group name' )

class AdjSid( Model ):
   index = Int( help="Index of the Segment Identifier" )
   isValue = Bool( help="Index is a value" )
   isIpV6 = Bool( help="SID is IPv6" )

class NodeSid( Model ):
   index = Int( help="Index of the Segment Identifier" )
   isIpV6 = Bool( help="SID is IPv6" )
   flexAlgoName = Str( help="Algorithm Name" )

class TeMaxResvBw( Model ):
   bwUnitType = Enum( values=( 'percent', 'mbps', 'gbps' ), help="Bandwidth units" )
   value = Int( help="Bandwidth" )

   def render( self ):
      if self.value:
         print( "Max reservable bandwidth:", self.value, self.bwUnitType )

class TeIntfModel( Model ):
   intf = Interface( help="Interface" )
   adjSids = List( valueType=AdjSid, help="List of Adjacency SIDs" )
   nodeSids = List( valueType=NodeSid, help="List of Node SIDs" )
   adminGroup = Int( help="Administrative groups" )
   adminGroupsExtended = List( valueType=int, help="Extended administrative groups" )
   adminGroupNames = List( valueType=str, help="List of Administrative group names" )
   delay = Submodel( valueType=TeDelay, help="Delay parameters" )
   srlg = List( valueType=IntfSrlg, help="List of Shared Risk Link Groups" )
   maxResvBw = Submodel( valueType=TeMaxResvBw, help="Max reservable bandwidth" )
   metric = Int( help="Traffic engineering metric" )

   def render( self ):
      print( "Interface %s:" % self.intf.stringValue )

      if self.adjSids:
         sidStrs = []
         for sid in self.adjSids:
            sidStr = str( sid.index ) + ' ['
            if sid.isIpV6:
               sidStr += 'F'
               if sid.isValue:
                  sidStr += ',V'
            elif sid.isValue:
               sidStr += 'V'
            sidStr += ']'
            sidStrs.append( sidStr )
         if sidStrs:
            print( "Adjacency SIDs:", ', '.join( sidStrs ) )

      if self.nodeSids:
         sidStrs = []
         for sid in self.nodeSids:
            sidStr = str( sid.index ) + ' ['
            if sid.isIpV6:
               sidStr += 'F'
            sidStr += ']'
            if sid.flexAlgoName:
               sidStr += ' algorithm: ' + sid.flexAlgoName
            sidStrs.append( sidStr )
         if sidStrs:
            print( "Node SIDs:", ', '.join( sidStrs ) )

      if self.maxResvBw:
         self.maxResvBw.render()
      if self.metric:
         print( "Traffic engineering metric:", self.metric )
      if self.adminGroup or self.adminGroupsExtended or self.adminGroupNames:
         if TeToggleLib.toggleExtendedAdminGroupEnabled():
            adminGroupListStr = adminGroupAndNameListToStr(
                  self.adminGroupsExtended, self.adminGroupNames )
         else:
            adminGroupList = adminGroupDictToDecimalList( { 0 : self.adminGroup } )
            adminGroupListStr = adminGroupAndNameListToStr( adminGroupList,
                                                            self.adminGroupNames )
         print( "Administrative groups:", adminGroupListStr )
      srlgs = ""
      first = True
      for srlg in self.srlg:
         if not first:
            srlgs += ', '
         first = False
         if srlg.name:
            srlgs += srlg.name
         else:
            srlgs += str( srlg.gid )
      if srlgs:
         print( "Shared Risk Link Groups:", srlgs )

      self.delay.render()

class TeIntfListModel( Model ):
   interfaces = List( valueType=TeIntfModel, help="List of interfaces" )

   def render( self ):
      for intf in self.interfaces:
         intf.render()
         print()

class AdvertisedBwOfPriorityClasses( Model ):
   bandwidths = Dict( keyType=int, valueType=int,
            help="Advertised unreserved bandwidth for each priotiy in bps" )

   def render( self ):
      indent_str = 4 * " "
      grpSize = 3
      info = ""
      count = 0
      for clsname, advBw in self.bandwidths.items():
         bw_value, units = bw_best_value_units( advBw )
         info += ( indent_str * 2 + "TE class " + "%s" % clsname + ":"
                  + "%s" % bw_value + units )
         count += 1
         if count % grpSize == 0:
            print( info )
            info = ""
      if info:
         print( info )

class BandwidthAdByIgp( Model ):
   instanceName = Str( help="IGP Instance name" )
   _igpname = Str( help="IGP Name (IS-IS/OSPF)" )
   bandwidthsInEachLevelOrArea = Dict( keyType=int,
         valueType=AdvertisedBwOfPriorityClasses,
         help=" Dictionary of advertised unreserved bandwidth "
              "in each IGP level or area" )

   def render( self ):
      indent_str = 4 * " "
      for levelId, igpAdvBw in self.bandwidthsInEachLevelOrArea.items():
         print( indent_str, "Last advertised unreserved BW:" )
         print( indent_str * 2 + "IGP name: ", self._igpname,
            ", instance: ", self.instanceName, ", level ", levelId )
         igpAdvBw.render( )

class TeInterfaceBwModel( Model ):
   maxBandwidth = Int( help="Interface maximum bandwidth in bps" )
   maxReservableBandwidth = Int( help="Interface maximum reservable"
                                 " bandwidth in bps" )
   unreservedBandwidths = Dict( keyType=int, valueType=int,
         help="Current unreserved bandwidth for each priority" )
   advertisedUnreservedBandwidths = Dict( keyType=str,
         valueType=BandwidthAdByIgp,
         help="Dictionary of advertised unreserved bandwidth by IGP" )

   def render( self ):
      indent_str = "    "
      grpSize = 3
      bw_value, units = bw_best_value_units( self.maxBandwidth )
      print( indent_str, "Maximum link BW:", bw_value, units )
      bw_value, units = bw_best_value_units( self.maxReservableBandwidth )
      print( indent_str, "Maximum reservable link BW:", bw_value, units )
      print( indent_str, "Current unreserved BW:" )

      info = ""
      count = 0
      for clsname, prioBw in self.unreservedBandwidths.items():
         bw_value, units = bw_best_value_units( prioBw )
         info += ( indent_str * 2 + "TE class " + "%s" % clsname + ":"
                  + "%s" % bw_value + units )
         count += 1
         if count % grpSize == 0:
            print( info )
            info = ""
      if info != "":
         print( info )

      for igpAdvBwList in self.advertisedUnreservedBandwidths.values():
         igpAdvBwList.render()

class TeInterfaceBwStatsModel( Model ):
   threshold = Int( help="IGP unreserved bandwidth "
                     "update threshold in percentage" )
   interfaces = Dict( keyType=Interface, valueType=TeInterfaceBwModel,
                  help="Dictionary of interface bandwidth statistics" )

   def render( self ):
      if self.threshold >= 0:
         print( "IGP BW update threshold:", self.threshold, "% unreserved" )
      else:
         print( "IGP BW update threshold:" )

      if self.interfaces:
         for intf in sortIntf( self.interfaces ):
            print( "Interface %s:" % intf )
            intfBwModel = self.interfaces[ intf ]
            intfBwModel.render( )
