#!/usr/bin/env python3
# Copyright (c) 2018 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# pylint: disable=too-many-nested-blocks
# pylint: disable=protected-access
# pylint: disable=relative-beyond-top-level

#-------------------------------------------------------------------------------
# This module implements Cspf show commands
#-------------------------------------------------------------------------------

from collections import defaultdict
import Tac
import Tracing
import SharedMem
import Smash
import SmashLazyMount
import SharkLazyMount
import BasicCli
import CliMatcher
import CliCommand
import CliModel
import LazyMount
import ShowCommand
import ipaddress
from CliPlugin.RoutingIsisCli import getHostName, isisKw, getFlexAlgoName
from CliPlugin.IsisCliModels import getRouterId
from CliPlugin.RoutingOspfCli import matcherOspfShow
from CliPlugin.IpAddrMatcher import IpAddrMatcher
from CliPlugin.TeCli import ( getSrlgIdToNameMap,
                              teConfiguration,
                              adminGroupDictToDecimalList )
import CliToken.Ip
from AgentCommandRequest import runCliPrintSocketCommand
from .CspfShowPathModel import PATH_NOT_FOUND_REASON_TO_TYPE_MAP, CspfPathLinksModel
from .CspfShowPathModel import CspfPathModel
from .CspfShowPathModel import CspfPathVrfModel
from .CspfShowPathModel import CspfPathAfModel
from .CspfShowPathModel import CspfPathDestIpModel
from .CspfShowPathModel import CspfPathEntryModel
from .CspfShowPathModel import CspfPathConstraintModel
from .CspfShowPathModel import IntfSrlg, SrlgIdToNameMapHelper
from .CspfShowPathModel import TilfaPathEntryModel, TilfaPathDestModel
from .CspfShowPathModel import TilfaPathAfModel, TilfaPathVrfModel
from .CspfShowPathModel import TilfaPathModel, SystemIdHostnameModel
from .CspfShowPathModel import TilfaPathTopoIdModel, TilfaPathDetails
from .CspfShowPathModel import TilfaPathConstraintModel, FlexAlgoPathConstraintModel
from .CspfShowPathModel import ( RouterIdHostnameModel,
                                 TilfaPathOspfAfModel,
                                 TilfaPathOspfDestModel,
                                 TilfaPathOspfEntryModel,
                                 TilfaPathOspfModel,
                                 TilfaPathOspfTopoIdModel,
                                 TilfaPathOspfVrfModel )
from .CspfShowPathModel import CspfPathHelperModel
from .CspfShowPathModel import CspfPathHopModel
from .CspfShowPathModel import CspfExplicitHopModel
from .CspfShowPathModel import CspfIncludeHopModel
from .CspfShowPathModel import CspfPathByIdLinksModel
from .CspfShowPathModel import CspfPathByIdModel
from .CspfShowPathModel import CspfPathByIdVrfModel
from .CspfShowPathModel import CspfPathByIdEntryModel
from .CspfShowPathModel import CspfPathP2mpModel
from .CspfShowPathModel import CspfPathVrfP2mpModel
from .CspfShowPathModel import CspfPathAfP2mpModel
from .CspfShowPathModel import CspfPathTreeModel
from .CspfShowPathModel import CspfPathTreeConstraintModel
from .CspfShowPathModel import CspfLeafPathModel
from .CspfShowPathModel import CspfLeafPathConstraintModel
from .CspfShowPathModel import PathNotFoundReasonModel
from .CspfShowPathModel import PathNotFoundReasonType

from TypeFuture import TacLazyType
from Toggles.gatedToggleLib import toggleOspfTilfaEnabled
from Toggles.TunnelToggleLib import toggleMvpnRsvpP2mpEnabled
from Toggles.CspfToggleLib import toggleCspfTunnelNamesEnabled
from Toggles import TeToggleLib

MetricType = TacLazyType( "Cspf::MetricType" )

__defaultTraceHandle__ = Tracing.Handle( 'CspfCli' )
t0 = Tracing.trace0

matcherDetail = CliMatcher.KeywordMatcher( 'detail',
                helpdesc='Show detailed path information' )
matcherTrafficEngineering = CliMatcher.KeywordMatcher( 'traffic-engineering',
                            helpdesc='Traffic Engineering related information' )
matcherPathId = CliMatcher.IntegerMatcher( 0, 4294967295,
                                           helpdesc='CSPF Path ID' )
matcherTreeId = CliMatcher.IntegerMatcher( 0, 4294967295,
                                           helpdesc='CSPF Tree ID' )
entityManager = None
flexAlgoConfig = None
clientDir = None
AddressFamily = Tac.Type( "Arnet::AddressFamily" )
TopoId = Tac.Type( "Cspf::TopoId" )

def showClientCspfPathLinks( mode, param, Model ):
   command = [ "show traffic-engineering cspf path", str( param ), "links" ]
   command = " ".join( command )
   runCliPrintSocketCommand( mode.entityManager, "Cspf", "CallbackWithF", command,
                             mode, asyncCommand=True )
   return CliModel.cliPrinted( Model )

def showClientCspfPathIpAddrLinks( mode, args ):
   return showClientCspfPathLinks( mode, args[ "IP_ADDR" ], CspfPathLinksModel )

def showClientCspfPathIdLinks( mode, args ):
   return showClientCspfPathLinks( mode, args[ "PATH_ID" ], CspfPathByIdLinksModel )

def getIntfSrlgList( intfName ):
   srlgIdList = []
   config = teConfiguration()
   if intfName not in config.intfConfig:
      return srlgIdList
   for groupId in config.intfConfig[ intfName ].srlgIdList:
      srlgIdList.append( groupId )
   for groupName in config.intfConfig[ intfName ].srlgNameList:
      groupIdEntry = config.srlgMap.get( groupName )
      if groupIdEntry:
         srlgIdList.append( groupIdEntry.srlgId )
   return srlgIdList

def getClientConfigStatus( clientName, vrf, af ):
   shmemEm = SharedMem.entityManager( sysdbEm=entityManager )
   smi = Smash.mountInfo( 'reader' )
   configPath = "te/cspf/config/vrf/" + vrf + "/" + af + "/" + clientName
   statusPath = "te/cspf/status/vrf/" + vrf + "/" + af + "/" + clientName
   cspfConfig = shmemEm.doMount( configPath, 'Cspf::Config', smi )
   cspfStatus = shmemEm.doMount( statusPath, 'Cspf::Status', smi )
   return ( cspfConfig, cspfStatus )

def getClientConfigStatusP2mp( clientName, vrf, af ):
   cspfConfig = SharkLazyMount.mount(
      entityManager, "te/cspf/config/vrf/" + vrf + "/" + af + "/" + clientName,
      'Cspf::CspfSharkConfig', SharkLazyMount.mountInfo( 'shadow' ), True )
   cspfStatus = SharkLazyMount.mount(
      entityManager, "te/cspf/status/vrf/" + vrf + "/" + af + "/" + clientName,
      'Cspf::CspfSharkStatus', SharkLazyMount.mountInfo( 'shadow' ), True )
   return ( cspfConfig, cspfStatus )

def getSrSmashStatus( igpString, vrf ):
   srSmashStatus = SmashLazyMount.mount( entityManager,
                                         f"segmentrouting/{igpString}/{vrf}",
                                         "Smash::SegmentRouting::Status",
                                         SmashLazyMount.mountInfo( 'reader' ),
                                         autoUnmount=True )
   return srSmashStatus

def populateExcludeSrlgOfPaths( pathEntries, srlgGroupOrder, constraintModel,
                                srlgIdToNameMap, pathStatus, detail ):
   """
   pathEntries - A list of tuples ( pathKey, srlgGroupOrder )
   srlgGroupOder - SrlgGroupOrder for the pathConfig for which excludeSrlgOfPaths
                   will be populated in constraintModel
   constraintModel - It hold constraint entries in CAPI model
   srlgIdToNameMap - A map from SRLG ids to SRLG group name
   detail - If set we will also populate SRLG in excludeSrlgOfPaths
   """
   for ( pathKey, srlgGpOrder ) in pathEntries:
      if srlgGroupOrder <= srlgGpOrder:
         continue
      pModel = CspfPathHelperModel( destination=pathKey.dstIp,
                                    pathId=pathKey.id,
                                    srlgIds=None )
      if detail:
         # SRLG value will be shown only in detail output
         pModel._srlgIdToNameMap = srlgIdToNameMap
         pStatus = pathStatus.get( pathKey )
         if pStatus and pStatus.pathSrlg:
            srlgs = set()
            for value in pStatus.pathSrlg.values():
               srlgs.update( value.srlg.values() )
            if srlgs:
               pModel.srlgIds = sorted( srlgs )
      constraintModel.excludeSrlgOfPaths.append( pModel )
   # excludeSrlgOfPaths is an optional attribute, if there is no entries
   # in it, we will not show it in CAPI output
   if not constraintModel.excludeSrlgOfPaths:
      constraintModel.excludeSrlgOfPaths = None

def getSharedBwGroupAndPathBySrlgGroupIdMap( pathConfigs ):
   vrfBwGroupMap = defaultdict( list )
   pathBySrlgGroup = defaultdict( list )
   for p, pathConfig in pathConfigs.items():
      if pathConfig.srlgGroupId != 0:
         pathBySrlgGroup[ pathConfig.srlgGroupId ].extend(
            [ ( p, pathConfig.srlgGroupOrder ) ] )
      sharedBwGroupId = pathConfig.constraint.sharedBwGroupId
      if not sharedBwGroupId:
         continue
      pathModel = CspfPathHelperModel( destination=p.dstIp, pathId=p.id,
                                       srlgIds=None )
      vrfBwGroupMap[ sharedBwGroupId ].extend( [ pathModel ] )
   return ( vrfBwGroupMap, pathBySrlgGroup )

def showClientCspfPathInfo( mode, args ):
   ipAddr = args.get( "IP_ADDR" )
   detail = "detail" in args
   pathModel = CspfPathModel()
   srlgIdToNameMap = getSrlgIdToNameMap()

   # Create vrf model for each vrf for the client
   for ( clientName, client ) in clientDir.entityPtr.items():
      # We are only interested in rsvpFrr and rsvpLer client
      if clientName not in [ "rsvpFrr", "rsvpLer" ]:
         continue
      getClientCspfPathInfo( client, clientName, ipAddr, detail, pathModel,
                             srlgIdToNameMap )
   return pathModel

def showClientCspfPathByIdInfo( mode, args ):
   pathId = args.get( "PATH_ID" )
   detail = "detail" in args
   pathModel = CspfPathByIdModel()
   srlgIdToNameMap = getSrlgIdToNameMap()

   # Create vrf model for each vrf for the client
   for ( clientName, client ) in clientDir.entityPtr.items():
      # We are only interested in rsvpFrr and rsvpLer client
      if clientName not in [ "rsvpFrr", "rsvpLer" ]:
         continue
      getClientCspfPathByIdInfo( client, clientName, pathId, detail, pathModel,
                                 srlgIdToNameMap )
   return pathModel

def showClientCspfPathP2mpInfo( mode, args ):
   treeId = args.get( "TREE_ID" )
   ipAddr = args.get( "IP_ADDR" )
   detail = "detail" in args
   pathModel = CspfPathP2mpModel()

   # Create vrf model for each vrf for the client
   for ( clientName, client ) in clientDir.entityPtr.items():
      # We are only interested in rsvpLerP2mp client
      if clientName != "rsvpLerP2mp":
         continue
      getClientCspfPathInfoP2mp( client, clientName, treeId, ipAddr, detail,
                                 pathModel )
   return pathModel

def getClientCspfPathInfoP2mp( client, clientName, treeId, ipAddr, detail,
                               pathModel ):
   for vrf in client.vrf:
      if vrf not in pathModel.vrfs:
         pathModel.vrfs[ vrf ] = CspfPathVrfP2mpModel()
      vrfModel = pathModel.vrfs[ vrf ]
      # Create af model for each af for the client
      for af in client.addressFamily:
         if af == AddressFamily.ipv4:
            if not vrfModel.v4Info:
               vrfModel.v4Info = CspfPathAfP2mpModel()
            afModel = vrfModel.v4Info
         elif af == AddressFamily.ipv6:
            if not vrfModel.v6Info:
               vrfModel.v6Info = CspfPathAfP2mpModel()
            afModel = vrfModel.v6Info
         else:
            assert False, "Client should not have ipunknown as addressFamily"
         afModel._detailsPresent = detail
         ( config, status ) = getClientConfigStatusP2mp( clientName, vrf, af )
         pathConfig = config.pathConfig
         pathStatus = status.pathStatus
         # Create tree model for each entry
         for pathKey, treeConfig in pathConfig.items():
            # These top-of-loop checks ensures that, in the case of a
            # non-existent treeId, the CspfPathAfP2mpModel in the output
            # does not contain an empty CspfPathTreeModel - instead, it contains
            # no tree models at all.
            if treeId is not None and treeId != pathKey.id:
               continue
            treeModel = CspfPathTreeModel()
            treeModel._treeId = pathKey.id
            treeModel._detailsPresent = detail
            statusEntry = pathStatus.get( pathKey )
            if detail:
               populateTreeModelDetails( treeConfig, statusEntry, treeModel )
            if toggleCspfTunnelNamesEnabled():
               treeModel.tunnelName = treeConfig.tunnelName
            for leafPathKey in treeConfig.p2mpLeafPath:
               if ipAddr and str( ipAddr ) != str( leafPathKey.dst ):
                  continue
               leafPathModel = CspfLeafPathModel()
               leafPath = None
               # There needs to be a pathStatus entry for both the tree and the leaf
               if statusEntry is not None:
                  leafPath = statusEntry.p2mpLeafPath.get( leafPathKey )
               treeModel.leafPaths[ leafPathKey.dst ] = populateLeafPathModel(
                  leafPathKey, treeConfig.p2mpLeafPath.get( leafPathKey ), leafPath,
                  detail, leafPathModel )
            afModel.trees[ pathKey.id ] = treeModel

def populateTreeModelDetails( pathConfig, pathStatus, treeModel ):
   treeModel.details = CspfPathTreeModel.Details()
   detailsModel = treeModel.details
   detailsModel.refreshReqSeq = pathConfig.refreshReqSeq
   detailsModel.explicitRefreshReqSeq = pathConfig.explicitRefreshReqSeq
   if pathStatus:
      detailsModel.treePathChangeCount = pathStatus.treePathChangeCount
      detailsModel.pendingRefreshRespSeq = pathStatus.refreshRespSeq < \
         pathConfig.refreshReqSeq
      detailsModel.pendingExplicitRespSeq = pathStatus.explicitRefreshRespSeq < \
         pathConfig.explicitRefreshReqSeq
   else:
      detailsModel.treePathChangeCount = 0
      detailsModel.pendingRefreshRespSeq = True
      detailsModel.pendingExplicitRespSeq = True
   constraint = CspfPathTreeConstraintModel()
   constraint.explicitPath = pathConfig.constraint.includeHopAllExplicit
   populateAdminGroupModel( constraint, pathConfig.constraint.adminGroupConstraint )
   detailsModel.constraint = constraint

def populateLeafPathModelDetails( leafPathConfig, leafPathStatus, leafModel ):
   leafModel.details = CspfLeafPathModel.Details()
   detailsModel = leafModel.details
   if leafPathStatus is not None:
      detailsModel.pathChangeCount = leafPathStatus.pathChangeCount
      detailsModel.lastUpdatedTime = leafPathStatus.lastUpdatedTime
      if len( leafPathStatus.path ) == 0:
         if leafPathStatus.pathNotFoundReason.type != \
            PathNotFoundReasonType.pathNotFoundReasonTypeNone:
            pathNotFoundReasonModel = PathNotFoundReasonModel()
            pathNotFoundReasonModel.reason = PATH_NOT_FOUND_REASON_TO_TYPE_MAP[
               leafPathStatus.pathNotFoundReason.type ]
            pathNotFoundReasonModel.details = leafPathStatus.pathNotFoundReason.data
            detailsModel.pathNotFoundReason = pathNotFoundReasonModel

   constraintModel = CspfLeafPathConstraintModel()
   for includeHop in leafPathConfig.includeHop.values():
      includeHopModel = CspfExplicitHopModel( hop=includeHop.hop,
                                              node=includeHop.node )
      constraintModel.explictPath.append( includeHopModel )
   if not constraintModel.explictPath:
      constraintModel.explictPath = None
   detailsModel.constraint = constraintModel

def populateLeafPathModel( key, path, leafPath, detail, leafPathModel ):
   leafPathModel._dstAddr = key.dst
   leafPathModel._detailsPresent = detail
   if detail:
      populateLeafPathModelDetails( path, leafPath, leafPathModel )
   populatePathEntryModel( leafPathModel, leafPath )
   return leafPathModel

def getClientCspfPathInfo( client, clientName, ipAddr, detail, pathModel,
                           srlgIdToNameMap ):
   for vrf in client.vrf:
      if vrf not in pathModel.vrfs:
         pathModel.vrfs[ vrf ] = CspfPathVrfModel()
      vrfModel = pathModel.vrfs[ vrf ]
      # Create af model for each af for the client
      for af in client.addressFamily:
         if af == AddressFamily.ipv4:
            if not vrfModel.v4Info:
               vrfModel.v4Info = CspfPathAfModel()
            afModel = vrfModel.v4Info
         elif af == AddressFamily.ipv6:
            if not vrfModel.v6Info:
               vrfModel.v6Info = CspfPathAfModel()
            afModel = vrfModel.v6Info
         else:
            assert False, "Client should not have ipunknown as addressFamily"
         if detail:
            afModel._detailsPresent = True
         ( config, status ) = getClientConfigStatus( clientName, vrf, af )
         # Mounted objects might change during the run
         # Create pathConfig snapshot in order to make it consistent
         pathConfigs = {}
         for p, pathConfig in config.pathConfig.items():
            pathConfigs[ p ] = pathConfig

         vrfSharedBwMap, pathBySrlgGroup = \
            getSharedBwGroupAndPathBySrlgGroupIdMap( pathConfigs )
         # Create ip model and entry model for each entry
         for p, pathConfig in pathConfigs.items():
            # These top-of-loop checks ensures that, in the case of a
            # non-existent (ipAddr, pathId) pair, the CspfPathAfModel in the output
            # does not contain an empty CspfpathDestIpModel - instead, it contains
            # no IP models at all.
            if ipAddr and str( ipAddr ) != str( p.dstIp ):
               continue
            if p.dstIp not in afModel.pathIps:
               ipModel = CspfPathDestIpModel()
               ipModel._dstAddr = p.dstIp
               afModel.pathIps[ p.dstIp ] = ipModel
            else:
               ipModel = afModel.pathIps[ p.dstIp ]
            if p.id not in ipModel.paths:
               entryModel = CspfPathEntryModel()
               populateCspfPathEntry( p, pathConfig, status, detail, srlgIdToNameMap,
                                      vrfSharedBwMap, pathBySrlgGroup, entryModel )
               ipModel.paths[ p.id ] = entryModel

def populateCspfPathEntry( p, pathConfig, status, detail, srlgIdToNameMap,
                           vrfSharedBwMap, pathBySrlgGroup, entryModel ):
   pathStatus = status.pathStatus.get( p )
   populatePathEntryModel( entryModel, pathStatus )
   if detail:
      details = CspfPathEntryModel.Details()
      populatePathEntryModelDetails( pathConfig, pathStatus, details )
      entryModel.details = details
   constraintModel = \
      getCspfPathConstraint( p, pathConfig, status, detail, srlgIdToNameMap,
                             vrfSharedBwMap, pathBySrlgGroup )
   entryModel.constraint = constraintModel

def getClientCspfPathByIdInfo( client, clientName, pathId, detail, pathModel,
                               srlgIdToNameMap ):
   for vrf in client.vrf:
      if vrf not in pathModel.vrfs:
         pathModel.vrfs[ vrf ] = CspfPathByIdVrfModel()
      vrfModel = pathModel.vrfs[ vrf ]
      # Create af model for each af for the client
      for af in client.addressFamily:
         ( config, status ) = getClientConfigStatus( clientName, vrf, af )
         # Mounted objects might change during the run
         # Create pathConfig snapshot in order to make it consistent
         pathConfigs = {}
         for p, pathConfig in config.pathConfig.items():
            pathConfigs[ p ] = pathConfig

         vrfSharedBwMap, pathBySrlgGroup = \
            getSharedBwGroupAndPathBySrlgGroupIdMap( pathConfigs )

         # Create ip model and entry model for given pathId
         for p, pathConfig in config.pathConfig.items():
            if pathId is not None and pathId == int( p.id ):
               if af == AddressFamily.ipv4:
                  if not vrfModel.v4Info:
                     vrfModel.v4Info = CspfPathByIdEntryModel()
                  entryModel = vrfModel.v4Info
               elif af == AddressFamily.ipv6:
                  if not vrfModel.v6Info:
                     vrfModel.v6Info = CspfPathByIdEntryModel()
                  entryModel = vrfModel.v6Info
               else:
                  assert False, "Client should not have ipunknown as addressFamily"
               if detail:
                  entryModel._detailsPresent = True
               entryModel.pathId = p.id
               entryModel.dstAddr = p.dstIp
               pathStatus = status.pathStatus.get( p )
               populatePathEntryModel( entryModel, pathStatus )
               if detail:
                  details = CspfPathByIdEntryModel.Details()
                  populatePathEntryModelDetails( pathConfig, pathStatus, details )
                  entryModel.details = details
               entryModel.constraint = \
                  getCspfPathConstraint( p, pathConfig, status, detail,
                                         srlgIdToNameMap, vrfSharedBwMap,
                                         pathBySrlgGroup )
               break

def populatePathEntryModel( entryModel, pathStatus ):
   if pathStatus:
      if len( pathStatus.path ):
         includeIpMap = defaultdict( list )
         for includeIp in pathStatus.includeIp.values():
            # Only incude entries from pathStatus.includeIp in which
            # incudeIP address is different from both teRouterId or the
            # ingress IP address a that hopIndex
            iip = includeIp.includeIp
            hopIndex = includeIp.hopIndex
            # pylint: disable-next=consider-using-in
            if ( iip != pathStatus.path[ hopIndex ] and
                 iip != pathStatus.pathTeRouterId[ hopIndex ] ):
               ( includeIpMap[ includeIp.hopIndex ]
                 .append( includeIp.includeIp ) )
         for idx, hopIpAddr in pathStatus.path.items():
            hopTeRouterId = pathStatus.pathTeRouterId[ idx ]
            hopIncludeIps = ( includeIpMap[ idx ] if idx in includeIpMap
                              else None )
            hopModel = CspfPathHopModel( ipAddr=hopIpAddr,
                                         teRouterId=hopTeRouterId,
                                         includeIps=hopIncludeIps )
            entryModel.hops.append( hopModel )
         entryModel.status = 'pathFound'
      else:
         entryModel.status = 'pathNotFound'
   else:
      entryModel.status = 'cspfPending'

def populatePathEntryModelDetails( pathConfig, pathStatus, details ):
   details.refreshReqSeq = pathConfig.refreshReqSeq
   details.autoUpdate = pathConfig.autoUpdate
   if pathStatus:
      details.refreshRespSeq = pathStatus.refreshRespSeq
      details.changeCount = pathStatus.changeCount
      details.lastUpdatedTime = pathStatus.lastUpdatedTime
      details.bwLocallyAccounted = \
         pathStatus.bwLocallyAccounted

def populateAdminGroupModel( model, constraint ):
   agConstraints = {
         'excludeAdminGroup' : constraint.excludeAdminGroup,
         'includeAllAdminGroup' : constraint.includeAllAdminGroup,
         'includeAnyAdminGroup' : constraint.includeAnyAdminGroup }
   for attrName, adminGroupColl in agConstraints.items():
      if any( ag != 0 for ag in adminGroupColl.values() ):
         adminGroup = adminGroupColl.get( 0, None )
         if adminGroup:
            setattr( model, attrName, adminGroup )
         if TeToggleLib.toggleExtendedAdminGroupEnabled():
            adminGroupExtended = adminGroupDictToDecimalList( adminGroupColl )
            setattr( model, attrName + 'sExtended', adminGroupExtended )
         else:
            setattr( model, attrName + 'sExtended', None )
      else:
         setattr( model, attrName + 'sExtended', None )

def getCspfPathConstraint( p, pathConfig, status, detail, srlgIdToNameMap,
                           vrfSharedBwMap, pathBySrlgGroup ):
   constraintModel = CspfPathConstraintModel()

   populateAdminGroupModel( constraintModel, pathConfig.constraint )

   excludeSrlgOfIntf = IntfSrlg()
   excludeSrlgOfIntf.srlgIds = None
   bandwidth = pathConfig.constraint.bandwidth
   bwSetupPriority = pathConfig.constraint.bwSetupPriority
   sharedBwGroupId = pathConfig.constraint.sharedBwGroupId
   if pathConfig.constraint.excludeIntf:
      constraintModel.excludeIntf = pathConfig.constraint.excludeIntf
      if pathConfig.constraint.excludeSrlg:
         excludeSrlgOfIntf.interface = \
            pathConfig.constraint.excludeIntf
   elif pathConfig.constraint.excludeNode:
      constraintModel.excludeNode = \
         pathConfig.constraint.excludeNode.node.stringValue
      if pathConfig.constraint.excludeSrlg:
         excludeSrlgOfIntf.interface = \
            pathConfig.constraint.excludeNode.nexthopIntf
   for excludeAddr in pathConfig.constraint.excludeLinkWithIp.values():
      constraintModel.excludeLinksWithAddress.append( excludeAddr )
   if not constraintModel.excludeLinksWithAddress:
      constraintModel.excludeLinksWithAddress = None
   if bandwidth:
      # Convert bandwidth to bits per second for show command
      constraintModel.bandwidth = bandwidth * 8
      # bwSetupPriority and sharedBwGroupId take effect only when
      # bandwidth constraint is present
      constraintModel.bwSetupPriority = bwSetupPriority
      if sharedBwGroupId:
         constraintModel.sharedBwGroupId = sharedBwGroupId
         constraintModel._sharedBwPaths = (
            vrfSharedBwMap[ sharedBwGroupId ] )
         # srlgIds is an optional attribute.
         pathKeyModel = CspfPathHelperModel( destination=p.dstIp,
                                             pathId=p.id, srlgIds=None )
         # Exclude the current path itself from shared paths
         constraintModel._sharedBwPaths.remove( pathKeyModel )
   if pathConfig.constraint.includeHopAllExplicit:
      for explicitHop in pathConfig.constraint.includeHop.values():
         constraintModel.explicitPath.append( explicitHop.hop )
   else:
      for includeHop in pathConfig.constraint.includeHop.values():
         includeHopModel = CspfIncludeHopModel( hop=includeHop.hop,
                                                loose=includeHop.loose )
         constraintModel.includeHops.append( includeHopModel )
   if not constraintModel.includeHops:
      constraintModel.includeHops = None
   if not constraintModel.explicitPath:
      constraintModel.explicitPath = None
   # 0 is not a valid srlgGroupId value.
   if pathConfig.srlgGroupId != 0:
      pathEntries = pathBySrlgGroup[ pathConfig.srlgGroupId ]
      populateExcludeSrlgOfPaths( pathEntries, pathConfig.srlgGroupOrder,
                                  constraintModel, srlgIdToNameMap,
                                  status.pathStatus, detail )
   else:
      # If srlgGroupId is 0 - srlgGroup is not set
      # In this case, excludeSrlgOfPaths attribute will not be shown
      # in CAPI model ( It is an optional attribute )
      constraintModel.excludeSrlgOfPaths = None

   if detail:
      if excludeSrlgOfIntf.interface:
         srlgIds = getIntfSrlgList(
            excludeSrlgOfIntf.interface.stringValue )
         if len( srlgIds ): # pylint: disable=use-implicit-booleaness-not-len
            excludeSrlgOfIntf.srlgIds = srlgIds
            excludeSrlgOfIntf._srlgIdToNameMap = srlgIdToNameMap

   if pathConfig.constraint.excludeSrlg and excludeSrlgOfIntf.interface:
      constraintModel.excludeSrlgOfIntf = excludeSrlgOfIntf
   return constraintModel

def updateFlexAlgoConstraints( constraintModel, pathConfig, srlgIdToNameMap ):
   pathConstraint = pathConfig.constraint
   faConstraintModel = FlexAlgoPathConstraintModel()
   faConstraintModel.metricType = 'igp'
   if pathConstraint.metricType == MetricType.minDelayMetric:
      faConstraintModel.metricType = 'minDelay'
   elif pathConstraint.metricType == MetricType.teMetric:
      faConstraintModel.metricType = 'te'

   if pathConstraint.excludeExplicitSrlg:
      # pylint: disable-msg=protected-access
      srlgIdToNameMapHelper = SrlgIdToNameMapHelper(
         _srlgIdToNameMap=srlgIdToNameMap )
      faConstraintModel._srlgIdToNameMap = srlgIdToNameMapHelper
      for s in pathConstraint.excludeExplicitSrlg.values():
         faConstraintModel.excludeSrlg.append( s )
      faConstraintModel.excludeSrlgMode = 'strict'
   else:
      faConstraintModel.excludeSrlg = None

   populateAdminGroupModel( faConstraintModel, pathConstraint )
   constraintModel.flexAlgo = faConstraintModel

def showClientTilfaPathInfo( mode, args ):
   pathModel = TilfaPathModel()
   reqDstId = str( args.get( "DST_ID", "" ) )
   detail = "detail" in args
   # Only show tilfa related path information in this show cmd.
   clientName = "tilfa" if not toggleOspfTilfaEnabled() else "tilfaisis"
   client = clientDir.entityPtr.get( clientName )
   srlgIdToNameMap = getSrlgIdToNameMap()
   if not ( client and clientName in clientDir.entryState ):
      return pathModel
   for vrf in client.vrf:
      vrfModel = TilfaPathVrfModel()
      pathModel.vrfs[ vrf ] = vrfModel
      for af in client.addressFamily:
         afModel = TilfaPathAfModel()
         if af == AddressFamily.ipv4:
            vrfModel.v4Info = afModel
         elif af == AddressFamily.ipv6:
            vrfModel.v6Info = afModel

         topoIdModelL1 = TilfaPathTopoIdModel()
         topoIdModelL2 = TilfaPathTopoIdModel()
         destAlgoConstraintDictL1 = {}
         destAlgoConstraintDictL2 = {}

         ( config, status ) = getClientConfigStatus( clientName, vrf, af )
         for key, pathConfig in config.pathConfig.items():
            dstId = key.dst.stringValue
            hostname = None
            if key.dst.type == 'vtxIsisSystemId':
               instanceId = pathConfig.constraint.topoId.instance
               hostname = getHostName( getRouterId( dstId, instanceId ) )
            if reqDstId and reqDstId not in ( dstId, hostname ):
               continue
            if pathConfig.constraint.topoId.protoId == 1:
               # level-1
               topoIdModel = topoIdModelL1
               destAlgoConstraintDict = destAlgoConstraintDictL1
            else:
               # level-2
               topoIdModel = topoIdModelL2
               destAlgoConstraintDict = destAlgoConstraintDictL2
            if detail:
               topoIdModel._detailsPresent = True
            afModel.topologies[ pathConfig.constraint.topoId.protoId ] = topoIdModel
            if dstId not in topoIdModel.destinations:
               sysIdModel = TilfaPathDestModel()
               sysIdModel._dstId = dstId
               sysIdModel.hostname = hostname
               topoIdModel.destinations[ sysIdModel._dstId ] = sysIdModel
            else:
               sysIdModel = topoIdModel.destinations[ dstId ]
            destAlgoConstraintTuple = None
            if pathConfig.constraint.excludeIntf:
               destAlgoConstraintTuple = ( dstId, pathConfig.constraint.algoId,
                                           pathConfig.constraint.excludeIntf )
            elif pathConfig.constraint.excludeNode:
               destAlgoConstraintTuple = ( dstId, pathConfig.constraint.algoId,
                                           pathConfig.constraint.excludeNode )

            if key.id not in sysIdModel.pathIds and \
               destAlgoConstraintTuple not in destAlgoConstraintDict:
               entryModel = TilfaPathEntryModel()
               constraintModel = TilfaPathConstraintModel()
               excludeSrlgOfIntf = IntfSrlg()
               excludeSrlgOfIntf.srlgIds = None
               if pathConfig.constraint.excludeIntf:
                  constraintModel.excludeIntf = pathConfig.constraint.excludeIntf
                  if pathConfig.constraint.excludeSrlg:
                     excludeSrlgOfIntf.interface = pathConfig.constraint.excludeIntf
               elif pathConfig.constraint.excludeNode:
                  constraintModel.excludeNode = \
                                 pathConfig.constraint.excludeNode.node.stringValue
                  if pathConfig.constraint.excludeSrlg:
                     excludeSrlgOfIntf.interface = \
                              pathConfig.constraint.excludeNode.nexthopIntf
               if pathConfig.constraint.excludeSrlg:
                  constraintModel.excludeSrlgOfIntf = excludeSrlgOfIntf
                  constraintModel.excludeSrlgMode = \
                        'strict' if pathConfig.constraint.srlgStrictMode else 'loose'

               # Add algorithm information
               constraintModel.algoId = pathConfig.constraint.algoId
               algoName = getFlexAlgoName( pathConfig.constraint.algoId )
               if algoName != str( constraintModel.algoId ):
                  constraintModel.algoName = algoName

               if detail and pathConfig.constraint.algoId != 0:
                  # Add Flex-Algo Constraints.
                  updateFlexAlgoConstraints( constraintModel, pathConfig,
                                             srlgIdToNameMap )
               entryModel.constraint = constraintModel
               if detail and excludeSrlgOfIntf.interface:
                  srlgIds = getIntfSrlgList(
                     excludeSrlgOfIntf.interface.stringValue )
                  # pylint: disable-next=use-implicit-booleaness-not-len
                  if len( srlgIds ):
                     excludeSrlgOfIntf.srlgIds = srlgIds
                     excludeSrlgOfIntf._srlgIdToNameMap = srlgIdToNameMap

               if not detail:
                  destAlgoConstraintDict[ destAlgoConstraintTuple ] = True
               pathStatus = status.pathStatus.get( key )
               lfaIdx = Tac.Type( "Cspf::Lfa::PqIdx" )
               statusStr = 'pathNotFound'
               if pathStatus:
                  if ( pathStatus.repairPathIdx() != lfaIdx.invalid and
                     status.pqInfo.get( pathStatus.repairPathIdx() ) ):
                     pqInfoNode = status.pqInfo.get( pathStatus.repairPathIdx() )
                     # Added an extra check when pqInfo is again evaluated.
                     if pqInfoNode:
                        for sysid in pqInfoNode.path.values():
                           sysidHostnameModel = SystemIdHostnameModel()
                           sysidHostnameModel.sysId = sysid.isisSystemId()
                           sysidHostnameModel.hostname = getHostName( getRouterId(
                              sysid.isisSystemId() ) )
                           entryModel.sysIds.append( sysidHostnameModel )
                        statusStr = 'pathFound'
               else:
                  statusStr = 'cspfPending'
               entryModel.status = statusStr
               if detail:
                  details = TilfaPathDetails()
                  details.refreshReqSeq = pathConfig.refreshReqSeq
                  if pathStatus:
                     details.refreshRespSeq = pathStatus.refreshRespSeq
                     details.changeCount = pathStatus.changeCount
                     details.lastUpdatedTime = pathStatus.lastUpdatedTime
                  entryModel.details = details
               entryModel._pathId = int( key.id )
               sysIdModel.pathIds[ int( key.id ) ] = entryModel

   return pathModel

def showClientOspfTilfaPathInfo( mode, args ):
   pathModel = TilfaPathOspfModel()
   reqRtrId = str( args.get( "RTR_ID", "" ) )
   detail = "detail" in args

   clientName = "tilfaospf"
   client = clientDir.entityPtr.get( clientName )
   srlgIdToNameMap = getSrlgIdToNameMap()
   if not ( client and clientName in clientDir.entryState ):
      return pathModel
   for vrf in client.vrf:
      vrfModel = TilfaPathOspfVrfModel()
      pathModel.vrfs[ vrf ] = vrfModel
      for af in client.addressFamily:
         afModel = TilfaPathOspfAfModel()
         if af == AddressFamily.ipv4:
            vrfModel.v4Info = afModel
         elif af == AddressFamily.ipv6:
            vrfModel.v6Info = afModel

         topoIdModelByProtoId = {}
         destAlgoConstraintDictByProtoId = {}

         ( config, status ) = getClientConfigStatus( clientName, vrf, af )
         for key, pathConfig in config.pathConfig.items():
            dstId = key.dst.stringValue
            if reqRtrId and reqRtrId not in ( dstId ):
               continue
            protoId = pathConfig.constraint.topoId.protoId
            protoId = str( ipaddress.ip_address( protoId ) )
            if protoId not in topoIdModelByProtoId:
               topoIdModelByProtoId[ protoId ] = TilfaPathOspfTopoIdModel()
               destAlgoConstraintDictByProtoId[ protoId ] = {}
            topoIdModel = topoIdModelByProtoId[ protoId ]
            destAlgoConstraintDict = destAlgoConstraintDictByProtoId[ protoId ]

            if detail:
               topoIdModel._detailsPresent = True
            afModel.topologies[ protoId ] = topoIdModel
            if dstId not in topoIdModel.destinations:
               rtrIdModel = TilfaPathOspfDestModel()
               rtrIdModel._dstId = dstId
               topoIdModel.destinations[ rtrIdModel._dstId ] = rtrIdModel
            else:
               rtrIdModel = topoIdModel.destinations[ dstId ]
            destAlgoConstraintTuple = None
            if pathConfig.constraint.excludeIntf:
               destAlgoConstraintTuple = ( dstId, pathConfig.constraint.algoId,
                                           pathConfig.constraint.excludeIntf )
            elif pathConfig.constraint.excludeNode:
               destAlgoConstraintTuple = ( dstId, pathConfig.constraint.algoId,
                                           pathConfig.constraint.excludeNode )

            if key.id not in rtrIdModel.pathIds and \
               destAlgoConstraintTuple not in destAlgoConstraintDict:
               entryModel = TilfaPathOspfEntryModel()
               constraintModel = TilfaPathConstraintModel()
               excludeSrlgOfIntf = IntfSrlg()
               excludeSrlgOfIntf.srlgIds = None
               if pathConfig.constraint.excludeIntf:
                  constraintModel.excludeIntf = pathConfig.constraint.excludeIntf
                  if pathConfig.constraint.excludeSrlg:
                     excludeSrlgOfIntf.interface = pathConfig.constraint.excludeIntf
               elif pathConfig.constraint.excludeNode:
                  constraintModel.excludeNode = \
                                 pathConfig.constraint.excludeNode.node.stringValue
                  if pathConfig.constraint.excludeSrlg:
                     excludeSrlgOfIntf.interface = \
                              pathConfig.constraint.excludeNode.nexthopIntf
               if pathConfig.constraint.excludeSrlg:
                  constraintModel.excludeSrlgOfIntf = excludeSrlgOfIntf
                  constraintModel.excludeSrlgMode = \
                        'strict' if pathConfig.constraint.srlgStrictMode else 'loose'

               # Add algorithm information
               constraintModel.algoId = pathConfig.constraint.algoId
               algoName = getFlexAlgoName( pathConfig.constraint.algoId )
               if algoName != str( constraintModel.algoId ):
                  constraintModel.algoName = algoName

               entryModel.constraint = constraintModel
               if detail and excludeSrlgOfIntf.interface:
                  srlgIds = getIntfSrlgList(
                     excludeSrlgOfIntf.interface.stringValue )
                  # pylint: disable-next=use-implicit-booleaness-not-len
                  if len( srlgIds ):
                     excludeSrlgOfIntf.srlgIds = srlgIds
                     excludeSrlgOfIntf._srlgIdToNameMap = srlgIdToNameMap

               if not detail:
                  destAlgoConstraintDict[ destAlgoConstraintTuple ] = True
               pathStatus = status.pathStatus.get( key )
               lfaIdx = Tac.Type( "Cspf::Lfa::PqIdx" )
               statusStr = 'pathNotFound'
               if pathStatus:
                  if ( pathStatus.repairPathIdx() != lfaIdx.invalid and
                     status.pqInfo.get( pathStatus.repairPathIdx() ) ):
                     pqInfoNode = status.pqInfo.get( pathStatus.repairPathIdx() )
                     # Added an extra check when pqInfo is again evaluated.
                     if pqInfoNode:
                        for rtrId in pqInfoNode.path.values():
                           rtrIdHostnameModel = RouterIdHostnameModel()
                           rtrIdHostnameModel.routerId = rtrId.stringValue
                           entryModel.routerIds.append( rtrIdHostnameModel )
                        statusStr = 'pathFound'
               else:
                  statusStr = 'cspfPending'
               entryModel.status = statusStr
               if detail:
                  details = TilfaPathDetails()
                  details.refreshReqSeq = pathConfig.refreshReqSeq
                  if pathStatus:
                     details.refreshRespSeq = pathStatus.refreshRespSeq
                     details.changeCount = pathStatus.changeCount
                     details.lastUpdatedTime = pathStatus.lastUpdatedTime
                  entryModel.details = details
               entryModel._pathId = int( key.id )
               rtrIdModel.pathIds[ int( key.id ) ] = entryModel

   return pathModel

#--------------------------------------------------------------------------------
# show isis ti-lfa path [ { ( DST_ID | detail ) } ]
#--------------------------------------------------------------------------------
dstIdMatcher = CliMatcher.PatternMatcher(
      pattern='^(?![dD]([eE]([tT]([aA]([iI]([lL])?)?)?)?)?$)[A-Za-z0-9,_:'
              '\\-\\./#%+]{1,255}',
      helpdesc='Destination system identifier, hostname or prefix',
      helpname='IDENTIFIER' )

class IsisTiLfaPathCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show isis ti-lfa path [ { ( DST_ID | detail ) } ]'
   data = {
      'isis' : isisKw,
      'ti-lfa' : 'TI-LFA related path information',
      'path' : 'TI-LFA path to destination',
      'DST_ID' : CliCommand.Node( matcher=dstIdMatcher, maxMatches=1 ),
      'detail' : CliCommand.Node( matcher=matcherDetail, maxMatches=1 ),
   }
   handler = showClientTilfaPathInfo
   cliModel = TilfaPathModel

BasicCli.addShowCommandClass( IsisTiLfaPathCmd )

#--------------------------------------------------------------------------------
# show traffic-engineering cspf path [ IP_ADDR ] [ detail ]
#--------------------------------------------------------------------------------
class TrafficEngineeringCspfPathCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show traffic-engineering cspf path [ IP_ADDR ] [ detail ]'
   data = {
      'traffic-engineering' : matcherTrafficEngineering,
      'cspf' : 'Constrained Shortest Path related information',
      'path' : 'CSPF path to destination',
      'IP_ADDR' : IpAddrMatcher( helpdesc='CSPF path to this destination '
                                          'IP address' ),
      'detail' : matcherDetail,
   }
   handler = showClientCspfPathInfo
   cliModel = CspfPathModel

BasicCli.addShowCommandClass( TrafficEngineeringCspfPathCmd )
# --------------------------------------------------------------------------------
# show traffic-engineering cspf path PATH_ID [ detail ]
# --------------------------------------------------------------------------------
class TrafficEngineeringCspfPathByIdCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show traffic-engineering cspf path PATH_ID [ detail ]'
   data = {
      'traffic-engineering' : matcherTrafficEngineering,
      'cspf' : 'Constrained Shortest Path related information',
      'path' : 'CSPF path to destination',
      'PATH_ID' : matcherPathId,
      'detail' : matcherDetail,
   }
   handler = showClientCspfPathByIdInfo
   cliModel = CspfPathByIdModel

BasicCli.addShowCommandClass( TrafficEngineeringCspfPathByIdCmd )

# --------------------------------------------------------------------------------
# show traffic-engineering cspf path IP_ADDR links
# --------------------------------------------------------------------------------
class TrafficEngineeringCspfPathLinksCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show traffic-engineering cspf path IP_ADDR links'
   data = {
      'traffic-engineering' : matcherTrafficEngineering,
      'cspf' : 'Constrained Shortest Path related information',
      'path' : 'CSPF path to destination',
      'IP_ADDR' : IpAddrMatcher( helpdesc='CSPF path to this destination '
                                          'IP address' ),
      'links' : 'Show current link attributes of the path',
   }
   handler = showClientCspfPathIpAddrLinks
   cliModel = CspfPathLinksModel

BasicCli.addShowCommandClass( TrafficEngineeringCspfPathLinksCmd )

# --------------------------------------------------------------------------------
# show traffic-engineering cspf path PATH_ID links
# --------------------------------------------------------------------------------
class TrafficEngineeringCspfPathByIdLinksCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show traffic-engineering cspf path PATH_ID links'
   data = {
      'traffic-engineering' : matcherTrafficEngineering,
      'cspf' : 'Constrained Shortest Path related information',
      'path' : 'CSPF path to destination',
      'PATH_ID' : matcherPathId,
      'links' : 'Show current link attributes of the path',
   }
   handler = showClientCspfPathIdLinks
   cliModel = CspfPathByIdLinksModel

BasicCli.addShowCommandClass( TrafficEngineeringCspfPathByIdLinksCmd )

# --------------------------------------------------------------------------------
# show traffic-engineering cspf path p2mp [ TREE_ID [ IP_ADDR ] ] [ detail ]
# --------------------------------------------------------------------------------
class TrafficEngineeringCspfPathP2mpCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show traffic-engineering cspf path p2mp' + \
      ' [ TREE_ID [ IP_ADDR ] ] [ detail ]'
   data = {
      'traffic-engineering' : matcherTrafficEngineering,
      'cspf' : 'Constrained Shortest Path related information',
      'path' : 'CSPF path to destination',
      'p2mp' : 'P2MP path',
      'TREE_ID' : matcherTreeId,
      'IP_ADDR' : IpAddrMatcher( helpdesc='CSPF leaf path to this destination '
                                          'IP address' ),
      'detail' : matcherDetail,
   }
   handler = showClientCspfPathP2mpInfo
   cliModel = CspfPathP2mpModel

if toggleMvpnRsvpP2mpEnabled():
   BasicCli.addShowCommandClass( TrafficEngineeringCspfPathP2mpCmd )

#--------------------------------------------------------------------------------
# show ip ospf ti-lfa path [ RTR_ID ] [detail ]
#--------------------------------------------------------------------------------
class OspfTiLfaPathCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show ip ospf ti-lfa path [ RTR_ID ] [ detail ]'
   data = {
      'ip' : CliToken.Ip.ipMatcherForShow,
      'ospf' : matcherOspfShow,
      'ti-lfa' : 'TI-LFA related path information',
      'path' : 'TI-LFA path to destination',
      'RTR_ID' : IpAddrMatcher( helpdesc='OSPF Router ID' ),
      'detail' : matcherDetail
   }
   handler = showClientOspfTilfaPathInfo
   cliModel = TilfaPathOspfModel

if toggleOspfTilfaEnabled():
   BasicCli.addShowCommandClass( OspfTiLfaPathCmd )

def Plugin( entMan ):
   global entityManager
   global clientDir
   global flexAlgoConfig
   entityManager = entMan
   mg = entityManager.mountGroup()
   clientDir = mg.mount( "te/cspf/client", "Tac::Dir", "ri" )
   flexAlgoConfig = LazyMount.mount( entityManager, 'te/flexalgo/config',
                                     'FlexAlgo::Config', 'r' )
   mg.close( None )
