#!/usr/bin/env python3
# Copyright (c) 2018 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# pylint: disable=consider-using-f-string

from CliModel import Model
from CliModel import ( Str,
                       Bool,
                       Enum,
                       Int,
                       Float,
                       Dict,
                       Submodel,
                       List )
from ArnetModel import IpGenericAddress, Ip4Address
from IntfModels import Interface
from CliPlugin.IsisCliModels import bw_best_value_units
from CliPlugin.TeCli import adminGroupToStr, adminGroupDecimalListToDict
from CliPlugin.CspfShowDbModel import ( InterfaceAddress,
                                        ReservablePriorityBandwidth,
                                        SharedRiskLinkGroup )
from Toggles import (
   CspfToggleLib,
   TeToggleLib,
)
import Ark
import Tac
import TableOutput
from itertools import zip_longest
from functools import partial

# pylint: disable-msg=R1702

# TODO Remove format strings with toggle CspfTunnelNames. See BUG976254
entryFormatStr = "%-15s %-10s %-50s %-15s"
entryFormatStrIpPathId = "%-15s %-11s"
entryFormatStrWithoutIp = "%-50s %-15s"

MAX_TUNNEL_NAME_COLUMN_WIDTH = 40

# Sample output
# rtr1#show traffic-engineering cspf path
#
# Destination     Path ID    Constraint                                    Path
# 20.0.0.1        0          exclude Ethernet1                             1.1.1.1
#                                                                          1.1.1.2
#                                                                          2.2.2.2
#                                                                          3.3.3.2
#
#                 1          bandwidth 18.75 Mbps                          1.1.1.1
#                            setup priority 4                              1.1.1.2
#                            share bandwidth with path 10.0.1.1 ID 8       2.2.2.2
#                            share bandwidth with path 10.1.1.1 ID 5       3.3.3.2
#
#                 2          exclude node 1.0.0.1                          20.0.2.2
#                            exclude link with address 1.0.1.3             20.1.2.2
#                                                                          20.2.2.2
#
# rtr1#show traffic-engineering cspf path detail
#
# Destination: 20.0.0.1
#   Path ID: 1
#    Path constraint: exclude Ethernet1
#       Request sequence number: 1
#       Response sequence number: 1
#       Number of times path updated: 2
#       Reoptimize: Always
#       Last updated: 00:01:58
#       Path:
#          1.1.1.1
#          1.1.1.2
#          2.2.2.2
#          3.3.3.2
#   Path ID: 2
#    Path constraint: exclude Ethernet
#       Request sequence number: 2
#       Response sequence number: 2
#       Number of times path updated: 3
#       Reoptimize: On request
#       Last updated: 00:00:38
#       Path:
#          1.1.1.1
#          1.1.1.2
#          2.2.2.2
#          3.3.3.2
#   Path ID: 3
#    Path constraint: exclude node 1.0.0.1
#       Request sequence number: 2
#       Response sequence number: 2
#       Number of times path updated: 3
#       Reoptimize: On request
#       Last updated: 00:00:38 ago
#       Path:
#          1.1.1.1
#          1.1.1.2
#          2.2.2.2
#          3.3.3.2
PATH_STATUS_MAP = Ark.ReversibleDict( { 'pathFound' : 'Path found',
                                        'pathNotFound' : 'Path not found',
                                        'cspfPending' : 'CSPF pending' } )

PathNotFoundReasonType = Tac.Type( 'Cspf::PathNotFoundReasonType' )
PathNotFoundReason = Tac.Type( 'Cspf::PathNotFoundReason' )

PATH_NOT_FOUND_REASON_MAP = Ark.ReversibleDict( {
   "none" : "",
   "remerge" : "Remerge detected at",
} )

PATH_NOT_FOUND_REASON_TO_TYPE_MAP = Ark.ReversibleDict( {
   PathNotFoundReasonType.pathNotFoundReasonTypeNone : "none",
   PathNotFoundReasonType.pathNotFoundReasonTypeRemerge : "remerge",
} )

SPACE_23 = " " * 23
SPACE_27 = " " * 27
MAX_WIDTH = 85

excludeAdminGroupInt = partial( Int, help="Admin Group excluded", optional=True )
excludeAdminGroupsExtendedList = partial(
   List, valueType=int, help="Extended Admin Groups excluded", optional=True )
includeAllAdminGroupInt = partial( Int, help="Admin Group included", optional=True )
includeAllAdminGroupsExtendedList = partial(
   List, valueType=int, help="Extended Admin Groups included", optional=True )
includeAnyAdminGroupInt = partial(
   Int, help="Any Admin Group included", optional=True )
includeAnyAdminGroupsExtendedList = partial(
   List, valueType=int, help="Any Extended Admin Groups included", optional=True )

def wrapText( string, width=MAX_WIDTH, startWidth=20, subsequent_indent=SPACE_23 ):
   output = ""
   constraintInfo = string.split( ',' )
   currWidth = startWidth
   for token in constraintInfo:
      if currWidth + len( token ) < width:
         output += token + ","
      else:
         currWidth = len( subsequent_indent )
         output += "\n" + subsequent_indent + token + ","
      currWidth += len( token + "," )
   return output[ : -1 ]

def toStr( id_, name ):
   text = str( id_ )
   return " %s (%s)" % ( name, text ) if name else " " + text

def addAdminGroupConstraint( constraintList, adminGroup, agDecimalList,
                             attrStr ):
   if TeToggleLib.toggleExtendedAdminGroupEnabled():
      adminGroup = adminGroupDecimalListToDict( agDecimalList )
   constraintList.append( f"{attrStr} Admin Group {adminGroupToStr(adminGroup)}" )

class SrlgIdToNameMapHelper( Model ):
   # _srlgIdToNameMap has srlgName for corresponding srlgId's. Name mapping not
   # needed in capi model so it is just a hidden attribute.
   _srlgIdToNameMap = Dict( keyType=int, valueType=str, valueOptional=True,
                            help="Ordered list of srlgId specifying srlgName",
                            optional=True )

   def printSrlgMap( self, groupIds=None ):
      output = ""
      groupIds = groupIds or self.srlgIds
      for groupId in groupIds:
         groupName = self._srlgIdToNameMap.get( groupId )
         output += toStr( groupId, groupName ) + ","
      return output[ : -1 ]

class IntfSrlg( SrlgIdToNameMapHelper ):
   interface = Interface( help="Interface name" )
   srlgIds = List( valueType=int, optional=True,
                   help="List of SRLG IDs of the interface" )

class CspfPathHelperModel( SrlgIdToNameMapHelper ):
   destination = IpGenericAddress( help="Destination address" )
   pathId = Int( help="Path index" )
   srlgIds = List( valueType=int, optional=True,
                   help="List of SRLG IDs for the path" )

class SharedRiskLinkGroupWithCost( SharedRiskLinkGroup ):
   cost = Int( help="Cost of SRLG Group" )

class CspfIncludeHopModel( Model ):
   hop = IpGenericAddress( help="Interface address or TE Router ID of the hop" )
   loose = Bool( help="The include hop constraint is loose" )

class CspfPathConstraintModel( Model ):
   excludeIntf = Interface( help="Interface excluded from this path",
                            optional=True )
   excludeNode = Str( help="Node excluded from this path", optional=True )
   excludeAdminGroup = excludeAdminGroupInt()
   excludeAdminGroupsExtended = excludeAdminGroupsExtendedList()
   includeAllAdminGroup = includeAllAdminGroupInt()
   includeAllAdminGroupsExtended = includeAllAdminGroupsExtendedList()
   includeAnyAdminGroup = includeAnyAdminGroupInt()
   includeAnyAdminGroupsExtended = includeAnyAdminGroupsExtendedList()
   excludeSrlgOfIntf = Submodel( valueType=IntfSrlg, optional=True,
                                 help="SRLG excluded of interface" )
   excludeLinksWithAddress = List( valueType=IpGenericAddress, optional=True,
                                   help="Addresses of links excluded" )
   bandwidth = Int( help="Bandwidth reserved for this path in bits per second",
                    optional=True )
   bwSetupPriority = Int( help="Priority of bandwidth reservation request",
                          optional=True )
   sharedBwGroupId = Int( help="Shared bandwidth group ID", optional=True )
   # _sharedBwPaths is only used to show the paths sharing bandwidth
   # instead of showing the shared bandwidth group ID. It is not
   # required in the CAPI model.
   _sharedBwPaths = List( help="Keys of paths with the same sharedBwGroupId",
                          valueType=CspfPathHelperModel, optional=True )
   excludeSrlgOfPaths = List( help="List of all paths whose SRLGs to be excluded",
                              valueType=CspfPathHelperModel, optional=True )
   includeHops = List( help="Nodes or links included in this path",
                       valueType=CspfIncludeHopModel, optional=True )
   explicitPath = List( help="Complete list of hops making up this path",
                        valueType=IpGenericAddress, optional=True )

   def getConstraintList( self ):
      constraintList = []
      if self.excludeIntf:
         constraintList.append( "exclude %s" % self.excludeIntf.stringValue )
      elif self.excludeNode:
         constraintList.append( "exclude node %s" % self.excludeNode )
      if self.excludeSrlgOfIntf:
         if self.excludeSrlgOfIntf.srlgIds:
            # printSrlgMap add an extra space at the begining of output.
            output = wrapText( "exclude SRLG of {}:{}".format(
                               self.excludeSrlgOfIntf.interface.stringValue,
                               self.excludeSrlgOfIntf.printSrlgMap() ) )
            constraintList.append( output )
         else:
            constraintList.append( "exclude SRLG of %s" %
                                   self.excludeSrlgOfIntf.interface.stringValue )
      if self.excludeSrlgOfPaths:
         for p in self.excludeSrlgOfPaths:
            if p.srlgIds:
               # printSrlgMap add an extra space at the begining of output.
               output = wrapText( "exclude SRLG of path %s ID %d:%s" % (
                                  p.destination, p.pathId, p.printSrlgMap() ) )
               constraintList.append( output )
            else:
               constraintList.append( "exclude SRLG of path %s ID %d" %
                                      ( p.destination, p.pathId ) )
      if self.includeAllAdminGroup or self.includeAllAdminGroupsExtended:
         addAdminGroupConstraint( constraintList, self.includeAllAdminGroup,
               self.includeAllAdminGroupsExtended, "include all" )

      if self.includeAnyAdminGroup or self.includeAnyAdminGroupsExtended:
         addAdminGroupConstraint( constraintList, self.includeAnyAdminGroup,
               self.includeAnyAdminGroupsExtended, "include any" )

      if self.excludeAdminGroup or self.excludeAdminGroupsExtended:
         addAdminGroupConstraint( constraintList, self.excludeAdminGroup,
               self.excludeAdminGroupsExtended, "exclude" )

      if self.excludeLinksWithAddress:
         for excludeAddr in sorted( self.excludeLinksWithAddress,
                                    key=lambda ip: ip.sortKey ):
            constraintList.append( "exclude link with address %s" % excludeAddr )
      if self.bandwidth:
         bwValueUnits = bw_best_value_units( self.bandwidth )
         constraintList.append( "bandwidth {:0.2f} {}".format( bwValueUnits[ 0 ],
                                                         bwValueUnits[ 1 ] ) )
         constraintList.append( "setup priority %d" % self.bwSetupPriority )
      for p in sorted( self._sharedBwPaths,
                       key=lambda p: ( p.destination.sortKey, p.pathId ) ):
         constraintList.append( "share bandwidth with path %s ID %d" %
                                ( p.destination, p.pathId ) )
      if self.includeHops:
         for includeHop in self.includeHops:
            constraintList.append( "include hop %s (%s)" %
                                  ( includeHop.hop,
                                    'loose' if includeHop.loose else 'strict' ) )
      if self.explicitPath:
         constraintList.append( "explicit path" )
         for explicitHop in self.explicitPath:
            constraintList.append( "   %s" % explicitHop )
      return constraintList

class CspfPathHopModel( Model ):
   ipAddr = IpGenericAddress( help="Interface IP address" )
   teRouterId = IpGenericAddress( help="TE Router ID" )
   includeIps = List( valueType=IpGenericAddress, optional=True,
                      help="Non-ingress IP addresses in includeHop constraint "
                      "corresponding to teRouterId" )

# TODO Remove with toggle CspfTunnelNames. See BUG976254
def genPathEntryStr( constraint, hops, status ):
   pathStr = ""
   constraintList = constraint.getConstraintList()
   if not constraintList:
      constraintList.append( 'None' )
   if len( hops ):
      first = True
      for constraint_, hop in zip_longest( constraintList, hops,
                                           fillvalue='' ):
         if first:
            pathStr += entryFormatStrWithoutIp % \
               ( constraint_, hop.ipAddr if hop else '' )
         else:
            pathStr += entryFormatStr % \
               ( '', '', constraint_, hop.ipAddr if hop else '' )
         pathStr += "\n"
         first = False
   else:
      first = True
      for constraint_ in constraintList:
         if first:
            pathStr += entryFormatStrWithoutIp % ( constraint_,
                                                PATH_STATUS_MAP[ status ] )
         else:
            pathStr += entryFormatStr % ( '', '', constraint_, '' )
         pathStr += "\n"
         first = False
   return pathStr

def genPathEntryStrDetail( constraint, hops, status, details ):
   pathStr = ""
   if len( constraint.getConstraintList() ):
      first = True
      for constaint_ in constraint.getConstraintList():
         if first:
            pathStr += f"   Path constraint: {constaint_}\n"
         else:
            pathStr += f"                    {constaint_}\n"
         first = False
   else:
      pathStr += "   Path constraint: None\n"
   pathStr += f"      Request sequence number: {details.refreshReqSeq}\n"
   if details.refreshRespSeq:
      pathStr += f"      Response sequence number: {details.refreshRespSeq}\n"
   if details.changeCount:
      pathStr += f"      Number of times path updated: {details.changeCount}\n"
   if details.lastUpdatedTime:
      pathStr += \
         f"      Last updated: {Ark.timestampToStr( details.lastUpdatedTime )}\n"
   reoptimizeString = "Always" if details.autoUpdate else "On request"
   pathStr += f"      Reoptimize: {reoptimizeString}\n"
   if len( constraint.getConstraintList() ) and \
      constraint.bandwidth:
      if details.bwLocallyAccounted:
         pathStr += "      Bandwidth is locally accounted as reserved\n"
      else:
         pathStr += "      Bandwidth is not locally accounted as reserved\n"
   if not hops:
      pathStr += "      Path: %s\n" % ( PATH_STATUS_MAP[ status ] )
   else:
      includeIpsPresent = False
      headings = ( ( 'Ingress IP', 'l' ), ( 'TE router ID', 'l' ),
                   ( 'Include IP', 'l' ) )
      pathTable = TableOutput.createTable( headings )
      for hop in hops:
         includeIpStr = ", ".join( str( i ) for i in hop.includeIps or [] )
         if includeIpStr:
            includeIpsPresent = True
         pathTable.newRow( str( hop.ipAddr ), str( hop.teRouterId ),
                           includeIpStr )
      pathEntries = pathTable.output().split( '\n' )
      if not includeIpsPresent:
         # Strip off the `Include IP` heading and its horizontal border
         pathEntries[ 0 ] = pathEntries[ 0 ].rsplit( 'Include IP' )[ 0 ]
         pathEntries[ 1 ] = pathEntries[ 1 ].rsplit( ' -', 1 )[ 0 ]
      pathStr += f"      Path: {pathEntries[ 0 ].rstrip()}\n"
      for pathEntry in pathEntries[ 1 : ]:
         pathStr += f"            {pathEntry}\n"
   return pathStr

class CspfPathEntryModel( Model ):
   constraint = Submodel( valueType=CspfPathConstraintModel,
                          help="Path constraint" )
   hops = List( valueType=CspfPathHopModel,
                help="Hops specifying the path to the destination" )
   status = Enum( values=PATH_STATUS_MAP,
                  help="Status of path computation" )

   class Details( Model ):
      refreshReqSeq = Int( help="Request sequence number" )
      refreshRespSeq = Int( help="Response sequence number", optional=True )
      changeCount = Int( help="Number of times path updated", optional=True )
      lastUpdatedTime = Float( help="UTC timestamp of the last path update",
                               optional=True )
      autoUpdate = Bool( help="Automatically re-optimize the path" )
      bwLocallyAccounted = Bool( help="Bandwidth is locally accounted as reserved",
                                 optional=True )

   details = Submodel( valueType=Details, help="Detailed path information",
                       optional=True )

class CspfPathDestIpModel( Model ):
   _dstAddr = IpGenericAddress( help="Address for destination" )
   paths = Dict( keyType=int, valueType=CspfPathEntryModel,
                 help="A mapping of an id representing a constraint to"
                      " the corresponding CSPF path" )
   _detailsPresent = Bool( default=False,
                           help="Private attribute to indicate that the"
                                " details submodel is present" )

   def renderEntryDetail( self ):
      dstStr = ""
      dstStr += "Destination: %s\n" % self._dstAddr
      for key in sorted( self.paths ):
         dstStr += "  Path ID: %d\n" % key
         path = self.paths[ key ]
         dstStr += genPathEntryStrDetail( path.constraint, path.hops,
                                            path.status, path.details )
      print( dstStr, end='' )

def renderTable( pathIps ):
   headings = [ 'Destination', 'Path ID', 'Constraint', 'Path' ]
   destinationTable = TableOutput.createTable( headings )
   terminalWidth = TableOutput.terminalWidth()
   destinationMaxWidth = 15
   pathIdMaxWidth = 10
   pathMaxWidth = 15
   constraintMaxWidth = terminalWidth - ( destinationMaxWidth +
                        pathIdMaxWidth + pathMaxWidth )
   if constraintMaxWidth <= 0:
      constraintMaxWidth = 1

   destinationFormat = TableOutput.Format( justify='left',
                                           maxWidth=destinationMaxWidth,
                                           wrap=True )
   destinationFormat.noPadLeftIs( True )
   pathIdFormat = TableOutput.Format( justify='right', maxWidth=pathIdMaxWidth )
   pathIdFormat.noPadLeftIs( True )
   constraintFormat = TableOutput.Format( justify='left',
                                          maxWidth=constraintMaxWidth,
                                          wrap=True )
   constraintFormat.noPadLeftIs( True )
   pathFormat = TableOutput.Format( justify='left',
                                    maxWidth=pathMaxWidth,
                                    wrap=True )
   pathFormat.noPadLeftIs( True )
   destinationTable.formatColumns( destinationFormat, pathIdFormat,
                                   constraintFormat, pathFormat )
   for dstAddr, entry in sorted( pathIps.items() ):
      paths = entry.paths
      constraintStr = ''
      pathStr = ''
      for pathId, path in sorted( paths.items() ):
         hops = path.hops
         status = path.status
         constraintList = path.constraint.getConstraintList()
         if not constraintList:
            constraintList.append( 'None' )
         constraintStr = ''
         pathStr = ''
         if hops:
            for constraint_, hop in zip_longest( constraintList,
                                                 hops, fillvalue='' ):
               constraintStr += f"{constraint_}\n"
               pathStr += f"{hop.ipAddr if hop else ''}\n"
         else:
            pathStr = f"{PATH_STATUS_MAP[ status ]}\n"
            for constraint_ in constraintList:
               constraintStr += f"{constraint_}\n"
         destinationTable.newRow( dstAddr, pathId, constraintStr, pathStr )
         dstAddr = ""
         destinationTable.newRow()
   print( destinationTable.output() )

class CspfPathAfModel( Model ):
   pathIps = Dict( keyType=IpGenericAddress, valueType=CspfPathDestIpModel,
                   help="A mapping of a destination IP to"
                        " all CSPF paths for that destination IP" )
   _detailsPresent = Bool( default=False,
                           help="Private attribute to indicate that the"
                                " details submodel is present" )

   def render( self ):
      if self._detailsPresent:
         for key in sorted( self.pathIps ):
            self.pathIps[ key ].renderEntryDetail()
      else:
         if CspfToggleLib.toggleCspfTunnelNamesEnabled():
            renderTable( self.pathIps )
         else:
            print( entryFormatStr % ( 'Destination', 'Path ID',
                                      'Constraint', 'Path' ) )
            for dstAddr in sorted( self.pathIps ):
               dstStr = ""
               dst = self.pathIps[ dstAddr ]
               first = True
               for key in sorted( dst.paths ):
                  dstStr += entryFormatStrIpPathId % ( dstAddr, str( key ) )
                  path = dst.paths[ key ]
                  dstStr += genPathEntryStr( path.constraint,
                                             path.hops,
                                             path.status )
                  dstStr += "\n"
                  if first:
                     first = False
                     dstAddr = ''
               print( dstStr, end='' )

class CspfPathVrfModel( Model ):
   v4Info = Submodel( valueType=CspfPathAfModel, optional=True,
                      help="CSPF P2MP path information for IPv4 address family" )
   v6Info = Submodel( valueType=CspfPathAfModel, optional=True,
                      help="CSPF P2MP path information for IPv6 address family" )

   def render( self ):
      for af in [ self.v4Info, self.v6Info ]:
         if af:
            af.render()

class CspfPathModel( Model ):
   __revision__ = 2
   vrfs = Dict( keyType=str, valueType=CspfPathVrfModel,
                help="A mapping between Vrf and information of all CSPF paths"
                     " in that vrf" )

   def render( self ):
      for key in sorted( self.vrfs ):
         self.vrfs[ key ].render()

   def degrade( self, dictRepr, revision ):
      if revision == 1:
         # Changes from revision 1 -> revsion 2
         # - Renames includeAdminGroup attribute to includeAllAdminGroup
         # - Adds a new attribute includeAnyAdminGroup
         # - Replaces ipAddrs dict attribute with hops list
         for cspfPathVrfs in dictRepr[ 'vrfs' ].values():
            for afInfo in [ 'v4Info', 'v6Info' ]:
               if afInfo in cspfPathVrfs:
                  for pathIp in cspfPathVrfs[ afInfo ][ 'pathIps' ].values():
                     for path in pathIp[ 'paths' ].values():
                        path[ 'ipAddrs' ] = {}
                        for idx, hop in enumerate( path[ 'hops' ] ):
                           path[ 'ipAddrs' ][ idx ] = hop[ 'ipAddr' ]
                        del path[ 'hops' ]
                        constraint = path[ 'constraint' ]
                        agValue = constraint.pop( 'includeAllAdminGroup', None )
                        constraint[ 'includeAdminGroup' ] = agValue
                        constraint.pop( 'includeAnyAdminGroup', None )
      return dictRepr

class CspfPathByIdEntryModel( Model ):

   constraint = Submodel( valueType=CspfPathConstraintModel, help="Path constraint" )
   hops = List( valueType=CspfPathHopModel,
                help="Hops specifying the path to the destination" )
   status = Enum( values=PATH_STATUS_MAP, help="Status of path computation" )

   class Details( Model ):
      refreshReqSeq = Int( help="Request sequence number" )
      refreshRespSeq = Int( help="Response sequence number", optional=True )
      changeCount = Int( help="Number of times path updated", optional=True )
      lastUpdatedTime = Float( help="UTC timestamp of the last path update",
                               optional=True )
      autoUpdate = Bool( help="Automatically re-optimize the path" )
      bwLocallyAccounted = Bool( help="Bandwidth is locally accounted as reserved",
                                 optional=True )

   details = Submodel( valueType=Details, help="Detailed path information",
                       optional=True )
   dstAddr = IpGenericAddress( help="Address for destination", optional=True )
   pathId = Int( help="Path ID", optional=True )
   _detailsPresent = Bool( default=False, help="Private attribute to indicate that"
                           " the details submodel is present" )

   def render( self ):
      if not self._detailsPresent:
         if CspfToggleLib.toggleCspfTunnelNamesEnabled():
            pathDestination = CspfPathDestIpModel()
            path = CspfPathEntryModel()
            path.constraint = self.constraint
            path.hops = self.hops
            path.status = self.status
            pathDestination.paths = {
               self.pathId : path
            }
            renderTable( { self.dstAddr : pathDestination } )
         else:
            pathStr = ""
            pathStr += entryFormatStr % \
               ( 'Destination', 'Path ID', 'Constraint', 'Path' )
            pathStr += "\n"
            if self.dstAddr:
               pathStr += entryFormatStrIpPathId % ( self.dstAddr, self.pathId )
               pathStr += genPathEntryStr( self.constraint, self.hops, self.status )
            pathStr += "\n"
            print( pathStr, end='' )
      elif self.dstAddr:
         pathStr = ""
         pathStr += "Destination: %s\n" % self.dstAddr
         pathStr += "  Path ID: %d\n" % self.pathId
         pathStr += genPathEntryStrDetail( self.constraint, self.hops,
                                             self.status, self.details )
         print( pathStr, end='' )

class CspfPathByIdVrfModel( Model ):
   v4Info = Submodel( valueType=CspfPathByIdEntryModel, optional=True,
                      help="CSPF path information for IPv4 address family" )
   v6Info = Submodel( valueType=CspfPathByIdEntryModel, optional=True,
                      help="CSPF path information for IPv6 address family" )

   def render( self ):
      for af in [ self.v4Info, self.v6Info ]:
         if af:
            af.render()

class CspfPathByIdModel( Model ):
   vrfs = Dict( keyType=str, valueType=CspfPathByIdVrfModel,
                help="A mapping between Vrf and information of all CSPF paths"
                     " in that vrf" )

   def render( self ):
      for key in sorted( self.vrfs ):
         self.vrfs[ key ].render()

class CspfExplicitHopModel( Model ):
   hop = IpGenericAddress( help="Interface address or TE Router ID of the hop" )
   node = Bool( help="The include hop constraint is a node" )

class CspfLeafPathConstraintModel( Model ):
   explictPath = List( help="Nodes or links included in this path",
                       valueType=CspfExplicitHopModel,
                       optional=True )

class PathNotFoundReasonModel( Model ):
   reason = Enum( values=PATH_NOT_FOUND_REASON_MAP, default='none',
                  help="The reason why path is not found" )
   details = Str( optional=True,
                  help="Additional data on the reason why path is not found" )

class CspfLeafPathModel( Model ):
   _dstAddr = IpGenericAddress( help="Address for destination" )
   _detailsPresent = Bool( default=False,
                           help="Private attribute to indicate that the"
                           " details submodel is present" )
   hops = List( valueType=CspfPathHopModel,
                help="Hops specifying the path to the destination" )
   status = Enum( values=PATH_STATUS_MAP, help="Status of path computation" )

   class Details( Model ):
      constraint = Submodel( valueType=CspfLeafPathConstraintModel,
                             help="Path constraint" )
      pathChangeCount = Int( help="Number of times path updated", default=0 )
      lastUpdatedTime = Float( help="UTC timestamp of the last path update",
                               default=0.0 )
      pathNotFoundReason = Submodel( valueType=PathNotFoundReasonModel,
                                     help="Reason why path is not found",
                                     optional=True )

   details = Submodel( valueType=Details,
                       help="Detailed path information",
                       optional=True )

   def getConstraintList( self ):
      constraintList = []
      if self.details.constraint.explictPath:
         for includeHop in self.details.constraint.explictPath:
            configOption = 'strict'
            if includeHop.node:
               configOption += ', node'
            constraintList.append( f"include hop {includeHop.hop} ({configOption})" )
      return constraintList

class CspfPathTreeConstraintModel( Model ):
   explicitPath = Bool( default=False,
                        help="Every hop in the tree must be specified" )
   excludeAdminGroup = excludeAdminGroupInt()
   excludeAdminGroupsExtended = excludeAdminGroupsExtendedList()
   includeAllAdminGroup = includeAllAdminGroupInt()
   includeAllAdminGroupsExtended = includeAllAdminGroupsExtendedList()
   includeAnyAdminGroup = includeAnyAdminGroupInt()
   includeAnyAdminGroupsExtended = includeAnyAdminGroupsExtendedList()

def getLeavesSortedByStatus( leafPaths ):
   sortedLeaves = sorted( leafPaths )
   leavesByStatus = { status : [ key for key in sortedLeaves
                      if leafPaths[ key ].status == status ] for status
                         in PATH_STATUS_MAP }
   return leavesByStatus[ 'pathFound' ] + leavesByStatus[ 'cspfPending' ] + \
      leavesByStatus[ 'pathNotFound' ]

class CspfPathTreeModel( Model ):
   _treeId = Int( help="Tree ID" )
   if CspfToggleLib.toggleCspfTunnelNamesEnabled():
      tunnelName = Str( help="Tunnel Name" )
   leafPaths = Dict( keyType=IpGenericAddress,
                     valueType=CspfLeafPathModel,
                     help="A mapping of a destination IP to"
                     " all CSPF leaf paths for that destination IP" )
   _detailsPresent = Bool( default=False,
                           help="Private attribute to indicate that the"
                           " details submodel is present" )

   class Details( Model ):
      treePathChangeCount = Int( help="Number of times tree's hops updated",
                                 default=0 )
      explicitRefreshReqSeq = Int( help="Reoptimization request sequence number",
                                   default=0 )
      refreshReqSeq = Int( help="Configuration request sequence number", default=0 )
      pendingExplicitRespSeq = Bool( help="Reoptimization response sequence "
                                     "number is pending", default=True )
      pendingRefreshRespSeq = Bool( help="Configuration response sequence number "
                                    "is pending", default=True )
      constraint = Submodel( valueType=CspfPathTreeConstraintModel,
                             help="Path constraint" )

   details = Submodel( valueType=Details,
                       help="Detailed path information",
                       optional=True )

   def getConstraintList( self ):
      constraintList = []
      if self.details.constraint.explicitPath:
         constraintList.append( "explicit path" )
      constraint = self.details.constraint
      if constraint.includeAllAdminGroup or constraint.includeAllAdminGroupsExtended:
         addAdminGroupConstraint( constraintList,
                                  constraint.includeAllAdminGroup,
                                  constraint.includeAllAdminGroupsExtended,
                                  "include all" )

      if constraint.includeAnyAdminGroup or constraint.includeAnyAdminGroupsExtended:
         addAdminGroupConstraint( constraintList,
                                  constraint.includeAnyAdminGroup,
                                  constraint.includeAnyAdminGroupsExtended,
                                  "include any" )

      if constraint.excludeAdminGroup or constraint.excludeAdminGroupsExtended:
         addAdminGroupConstraint( constraintList,
                                  constraint.excludeAdminGroup,
                                  constraint.excludeAdminGroupsExtended,
                                  "exclude" )
      return constraintList

   def renderEntryDetail( self ):
      dstStr = ""
      indent = 3 * space
      dstStr += f"Tree ID: {self._treeId}\n"
      first = True
      if CspfToggleLib.toggleCspfTunnelNamesEnabled():
         dstStr += f"{indent}Tunnel Name: {self.tunnelName}\n"
      for constraint in self.getConstraintList():
         if first:
            dstStr += f"{indent}Tree constraint: {constraint}\n"
            first = False
         else:
            dstStr += ( f"{space * 20}{constraint}\n" )
      if self.details.refreshReqSeq is not None:
         pending = ""
         if self.details.pendingRefreshRespSeq:
            pending = " (CSPF response pending)"
         dstStr += f"{indent}Configuration request sequence number: "\
            f"{self.details.refreshReqSeq}{pending}\n"
      if self.details.explicitRefreshReqSeq is not None:
         pending = ""
         if self.details.pendingExplicitRespSeq:
            pending = " (CSPF response pending)"
         dstStr += f"{indent}Reoptimization request sequence number: "\
            f"{self.details.explicitRefreshReqSeq}{pending}\n"
      if self.details.treePathChangeCount is not None:
         dstStr += f"{indent}Number of times tree's hops updated: "\
            f"{self.details.treePathChangeCount}\n"
      if not self.leafPaths:
         dstStr += f"{indent}No leaves are configured for this tree\n"
      for key in getLeavesSortedByStatus( self.leafPaths ):
         leafModel = self.leafPaths[ key ]
         dstStr += f"{indent}Destination: {str( key )}\n"
         first = True
         for constraint in leafModel.getConstraintList():
            if first:
               dstStr += f"{space * 6}Constraint: {constraint}\n"
               first = False
            else:
               dstStr += f"{space * 18}{constraint}\n"
         if leafModel.details.pathChangeCount:
            dstStr += f"{space * 6}Number of times path's hops updated: "\
               f"{leafModel.details.pathChangeCount}\n"
         if leafModel.details.lastUpdatedTime:
            dstStr += f"{space * 6}Last updated: "\
               f"{Ark.timestampToStr( leafModel.details.lastUpdatedTime )}\n"
         if len( leafModel.hops ) == 0:
            prefixPathLine = "Path: "
            dstStr += f"""{space * 6}{prefixPathLine}{PATH_STATUS_MAP[
               leafModel.status ]}\n"""
            pathNotFoundReason = leafModel.details.pathNotFoundReason
            if pathNotFoundReason is not None and \
               pathNotFoundReason.reason != "none":
               dstStr += f"""{space * (6 + len(prefixPathLine))}Reason: {
                  PATH_NOT_FOUND_REASON_MAP[pathNotFoundReason.reason]} {
                     pathNotFoundReason.details}\n"""
            continue
         includeIpsPresent = False
         headings = ( ( 'Ingress IP', 'l' ), ( 'TE router ID', 'l' ),
                        ( 'Include IP', 'l' ) )
         pathTable = TableOutput.createTable( headings )
         for hop in leafModel.hops:
            includeIpStr = ", ".join( str( i ) for i in hop.includeIps or [] )
            if includeIpStr:
               includeIpsPresent = True
            pathTable.newRow( str( hop.ipAddr ), str( hop.teRouterId ),
                              includeIpStr )
         pathEntries = pathTable.output().split( '\n' )
         if not includeIpsPresent:
            # Strip off the `Include IP` heading and its horizontal border
            pathEntries[ 0 ] = pathEntries[ 0 ].rsplit( 'Include IP' )[ 0 ]
            pathEntries[ 1 ] = pathEntries[ 1 ].rsplit( ' -', 1 )[ 0 ]
         dstStr += f"{space*6}Path: {pathEntries[ 0 ].rstrip()}\n"
         for pathEntry in pathEntries[ 1 : ]:
            dstStr += f"{space * 12}{pathEntry}\n"
      print( dstStr )

def renderedTunnelName( tunnelName ):
   # This theoretically should not happen,
   # but since the output is grouped on tunnel names,
   # a dash line is printed in the unlikely case a tunnel has
   # an empty name; this is to avoid confusion whether a path is
   # belonging to the prior tunnel or to an empty named one
   if not tunnelName:
      tunnelName = '-'
   return tunnelName if len( tunnelName ) < MAX_TUNNEL_NAME_COLUMN_WIDTH \
                     else f"{tunnelName[:MAX_TUNNEL_NAME_COLUMN_WIDTH-3]}..."

class CspfPathAfP2mpModel( Model ):
   trees = Dict( keyType=int, valueType=CspfPathTreeModel,
                 help="A mapping of a tree ID to all CSPF P2MP trees"
                      " with that tree ID" )
   _detailsPresent = Bool( default=False,
                           help="Private attribute to indicate that the"
                                " details submodel is present" )

   def render( self ):
      if self._detailsPresent:
         for treeId in sorted( self.trees ):
            self.trees[ treeId ].renderEntryDetail()
         return
      headings = [ 'Tree ID', 'Destination', 'Path' ]
      if CspfToggleLib.toggleCspfTunnelNamesEnabled():
         headings = [ 'Tunnel Name' ] + headings
      treeTable = TableOutput.createTable( headings )
      terminalWidth = TableOutput.terminalWidth()
      treeColumnMaxWidth = 10
      leafColumnMaxWidth = 15
      maxPathWidth = terminalWidth - ( treeColumnMaxWidth + leafColumnMaxWidth )
      if CspfToggleLib.toggleCspfTunnelNamesEnabled():
         maxPathWidth -= MAX_TUNNEL_NAME_COLUMN_WIDTH
      if maxPathWidth <= 0:
         maxPathWidth = 1
      tunnelFormat = TableOutput.Format( justify='left',
                                         maxWidth=MAX_TUNNEL_NAME_COLUMN_WIDTH )
      tunnelFormat.noPadLeftIs( True )
      treeFormat = TableOutput.Format( justify='right', maxWidth=treeColumnMaxWidth )
      leafFormat = TableOutput.Format( justify='left',
                                       maxWidth=leafColumnMaxWidth,
                                       wrap=True )
      pathFormat = TableOutput.Format(
         justify='left', maxWidth=maxPathWidth, wrap=True )
      if CspfToggleLib.toggleCspfTunnelNamesEnabled():
         treeTable.formatColumns( tunnelFormat, treeFormat, leafFormat, pathFormat )
         previousTunnelName = None
      else:
         treeTable.formatColumns( treeFormat, leafFormat, pathFormat )

      def getSortingKey( entry ):
         treeId, tree = entry
         if CspfToggleLib.toggleCspfTunnelNamesEnabled():
            return ( tree.tunnelName, treeId )
         return treeId

      for treeId, tree in sorted( self.trees.items(), key=getSortingKey ):
         tree = self.trees[ treeId ]
         if CspfToggleLib.toggleCspfTunnelNamesEnabled():
            tunnelName = tree.tunnelName
            tunnelName = renderedTunnelName( tunnelName )
            if tunnelName == previousTunnelName:
               tunnelName = ''
            else:
               previousTunnelName = tunnelName
         for leafIp in getLeavesSortedByStatus( tree.leafPaths ):
            leaf = tree.leafPaths[ leafIp ]
            if len( leaf.hops ):
               ipAddrList = [ str( hop.ipAddr ) for hop in leaf.hops ]
               hopStr = ", ".join( ipAddrList )
            else:
               hopStr = PATH_STATUS_MAP[ leaf.status ]
            if CspfToggleLib.toggleCspfTunnelNamesEnabled():
               treeTable.newRow( tunnelName, treeId, str( leafIp ), hopStr )
               tunnelName = ""
            else:
               treeTable.newRow( treeId, str( leafIp ), hopStr )
            treeId = ""
         if not tree.leafPaths:
            if CspfToggleLib.toggleCspfTunnelNamesEnabled():
               treeTable.newRow( tunnelName, treeId, '-', '-' )
            else:
               treeTable.newRow( treeId, '-', '-' )
         treeTable.newRow()
      print( treeTable.output() )

class CspfPathVrfP2mpModel( Model ):
   v4Info = Submodel( valueType=CspfPathAfP2mpModel, optional=True,
                      help="CSPF P2MP path information for IPv4 address family" )
   v6Info = Submodel( valueType=CspfPathAfP2mpModel, optional=True,
                      help="CSPF P2MP path information for IPv6 address family" )

   def render( self ):
      for af in [ self.v4Info, self.v6Info ]:
         if af:
            af.render()

class CspfPathP2mpModel( Model ):
   vrfs = Dict( keyType=str, valueType=CspfPathVrfP2mpModel,
                help="A mapping between Vrf and information of all CSPF P2MP paths"
                     " in that vrf" )

   def render( self ):
      for key in sorted( self.vrfs ):
         self.vrfs[ key ].render()

class IsisTopologyInfoModel( Model ):
   level = Int( help="IS-IS level" )

class OspfTopologyInfoModel( Model ):
   version = Int( help="OSPF version" )
   instanceId = Int( help="OSPF instance ID" )
   areaId = Ip4Address( help="OSPF area ID" )

class TopologyModel( Model ):
   isis = Submodel( valueType=IsisTopologyInfoModel, optional=True,
                    help="IS-IS topology information" )
   ospf = Submodel( valueType=OspfTopologyInfoModel, optional=True,
                    help="OSPF topology information" )

class LinkAttributesModel( Model ):
   interfaceAddresses = List( valueType=InterfaceAddress,
                              help="List of local Interface Addresses"
                                   " forming the adjacency" )
   neighborAddresses = List( valueType=InterfaceAddress,
                             help="List of remote Neighbor Addresses"
                                  " forming the adjacency" )
   topology = Submodel( valueType=TopologyModel,
                        help="Topology type information" )
   sourceTeRouterId = IpGenericAddress( help="Local TE Router ID" )
   destinationTeRouterId = IpGenericAddress( help="Remote TE Router ID" )
   sourceIgpRouterId = Str( help="Local IGP Router ID" )
   destinationIgpRouterId = Str( help="Remote IGP Router ID" )
   networkType = Enum( values=( "p2p", "lan" ), help="Network type" )
   metric = Int( optional=True, help="TE Link cost" )
   maxLinkBw = Int( optional=True,
                    help="Maximum bandwidth (bps) that can be used"
                         " on Directed Link" )
   maxReservableBw = Int( optional=True,
                          help="Maximum bandwidth (bps) that can be"
                               " reserved on Directed Link" )
   unreservedBw = Submodel( valueType=ReservablePriorityBandwidth,
                            optional=True,
                            help="Maximum bandwidth reservable for a priority" )
   sharedRiskLinkGroups = List( valueType=SharedRiskLinkGroupWithCost,
                                optional=True,
                                help="List of Shared Risk Link Groups" )
   administrativeGroup = Int( optional=True,
                              help="Administrative Group of the Link" )
   administrativeGroupsExtended = List( valueType=int, optional=True,
                                        help="Extended Administrative Groups of the"
                                             " Link" )

class CspfPathLinkAttrsModel( Model ):
   found = Bool( help="Link found in topology" )
   # hopAddress and linkAttributes are mutually exclusive -
   # the model should have one or the other
   hopAddress = IpGenericAddress( "Next hop address", optional=True )
   linkAttributes = Submodel( valueType=LinkAttributesModel,
                              help="Link attributes",
                              optional=True )

class CspfPathLinksEntryModel( Model ):
   status = Enum( values=PATH_STATUS_MAP,
                  help="Status of path computation" )
   destinationOnSourceNode = Bool( help="Destination IP address is on"
                                        " the source node" )
   # links is not included if destinationOnSourceNode is true
   # or if status != "pathFound"
   links = Dict( keyType=int, valueType=CspfPathLinkAttrsModel, optional=True,
                 help="TE information of each link in the CSPF path" )

class CspfPathLinksDestIpModel( Model ):
   _dstAddr = IpGenericAddress( help="Address for destination" )
   paths = Dict( keyType=int, valueType=CspfPathLinksEntryModel,
                 help="A mapping of an id representing a constraint to"
                      " the corresponding CSPF path" )

class CspfPathLinksAfModel( Model ):
   pathIps = Dict( keyType=IpGenericAddress, valueType=CspfPathLinksDestIpModel,
                   help="A mapping of a destination IP to"
                        " all CSPF paths for that destination IP" )

class CspfPathLinksVrfModel( Model ):
   v4Info = Submodel( valueType=CspfPathLinksAfModel, optional=True,
                      help="CSPF path information for IPv4 address family" )
   v6Info = Submodel( valueType=CspfPathLinksAfModel, optional=True,
                      help="CSPF path information for IPv6 address family" )

class CspfPathLinksModel( Model ):
   vrfs = Dict( keyType=str, valueType=CspfPathLinksVrfModel,
                    help="A mapping between VRF and information of all CSPF paths"
                         " in that VRF" )

class CspfPathByIdLinksAfModel( Model ):
   dstAddr = IpGenericAddress( help="Address for destination" )
   pathId = Int( help="Path ID" )
   status = Enum( values=PATH_STATUS_MAP,
                  help="Status of path computation" )
   destinationOnSourceNode = Bool( help="Destination IP address is on"
                                        " the source node" )
   # links is not included if destinationOnSourceNode is true
   # or if status != "pathFound"
   links = Dict( keyType=int, valueType=CspfPathLinkAttrsModel, optional=True,
                 help="TE information of each link in the CSPF path" )

class CspfPathByIdLinksVrfModel( Model ):
   v4Info = Submodel( valueType=CspfPathByIdLinksAfModel, optional=True,
                      help="CSPF path information for IPv4 address family" )
   v6Info = Submodel( valueType=CspfPathByIdLinksAfModel, optional=True,
                      help="CSPF path information for IPv6 address family" )

class CspfPathByIdLinksModel( Model ):
   vrfs = Dict( keyType=str, valueType=CspfPathByIdLinksVrfModel,
                help="A mapping between VRF and information of all CSPF paths"
                     " in that VRF" )


tilfaEntryFormatStr = "%s%-15s %-30s %-15s"
tilfaEntryFormatStrIp = "%s%-15s "
tilfaEntryFormatStrWithoutIp = "%-30s %-15s"
space = " "

class SystemIdHostnameModel( Model ):
   sysId = Str( help='System identifier' )
   hostname = Str( help='Hostname', optional=True )

class RouterIdHostnameModel( Model ):
   routerId = Str( help='OSPF Router ID' )
   hostname = Str( help='Hostname', optional=True )

class FlexAlgoPathConstraintModel( Model ):
   metricType = Enum( values=( 'igp', 'minDelay', 'te' ), help="Metric type" )
   excludeSrlg = List( valueType=int, optional=True,
                       help="List of SRLG IDs excluded" )
   excludeSrlgMode = Enum( values=( 'loose', 'strict' ), optional=True,
                           help="Type of SRLG exclusion" )
   includeAllAdminGroup = includeAllAdminGroupInt()
   includeAllAdminGroupsExtended = includeAllAdminGroupsExtendedList()
   includeAnyAdminGroup = includeAnyAdminGroupInt()
   includeAnyAdminGroupsExtended = includeAnyAdminGroupsExtendedList()
   excludeAdminGroup = excludeAdminGroupInt()
   excludeAdminGroupsExtended = excludeAdminGroupsExtendedList()
   # Hidden attribute to map srlgId to srlg name
   _srlgIdToNameMap = Submodel( valueType=SrlgIdToNameMapHelper, optional=True,
                                help="SrlgId to name map" )

   def getConstraintDetailList( self, algoId, algoName ):
      constraintList = []
      # toStr method add an extra space in front
      constraintList.append( "flex-algo algorithm%s" % toStr( algoId, algoName ) )

      metricTypeStr = 'IGP'
      if self.metricType == 'minDelay':
         metricTypeStr = 'MIN-DELAY'
      if self.metricType == 'te':
         metricTypeStr = 'TE'

      indent = " " * 2
      constraintList.append( indent + "metric type %s" % metricTypeStr )

      if self.excludeSrlg:
         startWidth = len( indent ) + 20
         subsequent_indent = indent + SPACE_23
         output = wrapText(
            indent + "exclude SRLG %s" % self._srlgIdToNameMap.printSrlgMap(
               self.excludeSrlg ),
            startWidth=startWidth,
            subsequent_indent=subsequent_indent )
         constraintList.append( output )
         constraintList.append( indent + "SRLG %s" % self.excludeSrlgMode )

      if self.includeAllAdminGroup or self.includeAllAdminGroupsExtended:
         addAdminGroupConstraint( constraintList, self.includeAllAdminGroup,
               self.includeAllAdminGroupsExtended, "include all" )

      if self.includeAnyAdminGroup or self.includeAnyAdminGroupsExtended:
         addAdminGroupConstraint( constraintList, self.includeAnyAdminGroup,
               self.includeAnyAdminGroupsExtended, "include any" )

      if self.excludeAdminGroup or self.excludeAdminGroupsExtended:
         addAdminGroupConstraint( constraintList, self.excludeAdminGroup,
               self.excludeAdminGroupsExtended, "exclude" )
      return constraintList

class TilfaPathConstraintModel( Model ):
   # TI-LFA constraints
   excludeIntf = Interface( help="Interface excluded from this path",
                            optional=True )
   excludeNode = Str( help="Node excluded from this path", optional=True )
   excludeSrlgOfIntf = Submodel( valueType=IntfSrlg, optional=True,
                                  help="SRLG exclude of interface" )
   excludeSrlgMode = Enum( values=( 'loose', 'strict' ), optional=True,
                           help="Type of SRLG exclusion" )
   algoId = Int( help='Algorithm Id' )
   algoName = Str( help='Algorithm name', optional=True )
   # Flex-Algo Constraints
   flexAlgo = Submodel( valueType=FlexAlgoPathConstraintModel, optional=True,
                        help="FlexAlgo constraint" )

   def getConstraintList( self ):
      constraintList = []
      if self.excludeIntf:
         constraintList.append( "exclude %s" % self.excludeIntf.stringValue )
      elif self.excludeNode:
         constraintList.append( "exclude node %s" % self.excludeNode )
      if self.excludeSrlgOfIntf:
         if self.excludeSrlgOfIntf.srlgIds:
            # printSrlgMap add an extra space at the begining of output.
            output = wrapText( "exclude SRLG of {}:{}".format(
                               self.excludeSrlgOfIntf.interface.stringValue,
                               self.excludeSrlgOfIntf.printSrlgMap() ) )
            constraintList.append( output )
            constraintList.append( "SRLG %s" % self.excludeSrlgMode )
         else:
            constraintList.append( "exclude SRLG of %s" %
                                   self.excludeSrlgOfIntf.interface.stringValue )
            constraintList.append( "SRLG %s" % self.excludeSrlgMode )

      return constraintList

   def getConstraintDetailList( self ):
      constraintList = self.getConstraintList()
      if self.algoId == 0:
         # toStr method add an extra space in front
         constraintList.append( "algorithm%s" % toStr( self.algoId, self.algoName ) )
      else:
         flexAlgoConstraintList = self.flexAlgo.getConstraintDetailList(
            self.algoId, self.algoName )
         constraintList += flexAlgoConstraintList
      return constraintList

class TilfaPathDetails( Model ):
   refreshReqSeq = Int( help="Request sequence number" )
   refreshRespSeq = Int( help="Response sequence number", optional=True )
   changeCount = Int( help="Number of times path updated", optional=True )
   lastUpdatedTime = Float( help="UTC timestamp of the last path update",
                               optional=True )

class TilfaPathEntryModelBase( Model ):
   _pathId = Int( help="Path ID" )
   constraint = Submodel( valueType=TilfaPathConstraintModel,
                          help="Path constraint" )
   status = Enum( values=PATH_STATUS_MAP,
                  help="Status of path computation" )
   details = Submodel( valueType=TilfaPathDetails, help="Detailed path information",
                       optional=True )

   def getPathLength( self ):
      raise NotImplementedError

   def getHopDetail( self, i ):
      raise NotImplementedError

   def getPqPathStr( self ):
      raise NotImplementedError

   def renderDetails( self, indent ):
      detailStr = ""
      detailStr += "%sRequest sequence number: %d\n" % ( indent,
                     self.details.refreshReqSeq )
      if self.details.refreshRespSeq is not None:
         detailStr += "%sResponse sequence number: %d\n" % ( indent,
                        self.details.refreshRespSeq )
      if self.details.changeCount is not None:
         detailStr += "%sNumber of times path updated: %d\n" % ( indent,
                        self.details.changeCount )
      if self.details.lastUpdatedTime:
         detailStr += "{}Last updated: {}\n".format( indent,
                                                     Ark.timestampToStr(
               self.details.lastUpdatedTime ) )
      detailStr += f"{indent}ID: 0x{self._pathId:x}\n"
      return detailStr

   def renderPathDetails( self, indent ):
      pathStr = ""
      lenSysIds = self.getPathLength()
      if lenSysIds == 1:
         suffix = ' [PQ-node]'
         hopDetail = self.getHopDetail( 0 )
         pathStr += f"{indent}{hopDetail}{suffix}\n"
      else:
         suffixP = ' [P-node]'
         suffixQ = ' [Q-node]'
         for key in range( lenSysIds ):
            if key == 0:
               suffix = suffixP
            elif key == ( lenSysIds - 1 ):
               suffix = suffixQ
            else:
               suffix = ''
            hopDetail = self.getHopDetail( key )
            pathStr += f"{indent}{hopDetail}{suffix}\n"
      return pathStr

   def summaryEntryGenerator( self ):
      constraints = self.constraint.getConstraintList()
      path = []
      if self.getPathLength() == 0:
         path = [ PATH_STATUS_MAP[ self.status ] ]
      else:
         path = self.getPqPathStr()

      algoName = ( self.constraint.algoName or str( self.constraint.algoId ) )
      yield from zip_longest( [ algoName ], constraints, path, fillvalue="" )

   def getPathEntryStrDetail( self ):
      pathEntryStr = ""
      lenPath = self.getPathLength()
      constraintList = self.constraint.getConstraintDetailList()
      first = True
      for constraint in constraintList:
         if first:
            pathEntryStr += ( "%sPath constraint: %s\n" %
                              ( space * 6, constraint ) )
            first = False
         else:
            pathEntryStr += ( f"{space * 23}{constraint}\n" )
      pathEntryStr += self.renderDetails( space * 9 )
      if lenPath == 0:
         pathEntryStr += f"{space * 9}Path: {PATH_STATUS_MAP[ self.status ]}\n"
      else:
         pathEntryStr += f"{space * 9}Path:\n"
         pathEntryStr += self.renderPathDetails( space * 12 )
      return pathEntryStr

class TilfaPathEntryModel( TilfaPathEntryModelBase ):
   sysIds = List( valueType=SystemIdHostnameModel,
                  help="Ordered list of system identifiers or hostname specifying "
                  "the path to the destination" )

   def getPathLength( self ):
      return len( self.sysIds )

   def getHopDetail( self, i ):
      return self.sysIds[ i ].hostname or self.sysIds[ i ].sysId

   def getPqPathStr( self ):
      return [ s.hostname or s.sysId for s in self.sysIds ]

class TilfaPathOspfEntryModel( TilfaPathEntryModelBase ):
   routerIds = List( valueType=RouterIdHostnameModel,
                     help="Ordered list of Router IDs or hostname specifying "
                     "the path to the destination" )

   def getPathLength( self ):
      return len( self.routerIds )

   def getHopDetail( self, i ):
      return self.routerIds[ i ].hostname or self.routerIds[ i ].routerId

   def getPqPathStr( self ):
      return [ s.hostname or s.routerId for s in self.routerIds ]

class TilfaPathDestModelBase( Model ):
   _dstId = Str( help="Destination System identifier" )
   hostname = Str( help="Hostname", optional=True )

   def summaryEntryGenerator( self ):
      dest = ( self.hostname or self._dstId )
      firstPrintDone = False
      for pathId in sorted( self.pathIds ):
         pathEntry = self.pathIds[ pathId ]
         for algoName, constraint, path in pathEntry.summaryEntryGenerator():
            yield "" if firstPrintDone else dest, algoName, constraint, path
            firstPrintDone = True
         # Print empty line between paths
         yield "", "", "", ""

   def renderDetailEntry( self ):
      dstStr = ""
      dstOrHostname = ( self.hostname or self._dstId )
      dstStr = f"{space * 3}Destination: {dstOrHostname}\n"
      for key in sorted( self.pathIds ):
         dstStr += self.pathIds[ key ].getPathEntryStrDetail()
      print( dstStr )

class TilfaPathDestModel( TilfaPathDestModelBase ):
   pathIds = Dict( keyType=int, valueType=TilfaPathEntryModel,
                   help="A mapping of an ID representing a constraint to the"
                   " corresponding TI-LFA path" )

class TilfaPathOspfDestModel( TilfaPathDestModelBase ):
   pathIds = Dict( keyType=int, valueType=TilfaPathOspfEntryModel,
                   help="mapping of ID representing a constraint to the"
                   " corresponding TI-LFA path" )

class TilfaPathTopoIdModelBase( Model ):
   _detailsPresent = Bool( default=False,
                           help="Private attribute to indicate that the detail"
                           " submodel is present" )

   def renderSummaryOutput( self ):
      headings = ( ( 'Destination', 'l' ), ( 'Algorithm', 'l' ),
                   ( 'Constraint', 'l' ), ( 'Path', 'l' ) )
      table = TableOutput.createTable( headings )
      for dstEntry in self.destinations.values():
         for dest, algoName, constraint, path in dstEntry.summaryEntryGenerator():
            table.newRow( dest, algoName, constraint, path )
      print( table.output() )

   def render( self ):
      if not self._detailsPresent:
         self.renderSummaryOutput()
         return
      for key in sorted( self.destinations ):
         self.destinations[ key ].renderDetailEntry()

class TilfaPathTopoIdModel( TilfaPathTopoIdModelBase ):
   destinations = Dict( keyType=str, valueType=TilfaPathDestModel,
                        help="A mapping of path systemIds to all TI-LFA "
                        "paths for that destination" )

class TilfaPathOspfTopoIdModel( TilfaPathTopoIdModelBase ):
   destinations = Dict( keyType=str, valueType=TilfaPathOspfDestModel,
                        help="A mapping of destination Router ID to TI-LFA "
                        "path for that destination" )

class TilfaPathAfModelBase( Model ):

   def render( self ):
      for topoId, topoModel in self.topologies.items():
         if topoModel.destinations:
            print( "%sTopology ID: Level-%d" % ( space * 3, topoId ) )
            topoModel.render()

class TilfaPathOspfAfModelBase( Model ):

   def render( self ):
      for area, topoModel in self.topologies.items():
         if topoModel.destinations:
            print( "%sTopology ID: Area %s" % ( space * 3, area ) )
            topoModel.render()

class TilfaPathAfModel( TilfaPathAfModelBase ):
   topologies = Dict( keyType=int, valueType=TilfaPathTopoIdModel,
                      help="A mapping of topology ID to all TI-LFA paths" )

class TilfaPathOspfAfModel( TilfaPathOspfAfModelBase ):
   topologies = Dict( keyType=str, valueType=TilfaPathOspfTopoIdModel,
                      help="A mapping of topology ID to all TI-LFA paths" )

class TilfaPathVrfModelBase( Model ):
   def render( self ):
      if self.v4Info and self.v4Info.topologies:
         print( "TI-LFA paths for IPv4 address family" )
         self.v4Info.render()
      if self.v6Info and self.v6Info.topologies:
         print( "TI-LFA paths for IPv6 address family" )
         self.v6Info.render()

class TilfaPathVrfModel( TilfaPathVrfModelBase ):
   v4Info = Submodel( valueType=TilfaPathAfModel, optional=True,
                       help="TI-LFA path information for IPv4 address family" )
   v6Info = Submodel( valueType=TilfaPathAfModel, optional=True,
                      help="TI-LFA path information for IPv6 address family" )

class TilfaPathOspfVrfModel( TilfaPathVrfModelBase ):
   v4Info = Submodel( valueType=TilfaPathOspfAfModel, optional=True,
                       help="TI-LFA path information for IPv4 address family" )
   v6Info = Submodel( valueType=TilfaPathOspfAfModel, optional=True,
                      help="TI-LFA path information for IPv6 address family" )

class TilfaPathModelBase( Model ):

   def render( self ):
      for key in sorted( self.vrfs ):
         self.vrfs[ key ].render()

class TilfaPathModel( TilfaPathModelBase ):
   vrfs = Dict( keyType=str, valueType=TilfaPathVrfModel,
                help="A mapping between a vrf and its TI-LFA paths" )

class TilfaPathOspfModel( TilfaPathModelBase ):
   vrfs = Dict( keyType=str, valueType=TilfaPathOspfVrfModel,
                help="A mapping between a vrf and its TI-LFA paths" )
