#!/usr/bin/env python3
# Copyright (c) 2020 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import BasicCli
import CliGlobal
import CliToken
from CliPlugin import RoutingIsisCli
from CliPlugin.VrfCli import VrfExprFactory
from CliPlugin.CspfShowTopologyModel import (
   SharedRiskLinkGroup,
   FlexAlgoAttributesModel,
   NeighborModel,
   ReachabilityModel,
   RouterModel,
   VrfModel,
   IgpInstanceInfoModel,
   IsisModel,
   CommonProtocolModel,
   TopologyModel,
   OspfModel,
   OspfInstanceInfoModel )
from CliPlugin.CspfShowDbCli import splitSmashPath
from CliPlugin.TeCli import getSrlgIdToNameMap, adminGroupDictToDecimalList
import LazyMount
import SharedMem
import ShowCommand
import SmashLazyMount
from IpLibConsts import DEFAULT_VRF
from TypeFuture import TacLazyType
from Toggles import TeToggleLib

gv = CliGlobal.CliGlobal(
   dict( entityManager=None, readerInfo=None, topoDbExport=None, shmemEm=None ) )

ReachabilityByOriginTable = TacLazyType(
   "Routing::Topology::ReachabilityByOriginTable" )
TopoDbHelper = TacLazyType( "Routing::Topology::TopoDbHelper" )

def mountIgpTopoDb( args ):
   topoDbs = {}
   if 'LEVEL' in args:
      levels = [ int( args[ 'LEVEL' ][ -1 ] ) ]
   else:
      levels = [ 1, 2 ]

   if 'ADDR_FAMILY' in args:
      afs = [ 'ip' if args[ 'ADDR_FAMILY' ] == 'ipv4' else 'ip6' ]
   else:
      afs = [ 'ip', 'ip6' ]

   if 'isis' in args:
      igp = [ 'isis' ]
   elif 'ospf' in args:
      igp = [ 'ospf' ]
   else:
      igp = [ 'isis', 'ospf' ]

   for key in gv.topoDbExport:
      path = key.replace( '-', '/' )
      proto, addrFamily, _, igpId = splitSmashPath( path )
      if proto not in igp:
         continue
      if addrFamily not in afs:
         continue
      if proto == 'isis':
         if igpId not in levels:
            continue

      # mount the topodb
      topoDb = SmashLazyMount.mount( gv.entityManager, path,
                                     'Routing::Topology::TopoDb', gv.readerInfo,
                                     autoUnmount=True, unmountTimeout=600 )
      topoDbs[ path ] = topoDb

   return topoDbs

def populateProtocolModel( protoIdModel, topoDb, af ):
   if af == 'ipv4':
      afModel = protoIdModel.ipv4
   elif af == 'ipv6':
      afModel = protoIdModel.ipv6
   srlgIdToNameMap = getSrlgIdToNameMap()
   reachByOrgTbl = ReachabilityByOriginTable()
   _ = TopoDbHelper( topoDb.force(), reachByOrgTbl )
   for routerId, router in topoDb.router.items():
      rtrInfoModel = RouterModel()
      afModel[ routerId.stringValue ] = rtrInfoModel
      # Populating Flex Algo names

      for flexAlgoId in router.flexAlgo.values():
         rtrInfoModel.flexAlgo.append( flexAlgoId )
      if not rtrInfoModel.flexAlgo:
         rtrInfoModel.flexAlgo = None
      for ngbKey in router.neighbor.values():
         ngb = topoDb.neighbor.get( ngbKey )
         if ngb is None:
            continue
         ngbModel = NeighborModel()
         ngbModel.metric = ngb.metric

         if ngbKey.neighborType == 'neighborTypeRouter':
            neighborId = ngbKey.neighborId.routerId.stringValue
            linksModel = rtrInfoModel.p2ps
         elif ngbKey.neighborType == 'neighborTypeNetwork':
            # pylint: disable-next=consider-using-f-string
            neighborId = '{}.{:x}'.format( ngbKey.networkId.drRouterId,
                                           ngbKey.networkId.drNetworkId )
            linksModel = rtrInfoModel.lans
         # pylint: disable-next=unsupported-assignment-operation
         linksModel[ neighborId ] = ngbModel

         for addr in ngb.interfaceAddr.values():
            ngbModel.interfaceAddresses.append( addr.stringValue )

         for addr in ngb.neighborAddr.values():
            ngbModel.neighborAddresses.append( addr.stringValue )

         # Exclude optional empty collections from output model
         if not ngbModel.interfaceAddresses:
            ngbModel.interfaceAddresses = None
         if not ngbModel.neighborAddresses:
            ngbModel.neighborAddresses = None

         # Populating the flex Algo Attributes model
         if ngb.flexAlgoTeAttr:
            faModel = FlexAlgoAttributesModel()
            if ngb.faAdminGroupPresent:
               if TeToggleLib.toggleExtendedAdminGroupEnabled():
                  faModel.administrativeGroupsExtended = \
                        adminGroupDictToDecimalList( ngb.faAdminGroup )
               else:
                  faModel.administrativeGroupsExtended = None
               faModel.administrativeGroup = ngb.faAdminGroup[ 0 ]
            else:
               faModel.administrativeGroup = None
               faModel.administrativeGroupsExtended = None
            if ngb.faMinDelayMetricPresent:
               faModel.minimumDelayMetric = ngb.faMinDelayMetric
            else:
               faModel.minimumDelayMetric = None
            if ngb.faTeMetricPresent:
               faModel.defaultTeMetric = ngb.faTeMetric
            else:
               faModel.defaultTeMetric = None
            if ngb.faSrlgCount:
               for srlgId in ngb.flexAlgoTeAttr.srlg.values():
                  srlg = SharedRiskLinkGroup()
                  srlg.srlgId = srlgId
                  srlg.srlgName = srlgIdToNameMap.get( srlgId, None )
                  faModel.sharedRiskLinkGroups.append( srlg )
            else:
               faModel.sharedRiskLinkGroups = None
            ngbModel.flexAlgoAttributes = faModel
         else:
            ngbModel.flexAlgoAttributes = None

      reachCollection = reachByOrgTbl.reachabilityByOrigin.get( routerId )
      if reachCollection:
         for prefix, pfxInfo in reachCollection.prefixInfo.items():
            reachModel = ReachabilityModel()
            reachModel.metric = pfxInfo.metric
            # pylint: disable=unsupported-assignment-operation
            rtrInfoModel.reachability[ prefix ] = reachModel
            # pylint: enable=unsupported-assignment-operation

      # Exclude optional empty collections from output model
      if not rtrInfoModel.p2ps:
         rtrInfoModel.p2ps = None
      if not rtrInfoModel.lans:
         rtrInfoModel.lans = None
      if not rtrInfoModel.reachability:
         rtrInfoModel.reachability = None

def populateLevelModel( levelModel, topoDb, af ):
   if levelModel is None:
      protoIdModel = CommonProtocolModel()
      populateProtocolModel( protoIdModel, topoDb, af )
      # Only return the new model if something was read from TopoDb
      if protoIdModel.ipv4 or protoIdModel.ipv6:
         return protoIdModel
      else:
         return None
   else:
      populateProtocolModel( levelModel, topoDb, af )
      return levelModel

def populateIsisModel( topoModel, topoDb, path ):
   vrfModel = topoModel.vrfs.get( DEFAULT_VRF )
   proto, addrFamily, instId, igpId = splitSmashPath( path )
   assert proto == 'isis'
   if vrfModel.isis is None:
      vrfModel.isis = IgpInstanceInfoModel()
   if instId not in vrfModel.isis.instances:
      vrfModel.isis.instances[ instId ] = IsisModel()

   isisModel = vrfModel.isis.instances[ instId ]
   af = 'ipv4' if addrFamily == 'ip' else 'ipv6'
   if igpId == 1:
      isisModel.level1 = populateLevelModel( isisModel.level1, topoDb, af )
   elif igpId == 2:
      isisModel.level2 = populateLevelModel( isisModel.level2, topoDb, af )

def populateOspfModel( topoModel, topoDb, path ):
   vrfModel = topoModel.vrfs.get( DEFAULT_VRF )
   proto, addrFamily, instId, areaId = splitSmashPath( path )
   assert proto == 'ospf'

   af = 'ipv4' if addrFamily == 'ip' else 'ipv6'

   if areaId:
      if vrfModel.ospf is None:
         vrfModel.ospf = OspfInstanceInfoModel()
      if instId not in vrfModel.ospf.instances:
         vrfModel.ospf.instances[ instId ] = OspfModel()
      ospfModel = vrfModel.ospf.instances[ instId ]
      model = populateLevelModel( None, topoDb, af )
      if model:
         ospfModel.areas[ areaId ] = model

def populateIgpModel( topoModel, topoDbs ):
   for path in topoDbs:
      if 'isis' in path:
         populateIsisModel( topoModel, topoDbs[ path ], path )
      elif 'ospf' in path:
         populateOspfModel( topoModel, topoDbs[ path ], path )

def populateTopologyModel( mode, args ):
   topoModel = TopologyModel()
   vrf = DEFAULT_VRF
   vrfModel = topoModel.vrfs.get( vrf )
   if vrfModel is None:
      vrfModel = VrfModel()
      topoModel.vrfs[ vrf ] = vrfModel

   topoDbs = mountIgpTopoDb( args )
   populateIgpModel( topoModel, topoDbs )
   return topoModel

# --------------------------------------------------------------------------------
# show igp topology
# --------------------------------------------------------------------------------
class TopologyCmd( ShowCommand.ShowCliCommandClass ):
   syntax = '''show router igp topology [ ( ospf | ( isis [ LEVEL ] ) ) ]
            [ ADDR_FAMILY ] [ VRF ]'''
   data = {
      'router' : CliToken.Router.routerMatcherForConfig,
      'igp' : 'Interior gateway protocols information',
      'topology' : 'Exported topology database',
      'isis' : 'IS-IS only',
      'ospf' : 'OSPF only',
      'LEVEL' : RoutingIsisCli.levelMatcherForShow,
      'ADDR_FAMILY' : RoutingIsisCli.afMatcher,
      'VRF' : VrfExprFactory( helpdesc='VRF name',
                              inclDefaultVrf=True,
                              inclAllVrf=True ),
   }
   handler = populateTopologyModel
   cliModel = TopologyModel

BasicCli.addShowCommandClass( TopologyCmd )

def Plugin( entityManager ):
   gv.entityManager = entityManager
   gv.readerInfo = SmashLazyMount.mountInfo( 'reader' )
   gv.topoDbExport = LazyMount.mount( entityManager,
                                      'routing/topoDbExport', 'Tac::Dir', 'ri' )
   gv.shmemEm = SharedMem.entityManager( sysdbEm=entityManager )
