#!/usr/bin/env python3
# Copyright (c) 2020 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# pylint: disable=consider-using-f-string

import ArPyUtils
from ArnetModel import IpGenericAddress, IpGenericPrefix
from CliModel import Model, Int, List, Dict, Submodel, Str
from CliPlugin.TeCli import adminGroupToStr, adminGroupDecimalListToDict
from CliPlugin.RoutingIsisCli import getFlexAlgoName
from Toggles import TeToggleLib

class SharedRiskLinkGroup( Model ):
   srlgId = Int( help='Shared Risk Link Group Identifier' )
   srlgName = Str( help='Shared Risk Link Group Name', optional=True )

class FlexAlgoAttributesModel( Model ):
   administrativeGroup = Int( optional=True,
                              help='Administrative Group of the Link' )
   administrativeGroupsExtended = List( valueType=int, optional=True,
         help='Extended Administrative Groups of the Link' )
   minimumDelayMetric = Int( optional=True,
                             help='Minimum Delay Metric in microseconds' )
   defaultTeMetric = Int( optional=True,
                             help='TE default metric' )
   sharedRiskLinkGroups = List( valueType=SharedRiskLinkGroup,
                                optional=True,
                                help='List of Shared Risk Link Group' )

   def render( self ):
      if self.administrativeGroup or self.administrativeGroupsExtended:
         if TeToggleLib.toggleExtendedAdminGroupEnabled():
            adminGroup = adminGroupDecimalListToDict(
                  self.administrativeGroupsExtended )
            print( '   Administrative groups: {}'.format( adminGroupToStr(
               adminGroup ) ) )
         else:
            print( '   Administrative groups: {} (0x{:x})'.format( adminGroupToStr(
                          self.administrativeGroup ), self.administrativeGroup ) )
      if self.minimumDelayMetric:
         print( '   Minimum delay metric: %s microseconds' % (
                       str( self.minimumDelayMetric ) ) )
      if self.defaultTeMetric:
         print( '   TE default metric: %s ' % (
                       str( self.defaultTeMetric ) ) )
      if self.sharedRiskLinkGroups:
         srlgData = []
         for srlg in self.sharedRiskLinkGroups:
            if srlg.srlgName:
               srlgData.append( f'{srlg.srlgName}({str( srlg.srlgId )})' )
            else:
               srlgData.append( str( srlg.srlgId ) )
         srlgStr = ', '.join( srlgData )
         print( '   Shared risk link groups: %s' % ( srlgStr ) )

class NeighborModel( Model ):
   metric = Int( help='Link cost' )
   interfaceAddresses = List( valueType=IpGenericAddress, optional=True,
                              help='Local interface IP addresses' )
   neighborAddresses = List( valueType=IpGenericPrefix, optional=True,
                             help='Remote interface IP addresses' )
   flexAlgoAttributes = Submodel( valueType=FlexAlgoAttributesModel, optional=True,
                                  help='flexAlgo related Attributes' )

   def renderData( self, addrFamily ):
      print( '  Metric:', self.metric )
      if self.flexAlgoAttributes:
         print( '  Flex Algo attributes:' )
         self.flexAlgoAttributes.render()
      if self.interfaceAddresses:
         print( '', addrFamily, 'interface addresses:' )
         # pylint: disable=not-an-iterable
         for addr in self.interfaceAddresses:
            # pylint: enable=not-an-iterable
            print( ' ', addr )
      if self.neighborAddresses:
         print( '', addrFamily, 'neighbor addresses:' )
         # pylint: disable=not-an-iterable
         for addr in self.neighborAddresses:
            # pylint: enable=not-an-iterable
            print( ' ', addr )

class ReachabilityModel( Model ):
   metric = Int( help='Link cost' )

class RouterModel( Model ):
   flexAlgo = List( valueType=int, optional=True,
                    help='List of all supported Flex Algos' )
   p2ps = Dict( keyType=str, valueType=NeighborModel, optional=True,
                help='Neighbor information per P2P neighbor' )
   lans = Dict( keyType=str, valueType=NeighborModel, optional=True,
                help='Neighbor information per LAN neighbor' )
   reachability = Dict( keyType=IpGenericPrefix, valueType=ReachabilityModel,
                        optional=True, help='Reachability information per prefix' )

   def renderData( self, addrFamily ):
      if self.flexAlgo is not None:
         flexAlgoNameList = [ getFlexAlgoName( algoId ) for algoId in self.flexAlgo ]
         flexAlgoData = ', '.join( flexAlgoNameList )
         print( 'Flex Algo:', flexAlgoData )
      linkModels = [ self.p2ps, self.lans ]
      linkTypes = [ 'P2P', 'LAN' ]
      for ( linkType, linkModel ) in zip( linkTypes, linkModels ):
         if linkModel:
            print( 'Network type:', linkType )
            for neighborId, neighbor in linkModel.items():
               print( '  Neighbor:', neighborId )
               neighbor.renderData( addrFamily )
      if self.reachability:
         print( 'Reachability:' )
         for prefix in ArPyUtils.naturalsorted( self.reachability ):
            # pylint: disable=unsubscriptable-object
            print( '  Prefix:', prefix,
                   'Metric:', self.reachability[ prefix ].metric )
            # pylint: enable=unsubscriptable-object

class CommonProtocolModel( Model ):
   ipv4 = Dict( keyType=str, valueType=RouterModel, optional=True,
                help='IPv4 topology information per router ID' )
   ipv6 = Dict( keyType=str, valueType=RouterModel, optional=True,
                help='IPv6 topology information per router ID' )

   def renderData( self, protocol, instanceId, identifier ):
      for routers, af in zip( [ self.ipv4, self.ipv6 ], [ 'IPv4', 'IPv6' ] ):
         if routers:
            print( '\nSource:', protocol, 'Instance ID', instanceId,
                   identifier, af, 'database' )
            for routerId in ArPyUtils.naturalsorted( routers ):
               print( 'Router-ID:', routerId )
               routers[ routerId ].renderData( af )

class IsisModel( Model ):
   level1 = Submodel( valueType=CommonProtocolModel, optional=True,
                      help='IS-IS Level-1 topology information' )
   level2 = Submodel( valueType=CommonProtocolModel, optional=True,
                      help='IS-IS Level-2 topology information' )

   def renderData( self, instanceId ):
      if self.level1 is not None:
         self.level1.renderData( 'IS-IS', instanceId, 'Level-1' )

      if self.level2 is not None:
         self.level2.renderData( 'IS-IS', instanceId, 'Level-2' )

class OspfModel( Model ):
   areas = Dict( keyType=str, valueType=CommonProtocolModel, optional=True,
                    help='OSPF topology information per area' )

   def renderData( self, instanceId ):
      for area in ArPyUtils.naturalsorted( self.areas ):
         identifier = 'Area {}'.format( area )
         ospfModel = self.areas[ area ]
         ospfModel.renderData( 'OSPF', instanceId, identifier )

class IgpInstanceInfoModel( Model ):
   instances = Dict( keyType=int, valueType=IsisModel,
                     help="A mapping of the IGP instance ID to the IGP "
                     "database" )

   def renderData( self ):
      if self.instances is not None:
         for instanceId, isisModel in sorted( self.instances.items() ):
            isisModel.renderData( instanceId )

class OspfInstanceInfoModel( Model ):
   instances = Dict( keyType=int, valueType=OspfModel, optional=True,
                     help="A mapping of OSPF instance ID to OSPF database" )

   def renderData( self ):
      if self.instances is not None:
         for instanceId, ospfModel in sorted( self.instances.items() ):
            ospfModel.renderData( instanceId )

class VrfModel( Model ):
   isis = Submodel( valueType=IgpInstanceInfoModel, optional=True,
                    help='IGP topology information for IS-IS' )
   ospf = Submodel( valueType=OspfInstanceInfoModel, optional=True,
                    help='IGP topology information for Ospf' )

   def render( self ):
      if self.isis is not None:
         self.isis.renderData()
      if self.ospf is not None:
         self.ospf.renderData()

class TopologyModel( Model ):
   __revision__ = 2
   vrfs = Dict( keyType=str, valueType=VrfModel,
                help='IGP topology database per VRF' )

   def render( self ):
      for vrf in ArPyUtils.naturalsorted( self.vrfs ):
         print( 'VRF:', vrf )
         self.vrfs[ vrf ].render()

   def degrade( self, dictRepr, revision ):
      # Revision 1 only supports IS-IS output for a single instance ( ID 0 )
      if revision == 1:
         for topoVrfModel in dictRepr[ 'vrfs' ].values():
            for kw, isisInstanceInfoModel in topoVrfModel.items():
               if kw == 'ospf':
                  continue
               isisDict = isisInstanceInfoModel.get( 'instances', {} ).get( '0', {} )
               isisL1 = isisDict.get( 'level1', {} )
               isisL2 = isisDict.get( 'level2', {} )
               topoVrfModel[ 'isis' ] = {}
               if isisL1:
                  topoVrfModel[ 'isis' ][ 'level1' ] = isisL1
               if isisL2:
                  topoVrfModel[ 'isis' ][ 'level2' ] = isisL2
      return dictRepr
